/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.frauddetector.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details of the impact of aggregated variables on the prediction score.
 * </p>
 * <p>
 * Account Takeover Insights (ATI) model uses the login data you provide to continuously calculate a set of variables
 * (aggregated variables) based on historical events. For example, the model might calculate the number of times an user
 * has logged in using the same IP address. In this case, event variables used to derive the aggregated variables are
 * <code>IP address</code> and <code>user</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AggregatedVariablesImpactExplanation implements SdkPojo, Serializable,
        ToCopyableBuilder<AggregatedVariablesImpactExplanation.Builder, AggregatedVariablesImpactExplanation> {
    private static final SdkField<List<String>> EVENT_VARIABLE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("eventVariableNames")
            .getter(getter(AggregatedVariablesImpactExplanation::eventVariableNames))
            .setter(setter(Builder::eventVariableNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventVariableNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RELATIVE_IMPACT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("relativeImpact").getter(getter(AggregatedVariablesImpactExplanation::relativeImpact))
            .setter(setter(Builder::relativeImpact))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relativeImpact").build()).build();

    private static final SdkField<Float> LOG_ODDS_IMPACT_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("logOddsImpact").getter(getter(AggregatedVariablesImpactExplanation::logOddsImpact))
            .setter(setter(Builder::logOddsImpact))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logOddsImpact").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_VARIABLE_NAMES_FIELD,
            RELATIVE_IMPACT_FIELD, LOG_ODDS_IMPACT_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> eventVariableNames;

    private final String relativeImpact;

    private final Float logOddsImpact;

    private AggregatedVariablesImpactExplanation(BuilderImpl builder) {
        this.eventVariableNames = builder.eventVariableNames;
        this.relativeImpact = builder.relativeImpact;
        this.logOddsImpact = builder.logOddsImpact;
    }

    /**
     * For responses, this returns true if the service returned a value for the EventVariableNames property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEventVariableNames() {
        return eventVariableNames != null && !(eventVariableNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of all the event variables that were used to derive the aggregated variables.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventVariableNames} method.
     * </p>
     * 
     * @return The names of all the event variables that were used to derive the aggregated variables.
     */
    public final List<String> eventVariableNames() {
        return eventVariableNames;
    }

    /**
     * <p>
     * The relative impact of the aggregated variables in terms of magnitude on the prediction scores.
     * </p>
     * 
     * @return The relative impact of the aggregated variables in terms of magnitude on the prediction scores.
     */
    public final String relativeImpact() {
        return relativeImpact;
    }

    /**
     * <p>
     * The raw, uninterpreted value represented as log-odds of the fraud. These values are usually between -10 to +10,
     * but range from -infinity to +infinity.
     * </p>
     * <ul>
     * <li>
     * <p>
     * A positive value indicates that the variables drove the risk score up.
     * </p>
     * </li>
     * <li>
     * <p>
     * A negative value indicates that the variables drove the risk score down.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The raw, uninterpreted value represented as log-odds of the fraud. These values are usually between -10
     *         to +10, but range from -infinity to +infinity.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A positive value indicates that the variables drove the risk score up.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A negative value indicates that the variables drove the risk score down.
     *         </p>
     *         </li>
     */
    public final Float logOddsImpact() {
        return logOddsImpact;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasEventVariableNames() ? eventVariableNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(relativeImpact());
        hashCode = 31 * hashCode + Objects.hashCode(logOddsImpact());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AggregatedVariablesImpactExplanation)) {
            return false;
        }
        AggregatedVariablesImpactExplanation other = (AggregatedVariablesImpactExplanation) obj;
        return hasEventVariableNames() == other.hasEventVariableNames()
                && Objects.equals(eventVariableNames(), other.eventVariableNames())
                && Objects.equals(relativeImpact(), other.relativeImpact())
                && Objects.equals(logOddsImpact(), other.logOddsImpact());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AggregatedVariablesImpactExplanation")
                .add("EventVariableNames", hasEventVariableNames() ? eventVariableNames() : null)
                .add("RelativeImpact", relativeImpact()).add("LogOddsImpact", logOddsImpact()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "eventVariableNames":
            return Optional.ofNullable(clazz.cast(eventVariableNames()));
        case "relativeImpact":
            return Optional.ofNullable(clazz.cast(relativeImpact()));
        case "logOddsImpact":
            return Optional.ofNullable(clazz.cast(logOddsImpact()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AggregatedVariablesImpactExplanation, T> g) {
        return obj -> g.apply((AggregatedVariablesImpactExplanation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AggregatedVariablesImpactExplanation> {
        /**
         * <p>
         * The names of all the event variables that were used to derive the aggregated variables.
         * </p>
         * 
         * @param eventVariableNames
         *        The names of all the event variables that were used to derive the aggregated variables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventVariableNames(Collection<String> eventVariableNames);

        /**
         * <p>
         * The names of all the event variables that were used to derive the aggregated variables.
         * </p>
         * 
         * @param eventVariableNames
         *        The names of all the event variables that were used to derive the aggregated variables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventVariableNames(String... eventVariableNames);

        /**
         * <p>
         * The relative impact of the aggregated variables in terms of magnitude on the prediction scores.
         * </p>
         * 
         * @param relativeImpact
         *        The relative impact of the aggregated variables in terms of magnitude on the prediction scores.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relativeImpact(String relativeImpact);

        /**
         * <p>
         * The raw, uninterpreted value represented as log-odds of the fraud. These values are usually between -10 to
         * +10, but range from -infinity to +infinity.
         * </p>
         * <ul>
         * <li>
         * <p>
         * A positive value indicates that the variables drove the risk score up.
         * </p>
         * </li>
         * <li>
         * <p>
         * A negative value indicates that the variables drove the risk score down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param logOddsImpact
         *        The raw, uninterpreted value represented as log-odds of the fraud. These values are usually between
         *        -10 to +10, but range from -infinity to +infinity.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        A positive value indicates that the variables drove the risk score up.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A negative value indicates that the variables drove the risk score down.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logOddsImpact(Float logOddsImpact);
    }

    static final class BuilderImpl implements Builder {
        private List<String> eventVariableNames = DefaultSdkAutoConstructList.getInstance();

        private String relativeImpact;

        private Float logOddsImpact;

        private BuilderImpl() {
        }

        private BuilderImpl(AggregatedVariablesImpactExplanation model) {
            eventVariableNames(model.eventVariableNames);
            relativeImpact(model.relativeImpact);
            logOddsImpact(model.logOddsImpact);
        }

        public final Collection<String> getEventVariableNames() {
            if (eventVariableNames instanceof SdkAutoConstructList) {
                return null;
            }
            return eventVariableNames;
        }

        public final void setEventVariableNames(Collection<String> eventVariableNames) {
            this.eventVariableNames = ListOfStringsCopier.copy(eventVariableNames);
        }

        @Override
        public final Builder eventVariableNames(Collection<String> eventVariableNames) {
            this.eventVariableNames = ListOfStringsCopier.copy(eventVariableNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventVariableNames(String... eventVariableNames) {
            eventVariableNames(Arrays.asList(eventVariableNames));
            return this;
        }

        public final String getRelativeImpact() {
            return relativeImpact;
        }

        public final void setRelativeImpact(String relativeImpact) {
            this.relativeImpact = relativeImpact;
        }

        @Override
        public final Builder relativeImpact(String relativeImpact) {
            this.relativeImpact = relativeImpact;
            return this;
        }

        public final Float getLogOddsImpact() {
            return logOddsImpact;
        }

        public final void setLogOddsImpact(Float logOddsImpact) {
            this.logOddsImpact = logOddsImpact;
        }

        @Override
        public final Builder logOddsImpact(Float logOddsImpact) {
            this.logOddsImpact = logOddsImpact;
            return this;
        }

        @Override
        public AggregatedVariablesImpactExplanation build() {
            return new AggregatedVariablesImpactExplanation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
