/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreatePredictorRequest extends ForecastRequest implements
        ToCopyableBuilder<CreatePredictorRequest.Builder, CreatePredictorRequest> {
    private static final SdkField<String> PREDICTOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PredictorName").getter(getter(CreatePredictorRequest::predictorName))
            .setter(setter(Builder::predictorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PredictorName").build()).build();

    private static final SdkField<String> ALGORITHM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlgorithmArn").getter(getter(CreatePredictorRequest::algorithmArn))
            .setter(setter(Builder::algorithmArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlgorithmArn").build()).build();

    private static final SdkField<Integer> FORECAST_HORIZON_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ForecastHorizon").getter(getter(CreatePredictorRequest::forecastHorizon))
            .setter(setter(Builder::forecastHorizon))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastHorizon").build()).build();

    private static final SdkField<List<String>> FORECAST_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ForecastTypes")
            .getter(getter(CreatePredictorRequest::forecastTypes))
            .setter(setter(Builder::forecastTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> PERFORM_AUTO_ML_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PerformAutoML").getter(getter(CreatePredictorRequest::performAutoML))
            .setter(setter(Builder::performAutoML))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerformAutoML").build()).build();

    private static final SdkField<String> AUTO_ML_OVERRIDE_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AutoMLOverrideStrategy").getter(getter(CreatePredictorRequest::autoMLOverrideStrategyAsString))
            .setter(setter(Builder::autoMLOverrideStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoMLOverrideStrategy").build())
            .build();

    private static final SdkField<Boolean> PERFORM_HPO_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PerformHPO").getter(getter(CreatePredictorRequest::performHPO)).setter(setter(Builder::performHPO))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerformHPO").build()).build();

    private static final SdkField<Map<String, String>> TRAINING_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("TrainingParameters")
            .getter(getter(CreatePredictorRequest::trainingParameters))
            .setter(setter(Builder::trainingParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<EvaluationParameters> EVALUATION_PARAMETERS_FIELD = SdkField
            .<EvaluationParameters> builder(MarshallingType.SDK_POJO).memberName("EvaluationParameters")
            .getter(getter(CreatePredictorRequest::evaluationParameters)).setter(setter(Builder::evaluationParameters))
            .constructor(EvaluationParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationParameters").build())
            .build();

    private static final SdkField<HyperParameterTuningJobConfig> HPO_CONFIG_FIELD = SdkField
            .<HyperParameterTuningJobConfig> builder(MarshallingType.SDK_POJO).memberName("HPOConfig")
            .getter(getter(CreatePredictorRequest::hpoConfig)).setter(setter(Builder::hpoConfig))
            .constructor(HyperParameterTuningJobConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HPOConfig").build()).build();

    private static final SdkField<InputDataConfig> INPUT_DATA_CONFIG_FIELD = SdkField
            .<InputDataConfig> builder(MarshallingType.SDK_POJO).memberName("InputDataConfig")
            .getter(getter(CreatePredictorRequest::inputDataConfig)).setter(setter(Builder::inputDataConfig))
            .constructor(InputDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputDataConfig").build()).build();

    private static final SdkField<FeaturizationConfig> FEATURIZATION_CONFIG_FIELD = SdkField
            .<FeaturizationConfig> builder(MarshallingType.SDK_POJO).memberName("FeaturizationConfig")
            .getter(getter(CreatePredictorRequest::featurizationConfig)).setter(setter(Builder::featurizationConfig))
            .constructor(FeaturizationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeaturizationConfig").build())
            .build();

    private static final SdkField<EncryptionConfig> ENCRYPTION_CONFIG_FIELD = SdkField
            .<EncryptionConfig> builder(MarshallingType.SDK_POJO).memberName("EncryptionConfig")
            .getter(getter(CreatePredictorRequest::encryptionConfig)).setter(setter(Builder::encryptionConfig))
            .constructor(EncryptionConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfig").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreatePredictorRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OPTIMIZATION_METRIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OptimizationMetric").getter(getter(CreatePredictorRequest::optimizationMetricAsString))
            .setter(setter(Builder::optimizationMetric))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptimizationMetric").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PREDICTOR_NAME_FIELD,
            ALGORITHM_ARN_FIELD, FORECAST_HORIZON_FIELD, FORECAST_TYPES_FIELD, PERFORM_AUTO_ML_FIELD,
            AUTO_ML_OVERRIDE_STRATEGY_FIELD, PERFORM_HPO_FIELD, TRAINING_PARAMETERS_FIELD, EVALUATION_PARAMETERS_FIELD,
            HPO_CONFIG_FIELD, INPUT_DATA_CONFIG_FIELD, FEATURIZATION_CONFIG_FIELD, ENCRYPTION_CONFIG_FIELD, TAGS_FIELD,
            OPTIMIZATION_METRIC_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String predictorName;

    private final String algorithmArn;

    private final Integer forecastHorizon;

    private final List<String> forecastTypes;

    private final Boolean performAutoML;

    private final String autoMLOverrideStrategy;

    private final Boolean performHPO;

    private final Map<String, String> trainingParameters;

    private final EvaluationParameters evaluationParameters;

    private final HyperParameterTuningJobConfig hpoConfig;

    private final InputDataConfig inputDataConfig;

    private final FeaturizationConfig featurizationConfig;

    private final EncryptionConfig encryptionConfig;

    private final List<Tag> tags;

    private final String optimizationMetric;

    private CreatePredictorRequest(BuilderImpl builder) {
        super(builder);
        this.predictorName = builder.predictorName;
        this.algorithmArn = builder.algorithmArn;
        this.forecastHorizon = builder.forecastHorizon;
        this.forecastTypes = builder.forecastTypes;
        this.performAutoML = builder.performAutoML;
        this.autoMLOverrideStrategy = builder.autoMLOverrideStrategy;
        this.performHPO = builder.performHPO;
        this.trainingParameters = builder.trainingParameters;
        this.evaluationParameters = builder.evaluationParameters;
        this.hpoConfig = builder.hpoConfig;
        this.inputDataConfig = builder.inputDataConfig;
        this.featurizationConfig = builder.featurizationConfig;
        this.encryptionConfig = builder.encryptionConfig;
        this.tags = builder.tags;
        this.optimizationMetric = builder.optimizationMetric;
    }

    /**
     * <p>
     * A name for the predictor.
     * </p>
     * 
     * @return A name for the predictor.
     */
    public final String predictorName() {
        return predictorName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the algorithm to use for model training. Required if <code>PerformAutoML</code>
     * is not set to <code>true</code>.
     * </p>
     * <p class="title">
     * <b>Supported algorithms:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:forecast:::algorithm/ARIMA</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>arn:aws:forecast:::algorithm/CNN-QR</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>arn:aws:forecast:::algorithm/Deep_AR_Plus</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>arn:aws:forecast:::algorithm/ETS</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>arn:aws:forecast:::algorithm/NPTS</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>arn:aws:forecast:::algorithm/Prophet</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Resource Name (ARN) of the algorithm to use for model training. Required if
     *         <code>PerformAutoML</code> is not set to <code>true</code>.</p>
     *         <p class="title">
     *         <b>Supported algorithms:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:forecast:::algorithm/ARIMA</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>arn:aws:forecast:::algorithm/CNN-QR</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>arn:aws:forecast:::algorithm/Deep_AR_Plus</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>arn:aws:forecast:::algorithm/ETS</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>arn:aws:forecast:::algorithm/NPTS</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>arn:aws:forecast:::algorithm/Prophet</code>
     *         </p>
     *         </li>
     */
    public final String algorithmArn() {
        return algorithmArn;
    }

    /**
     * <p>
     * Specifies the number of time-steps that the model is trained to predict. The forecast horizon is also called the
     * prediction length.
     * </p>
     * <p>
     * For example, if you configure a dataset for daily data collection (using the <code>DataFrequency</code> parameter
     * of the <a>CreateDataset</a> operation) and set the forecast horizon to 10, the model returns predictions for 10
     * days.
     * </p>
     * <p>
     * The maximum forecast horizon is the lesser of 500 time-steps or 1/3 of the TARGET_TIME_SERIES dataset length.
     * </p>
     * 
     * @return Specifies the number of time-steps that the model is trained to predict. The forecast horizon is also
     *         called the prediction length.</p>
     *         <p>
     *         For example, if you configure a dataset for daily data collection (using the <code>DataFrequency</code>
     *         parameter of the <a>CreateDataset</a> operation) and set the forecast horizon to 10, the model returns
     *         predictions for 10 days.
     *         </p>
     *         <p>
     *         The maximum forecast horizon is the lesser of 500 time-steps or 1/3 of the TARGET_TIME_SERIES dataset
     *         length.
     */
    public final Integer forecastHorizon() {
        return forecastHorizon;
    }

    /**
     * For responses, this returns true if the service returned a value for the ForecastTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasForecastTypes() {
        return forecastTypes != null && !(forecastTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the forecast types used to train a predictor. You can specify up to five forecast types. Forecast types
     * can be quantiles from 0.01 to 0.99, by increments of 0.01 or higher. You can also specify the mean forecast with
     * <code>mean</code>.
     * </p>
     * <p>
     * The default value is <code>["0.10", "0.50", "0.9"]</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasForecastTypes} method.
     * </p>
     * 
     * @return Specifies the forecast types used to train a predictor. You can specify up to five forecast types.
     *         Forecast types can be quantiles from 0.01 to 0.99, by increments of 0.01 or higher. You can also specify
     *         the mean forecast with <code>mean</code>. </p>
     *         <p>
     *         The default value is <code>["0.10", "0.50", "0.9"]</code>.
     */
    public final List<String> forecastTypes() {
        return forecastTypes;
    }

    /**
     * <p>
     * Whether to perform AutoML. When Amazon Forecast performs AutoML, it evaluates the algorithms it provides and
     * chooses the best algorithm and configuration for your training dataset.
     * </p>
     * <p>
     * The default value is <code>false</code>. In this case, you are required to specify an algorithm.
     * </p>
     * <p>
     * Set <code>PerformAutoML</code> to <code>true</code> to have Amazon Forecast perform AutoML. This is a good option
     * if you aren't sure which algorithm is suitable for your training data. In this case, <code>PerformHPO</code> must
     * be false.
     * </p>
     * 
     * @return Whether to perform AutoML. When Amazon Forecast performs AutoML, it evaluates the algorithms it provides
     *         and chooses the best algorithm and configuration for your training dataset.</p>
     *         <p>
     *         The default value is <code>false</code>. In this case, you are required to specify an algorithm.
     *         </p>
     *         <p>
     *         Set <code>PerformAutoML</code> to <code>true</code> to have Amazon Forecast perform AutoML. This is a
     *         good option if you aren't sure which algorithm is suitable for your training data. In this case,
     *         <code>PerformHPO</code> must be false.
     */
    public final Boolean performAutoML() {
        return performAutoML;
    }

    /**
     * <note>
     * <p>
     * The <code>LatencyOptimized</code> AutoML override strategy is only available in private beta. Contact Amazon Web
     * Services Support or your account manager to learn more about access privileges.
     * </p>
     * </note>
     * <p>
     * Used to overide the default AutoML strategy, which is to optimize predictor accuracy. To apply an AutoML strategy
     * that minimizes training time, use <code>LatencyOptimized</code>.
     * </p>
     * <p>
     * This parameter is only valid for predictors trained using AutoML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #autoMLOverrideStrategy} will return {@link AutoMLOverrideStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #autoMLOverrideStrategyAsString}.
     * </p>
     * 
     * @return <p>
     *         The <code>LatencyOptimized</code> AutoML override strategy is only available in private beta. Contact
     *         Amazon Web Services Support or your account manager to learn more about access privileges.
     *         </p>
     *         </note>
     *         <p>
     *         Used to overide the default AutoML strategy, which is to optimize predictor accuracy. To apply an AutoML
     *         strategy that minimizes training time, use <code>LatencyOptimized</code>.
     *         </p>
     *         <p>
     *         This parameter is only valid for predictors trained using AutoML.
     * @see AutoMLOverrideStrategy
     */
    public final AutoMLOverrideStrategy autoMLOverrideStrategy() {
        return AutoMLOverrideStrategy.fromValue(autoMLOverrideStrategy);
    }

    /**
     * <note>
     * <p>
     * The <code>LatencyOptimized</code> AutoML override strategy is only available in private beta. Contact Amazon Web
     * Services Support or your account manager to learn more about access privileges.
     * </p>
     * </note>
     * <p>
     * Used to overide the default AutoML strategy, which is to optimize predictor accuracy. To apply an AutoML strategy
     * that minimizes training time, use <code>LatencyOptimized</code>.
     * </p>
     * <p>
     * This parameter is only valid for predictors trained using AutoML.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #autoMLOverrideStrategy} will return {@link AutoMLOverrideStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #autoMLOverrideStrategyAsString}.
     * </p>
     * 
     * @return <p>
     *         The <code>LatencyOptimized</code> AutoML override strategy is only available in private beta. Contact
     *         Amazon Web Services Support or your account manager to learn more about access privileges.
     *         </p>
     *         </note>
     *         <p>
     *         Used to overide the default AutoML strategy, which is to optimize predictor accuracy. To apply an AutoML
     *         strategy that minimizes training time, use <code>LatencyOptimized</code>.
     *         </p>
     *         <p>
     *         This parameter is only valid for predictors trained using AutoML.
     * @see AutoMLOverrideStrategy
     */
    public final String autoMLOverrideStrategyAsString() {
        return autoMLOverrideStrategy;
    }

    /**
     * <p>
     * Whether to perform hyperparameter optimization (HPO). HPO finds optimal hyperparameter values for your training
     * data. The process of performing HPO is known as running a hyperparameter tuning job.
     * </p>
     * <p>
     * The default value is <code>false</code>. In this case, Amazon Forecast uses default hyperparameter values from
     * the chosen algorithm.
     * </p>
     * <p>
     * To override the default values, set <code>PerformHPO</code> to <code>true</code> and, optionally, supply the
     * <a>HyperParameterTuningJobConfig</a> object. The tuning job specifies a metric to optimize, which hyperparameters
     * participate in tuning, and the valid range for each tunable hyperparameter. In this case, you are required to
     * specify an algorithm and <code>PerformAutoML</code> must be false.
     * </p>
     * <p>
     * The following algorithms support HPO:
     * </p>
     * <ul>
     * <li>
     * <p>
     * DeepAR+
     * </p>
     * </li>
     * <li>
     * <p>
     * CNN-QR
     * </p>
     * </li>
     * </ul>
     * 
     * @return Whether to perform hyperparameter optimization (HPO). HPO finds optimal hyperparameter values for your
     *         training data. The process of performing HPO is known as running a hyperparameter tuning job.</p>
     *         <p>
     *         The default value is <code>false</code>. In this case, Amazon Forecast uses default hyperparameter values
     *         from the chosen algorithm.
     *         </p>
     *         <p>
     *         To override the default values, set <code>PerformHPO</code> to <code>true</code> and, optionally, supply
     *         the <a>HyperParameterTuningJobConfig</a> object. The tuning job specifies a metric to optimize, which
     *         hyperparameters participate in tuning, and the valid range for each tunable hyperparameter. In this case,
     *         you are required to specify an algorithm and <code>PerformAutoML</code> must be false.
     *         </p>
     *         <p>
     *         The following algorithms support HPO:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         DeepAR+
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         CNN-QR
     *         </p>
     *         </li>
     */
    public final Boolean performHPO() {
        return performHPO;
    }

    /**
     * For responses, this returns true if the service returned a value for the TrainingParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTrainingParameters() {
        return trainingParameters != null && !(trainingParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The hyperparameters to override for model training. The hyperparameters that you can override are listed in the
     * individual algorithms. For the list of supported algorithms, see <a>aws-forecast-choosing-recipes</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTrainingParameters} method.
     * </p>
     * 
     * @return The hyperparameters to override for model training. The hyperparameters that you can override are listed
     *         in the individual algorithms. For the list of supported algorithms, see
     *         <a>aws-forecast-choosing-recipes</a>.
     */
    public final Map<String, String> trainingParameters() {
        return trainingParameters;
    }

    /**
     * <p>
     * Used to override the default evaluation parameters of the specified algorithm. Amazon Forecast evaluates a
     * predictor by splitting a dataset into training data and testing data. The evaluation parameters define how to
     * perform the split and the number of iterations.
     * </p>
     * 
     * @return Used to override the default evaluation parameters of the specified algorithm. Amazon Forecast evaluates
     *         a predictor by splitting a dataset into training data and testing data. The evaluation parameters define
     *         how to perform the split and the number of iterations.
     */
    public final EvaluationParameters evaluationParameters() {
        return evaluationParameters;
    }

    /**
     * <p>
     * Provides hyperparameter override values for the algorithm. If you don't provide this parameter, Amazon Forecast
     * uses default values. The individual algorithms specify which hyperparameters support hyperparameter optimization
     * (HPO). For more information, see <a>aws-forecast-choosing-recipes</a>.
     * </p>
     * <p>
     * If you included the <code>HPOConfig</code> object, you must set <code>PerformHPO</code> to true.
     * </p>
     * 
     * @return Provides hyperparameter override values for the algorithm. If you don't provide this parameter, Amazon
     *         Forecast uses default values. The individual algorithms specify which hyperparameters support
     *         hyperparameter optimization (HPO). For more information, see <a>aws-forecast-choosing-recipes</a>.</p>
     *         <p>
     *         If you included the <code>HPOConfig</code> object, you must set <code>PerformHPO</code> to true.
     */
    public final HyperParameterTuningJobConfig hpoConfig() {
        return hpoConfig;
    }

    /**
     * <p>
     * Describes the dataset group that contains the data to use to train the predictor.
     * </p>
     * 
     * @return Describes the dataset group that contains the data to use to train the predictor.
     */
    public final InputDataConfig inputDataConfig() {
        return inputDataConfig;
    }

    /**
     * <p>
     * The featurization configuration.
     * </p>
     * 
     * @return The featurization configuration.
     */
    public final FeaturizationConfig featurizationConfig() {
        return featurizationConfig;
    }

    /**
     * <p>
     * An Key Management Service (KMS) key and the Identity and Access Management (IAM) role that Amazon Forecast can
     * assume to access the key.
     * </p>
     * 
     * @return An Key Management Service (KMS) key and the Identity and Access Management (IAM) role that Amazon
     *         Forecast can assume to access the key.
     */
    public final EncryptionConfig encryptionConfig() {
        return encryptionConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The optional metadata that you apply to the predictor to help you categorize and organize them. Each tag consists
     * of a key and an optional value, both of which you define.
     * </p>
     * <p>
     * The following basic restrictions apply to tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of tags per resource - 50.
     * </p>
     * </li>
     * <li>
     * <p>
     * For each resource, each tag key must be unique, and each tag key can have only one value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length - 128 Unicode characters in UTF-8.
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length - 256 Unicode characters in UTF-8.
     * </p>
     * </li>
     * <li>
     * <p>
     * If your tagging schema is used across multiple services and resources, remember that other services may have
     * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable
     * in UTF-8, and the following characters: + - = . _ : / @.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tag keys and values are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix for
     * keys as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys with this prefix. Values
     * can have this prefix. If a tag value has <code>aws</code> as its prefix but the key does not, then Forecast
     * considers it to be a user tag and will count against the limit of 50 tags. Tags with only the key prefix of
     * <code>aws</code> do not count against your tags per resource limit.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The optional metadata that you apply to the predictor to help you categorize and organize them. Each tag
     *         consists of a key and an optional value, both of which you define.</p>
     *         <p>
     *         The following basic restrictions apply to tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Maximum number of tags per resource - 50.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For each resource, each tag key must be unique, and each tag key can have only one value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum key length - 128 Unicode characters in UTF-8.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum value length - 256 Unicode characters in UTF-8.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If your tagging schema is used across multiple services and resources, remember that other services may
     *         have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
     *         representable in UTF-8, and the following characters: + - = . _ : / @.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tag keys and values are case sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
     *         prefix for keys as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys with
     *         this prefix. Values can have this prefix. If a tag value has <code>aws</code> as its prefix but the key
     *         does not, then Forecast considers it to be a user tag and will count against the limit of 50 tags. Tags
     *         with only the key prefix of <code>aws</code> do not count against your tags per resource limit.
     *         </p>
     *         </li>
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The accuracy metric used to optimize the predictor.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #optimizationMetric} will return {@link OptimizationMetric#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #optimizationMetricAsString}.
     * </p>
     * 
     * @return The accuracy metric used to optimize the predictor.
     * @see OptimizationMetric
     */
    public final OptimizationMetric optimizationMetric() {
        return OptimizationMetric.fromValue(optimizationMetric);
    }

    /**
     * <p>
     * The accuracy metric used to optimize the predictor.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #optimizationMetric} will return {@link OptimizationMetric#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #optimizationMetricAsString}.
     * </p>
     * 
     * @return The accuracy metric used to optimize the predictor.
     * @see OptimizationMetric
     */
    public final String optimizationMetricAsString() {
        return optimizationMetric;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(predictorName());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmArn());
        hashCode = 31 * hashCode + Objects.hashCode(forecastHorizon());
        hashCode = 31 * hashCode + Objects.hashCode(hasForecastTypes() ? forecastTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(performAutoML());
        hashCode = 31 * hashCode + Objects.hashCode(autoMLOverrideStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(performHPO());
        hashCode = 31 * hashCode + Objects.hashCode(hasTrainingParameters() ? trainingParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(evaluationParameters());
        hashCode = 31 * hashCode + Objects.hashCode(hpoConfig());
        hashCode = 31 * hashCode + Objects.hashCode(inputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(featurizationConfig());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(optimizationMetricAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreatePredictorRequest)) {
            return false;
        }
        CreatePredictorRequest other = (CreatePredictorRequest) obj;
        return Objects.equals(predictorName(), other.predictorName()) && Objects.equals(algorithmArn(), other.algorithmArn())
                && Objects.equals(forecastHorizon(), other.forecastHorizon()) && hasForecastTypes() == other.hasForecastTypes()
                && Objects.equals(forecastTypes(), other.forecastTypes())
                && Objects.equals(performAutoML(), other.performAutoML())
                && Objects.equals(autoMLOverrideStrategyAsString(), other.autoMLOverrideStrategyAsString())
                && Objects.equals(performHPO(), other.performHPO()) && hasTrainingParameters() == other.hasTrainingParameters()
                && Objects.equals(trainingParameters(), other.trainingParameters())
                && Objects.equals(evaluationParameters(), other.evaluationParameters())
                && Objects.equals(hpoConfig(), other.hpoConfig()) && Objects.equals(inputDataConfig(), other.inputDataConfig())
                && Objects.equals(featurizationConfig(), other.featurizationConfig())
                && Objects.equals(encryptionConfig(), other.encryptionConfig()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(optimizationMetricAsString(), other.optimizationMetricAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreatePredictorRequest").add("PredictorName", predictorName())
                .add("AlgorithmArn", algorithmArn()).add("ForecastHorizon", forecastHorizon())
                .add("ForecastTypes", hasForecastTypes() ? forecastTypes() : null).add("PerformAutoML", performAutoML())
                .add("AutoMLOverrideStrategy", autoMLOverrideStrategyAsString()).add("PerformHPO", performHPO())
                .add("TrainingParameters", hasTrainingParameters() ? trainingParameters() : null)
                .add("EvaluationParameters", evaluationParameters()).add("HPOConfig", hpoConfig())
                .add("InputDataConfig", inputDataConfig()).add("FeaturizationConfig", featurizationConfig())
                .add("EncryptionConfig", encryptionConfig()).add("Tags", hasTags() ? tags() : null)
                .add("OptimizationMetric", optimizationMetricAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PredictorName":
            return Optional.ofNullable(clazz.cast(predictorName()));
        case "AlgorithmArn":
            return Optional.ofNullable(clazz.cast(algorithmArn()));
        case "ForecastHorizon":
            return Optional.ofNullable(clazz.cast(forecastHorizon()));
        case "ForecastTypes":
            return Optional.ofNullable(clazz.cast(forecastTypes()));
        case "PerformAutoML":
            return Optional.ofNullable(clazz.cast(performAutoML()));
        case "AutoMLOverrideStrategy":
            return Optional.ofNullable(clazz.cast(autoMLOverrideStrategyAsString()));
        case "PerformHPO":
            return Optional.ofNullable(clazz.cast(performHPO()));
        case "TrainingParameters":
            return Optional.ofNullable(clazz.cast(trainingParameters()));
        case "EvaluationParameters":
            return Optional.ofNullable(clazz.cast(evaluationParameters()));
        case "HPOConfig":
            return Optional.ofNullable(clazz.cast(hpoConfig()));
        case "InputDataConfig":
            return Optional.ofNullable(clazz.cast(inputDataConfig()));
        case "FeaturizationConfig":
            return Optional.ofNullable(clazz.cast(featurizationConfig()));
        case "EncryptionConfig":
            return Optional.ofNullable(clazz.cast(encryptionConfig()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "OptimizationMetric":
            return Optional.ofNullable(clazz.cast(optimizationMetricAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("PredictorName", PREDICTOR_NAME_FIELD);
        map.put("AlgorithmArn", ALGORITHM_ARN_FIELD);
        map.put("ForecastHorizon", FORECAST_HORIZON_FIELD);
        map.put("ForecastTypes", FORECAST_TYPES_FIELD);
        map.put("PerformAutoML", PERFORM_AUTO_ML_FIELD);
        map.put("AutoMLOverrideStrategy", AUTO_ML_OVERRIDE_STRATEGY_FIELD);
        map.put("PerformHPO", PERFORM_HPO_FIELD);
        map.put("TrainingParameters", TRAINING_PARAMETERS_FIELD);
        map.put("EvaluationParameters", EVALUATION_PARAMETERS_FIELD);
        map.put("HPOConfig", HPO_CONFIG_FIELD);
        map.put("InputDataConfig", INPUT_DATA_CONFIG_FIELD);
        map.put("FeaturizationConfig", FEATURIZATION_CONFIG_FIELD);
        map.put("EncryptionConfig", ENCRYPTION_CONFIG_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("OptimizationMetric", OPTIMIZATION_METRIC_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreatePredictorRequest, T> g) {
        return obj -> g.apply((CreatePredictorRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends ForecastRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreatePredictorRequest> {
        /**
         * <p>
         * A name for the predictor.
         * </p>
         * 
         * @param predictorName
         *        A name for the predictor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predictorName(String predictorName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the algorithm to use for model training. Required if
         * <code>PerformAutoML</code> is not set to <code>true</code>.
         * </p>
         * <p class="title">
         * <b>Supported algorithms:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:forecast:::algorithm/ARIMA</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>arn:aws:forecast:::algorithm/CNN-QR</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>arn:aws:forecast:::algorithm/Deep_AR_Plus</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>arn:aws:forecast:::algorithm/ETS</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>arn:aws:forecast:::algorithm/NPTS</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>arn:aws:forecast:::algorithm/Prophet</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param algorithmArn
         *        The Amazon Resource Name (ARN) of the algorithm to use for model training. Required if
         *        <code>PerformAutoML</code> is not set to <code>true</code>.</p>
         *        <p class="title">
         *        <b>Supported algorithms:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:forecast:::algorithm/ARIMA</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>arn:aws:forecast:::algorithm/CNN-QR</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>arn:aws:forecast:::algorithm/Deep_AR_Plus</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>arn:aws:forecast:::algorithm/ETS</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>arn:aws:forecast:::algorithm/NPTS</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>arn:aws:forecast:::algorithm/Prophet</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmArn(String algorithmArn);

        /**
         * <p>
         * Specifies the number of time-steps that the model is trained to predict. The forecast horizon is also called
         * the prediction length.
         * </p>
         * <p>
         * For example, if you configure a dataset for daily data collection (using the <code>DataFrequency</code>
         * parameter of the <a>CreateDataset</a> operation) and set the forecast horizon to 10, the model returns
         * predictions for 10 days.
         * </p>
         * <p>
         * The maximum forecast horizon is the lesser of 500 time-steps or 1/3 of the TARGET_TIME_SERIES dataset length.
         * </p>
         * 
         * @param forecastHorizon
         *        Specifies the number of time-steps that the model is trained to predict. The forecast horizon is also
         *        called the prediction length.</p>
         *        <p>
         *        For example, if you configure a dataset for daily data collection (using the
         *        <code>DataFrequency</code> parameter of the <a>CreateDataset</a> operation) and set the forecast
         *        horizon to 10, the model returns predictions for 10 days.
         *        </p>
         *        <p>
         *        The maximum forecast horizon is the lesser of 500 time-steps or 1/3 of the TARGET_TIME_SERIES dataset
         *        length.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastHorizon(Integer forecastHorizon);

        /**
         * <p>
         * Specifies the forecast types used to train a predictor. You can specify up to five forecast types. Forecast
         * types can be quantiles from 0.01 to 0.99, by increments of 0.01 or higher. You can also specify the mean
         * forecast with <code>mean</code>.
         * </p>
         * <p>
         * The default value is <code>["0.10", "0.50", "0.9"]</code>.
         * </p>
         * 
         * @param forecastTypes
         *        Specifies the forecast types used to train a predictor. You can specify up to five forecast types.
         *        Forecast types can be quantiles from 0.01 to 0.99, by increments of 0.01 or higher. You can also
         *        specify the mean forecast with <code>mean</code>. </p>
         *        <p>
         *        The default value is <code>["0.10", "0.50", "0.9"]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastTypes(Collection<String> forecastTypes);

        /**
         * <p>
         * Specifies the forecast types used to train a predictor. You can specify up to five forecast types. Forecast
         * types can be quantiles from 0.01 to 0.99, by increments of 0.01 or higher. You can also specify the mean
         * forecast with <code>mean</code>.
         * </p>
         * <p>
         * The default value is <code>["0.10", "0.50", "0.9"]</code>.
         * </p>
         * 
         * @param forecastTypes
         *        Specifies the forecast types used to train a predictor. You can specify up to five forecast types.
         *        Forecast types can be quantiles from 0.01 to 0.99, by increments of 0.01 or higher. You can also
         *        specify the mean forecast with <code>mean</code>. </p>
         *        <p>
         *        The default value is <code>["0.10", "0.50", "0.9"]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastTypes(String... forecastTypes);

        /**
         * <p>
         * Whether to perform AutoML. When Amazon Forecast performs AutoML, it evaluates the algorithms it provides and
         * chooses the best algorithm and configuration for your training dataset.
         * </p>
         * <p>
         * The default value is <code>false</code>. In this case, you are required to specify an algorithm.
         * </p>
         * <p>
         * Set <code>PerformAutoML</code> to <code>true</code> to have Amazon Forecast perform AutoML. This is a good
         * option if you aren't sure which algorithm is suitable for your training data. In this case,
         * <code>PerformHPO</code> must be false.
         * </p>
         * 
         * @param performAutoML
         *        Whether to perform AutoML. When Amazon Forecast performs AutoML, it evaluates the algorithms it
         *        provides and chooses the best algorithm and configuration for your training dataset.</p>
         *        <p>
         *        The default value is <code>false</code>. In this case, you are required to specify an algorithm.
         *        </p>
         *        <p>
         *        Set <code>PerformAutoML</code> to <code>true</code> to have Amazon Forecast perform AutoML. This is a
         *        good option if you aren't sure which algorithm is suitable for your training data. In this case,
         *        <code>PerformHPO</code> must be false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performAutoML(Boolean performAutoML);

        /**
         * <note>
         * <p>
         * The <code>LatencyOptimized</code> AutoML override strategy is only available in private beta. Contact Amazon
         * Web Services Support or your account manager to learn more about access privileges.
         * </p>
         * </note>
         * <p>
         * Used to overide the default AutoML strategy, which is to optimize predictor accuracy. To apply an AutoML
         * strategy that minimizes training time, use <code>LatencyOptimized</code>.
         * </p>
         * <p>
         * This parameter is only valid for predictors trained using AutoML.
         * </p>
         * 
         * @param autoMLOverrideStrategy
         *        <p>
         *        The <code>LatencyOptimized</code> AutoML override strategy is only available in private beta. Contact
         *        Amazon Web Services Support or your account manager to learn more about access privileges.
         *        </p>
         *        </note>
         *        <p>
         *        Used to overide the default AutoML strategy, which is to optimize predictor accuracy. To apply an
         *        AutoML strategy that minimizes training time, use <code>LatencyOptimized</code>.
         *        </p>
         *        <p>
         *        This parameter is only valid for predictors trained using AutoML.
         * @see AutoMLOverrideStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMLOverrideStrategy
         */
        Builder autoMLOverrideStrategy(String autoMLOverrideStrategy);

        /**
         * <note>
         * <p>
         * The <code>LatencyOptimized</code> AutoML override strategy is only available in private beta. Contact Amazon
         * Web Services Support or your account manager to learn more about access privileges.
         * </p>
         * </note>
         * <p>
         * Used to overide the default AutoML strategy, which is to optimize predictor accuracy. To apply an AutoML
         * strategy that minimizes training time, use <code>LatencyOptimized</code>.
         * </p>
         * <p>
         * This parameter is only valid for predictors trained using AutoML.
         * </p>
         * 
         * @param autoMLOverrideStrategy
         *        <p>
         *        The <code>LatencyOptimized</code> AutoML override strategy is only available in private beta. Contact
         *        Amazon Web Services Support or your account manager to learn more about access privileges.
         *        </p>
         *        </note>
         *        <p>
         *        Used to overide the default AutoML strategy, which is to optimize predictor accuracy. To apply an
         *        AutoML strategy that minimizes training time, use <code>LatencyOptimized</code>.
         *        </p>
         *        <p>
         *        This parameter is only valid for predictors trained using AutoML.
         * @see AutoMLOverrideStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMLOverrideStrategy
         */
        Builder autoMLOverrideStrategy(AutoMLOverrideStrategy autoMLOverrideStrategy);

        /**
         * <p>
         * Whether to perform hyperparameter optimization (HPO). HPO finds optimal hyperparameter values for your
         * training data. The process of performing HPO is known as running a hyperparameter tuning job.
         * </p>
         * <p>
         * The default value is <code>false</code>. In this case, Amazon Forecast uses default hyperparameter values
         * from the chosen algorithm.
         * </p>
         * <p>
         * To override the default values, set <code>PerformHPO</code> to <code>true</code> and, optionally, supply the
         * <a>HyperParameterTuningJobConfig</a> object. The tuning job specifies a metric to optimize, which
         * hyperparameters participate in tuning, and the valid range for each tunable hyperparameter. In this case, you
         * are required to specify an algorithm and <code>PerformAutoML</code> must be false.
         * </p>
         * <p>
         * The following algorithms support HPO:
         * </p>
         * <ul>
         * <li>
         * <p>
         * DeepAR+
         * </p>
         * </li>
         * <li>
         * <p>
         * CNN-QR
         * </p>
         * </li>
         * </ul>
         * 
         * @param performHPO
         *        Whether to perform hyperparameter optimization (HPO). HPO finds optimal hyperparameter values for your
         *        training data. The process of performing HPO is known as running a hyperparameter tuning job.</p>
         *        <p>
         *        The default value is <code>false</code>. In this case, Amazon Forecast uses default hyperparameter
         *        values from the chosen algorithm.
         *        </p>
         *        <p>
         *        To override the default values, set <code>PerformHPO</code> to <code>true</code> and, optionally,
         *        supply the <a>HyperParameterTuningJobConfig</a> object. The tuning job specifies a metric to optimize,
         *        which hyperparameters participate in tuning, and the valid range for each tunable hyperparameter. In
         *        this case, you are required to specify an algorithm and <code>PerformAutoML</code> must be false.
         *        </p>
         *        <p>
         *        The following algorithms support HPO:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        DeepAR+
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        CNN-QR
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performHPO(Boolean performHPO);

        /**
         * <p>
         * The hyperparameters to override for model training. The hyperparameters that you can override are listed in
         * the individual algorithms. For the list of supported algorithms, see <a>aws-forecast-choosing-recipes</a>.
         * </p>
         * 
         * @param trainingParameters
         *        The hyperparameters to override for model training. The hyperparameters that you can override are
         *        listed in the individual algorithms. For the list of supported algorithms, see
         *        <a>aws-forecast-choosing-recipes</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingParameters(Map<String, String> trainingParameters);

        /**
         * <p>
         * Used to override the default evaluation parameters of the specified algorithm. Amazon Forecast evaluates a
         * predictor by splitting a dataset into training data and testing data. The evaluation parameters define how to
         * perform the split and the number of iterations.
         * </p>
         * 
         * @param evaluationParameters
         *        Used to override the default evaluation parameters of the specified algorithm. Amazon Forecast
         *        evaluates a predictor by splitting a dataset into training data and testing data. The evaluation
         *        parameters define how to perform the split and the number of iterations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationParameters(EvaluationParameters evaluationParameters);

        /**
         * <p>
         * Used to override the default evaluation parameters of the specified algorithm. Amazon Forecast evaluates a
         * predictor by splitting a dataset into training data and testing data. The evaluation parameters define how to
         * perform the split and the number of iterations.
         * </p>
         * This is a convenience method that creates an instance of the {@link EvaluationParameters.Builder} avoiding
         * the need to create one manually via {@link EvaluationParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EvaluationParameters.Builder#build()} is called immediately and
         * its result is passed to {@link #evaluationParameters(EvaluationParameters)}.
         * 
         * @param evaluationParameters
         *        a consumer that will call methods on {@link EvaluationParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #evaluationParameters(EvaluationParameters)
         */
        default Builder evaluationParameters(Consumer<EvaluationParameters.Builder> evaluationParameters) {
            return evaluationParameters(EvaluationParameters.builder().applyMutation(evaluationParameters).build());
        }

        /**
         * <p>
         * Provides hyperparameter override values for the algorithm. If you don't provide this parameter, Amazon
         * Forecast uses default values. The individual algorithms specify which hyperparameters support hyperparameter
         * optimization (HPO). For more information, see <a>aws-forecast-choosing-recipes</a>.
         * </p>
         * <p>
         * If you included the <code>HPOConfig</code> object, you must set <code>PerformHPO</code> to true.
         * </p>
         * 
         * @param hpoConfig
         *        Provides hyperparameter override values for the algorithm. If you don't provide this parameter, Amazon
         *        Forecast uses default values. The individual algorithms specify which hyperparameters support
         *        hyperparameter optimization (HPO). For more information, see <a>aws-forecast-choosing-recipes</a>.</p>
         *        <p>
         *        If you included the <code>HPOConfig</code> object, you must set <code>PerformHPO</code> to true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hpoConfig(HyperParameterTuningJobConfig hpoConfig);

        /**
         * <p>
         * Provides hyperparameter override values for the algorithm. If you don't provide this parameter, Amazon
         * Forecast uses default values. The individual algorithms specify which hyperparameters support hyperparameter
         * optimization (HPO). For more information, see <a>aws-forecast-choosing-recipes</a>.
         * </p>
         * <p>
         * If you included the <code>HPOConfig</code> object, you must set <code>PerformHPO</code> to true.
         * </p>
         * This is a convenience method that creates an instance of the {@link HyperParameterTuningJobConfig.Builder}
         * avoiding the need to create one manually via {@link HyperParameterTuningJobConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HyperParameterTuningJobConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #hpoConfig(HyperParameterTuningJobConfig)}.
         * 
         * @param hpoConfig
         *        a consumer that will call methods on {@link HyperParameterTuningJobConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hpoConfig(HyperParameterTuningJobConfig)
         */
        default Builder hpoConfig(Consumer<HyperParameterTuningJobConfig.Builder> hpoConfig) {
            return hpoConfig(HyperParameterTuningJobConfig.builder().applyMutation(hpoConfig).build());
        }

        /**
         * <p>
         * Describes the dataset group that contains the data to use to train the predictor.
         * </p>
         * 
         * @param inputDataConfig
         *        Describes the dataset group that contains the data to use to train the predictor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataConfig(InputDataConfig inputDataConfig);

        /**
         * <p>
         * Describes the dataset group that contains the data to use to train the predictor.
         * </p>
         * This is a convenience method that creates an instance of the {@link InputDataConfig.Builder} avoiding the
         * need to create one manually via {@link InputDataConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InputDataConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #inputDataConfig(InputDataConfig)}.
         * 
         * @param inputDataConfig
         *        a consumer that will call methods on {@link InputDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputDataConfig(InputDataConfig)
         */
        default Builder inputDataConfig(Consumer<InputDataConfig.Builder> inputDataConfig) {
            return inputDataConfig(InputDataConfig.builder().applyMutation(inputDataConfig).build());
        }

        /**
         * <p>
         * The featurization configuration.
         * </p>
         * 
         * @param featurizationConfig
         *        The featurization configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featurizationConfig(FeaturizationConfig featurizationConfig);

        /**
         * <p>
         * The featurization configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link FeaturizationConfig.Builder} avoiding the
         * need to create one manually via {@link FeaturizationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FeaturizationConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #featurizationConfig(FeaturizationConfig)}.
         * 
         * @param featurizationConfig
         *        a consumer that will call methods on {@link FeaturizationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #featurizationConfig(FeaturizationConfig)
         */
        default Builder featurizationConfig(Consumer<FeaturizationConfig.Builder> featurizationConfig) {
            return featurizationConfig(FeaturizationConfig.builder().applyMutation(featurizationConfig).build());
        }

        /**
         * <p>
         * An Key Management Service (KMS) key and the Identity and Access Management (IAM) role that Amazon Forecast
         * can assume to access the key.
         * </p>
         * 
         * @param encryptionConfig
         *        An Key Management Service (KMS) key and the Identity and Access Management (IAM) role that Amazon
         *        Forecast can assume to access the key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfig(EncryptionConfig encryptionConfig);

        /**
         * <p>
         * An Key Management Service (KMS) key and the Identity and Access Management (IAM) role that Amazon Forecast
         * can assume to access the key.
         * </p>
         * This is a convenience method that creates an instance of the {@link EncryptionConfig.Builder} avoiding the
         * need to create one manually via {@link EncryptionConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #encryptionConfig(EncryptionConfig)}.
         * 
         * @param encryptionConfig
         *        a consumer that will call methods on {@link EncryptionConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfig(EncryptionConfig)
         */
        default Builder encryptionConfig(Consumer<EncryptionConfig.Builder> encryptionConfig) {
            return encryptionConfig(EncryptionConfig.builder().applyMutation(encryptionConfig).build());
        }

        /**
         * <p>
         * The optional metadata that you apply to the predictor to help you categorize and organize them. Each tag
         * consists of a key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50.
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8.
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for keys as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys with this prefix.
         * Values can have this prefix. If a tag value has <code>aws</code> as its prefix but the key does not, then
         * Forecast considers it to be a user tag and will count against the limit of 50 tags. Tags with only the key
         * prefix of <code>aws</code> do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        The optional metadata that you apply to the predictor to help you categorize and organize them. Each
         *        tag consists of a key and an optional value, both of which you define.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource - 50.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length - 128 Unicode characters in UTF-8.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length - 256 Unicode characters in UTF-8.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for keys as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys with
         *        this prefix. Values can have this prefix. If a tag value has <code>aws</code> as its prefix but the
         *        key does not, then Forecast considers it to be a user tag and will count against the limit of 50 tags.
         *        Tags with only the key prefix of <code>aws</code> do not count against your tags per resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The optional metadata that you apply to the predictor to help you categorize and organize them. Each tag
         * consists of a key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50.
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8.
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for keys as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys with this prefix.
         * Values can have this prefix. If a tag value has <code>aws</code> as its prefix but the key does not, then
         * Forecast considers it to be a user tag and will count against the limit of 50 tags. Tags with only the key
         * prefix of <code>aws</code> do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        The optional metadata that you apply to the predictor to help you categorize and organize them. Each
         *        tag consists of a key and an optional value, both of which you define.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource - 50.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length - 128 Unicode characters in UTF-8.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length - 256 Unicode characters in UTF-8.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for keys as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys with
         *        this prefix. Values can have this prefix. If a tag value has <code>aws</code> as its prefix but the
         *        key does not, then Forecast considers it to be a user tag and will count against the limit of 50 tags.
         *        Tags with only the key prefix of <code>aws</code> do not count against your tags per resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The optional metadata that you apply to the predictor to help you categorize and organize them. Each tag
         * consists of a key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50.
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8.
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for keys as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys with this prefix.
         * Values can have this prefix. If a tag value has <code>aws</code> as its prefix but the key does not, then
         * Forecast considers it to be a user tag and will count against the limit of 50 tags. Tags with only the key
         * prefix of <code>aws</code> do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.forecast.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.forecast.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.forecast.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.forecast.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The accuracy metric used to optimize the predictor.
         * </p>
         * 
         * @param optimizationMetric
         *        The accuracy metric used to optimize the predictor.
         * @see OptimizationMetric
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OptimizationMetric
         */
        Builder optimizationMetric(String optimizationMetric);

        /**
         * <p>
         * The accuracy metric used to optimize the predictor.
         * </p>
         * 
         * @param optimizationMetric
         *        The accuracy metric used to optimize the predictor.
         * @see OptimizationMetric
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OptimizationMetric
         */
        Builder optimizationMetric(OptimizationMetric optimizationMetric);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ForecastRequest.BuilderImpl implements Builder {
        private String predictorName;

        private String algorithmArn;

        private Integer forecastHorizon;

        private List<String> forecastTypes = DefaultSdkAutoConstructList.getInstance();

        private Boolean performAutoML;

        private String autoMLOverrideStrategy;

        private Boolean performHPO;

        private Map<String, String> trainingParameters = DefaultSdkAutoConstructMap.getInstance();

        private EvaluationParameters evaluationParameters;

        private HyperParameterTuningJobConfig hpoConfig;

        private InputDataConfig inputDataConfig;

        private FeaturizationConfig featurizationConfig;

        private EncryptionConfig encryptionConfig;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String optimizationMetric;

        private BuilderImpl() {
        }

        private BuilderImpl(CreatePredictorRequest model) {
            super(model);
            predictorName(model.predictorName);
            algorithmArn(model.algorithmArn);
            forecastHorizon(model.forecastHorizon);
            forecastTypes(model.forecastTypes);
            performAutoML(model.performAutoML);
            autoMLOverrideStrategy(model.autoMLOverrideStrategy);
            performHPO(model.performHPO);
            trainingParameters(model.trainingParameters);
            evaluationParameters(model.evaluationParameters);
            hpoConfig(model.hpoConfig);
            inputDataConfig(model.inputDataConfig);
            featurizationConfig(model.featurizationConfig);
            encryptionConfig(model.encryptionConfig);
            tags(model.tags);
            optimizationMetric(model.optimizationMetric);
        }

        public final String getPredictorName() {
            return predictorName;
        }

        public final void setPredictorName(String predictorName) {
            this.predictorName = predictorName;
        }

        @Override
        public final Builder predictorName(String predictorName) {
            this.predictorName = predictorName;
            return this;
        }

        public final String getAlgorithmArn() {
            return algorithmArn;
        }

        public final void setAlgorithmArn(String algorithmArn) {
            this.algorithmArn = algorithmArn;
        }

        @Override
        public final Builder algorithmArn(String algorithmArn) {
            this.algorithmArn = algorithmArn;
            return this;
        }

        public final Integer getForecastHorizon() {
            return forecastHorizon;
        }

        public final void setForecastHorizon(Integer forecastHorizon) {
            this.forecastHorizon = forecastHorizon;
        }

        @Override
        public final Builder forecastHorizon(Integer forecastHorizon) {
            this.forecastHorizon = forecastHorizon;
            return this;
        }

        public final Collection<String> getForecastTypes() {
            if (forecastTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return forecastTypes;
        }

        public final void setForecastTypes(Collection<String> forecastTypes) {
            this.forecastTypes = ForecastTypesCopier.copy(forecastTypes);
        }

        @Override
        public final Builder forecastTypes(Collection<String> forecastTypes) {
            this.forecastTypes = ForecastTypesCopier.copy(forecastTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder forecastTypes(String... forecastTypes) {
            forecastTypes(Arrays.asList(forecastTypes));
            return this;
        }

        public final Boolean getPerformAutoML() {
            return performAutoML;
        }

        public final void setPerformAutoML(Boolean performAutoML) {
            this.performAutoML = performAutoML;
        }

        @Override
        public final Builder performAutoML(Boolean performAutoML) {
            this.performAutoML = performAutoML;
            return this;
        }

        public final String getAutoMLOverrideStrategy() {
            return autoMLOverrideStrategy;
        }

        public final void setAutoMLOverrideStrategy(String autoMLOverrideStrategy) {
            this.autoMLOverrideStrategy = autoMLOverrideStrategy;
        }

        @Override
        public final Builder autoMLOverrideStrategy(String autoMLOverrideStrategy) {
            this.autoMLOverrideStrategy = autoMLOverrideStrategy;
            return this;
        }

        @Override
        public final Builder autoMLOverrideStrategy(AutoMLOverrideStrategy autoMLOverrideStrategy) {
            this.autoMLOverrideStrategy(autoMLOverrideStrategy == null ? null : autoMLOverrideStrategy.toString());
            return this;
        }

        public final Boolean getPerformHPO() {
            return performHPO;
        }

        public final void setPerformHPO(Boolean performHPO) {
            this.performHPO = performHPO;
        }

        @Override
        public final Builder performHPO(Boolean performHPO) {
            this.performHPO = performHPO;
            return this;
        }

        public final Map<String, String> getTrainingParameters() {
            if (trainingParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return trainingParameters;
        }

        public final void setTrainingParameters(Map<String, String> trainingParameters) {
            this.trainingParameters = TrainingParametersCopier.copy(trainingParameters);
        }

        @Override
        public final Builder trainingParameters(Map<String, String> trainingParameters) {
            this.trainingParameters = TrainingParametersCopier.copy(trainingParameters);
            return this;
        }

        public final EvaluationParameters.Builder getEvaluationParameters() {
            return evaluationParameters != null ? evaluationParameters.toBuilder() : null;
        }

        public final void setEvaluationParameters(EvaluationParameters.BuilderImpl evaluationParameters) {
            this.evaluationParameters = evaluationParameters != null ? evaluationParameters.build() : null;
        }

        @Override
        public final Builder evaluationParameters(EvaluationParameters evaluationParameters) {
            this.evaluationParameters = evaluationParameters;
            return this;
        }

        public final HyperParameterTuningJobConfig.Builder getHpoConfig() {
            return hpoConfig != null ? hpoConfig.toBuilder() : null;
        }

        public final void setHpoConfig(HyperParameterTuningJobConfig.BuilderImpl hpoConfig) {
            this.hpoConfig = hpoConfig != null ? hpoConfig.build() : null;
        }

        @Override
        public final Builder hpoConfig(HyperParameterTuningJobConfig hpoConfig) {
            this.hpoConfig = hpoConfig;
            return this;
        }

        public final InputDataConfig.Builder getInputDataConfig() {
            return inputDataConfig != null ? inputDataConfig.toBuilder() : null;
        }

        public final void setInputDataConfig(InputDataConfig.BuilderImpl inputDataConfig) {
            this.inputDataConfig = inputDataConfig != null ? inputDataConfig.build() : null;
        }

        @Override
        public final Builder inputDataConfig(InputDataConfig inputDataConfig) {
            this.inputDataConfig = inputDataConfig;
            return this;
        }

        public final FeaturizationConfig.Builder getFeaturizationConfig() {
            return featurizationConfig != null ? featurizationConfig.toBuilder() : null;
        }

        public final void setFeaturizationConfig(FeaturizationConfig.BuilderImpl featurizationConfig) {
            this.featurizationConfig = featurizationConfig != null ? featurizationConfig.build() : null;
        }

        @Override
        public final Builder featurizationConfig(FeaturizationConfig featurizationConfig) {
            this.featurizationConfig = featurizationConfig;
            return this;
        }

        public final EncryptionConfig.Builder getEncryptionConfig() {
            return encryptionConfig != null ? encryptionConfig.toBuilder() : null;
        }

        public final void setEncryptionConfig(EncryptionConfig.BuilderImpl encryptionConfig) {
            this.encryptionConfig = encryptionConfig != null ? encryptionConfig.build() : null;
        }

        @Override
        public final Builder encryptionConfig(EncryptionConfig encryptionConfig) {
            this.encryptionConfig = encryptionConfig;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getOptimizationMetric() {
            return optimizationMetric;
        }

        public final void setOptimizationMetric(String optimizationMetric) {
            this.optimizationMetric = optimizationMetric;
        }

        @Override
        public final Builder optimizationMetric(String optimizationMetric) {
            this.optimizationMetric = optimizationMetric;
            return this;
        }

        @Override
        public final Builder optimizationMetric(OptimizationMetric optimizationMetric) {
            this.optimizationMetric(optimizationMetric == null ? null : optimizationMetric.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreatePredictorRequest build() {
            return new CreatePredictorRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
