/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.forecast.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This object belongs to the <a>CreatePredictor</a> operation. If you created your predictor with
 * <a>CreateAutoPredictor</a>, see <a>AttributeConfig</a>.
 * </p>
 * </note>
 * <p>
 * In a <a>CreatePredictor</a> operation, the specified algorithm trains a model using the specified dataset group. You
 * can optionally tell the operation to modify data fields prior to training a model. These modifications are referred
 * to as <i>featurization</i>.
 * </p>
 * <p>
 * You define featurization using the <code>FeaturizationConfig</code> object. You specify an array of transformations,
 * one for each field that you want to featurize. You then include the <code>FeaturizationConfig</code> object in your
 * <code>CreatePredictor</code> request. Amazon Forecast applies the featurization to the
 * <code>TARGET_TIME_SERIES</code> and <code>RELATED_TIME_SERIES</code> datasets before model training.
 * </p>
 * <p>
 * You can create multiple featurization configurations. For example, you might call the <code>CreatePredictor</code>
 * operation twice by specifying different featurization configurations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FeaturizationConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<FeaturizationConfig.Builder, FeaturizationConfig> {
    private static final SdkField<String> FORECAST_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ForecastFrequency").getter(getter(FeaturizationConfig::forecastFrequency))
            .setter(setter(Builder::forecastFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastFrequency").build()).build();

    private static final SdkField<List<String>> FORECAST_DIMENSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ForecastDimensions")
            .getter(getter(FeaturizationConfig::forecastDimensions))
            .setter(setter(Builder::forecastDimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastDimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Featurization>> FEATURIZATIONS_FIELD = SdkField
            .<List<Featurization>> builder(MarshallingType.LIST)
            .memberName("Featurizations")
            .getter(getter(FeaturizationConfig::featurizations))
            .setter(setter(Builder::featurizations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Featurizations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Featurization> builder(MarshallingType.SDK_POJO)
                                            .constructor(Featurization::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FORECAST_FREQUENCY_FIELD,
            FORECAST_DIMENSIONS_FIELD, FEATURIZATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String forecastFrequency;

    private final List<String> forecastDimensions;

    private final List<Featurization> featurizations;

    private FeaturizationConfig(BuilderImpl builder) {
        this.forecastFrequency = builder.forecastFrequency;
        this.forecastDimensions = builder.forecastDimensions;
        this.featurizations = builder.featurizations;
    }

    /**
     * <p>
     * The frequency of predictions in a forecast.
     * </p>
     * <p>
     * Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min (Minute).
     * For example, "1D" indicates every day and "15min" indicates every 15 minutes. You cannot specify a value that
     * would overlap with the next larger frequency. That means, for example, you cannot specify a frequency of 60
     * minutes, because that is equivalent to 1 hour. The valid values for each frequency are the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Minute - 1-59
     * </p>
     * </li>
     * <li>
     * <p>
     * Hour - 1-23
     * </p>
     * </li>
     * <li>
     * <p>
     * Day - 1-6
     * </p>
     * </li>
     * <li>
     * <p>
     * Week - 1-4
     * </p>
     * </li>
     * <li>
     * <p>
     * Month - 1-11
     * </p>
     * </li>
     * <li>
     * <p>
     * Year - 1
     * </p>
     * </li>
     * </ul>
     * <p>
     * Thus, if you want every other week forecasts, specify "2W". Or, if you want quarterly forecasts, you specify
     * "3M".
     * </p>
     * <p>
     * The frequency must be greater than or equal to the TARGET_TIME_SERIES dataset frequency.
     * </p>
     * <p>
     * When a RELATED_TIME_SERIES dataset is provided, the frequency must be equal to the TARGET_TIME_SERIES dataset
     * frequency.
     * </p>
     * 
     * @return The frequency of predictions in a forecast.</p>
     *         <p>
     *         Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
     *         (Minute). For example, "1D" indicates every day and "15min" indicates every 15 minutes. You cannot
     *         specify a value that would overlap with the next larger frequency. That means, for example, you cannot
     *         specify a frequency of 60 minutes, because that is equivalent to 1 hour. The valid values for each
     *         frequency are the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Minute - 1-59
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Hour - 1-23
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Day - 1-6
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Week - 1-4
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Month - 1-11
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Year - 1
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Thus, if you want every other week forecasts, specify "2W". Or, if you want quarterly forecasts, you
     *         specify "3M".
     *         </p>
     *         <p>
     *         The frequency must be greater than or equal to the TARGET_TIME_SERIES dataset frequency.
     *         </p>
     *         <p>
     *         When a RELATED_TIME_SERIES dataset is provided, the frequency must be equal to the TARGET_TIME_SERIES
     *         dataset frequency.
     */
    public final String forecastFrequency() {
        return forecastFrequency;
    }

    /**
     * For responses, this returns true if the service returned a value for the ForecastDimensions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasForecastDimensions() {
        return forecastDimensions != null && !(forecastDimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of dimension (field) names that specify how to group the generated forecast.
     * </p>
     * <p>
     * For example, suppose that you are generating a forecast for item sales across all of your stores, and your
     * dataset contains a <code>store_id</code> field. If you want the sales forecast for each item by store, you would
     * specify <code>store_id</code> as the dimension.
     * </p>
     * <p>
     * All forecast dimensions specified in the <code>TARGET_TIME_SERIES</code> dataset don't need to be specified in
     * the <code>CreatePredictor</code> request. All forecast dimensions specified in the
     * <code>RELATED_TIME_SERIES</code> dataset must be specified in the <code>CreatePredictor</code> request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasForecastDimensions} method.
     * </p>
     * 
     * @return An array of dimension (field) names that specify how to group the generated forecast.</p>
     *         <p>
     *         For example, suppose that you are generating a forecast for item sales across all of your stores, and
     *         your dataset contains a <code>store_id</code> field. If you want the sales forecast for each item by
     *         store, you would specify <code>store_id</code> as the dimension.
     *         </p>
     *         <p>
     *         All forecast dimensions specified in the <code>TARGET_TIME_SERIES</code> dataset don't need to be
     *         specified in the <code>CreatePredictor</code> request. All forecast dimensions specified in the
     *         <code>RELATED_TIME_SERIES</code> dataset must be specified in the <code>CreatePredictor</code> request.
     */
    public final List<String> forecastDimensions() {
        return forecastDimensions;
    }

    /**
     * For responses, this returns true if the service returned a value for the Featurizations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFeaturizations() {
        return featurizations != null && !(featurizations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of featurization (transformation) information for the fields of a dataset.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeaturizations} method.
     * </p>
     * 
     * @return An array of featurization (transformation) information for the fields of a dataset.
     */
    public final List<Featurization> featurizations() {
        return featurizations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(forecastFrequency());
        hashCode = 31 * hashCode + Objects.hashCode(hasForecastDimensions() ? forecastDimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFeaturizations() ? featurizations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FeaturizationConfig)) {
            return false;
        }
        FeaturizationConfig other = (FeaturizationConfig) obj;
        return Objects.equals(forecastFrequency(), other.forecastFrequency())
                && hasForecastDimensions() == other.hasForecastDimensions()
                && Objects.equals(forecastDimensions(), other.forecastDimensions())
                && hasFeaturizations() == other.hasFeaturizations() && Objects.equals(featurizations(), other.featurizations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FeaturizationConfig").add("ForecastFrequency", forecastFrequency())
                .add("ForecastDimensions", hasForecastDimensions() ? forecastDimensions() : null)
                .add("Featurizations", hasFeaturizations() ? featurizations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ForecastFrequency":
            return Optional.ofNullable(clazz.cast(forecastFrequency()));
        case "ForecastDimensions":
            return Optional.ofNullable(clazz.cast(forecastDimensions()));
        case "Featurizations":
            return Optional.ofNullable(clazz.cast(featurizations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FeaturizationConfig, T> g) {
        return obj -> g.apply((FeaturizationConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FeaturizationConfig> {
        /**
         * <p>
         * The frequency of predictions in a forecast.
         * </p>
         * <p>
         * Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
         * (Minute). For example, "1D" indicates every day and "15min" indicates every 15 minutes. You cannot specify a
         * value that would overlap with the next larger frequency. That means, for example, you cannot specify a
         * frequency of 60 minutes, because that is equivalent to 1 hour. The valid values for each frequency are the
         * following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Minute - 1-59
         * </p>
         * </li>
         * <li>
         * <p>
         * Hour - 1-23
         * </p>
         * </li>
         * <li>
         * <p>
         * Day - 1-6
         * </p>
         * </li>
         * <li>
         * <p>
         * Week - 1-4
         * </p>
         * </li>
         * <li>
         * <p>
         * Month - 1-11
         * </p>
         * </li>
         * <li>
         * <p>
         * Year - 1
         * </p>
         * </li>
         * </ul>
         * <p>
         * Thus, if you want every other week forecasts, specify "2W". Or, if you want quarterly forecasts, you specify
         * "3M".
         * </p>
         * <p>
         * The frequency must be greater than or equal to the TARGET_TIME_SERIES dataset frequency.
         * </p>
         * <p>
         * When a RELATED_TIME_SERIES dataset is provided, the frequency must be equal to the TARGET_TIME_SERIES dataset
         * frequency.
         * </p>
         * 
         * @param forecastFrequency
         *        The frequency of predictions in a forecast.</p>
         *        <p>
         *        Valid intervals are an integer followed by Y (Year), M (Month), W (Week), D (Day), H (Hour), and min
         *        (Minute). For example, "1D" indicates every day and "15min" indicates every 15 minutes. You cannot
         *        specify a value that would overlap with the next larger frequency. That means, for example, you cannot
         *        specify a frequency of 60 minutes, because that is equivalent to 1 hour. The valid values for each
         *        frequency are the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Minute - 1-59
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Hour - 1-23
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Day - 1-6
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Week - 1-4
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Month - 1-11
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Year - 1
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Thus, if you want every other week forecasts, specify "2W". Or, if you want quarterly forecasts, you
         *        specify "3M".
         *        </p>
         *        <p>
         *        The frequency must be greater than or equal to the TARGET_TIME_SERIES dataset frequency.
         *        </p>
         *        <p>
         *        When a RELATED_TIME_SERIES dataset is provided, the frequency must be equal to the TARGET_TIME_SERIES
         *        dataset frequency.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastFrequency(String forecastFrequency);

        /**
         * <p>
         * An array of dimension (field) names that specify how to group the generated forecast.
         * </p>
         * <p>
         * For example, suppose that you are generating a forecast for item sales across all of your stores, and your
         * dataset contains a <code>store_id</code> field. If you want the sales forecast for each item by store, you
         * would specify <code>store_id</code> as the dimension.
         * </p>
         * <p>
         * All forecast dimensions specified in the <code>TARGET_TIME_SERIES</code> dataset don't need to be specified
         * in the <code>CreatePredictor</code> request. All forecast dimensions specified in the
         * <code>RELATED_TIME_SERIES</code> dataset must be specified in the <code>CreatePredictor</code> request.
         * </p>
         * 
         * @param forecastDimensions
         *        An array of dimension (field) names that specify how to group the generated forecast.</p>
         *        <p>
         *        For example, suppose that you are generating a forecast for item sales across all of your stores, and
         *        your dataset contains a <code>store_id</code> field. If you want the sales forecast for each item by
         *        store, you would specify <code>store_id</code> as the dimension.
         *        </p>
         *        <p>
         *        All forecast dimensions specified in the <code>TARGET_TIME_SERIES</code> dataset don't need to be
         *        specified in the <code>CreatePredictor</code> request. All forecast dimensions specified in the
         *        <code>RELATED_TIME_SERIES</code> dataset must be specified in the <code>CreatePredictor</code>
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastDimensions(Collection<String> forecastDimensions);

        /**
         * <p>
         * An array of dimension (field) names that specify how to group the generated forecast.
         * </p>
         * <p>
         * For example, suppose that you are generating a forecast for item sales across all of your stores, and your
         * dataset contains a <code>store_id</code> field. If you want the sales forecast for each item by store, you
         * would specify <code>store_id</code> as the dimension.
         * </p>
         * <p>
         * All forecast dimensions specified in the <code>TARGET_TIME_SERIES</code> dataset don't need to be specified
         * in the <code>CreatePredictor</code> request. All forecast dimensions specified in the
         * <code>RELATED_TIME_SERIES</code> dataset must be specified in the <code>CreatePredictor</code> request.
         * </p>
         * 
         * @param forecastDimensions
         *        An array of dimension (field) names that specify how to group the generated forecast.</p>
         *        <p>
         *        For example, suppose that you are generating a forecast for item sales across all of your stores, and
         *        your dataset contains a <code>store_id</code> field. If you want the sales forecast for each item by
         *        store, you would specify <code>store_id</code> as the dimension.
         *        </p>
         *        <p>
         *        All forecast dimensions specified in the <code>TARGET_TIME_SERIES</code> dataset don't need to be
         *        specified in the <code>CreatePredictor</code> request. All forecast dimensions specified in the
         *        <code>RELATED_TIME_SERIES</code> dataset must be specified in the <code>CreatePredictor</code>
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastDimensions(String... forecastDimensions);

        /**
         * <p>
         * An array of featurization (transformation) information for the fields of a dataset.
         * </p>
         * 
         * @param featurizations
         *        An array of featurization (transformation) information for the fields of a dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featurizations(Collection<Featurization> featurizations);

        /**
         * <p>
         * An array of featurization (transformation) information for the fields of a dataset.
         * </p>
         * 
         * @param featurizations
         *        An array of featurization (transformation) information for the fields of a dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featurizations(Featurization... featurizations);

        /**
         * <p>
         * An array of featurization (transformation) information for the fields of a dataset.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.forecast.model.Featurization.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.forecast.model.Featurization#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.forecast.model.Featurization.Builder#build()} is called immediately
         * and its result is passed to {@link #featurizations(List<Featurization>)}.
         * 
         * @param featurizations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.forecast.model.Featurization.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #featurizations(java.util.Collection<Featurization>)
         */
        Builder featurizations(Consumer<Featurization.Builder>... featurizations);
    }

    static final class BuilderImpl implements Builder {
        private String forecastFrequency;

        private List<String> forecastDimensions = DefaultSdkAutoConstructList.getInstance();

        private List<Featurization> featurizations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FeaturizationConfig model) {
            forecastFrequency(model.forecastFrequency);
            forecastDimensions(model.forecastDimensions);
            featurizations(model.featurizations);
        }

        public final String getForecastFrequency() {
            return forecastFrequency;
        }

        public final void setForecastFrequency(String forecastFrequency) {
            this.forecastFrequency = forecastFrequency;
        }

        @Override
        public final Builder forecastFrequency(String forecastFrequency) {
            this.forecastFrequency = forecastFrequency;
            return this;
        }

        public final Collection<String> getForecastDimensions() {
            if (forecastDimensions instanceof SdkAutoConstructList) {
                return null;
            }
            return forecastDimensions;
        }

        public final void setForecastDimensions(Collection<String> forecastDimensions) {
            this.forecastDimensions = ForecastDimensionsCopier.copy(forecastDimensions);
        }

        @Override
        public final Builder forecastDimensions(Collection<String> forecastDimensions) {
            this.forecastDimensions = ForecastDimensionsCopier.copy(forecastDimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder forecastDimensions(String... forecastDimensions) {
            forecastDimensions(Arrays.asList(forecastDimensions));
            return this;
        }

        public final List<Featurization.Builder> getFeaturizations() {
            List<Featurization.Builder> result = FeaturizationsCopier.copyToBuilder(this.featurizations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFeaturizations(Collection<Featurization.BuilderImpl> featurizations) {
            this.featurizations = FeaturizationsCopier.copyFromBuilder(featurizations);
        }

        @Override
        public final Builder featurizations(Collection<Featurization> featurizations) {
            this.featurizations = FeaturizationsCopier.copy(featurizations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featurizations(Featurization... featurizations) {
            featurizations(Arrays.asList(featurizations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featurizations(Consumer<Featurization.Builder>... featurizations) {
            featurizations(Stream.of(featurizations).map(c -> Featurization.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public FeaturizationConfig build() {
            return new FeaturizationConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
