/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The deserializer you want Firehose to use for converting the input data from JSON. Firehose then serializes the data
 * to its final format using the <a>Serializer</a>. Firehose supports two types of deserializers: the <a
 * href="https://cwiki.apache.org/confluence/display/Hive/LanguageManual+DDL#LanguageManualDDL-JSON">Apache Hive JSON
 * SerDe</a> and the <a href="https://github.com/rcongiu/Hive-JSON-Serde">OpenX JSON SerDe</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Deserializer implements SdkPojo, Serializable, ToCopyableBuilder<Deserializer.Builder, Deserializer> {
    private static final SdkField<OpenXJsonSerDe> OPEN_X_JSON_SER_DE_FIELD = SdkField
            .<OpenXJsonSerDe> builder(MarshallingType.SDK_POJO).memberName("OpenXJsonSerDe")
            .getter(getter(Deserializer::openXJsonSerDe)).setter(setter(Builder::openXJsonSerDe))
            .constructor(OpenXJsonSerDe::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OpenXJsonSerDe").build()).build();

    private static final SdkField<HiveJsonSerDe> HIVE_JSON_SER_DE_FIELD = SdkField
            .<HiveJsonSerDe> builder(MarshallingType.SDK_POJO).memberName("HiveJsonSerDe")
            .getter(getter(Deserializer::hiveJsonSerDe)).setter(setter(Builder::hiveJsonSerDe))
            .constructor(HiveJsonSerDe::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HiveJsonSerDe").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OPEN_X_JSON_SER_DE_FIELD,
            HIVE_JSON_SER_DE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final OpenXJsonSerDe openXJsonSerDe;

    private final HiveJsonSerDe hiveJsonSerDe;

    private Deserializer(BuilderImpl builder) {
        this.openXJsonSerDe = builder.openXJsonSerDe;
        this.hiveJsonSerDe = builder.hiveJsonSerDe;
    }

    /**
     * <p>
     * The OpenX SerDe. Used by Firehose for deserializing data, which means converting it from the JSON format in
     * preparation for serializing it to the Parquet or ORC format. This is one of two deserializers you can choose,
     * depending on which one offers the functionality you need. The other option is the native Hive / HCatalog
     * JsonSerDe.
     * </p>
     * 
     * @return The OpenX SerDe. Used by Firehose for deserializing data, which means converting it from the JSON format
     *         in preparation for serializing it to the Parquet or ORC format. This is one of two deserializers you can
     *         choose, depending on which one offers the functionality you need. The other option is the native Hive /
     *         HCatalog JsonSerDe.
     */
    public final OpenXJsonSerDe openXJsonSerDe() {
        return openXJsonSerDe;
    }

    /**
     * <p>
     * The native Hive / HCatalog JsonSerDe. Used by Firehose for deserializing data, which means converting it from the
     * JSON format in preparation for serializing it to the Parquet or ORC format. This is one of two deserializers you
     * can choose, depending on which one offers the functionality you need. The other option is the OpenX SerDe.
     * </p>
     * 
     * @return The native Hive / HCatalog JsonSerDe. Used by Firehose for deserializing data, which means converting it
     *         from the JSON format in preparation for serializing it to the Parquet or ORC format. This is one of two
     *         deserializers you can choose, depending on which one offers the functionality you need. The other option
     *         is the OpenX SerDe.
     */
    public final HiveJsonSerDe hiveJsonSerDe() {
        return hiveJsonSerDe;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(openXJsonSerDe());
        hashCode = 31 * hashCode + Objects.hashCode(hiveJsonSerDe());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Deserializer)) {
            return false;
        }
        Deserializer other = (Deserializer) obj;
        return Objects.equals(openXJsonSerDe(), other.openXJsonSerDe()) && Objects.equals(hiveJsonSerDe(), other.hiveJsonSerDe());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Deserializer").add("OpenXJsonSerDe", openXJsonSerDe()).add("HiveJsonSerDe", hiveJsonSerDe())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OpenXJsonSerDe":
            return Optional.ofNullable(clazz.cast(openXJsonSerDe()));
        case "HiveJsonSerDe":
            return Optional.ofNullable(clazz.cast(hiveJsonSerDe()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("OpenXJsonSerDe", OPEN_X_JSON_SER_DE_FIELD);
        map.put("HiveJsonSerDe", HIVE_JSON_SER_DE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Deserializer, T> g) {
        return obj -> g.apply((Deserializer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Deserializer> {
        /**
         * <p>
         * The OpenX SerDe. Used by Firehose for deserializing data, which means converting it from the JSON format in
         * preparation for serializing it to the Parquet or ORC format. This is one of two deserializers you can choose,
         * depending on which one offers the functionality you need. The other option is the native Hive / HCatalog
         * JsonSerDe.
         * </p>
         * 
         * @param openXJsonSerDe
         *        The OpenX SerDe. Used by Firehose for deserializing data, which means converting it from the JSON
         *        format in preparation for serializing it to the Parquet or ORC format. This is one of two
         *        deserializers you can choose, depending on which one offers the functionality you need. The other
         *        option is the native Hive / HCatalog JsonSerDe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder openXJsonSerDe(OpenXJsonSerDe openXJsonSerDe);

        /**
         * <p>
         * The OpenX SerDe. Used by Firehose for deserializing data, which means converting it from the JSON format in
         * preparation for serializing it to the Parquet or ORC format. This is one of two deserializers you can choose,
         * depending on which one offers the functionality you need. The other option is the native Hive / HCatalog
         * JsonSerDe.
         * </p>
         * This is a convenience method that creates an instance of the {@link OpenXJsonSerDe.Builder} avoiding the need
         * to create one manually via {@link OpenXJsonSerDe#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OpenXJsonSerDe.Builder#build()} is called immediately and its
         * result is passed to {@link #openXJsonSerDe(OpenXJsonSerDe)}.
         * 
         * @param openXJsonSerDe
         *        a consumer that will call methods on {@link OpenXJsonSerDe.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #openXJsonSerDe(OpenXJsonSerDe)
         */
        default Builder openXJsonSerDe(Consumer<OpenXJsonSerDe.Builder> openXJsonSerDe) {
            return openXJsonSerDe(OpenXJsonSerDe.builder().applyMutation(openXJsonSerDe).build());
        }

        /**
         * <p>
         * The native Hive / HCatalog JsonSerDe. Used by Firehose for deserializing data, which means converting it from
         * the JSON format in preparation for serializing it to the Parquet or ORC format. This is one of two
         * deserializers you can choose, depending on which one offers the functionality you need. The other option is
         * the OpenX SerDe.
         * </p>
         * 
         * @param hiveJsonSerDe
         *        The native Hive / HCatalog JsonSerDe. Used by Firehose for deserializing data, which means converting
         *        it from the JSON format in preparation for serializing it to the Parquet or ORC format. This is one of
         *        two deserializers you can choose, depending on which one offers the functionality you need. The other
         *        option is the OpenX SerDe.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hiveJsonSerDe(HiveJsonSerDe hiveJsonSerDe);

        /**
         * <p>
         * The native Hive / HCatalog JsonSerDe. Used by Firehose for deserializing data, which means converting it from
         * the JSON format in preparation for serializing it to the Parquet or ORC format. This is one of two
         * deserializers you can choose, depending on which one offers the functionality you need. The other option is
         * the OpenX SerDe.
         * </p>
         * This is a convenience method that creates an instance of the {@link HiveJsonSerDe.Builder} avoiding the need
         * to create one manually via {@link HiveJsonSerDe#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HiveJsonSerDe.Builder#build()} is called immediately and its
         * result is passed to {@link #hiveJsonSerDe(HiveJsonSerDe)}.
         * 
         * @param hiveJsonSerDe
         *        a consumer that will call methods on {@link HiveJsonSerDe.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hiveJsonSerDe(HiveJsonSerDe)
         */
        default Builder hiveJsonSerDe(Consumer<HiveJsonSerDe.Builder> hiveJsonSerDe) {
            return hiveJsonSerDe(HiveJsonSerDe.builder().applyMutation(hiveJsonSerDe).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private OpenXJsonSerDe openXJsonSerDe;

        private HiveJsonSerDe hiveJsonSerDe;

        private BuilderImpl() {
        }

        private BuilderImpl(Deserializer model) {
            openXJsonSerDe(model.openXJsonSerDe);
            hiveJsonSerDe(model.hiveJsonSerDe);
        }

        public final OpenXJsonSerDe.Builder getOpenXJsonSerDe() {
            return openXJsonSerDe != null ? openXJsonSerDe.toBuilder() : null;
        }

        public final void setOpenXJsonSerDe(OpenXJsonSerDe.BuilderImpl openXJsonSerDe) {
            this.openXJsonSerDe = openXJsonSerDe != null ? openXJsonSerDe.build() : null;
        }

        @Override
        public final Builder openXJsonSerDe(OpenXJsonSerDe openXJsonSerDe) {
            this.openXJsonSerDe = openXJsonSerDe;
            return this;
        }

        public final HiveJsonSerDe.Builder getHiveJsonSerDe() {
            return hiveJsonSerDe != null ? hiveJsonSerDe.toBuilder() : null;
        }

        public final void setHiveJsonSerDe(HiveJsonSerDe.BuilderImpl hiveJsonSerDe) {
            this.hiveJsonSerDe = hiveJsonSerDe != null ? hiveJsonSerDe.build() : null;
        }

        @Override
        public final Builder hiveJsonSerDe(HiveJsonSerDe hiveJsonSerDe) {
            this.hiveJsonSerDe = hiveJsonSerDe;
            return this;
        }

        @Override
        public Deserializer build() {
            return new Deserializer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
