/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The top level object for configuring streams with database as a source.
 * </p>
 * <p>
 * Amazon Data Firehose is in preview release and is subject to change.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DatabaseSourceConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<DatabaseSourceConfiguration.Builder, DatabaseSourceConfiguration> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DatabaseSourceConfiguration::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Endpoint").getter(getter(DatabaseSourceConfiguration::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(DatabaseSourceConfiguration::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> SSL_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SSLMode").getter(getter(DatabaseSourceConfiguration::sslModeAsString)).setter(setter(Builder::sslMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SSLMode").build()).build();

    private static final SdkField<DatabaseList> DATABASES_FIELD = SdkField.<DatabaseList> builder(MarshallingType.SDK_POJO)
            .memberName("Databases").getter(getter(DatabaseSourceConfiguration::databases)).setter(setter(Builder::databases))
            .constructor(DatabaseList::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Databases").build()).build();

    private static final SdkField<DatabaseTableList> TABLES_FIELD = SdkField
            .<DatabaseTableList> builder(MarshallingType.SDK_POJO).memberName("Tables")
            .getter(getter(DatabaseSourceConfiguration::tables)).setter(setter(Builder::tables))
            .constructor(DatabaseTableList::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tables").build()).build();

    private static final SdkField<DatabaseColumnList> COLUMNS_FIELD = SdkField
            .<DatabaseColumnList> builder(MarshallingType.SDK_POJO).memberName("Columns")
            .getter(getter(DatabaseSourceConfiguration::columns)).setter(setter(Builder::columns))
            .constructor(DatabaseColumnList::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Columns").build()).build();

    private static final SdkField<List<String>> SURROGATE_KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SurrogateKeys")
            .getter(getter(DatabaseSourceConfiguration::surrogateKeys))
            .setter(setter(Builder::surrogateKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SurrogateKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SNAPSHOT_WATERMARK_TABLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnapshotWatermarkTable").getter(getter(DatabaseSourceConfiguration::snapshotWatermarkTable))
            .setter(setter(Builder::snapshotWatermarkTable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotWatermarkTable").build())
            .build();

    private static final SdkField<DatabaseSourceAuthenticationConfiguration> DATABASE_SOURCE_AUTHENTICATION_CONFIGURATION_FIELD = SdkField
            .<DatabaseSourceAuthenticationConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("DatabaseSourceAuthenticationConfiguration")
            .getter(getter(DatabaseSourceConfiguration::databaseSourceAuthenticationConfiguration))
            .setter(setter(Builder::databaseSourceAuthenticationConfiguration))
            .constructor(DatabaseSourceAuthenticationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("DatabaseSourceAuthenticationConfiguration").build()).build();

    private static final SdkField<DatabaseSourceVPCConfiguration> DATABASE_SOURCE_VPC_CONFIGURATION_FIELD = SdkField
            .<DatabaseSourceVPCConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("DatabaseSourceVPCConfiguration")
            .getter(getter(DatabaseSourceConfiguration::databaseSourceVPCConfiguration))
            .setter(setter(Builder::databaseSourceVPCConfiguration))
            .constructor(DatabaseSourceVPCConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseSourceVPCConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, ENDPOINT_FIELD,
            PORT_FIELD, SSL_MODE_FIELD, DATABASES_FIELD, TABLES_FIELD, COLUMNS_FIELD, SURROGATE_KEYS_FIELD,
            SNAPSHOT_WATERMARK_TABLE_FIELD, DATABASE_SOURCE_AUTHENTICATION_CONFIGURATION_FIELD,
            DATABASE_SOURCE_VPC_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String endpoint;

    private final Integer port;

    private final String sslMode;

    private final DatabaseList databases;

    private final DatabaseTableList tables;

    private final DatabaseColumnList columns;

    private final List<String> surrogateKeys;

    private final String snapshotWatermarkTable;

    private final DatabaseSourceAuthenticationConfiguration databaseSourceAuthenticationConfiguration;

    private final DatabaseSourceVPCConfiguration databaseSourceVPCConfiguration;

    private DatabaseSourceConfiguration(BuilderImpl builder) {
        this.type = builder.type;
        this.endpoint = builder.endpoint;
        this.port = builder.port;
        this.sslMode = builder.sslMode;
        this.databases = builder.databases;
        this.tables = builder.tables;
        this.columns = builder.columns;
        this.surrogateKeys = builder.surrogateKeys;
        this.snapshotWatermarkTable = builder.snapshotWatermarkTable;
        this.databaseSourceAuthenticationConfiguration = builder.databaseSourceAuthenticationConfiguration;
        this.databaseSourceVPCConfiguration = builder.databaseSourceVPCConfiguration;
    }

    /**
     * <p>
     * The type of database engine. This can be one of the following values.
     * </p>
     * <ul>
     * <li>
     * <p>
     * MySQL
     * </p>
     * </li>
     * <li>
     * <p>
     * PostgreSQL
     * </p>
     * </li>
     * </ul>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DatabaseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of database engine. This can be one of the following values. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         MySQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PostgreSQL
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     * @see DatabaseType
     */
    public final DatabaseType type() {
        return DatabaseType.fromValue(type);
    }

    /**
     * <p>
     * The type of database engine. This can be one of the following values.
     * </p>
     * <ul>
     * <li>
     * <p>
     * MySQL
     * </p>
     * </li>
     * <li>
     * <p>
     * PostgreSQL
     * </p>
     * </li>
     * </ul>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DatabaseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of database engine. This can be one of the following values. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         MySQL
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PostgreSQL
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     * @see DatabaseType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The endpoint of the database server.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * 
     * @return The endpoint of the database server. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * The port of the database. This can be one of the following values.
     * </p>
     * <ul>
     * <li>
     * <p>
     * 3306 for MySQL database type
     * </p>
     * </li>
     * <li>
     * <p>
     * 5432 for PostgreSQL database type
     * </p>
     * </li>
     * </ul>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * 
     * @return The port of the database. This can be one of the following values.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         3306 for MySQL database type
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         5432 for PostgreSQL database type
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The mode to enable or disable SSL when Firehose connects to the database endpoint.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sslMode} will
     * return {@link SSLMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sslModeAsString}.
     * </p>
     * 
     * @return The mode to enable or disable SSL when Firehose connects to the database endpoint. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     * @see SSLMode
     */
    public final SSLMode sslMode() {
        return SSLMode.fromValue(sslMode);
    }

    /**
     * <p>
     * The mode to enable or disable SSL when Firehose connects to the database endpoint.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sslMode} will
     * return {@link SSLMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sslModeAsString}.
     * </p>
     * 
     * @return The mode to enable or disable SSL when Firehose connects to the database endpoint. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     * @see SSLMode
     */
    public final String sslModeAsString() {
        return sslMode;
    }

    /**
     * <p>
     * The list of database patterns in source database endpoint for Firehose to read from.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * 
     * @return The list of database patterns in source database endpoint for Firehose to read from. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final DatabaseList databases() {
        return databases;
    }

    /**
     * <p>
     * The list of table patterns in source database endpoint for Firehose to read from.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * 
     * @return The list of table patterns in source database endpoint for Firehose to read from. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final DatabaseTableList tables() {
        return tables;
    }

    /**
     * <p>
     * The list of column patterns in source database endpoint for Firehose to read from.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * 
     * @return The list of column patterns in source database endpoint for Firehose to read from. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final DatabaseColumnList columns() {
        return columns;
    }

    /**
     * For responses, this returns true if the service returned a value for the SurrogateKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSurrogateKeys() {
        return surrogateKeys != null && !(surrogateKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The optional list of table and column names used as unique key columns when taking snapshot if the tables don’t
     * have primary keys configured.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSurrogateKeys} method.
     * </p>
     * 
     * @return The optional list of table and column names used as unique key columns when taking snapshot if the tables
     *         don’t have primary keys configured. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final List<String> surrogateKeys() {
        return surrogateKeys;
    }

    /**
     * <p>
     * The fully qualified name of the table in source database endpoint that Firehose uses to track snapshot progress.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * 
     * @return The fully qualified name of the table in source database endpoint that Firehose uses to track snapshot
     *         progress. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final String snapshotWatermarkTable() {
        return snapshotWatermarkTable;
    }

    /**
     * <p>
     * The structure to configure the authentication methods for Firehose to connect to source database endpoint.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * 
     * @return The structure to configure the authentication methods for Firehose to connect to source database
     *         endpoint. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final DatabaseSourceAuthenticationConfiguration databaseSourceAuthenticationConfiguration() {
        return databaseSourceAuthenticationConfiguration;
    }

    /**
     * <p>
     * The details of the VPC Endpoint Service which Firehose uses to create a PrivateLink to the database.
     * </p>
     * <p>
     * Amazon Data Firehose is in preview release and is subject to change.
     * </p>
     * 
     * @return The details of the VPC Endpoint Service which Firehose uses to create a PrivateLink to the database. </p>
     *         <p>
     *         Amazon Data Firehose is in preview release and is subject to change.
     */
    public final DatabaseSourceVPCConfiguration databaseSourceVPCConfiguration() {
        return databaseSourceVPCConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(sslModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(databases());
        hashCode = 31 * hashCode + Objects.hashCode(tables());
        hashCode = 31 * hashCode + Objects.hashCode(columns());
        hashCode = 31 * hashCode + Objects.hashCode(hasSurrogateKeys() ? surrogateKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(snapshotWatermarkTable());
        hashCode = 31 * hashCode + Objects.hashCode(databaseSourceAuthenticationConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(databaseSourceVPCConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DatabaseSourceConfiguration)) {
            return false;
        }
        DatabaseSourceConfiguration other = (DatabaseSourceConfiguration) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(endpoint(), other.endpoint())
                && Objects.equals(port(), other.port()) && Objects.equals(sslModeAsString(), other.sslModeAsString())
                && Objects.equals(databases(), other.databases()) && Objects.equals(tables(), other.tables())
                && Objects.equals(columns(), other.columns()) && hasSurrogateKeys() == other.hasSurrogateKeys()
                && Objects.equals(surrogateKeys(), other.surrogateKeys())
                && Objects.equals(snapshotWatermarkTable(), other.snapshotWatermarkTable())
                && Objects.equals(databaseSourceAuthenticationConfiguration(), other.databaseSourceAuthenticationConfiguration())
                && Objects.equals(databaseSourceVPCConfiguration(), other.databaseSourceVPCConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DatabaseSourceConfiguration").add("Type", typeAsString()).add("Endpoint", endpoint())
                .add("Port", port()).add("SSLMode", sslModeAsString()).add("Databases", databases()).add("Tables", tables())
                .add("Columns", columns()).add("SurrogateKeys", hasSurrogateKeys() ? surrogateKeys() : null)
                .add("SnapshotWatermarkTable", snapshotWatermarkTable())
                .add("DatabaseSourceAuthenticationConfiguration", databaseSourceAuthenticationConfiguration())
                .add("DatabaseSourceVPCConfiguration", databaseSourceVPCConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "SSLMode":
            return Optional.ofNullable(clazz.cast(sslModeAsString()));
        case "Databases":
            return Optional.ofNullable(clazz.cast(databases()));
        case "Tables":
            return Optional.ofNullable(clazz.cast(tables()));
        case "Columns":
            return Optional.ofNullable(clazz.cast(columns()));
        case "SurrogateKeys":
            return Optional.ofNullable(clazz.cast(surrogateKeys()));
        case "SnapshotWatermarkTable":
            return Optional.ofNullable(clazz.cast(snapshotWatermarkTable()));
        case "DatabaseSourceAuthenticationConfiguration":
            return Optional.ofNullable(clazz.cast(databaseSourceAuthenticationConfiguration()));
        case "DatabaseSourceVPCConfiguration":
            return Optional.ofNullable(clazz.cast(databaseSourceVPCConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Type", TYPE_FIELD);
        map.put("Endpoint", ENDPOINT_FIELD);
        map.put("Port", PORT_FIELD);
        map.put("SSLMode", SSL_MODE_FIELD);
        map.put("Databases", DATABASES_FIELD);
        map.put("Tables", TABLES_FIELD);
        map.put("Columns", COLUMNS_FIELD);
        map.put("SurrogateKeys", SURROGATE_KEYS_FIELD);
        map.put("SnapshotWatermarkTable", SNAPSHOT_WATERMARK_TABLE_FIELD);
        map.put("DatabaseSourceAuthenticationConfiguration", DATABASE_SOURCE_AUTHENTICATION_CONFIGURATION_FIELD);
        map.put("DatabaseSourceVPCConfiguration", DATABASE_SOURCE_VPC_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DatabaseSourceConfiguration, T> g) {
        return obj -> g.apply((DatabaseSourceConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DatabaseSourceConfiguration> {
        /**
         * <p>
         * The type of database engine. This can be one of the following values.
         * </p>
         * <ul>
         * <li>
         * <p>
         * MySQL
         * </p>
         * </li>
         * <li>
         * <p>
         * PostgreSQL
         * </p>
         * </li>
         * </ul>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param type
         *        The type of database engine. This can be one of the following values. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        MySQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PostgreSQL
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @see DatabaseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DatabaseType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of database engine. This can be one of the following values.
         * </p>
         * <ul>
         * <li>
         * <p>
         * MySQL
         * </p>
         * </li>
         * <li>
         * <p>
         * PostgreSQL
         * </p>
         * </li>
         * </ul>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param type
         *        The type of database engine. This can be one of the following values. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        MySQL
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PostgreSQL
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @see DatabaseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DatabaseType
         */
        Builder type(DatabaseType type);

        /**
         * <p>
         * The endpoint of the database server.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param endpoint
         *        The endpoint of the database server. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * The port of the database. This can be one of the following values.
         * </p>
         * <ul>
         * <li>
         * <p>
         * 3306 for MySQL database type
         * </p>
         * </li>
         * <li>
         * <p>
         * 5432 for PostgreSQL database type
         * </p>
         * </li>
         * </ul>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param port
         *        The port of the database. This can be one of the following values.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        3306 for MySQL database type
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        5432 for PostgreSQL database type
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The mode to enable or disable SSL when Firehose connects to the database endpoint.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param sslMode
         *        The mode to enable or disable SSL when Firehose connects to the database endpoint. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @see SSLMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSLMode
         */
        Builder sslMode(String sslMode);

        /**
         * <p>
         * The mode to enable or disable SSL when Firehose connects to the database endpoint.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param sslMode
         *        The mode to enable or disable SSL when Firehose connects to the database endpoint. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @see SSLMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSLMode
         */
        Builder sslMode(SSLMode sslMode);

        /**
         * <p>
         * The list of database patterns in source database endpoint for Firehose to read from.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param databases
         *        The list of database patterns in source database endpoint for Firehose to read from. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databases(DatabaseList databases);

        /**
         * <p>
         * The list of database patterns in source database endpoint for Firehose to read from.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * This is a convenience method that creates an instance of the {@link DatabaseList.Builder} avoiding the need
         * to create one manually via {@link DatabaseList#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatabaseList.Builder#build()} is called immediately and its
         * result is passed to {@link #databases(DatabaseList)}.
         * 
         * @param databases
         *        a consumer that will call methods on {@link DatabaseList.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databases(DatabaseList)
         */
        default Builder databases(Consumer<DatabaseList.Builder> databases) {
            return databases(DatabaseList.builder().applyMutation(databases).build());
        }

        /**
         * <p>
         * The list of table patterns in source database endpoint for Firehose to read from.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param tables
         *        The list of table patterns in source database endpoint for Firehose to read from. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tables(DatabaseTableList tables);

        /**
         * <p>
         * The list of table patterns in source database endpoint for Firehose to read from.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * This is a convenience method that creates an instance of the {@link DatabaseTableList.Builder} avoiding the
         * need to create one manually via {@link DatabaseTableList#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatabaseTableList.Builder#build()} is called immediately and its
         * result is passed to {@link #tables(DatabaseTableList)}.
         * 
         * @param tables
         *        a consumer that will call methods on {@link DatabaseTableList.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tables(DatabaseTableList)
         */
        default Builder tables(Consumer<DatabaseTableList.Builder> tables) {
            return tables(DatabaseTableList.builder().applyMutation(tables).build());
        }

        /**
         * <p>
         * The list of column patterns in source database endpoint for Firehose to read from.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param columns
         *        The list of column patterns in source database endpoint for Firehose to read from. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columns(DatabaseColumnList columns);

        /**
         * <p>
         * The list of column patterns in source database endpoint for Firehose to read from.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * This is a convenience method that creates an instance of the {@link DatabaseColumnList.Builder} avoiding the
         * need to create one manually via {@link DatabaseColumnList#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatabaseColumnList.Builder#build()} is called immediately and its
         * result is passed to {@link #columns(DatabaseColumnList)}.
         * 
         * @param columns
         *        a consumer that will call methods on {@link DatabaseColumnList.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #columns(DatabaseColumnList)
         */
        default Builder columns(Consumer<DatabaseColumnList.Builder> columns) {
            return columns(DatabaseColumnList.builder().applyMutation(columns).build());
        }

        /**
         * <p>
         * The optional list of table and column names used as unique key columns when taking snapshot if the tables
         * don’t have primary keys configured.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param surrogateKeys
         *        The optional list of table and column names used as unique key columns when taking snapshot if the
         *        tables don’t have primary keys configured. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder surrogateKeys(Collection<String> surrogateKeys);

        /**
         * <p>
         * The optional list of table and column names used as unique key columns when taking snapshot if the tables
         * don’t have primary keys configured.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param surrogateKeys
         *        The optional list of table and column names used as unique key columns when taking snapshot if the
         *        tables don’t have primary keys configured. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder surrogateKeys(String... surrogateKeys);

        /**
         * <p>
         * The fully qualified name of the table in source database endpoint that Firehose uses to track snapshot
         * progress.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param snapshotWatermarkTable
         *        The fully qualified name of the table in source database endpoint that Firehose uses to track snapshot
         *        progress. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotWatermarkTable(String snapshotWatermarkTable);

        /**
         * <p>
         * The structure to configure the authentication methods for Firehose to connect to source database endpoint.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param databaseSourceAuthenticationConfiguration
         *        The structure to configure the authentication methods for Firehose to connect to source database
         *        endpoint. </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseSourceAuthenticationConfiguration(
                DatabaseSourceAuthenticationConfiguration databaseSourceAuthenticationConfiguration);

        /**
         * <p>
         * The structure to configure the authentication methods for Firehose to connect to source database endpoint.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link DatabaseSourceAuthenticationConfiguration.Builder} avoiding the need to create one manually via
         * {@link DatabaseSourceAuthenticationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatabaseSourceAuthenticationConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #databaseSourceAuthenticationConfiguration(DatabaseSourceAuthenticationConfiguration)}.
         * 
         * @param databaseSourceAuthenticationConfiguration
         *        a consumer that will call methods on {@link DatabaseSourceAuthenticationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseSourceAuthenticationConfiguration(DatabaseSourceAuthenticationConfiguration)
         */
        default Builder databaseSourceAuthenticationConfiguration(
                Consumer<DatabaseSourceAuthenticationConfiguration.Builder> databaseSourceAuthenticationConfiguration) {
            return databaseSourceAuthenticationConfiguration(DatabaseSourceAuthenticationConfiguration.builder()
                    .applyMutation(databaseSourceAuthenticationConfiguration).build());
        }

        /**
         * <p>
         * The details of the VPC Endpoint Service which Firehose uses to create a PrivateLink to the database.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * 
         * @param databaseSourceVPCConfiguration
         *        The details of the VPC Endpoint Service which Firehose uses to create a PrivateLink to the database.
         *        </p>
         *        <p>
         *        Amazon Data Firehose is in preview release and is subject to change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseSourceVPCConfiguration(DatabaseSourceVPCConfiguration databaseSourceVPCConfiguration);

        /**
         * <p>
         * The details of the VPC Endpoint Service which Firehose uses to create a PrivateLink to the database.
         * </p>
         * <p>
         * Amazon Data Firehose is in preview release and is subject to change.
         * </p>
         * This is a convenience method that creates an instance of the {@link DatabaseSourceVPCConfiguration.Builder}
         * avoiding the need to create one manually via {@link DatabaseSourceVPCConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatabaseSourceVPCConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #databaseSourceVPCConfiguration(DatabaseSourceVPCConfiguration)}.
         * 
         * @param databaseSourceVPCConfiguration
         *        a consumer that will call methods on {@link DatabaseSourceVPCConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseSourceVPCConfiguration(DatabaseSourceVPCConfiguration)
         */
        default Builder databaseSourceVPCConfiguration(
                Consumer<DatabaseSourceVPCConfiguration.Builder> databaseSourceVPCConfiguration) {
            return databaseSourceVPCConfiguration(DatabaseSourceVPCConfiguration.builder()
                    .applyMutation(databaseSourceVPCConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String endpoint;

        private Integer port;

        private String sslMode;

        private DatabaseList databases;

        private DatabaseTableList tables;

        private DatabaseColumnList columns;

        private List<String> surrogateKeys = DefaultSdkAutoConstructList.getInstance();

        private String snapshotWatermarkTable;

        private DatabaseSourceAuthenticationConfiguration databaseSourceAuthenticationConfiguration;

        private DatabaseSourceVPCConfiguration databaseSourceVPCConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(DatabaseSourceConfiguration model) {
            type(model.type);
            endpoint(model.endpoint);
            port(model.port);
            sslMode(model.sslMode);
            databases(model.databases);
            tables(model.tables);
            columns(model.columns);
            surrogateKeys(model.surrogateKeys);
            snapshotWatermarkTable(model.snapshotWatermarkTable);
            databaseSourceAuthenticationConfiguration(model.databaseSourceAuthenticationConfiguration);
            databaseSourceVPCConfiguration(model.databaseSourceVPCConfiguration);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DatabaseType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getSslMode() {
            return sslMode;
        }

        public final void setSslMode(String sslMode) {
            this.sslMode = sslMode;
        }

        @Override
        public final Builder sslMode(String sslMode) {
            this.sslMode = sslMode;
            return this;
        }

        @Override
        public final Builder sslMode(SSLMode sslMode) {
            this.sslMode(sslMode == null ? null : sslMode.toString());
            return this;
        }

        public final DatabaseList.Builder getDatabases() {
            return databases != null ? databases.toBuilder() : null;
        }

        public final void setDatabases(DatabaseList.BuilderImpl databases) {
            this.databases = databases != null ? databases.build() : null;
        }

        @Override
        public final Builder databases(DatabaseList databases) {
            this.databases = databases;
            return this;
        }

        public final DatabaseTableList.Builder getTables() {
            return tables != null ? tables.toBuilder() : null;
        }

        public final void setTables(DatabaseTableList.BuilderImpl tables) {
            this.tables = tables != null ? tables.build() : null;
        }

        @Override
        public final Builder tables(DatabaseTableList tables) {
            this.tables = tables;
            return this;
        }

        public final DatabaseColumnList.Builder getColumns() {
            return columns != null ? columns.toBuilder() : null;
        }

        public final void setColumns(DatabaseColumnList.BuilderImpl columns) {
            this.columns = columns != null ? columns.build() : null;
        }

        @Override
        public final Builder columns(DatabaseColumnList columns) {
            this.columns = columns;
            return this;
        }

        public final Collection<String> getSurrogateKeys() {
            if (surrogateKeys instanceof SdkAutoConstructList) {
                return null;
            }
            return surrogateKeys;
        }

        public final void setSurrogateKeys(Collection<String> surrogateKeys) {
            this.surrogateKeys = DatabaseSurrogateKeyListCopier.copy(surrogateKeys);
        }

        @Override
        public final Builder surrogateKeys(Collection<String> surrogateKeys) {
            this.surrogateKeys = DatabaseSurrogateKeyListCopier.copy(surrogateKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder surrogateKeys(String... surrogateKeys) {
            surrogateKeys(Arrays.asList(surrogateKeys));
            return this;
        }

        public final String getSnapshotWatermarkTable() {
            return snapshotWatermarkTable;
        }

        public final void setSnapshotWatermarkTable(String snapshotWatermarkTable) {
            this.snapshotWatermarkTable = snapshotWatermarkTable;
        }

        @Override
        public final Builder snapshotWatermarkTable(String snapshotWatermarkTable) {
            this.snapshotWatermarkTable = snapshotWatermarkTable;
            return this;
        }

        public final DatabaseSourceAuthenticationConfiguration.Builder getDatabaseSourceAuthenticationConfiguration() {
            return databaseSourceAuthenticationConfiguration != null ? databaseSourceAuthenticationConfiguration.toBuilder()
                    : null;
        }

        public final void setDatabaseSourceAuthenticationConfiguration(
                DatabaseSourceAuthenticationConfiguration.BuilderImpl databaseSourceAuthenticationConfiguration) {
            this.databaseSourceAuthenticationConfiguration = databaseSourceAuthenticationConfiguration != null ? databaseSourceAuthenticationConfiguration
                    .build() : null;
        }

        @Override
        public final Builder databaseSourceAuthenticationConfiguration(
                DatabaseSourceAuthenticationConfiguration databaseSourceAuthenticationConfiguration) {
            this.databaseSourceAuthenticationConfiguration = databaseSourceAuthenticationConfiguration;
            return this;
        }

        public final DatabaseSourceVPCConfiguration.Builder getDatabaseSourceVPCConfiguration() {
            return databaseSourceVPCConfiguration != null ? databaseSourceVPCConfiguration.toBuilder() : null;
        }

        public final void setDatabaseSourceVPCConfiguration(
                DatabaseSourceVPCConfiguration.BuilderImpl databaseSourceVPCConfiguration) {
            this.databaseSourceVPCConfiguration = databaseSourceVPCConfiguration != null ? databaseSourceVPCConfiguration.build()
                    : null;
        }

        @Override
        public final Builder databaseSourceVPCConfiguration(DatabaseSourceVPCConfiguration databaseSourceVPCConfiguration) {
            this.databaseSourceVPCConfiguration = databaseSourceVPCConfiguration;
            return this;
        }

        @Override
        public DatabaseSourceConfiguration build() {
            return new DatabaseSourceConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
