/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eventbridge.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutRuleRequest extends EventBridgeRequest implements ToCopyableBuilder<PutRuleRequest.Builder, PutRuleRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(PutRuleRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> SCHEDULE_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScheduleExpression").getter(getter(PutRuleRequest::scheduleExpression))
            .setter(setter(Builder::scheduleExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleExpression").build())
            .build();

    private static final SdkField<String> EVENT_PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventPattern").getter(getter(PutRuleRequest::eventPattern)).setter(setter(Builder::eventPattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventPattern").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(PutRuleRequest::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(PutRuleRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(PutRuleRequest::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(PutRuleRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> EVENT_BUS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventBusName").getter(getter(PutRuleRequest::eventBusName)).setter(setter(Builder::eventBusName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventBusName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            SCHEDULE_EXPRESSION_FIELD, EVENT_PATTERN_FIELD, STATE_FIELD, DESCRIPTION_FIELD, ROLE_ARN_FIELD, TAGS_FIELD,
            EVENT_BUS_NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String name;

    private final String scheduleExpression;

    private final String eventPattern;

    private final String state;

    private final String description;

    private final String roleArn;

    private final List<Tag> tags;

    private final String eventBusName;

    private PutRuleRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.scheduleExpression = builder.scheduleExpression;
        this.eventPattern = builder.eventPattern;
        this.state = builder.state;
        this.description = builder.description;
        this.roleArn = builder.roleArn;
        this.tags = builder.tags;
        this.eventBusName = builder.eventBusName;
    }

    /**
     * <p>
     * The name of the rule that you are creating or updating.
     * </p>
     * 
     * @return The name of the rule that you are creating or updating.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The scheduling expression. For example, "cron(0 20 * * ? *)" or "rate(5 minutes)".
     * </p>
     * 
     * @return The scheduling expression. For example, "cron(0 20 * * ? *)" or "rate(5 minutes)".
     */
    public final String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * The event pattern. For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-event-patterns.html">Amazon EventBridge event
     * patterns</a> in the <i> <i>Amazon EventBridge User Guide</i> </i>.
     * </p>
     * 
     * @return The event pattern. For more information, see <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-event-patterns.html">Amazon EventBridge
     *         event patterns</a> in the <i> <i>Amazon EventBridge User Guide</i> </i>.
     */
    public final String eventPattern() {
        return eventPattern;
    }

    /**
     * <p>
     * The state of the rule.
     * </p>
     * <p>
     * Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DISABLED</code>: The rule is disabled. EventBridge does not match any events against the rule.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code>: The rule is enabled. EventBridge matches events against the rule, <i>except</i> for Amazon
     * Web Services management events delivered through CloudTrail.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled for all events, including Amazon
     * Web Services management events delivered through CloudTrail.
     * </p>
     * <p>
     * Management events provide visibility into management operations that are performed on resources in your Amazon
     * Web Services account. These are also known as control plane operations. For more information, see <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events"
     * >Logging management events</a> in the <i>CloudTrail User Guide</i>, and <a href=
     * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail"
     * >Filtering management events from Amazon Web Services services</a> in the <i> <i>Amazon EventBridge User
     * Guide</i> </i>.
     * </p>
     * <p>
     * This value is only valid for rules on the <a href=
     * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses"
     * >default</a> event bus or <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom event buses</a>.
     * It does not apply to <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner
     * event buses</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RuleState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the rule.</p>
     *         <p>
     *         Valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DISABLED</code>: The rule is disabled. EventBridge does not match any events against the rule.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: The rule is enabled. EventBridge matches events against the rule, <i>except</i> for
     *         Amazon Web Services management events delivered through CloudTrail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled for all events, including
     *         Amazon Web Services management events delivered through CloudTrail.
     *         </p>
     *         <p>
     *         Management events provide visibility into management operations that are performed on resources in your
     *         Amazon Web Services account. These are also known as control plane operations. For more information, see
     *         <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events"
     *         >Logging management events</a> in the <i>CloudTrail User Guide</i>, and <a href=
     *         "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail"
     *         >Filtering management events from Amazon Web Services services</a> in the <i> <i>Amazon EventBridge User
     *         Guide</i> </i>.
     *         </p>
     *         <p>
     *         This value is only valid for rules on the <a href=
     *         "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses"
     *         >default</a> event bus or <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom event
     *         buses</a>. It does not apply to <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner event buses</a>.
     *         </p>
     *         </li>
     * @see RuleState
     */
    public final RuleState state() {
        return RuleState.fromValue(state);
    }

    /**
     * <p>
     * The state of the rule.
     * </p>
     * <p>
     * Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DISABLED</code>: The rule is disabled. EventBridge does not match any events against the rule.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code>: The rule is enabled. EventBridge matches events against the rule, <i>except</i> for Amazon
     * Web Services management events delivered through CloudTrail.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled for all events, including Amazon
     * Web Services management events delivered through CloudTrail.
     * </p>
     * <p>
     * Management events provide visibility into management operations that are performed on resources in your Amazon
     * Web Services account. These are also known as control plane operations. For more information, see <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events"
     * >Logging management events</a> in the <i>CloudTrail User Guide</i>, and <a href=
     * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail"
     * >Filtering management events from Amazon Web Services services</a> in the <i> <i>Amazon EventBridge User
     * Guide</i> </i>.
     * </p>
     * <p>
     * This value is only valid for rules on the <a href=
     * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses"
     * >default</a> event bus or <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom event buses</a>.
     * It does not apply to <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner
     * event buses</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RuleState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the rule.</p>
     *         <p>
     *         Valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DISABLED</code>: The rule is disabled. EventBridge does not match any events against the rule.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code>: The rule is enabled. EventBridge matches events against the rule, <i>except</i> for
     *         Amazon Web Services management events delivered through CloudTrail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled for all events, including
     *         Amazon Web Services management events delivered through CloudTrail.
     *         </p>
     *         <p>
     *         Management events provide visibility into management operations that are performed on resources in your
     *         Amazon Web Services account. These are also known as control plane operations. For more information, see
     *         <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events"
     *         >Logging management events</a> in the <i>CloudTrail User Guide</i>, and <a href=
     *         "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail"
     *         >Filtering management events from Amazon Web Services services</a> in the <i> <i>Amazon EventBridge User
     *         Guide</i> </i>.
     *         </p>
     *         <p>
     *         This value is only valid for rules on the <a href=
     *         "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses"
     *         >default</a> event bus or <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom event
     *         buses</a>. It does not apply to <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner event buses</a>.
     *         </p>
     *         </li>
     * @see RuleState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * A description of the rule.
     * </p>
     * 
     * @return A description of the rule.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role associated with the rule.
     * </p>
     * <p>
     * If you're setting an event bus in another account as the target and that account granted permission to your
     * account through an organization instead of directly by the account ID, you must specify a <code>RoleArn</code>
     * with proper permissions in the <code>Target</code> structure, instead of here in this parameter.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role associated with the rule.</p>
     *         <p>
     *         If you're setting an event bus in another account as the target and that account granted permission to
     *         your account through an organization instead of directly by the account ID, you must specify a
     *         <code>RoleArn</code> with proper permissions in the <code>Target</code> structure, instead of here in
     *         this parameter.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of key-value pairs to associate with the rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of key-value pairs to associate with the rule.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name or ARN of the event bus to associate with this rule. If you omit this, the default event bus is used.
     * </p>
     * 
     * @return The name or ARN of the event bus to associate with this rule. If you omit this, the default event bus is
     *         used.
     */
    public final String eventBusName() {
        return eventBusName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(eventPattern());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(eventBusName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutRuleRequest)) {
            return false;
        }
        PutRuleRequest other = (PutRuleRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(eventPattern(), other.eventPattern()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(roleArn(), other.roleArn())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(eventBusName(), other.eventBusName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutRuleRequest").add("Name", name()).add("ScheduleExpression", scheduleExpression())
                .add("EventPattern", eventPattern()).add("State", stateAsString()).add("Description", description())
                .add("RoleArn", roleArn()).add("Tags", hasTags() ? tags() : null).add("EventBusName", eventBusName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ScheduleExpression":
            return Optional.ofNullable(clazz.cast(scheduleExpression()));
        case "EventPattern":
            return Optional.ofNullable(clazz.cast(eventPattern()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "EventBusName":
            return Optional.ofNullable(clazz.cast(eventBusName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("ScheduleExpression", SCHEDULE_EXPRESSION_FIELD);
        map.put("EventPattern", EVENT_PATTERN_FIELD);
        map.put("State", STATE_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("RoleArn", ROLE_ARN_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("EventBusName", EVENT_BUS_NAME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PutRuleRequest, T> g) {
        return obj -> g.apply((PutRuleRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends EventBridgeRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutRuleRequest> {
        /**
         * <p>
         * The name of the rule that you are creating or updating.
         * </p>
         * 
         * @param name
         *        The name of the rule that you are creating or updating.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The scheduling expression. For example, "cron(0 20 * * ? *)" or "rate(5 minutes)".
         * </p>
         * 
         * @param scheduleExpression
         *        The scheduling expression. For example, "cron(0 20 * * ? *)" or "rate(5 minutes)".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * The event pattern. For more information, see <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-event-patterns.html">Amazon EventBridge
         * event patterns</a> in the <i> <i>Amazon EventBridge User Guide</i> </i>.
         * </p>
         * 
         * @param eventPattern
         *        The event pattern. For more information, see <a
         *        href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-event-patterns.html">Amazon
         *        EventBridge event patterns</a> in the <i> <i>Amazon EventBridge User Guide</i> </i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventPattern(String eventPattern);

        /**
         * <p>
         * The state of the rule.
         * </p>
         * <p>
         * Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DISABLED</code>: The rule is disabled. EventBridge does not match any events against the rule.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code>: The rule is enabled. EventBridge matches events against the rule, <i>except</i> for
         * Amazon Web Services management events delivered through CloudTrail.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled for all events, including
         * Amazon Web Services management events delivered through CloudTrail.
         * </p>
         * <p>
         * Management events provide visibility into management operations that are performed on resources in your
         * Amazon Web Services account. These are also known as control plane operations. For more information, see <a
         * href=
         * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events"
         * >Logging management events</a> in the <i>CloudTrail User Guide</i>, and <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail"
         * >Filtering management events from Amazon Web Services services</a> in the <i> <i>Amazon EventBridge User
         * Guide</i> </i>.
         * </p>
         * <p>
         * This value is only valid for rules on the <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses"
         * >default</a> event bus or <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom event
         * buses</a>. It does not apply to <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner event buses</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the rule.</p>
         *        <p>
         *        Valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DISABLED</code>: The rule is disabled. EventBridge does not match any events against the rule.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: The rule is enabled. EventBridge matches events against the rule, <i>except</i>
         *        for Amazon Web Services management events delivered through CloudTrail.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled for all events,
         *        including Amazon Web Services management events delivered through CloudTrail.
         *        </p>
         *        <p>
         *        Management events provide visibility into management operations that are performed on resources in
         *        your Amazon Web Services account. These are also known as control plane operations. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events"
         *        >Logging management events</a> in the <i>CloudTrail User Guide</i>, and <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail"
         *        >Filtering management events from Amazon Web Services services</a> in the <i> <i>Amazon EventBridge
         *        User Guide</i> </i>.
         *        </p>
         *        <p>
         *        This value is only valid for rules on the <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses"
         *        >default</a> event bus or <a
         *        href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom event
         *        buses</a>. It does not apply to <a
         *        href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner event buses</a>.
         *        </p>
         *        </li>
         * @see RuleState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RuleState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the rule.
         * </p>
         * <p>
         * Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DISABLED</code>: The rule is disabled. EventBridge does not match any events against the rule.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code>: The rule is enabled. EventBridge matches events against the rule, <i>except</i> for
         * Amazon Web Services management events delivered through CloudTrail.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled for all events, including
         * Amazon Web Services management events delivered through CloudTrail.
         * </p>
         * <p>
         * Management events provide visibility into management operations that are performed on resources in your
         * Amazon Web Services account. These are also known as control plane operations. For more information, see <a
         * href=
         * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events"
         * >Logging management events</a> in the <i>CloudTrail User Guide</i>, and <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail"
         * >Filtering management events from Amazon Web Services services</a> in the <i> <i>Amazon EventBridge User
         * Guide</i> </i>.
         * </p>
         * <p>
         * This value is only valid for rules on the <a href=
         * "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses"
         * >default</a> event bus or <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom event
         * buses</a>. It does not apply to <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner event buses</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the rule.</p>
         *        <p>
         *        Valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DISABLED</code>: The rule is disabled. EventBridge does not match any events against the rule.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code>: The rule is enabled. EventBridge matches events against the rule, <i>except</i>
         *        for Amazon Web Services management events delivered through CloudTrail.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled for all events,
         *        including Amazon Web Services management events delivered through CloudTrail.
         *        </p>
         *        <p>
         *        Management events provide visibility into management operations that are performed on resources in
         *        your Amazon Web Services account. These are also known as control plane operations. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events"
         *        >Logging management events</a> in the <i>CloudTrail User Guide</i>, and <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail"
         *        >Filtering management events from Amazon Web Services services</a> in the <i> <i>Amazon EventBridge
         *        User Guide</i> </i>.
         *        </p>
         *        <p>
         *        This value is only valid for rules on the <a href=
         *        "https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses"
         *        >default</a> event bus or <a
         *        href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom event
         *        buses</a>. It does not apply to <a
         *        href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner event buses</a>.
         *        </p>
         *        </li>
         * @see RuleState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RuleState
         */
        Builder state(RuleState state);

        /**
         * <p>
         * A description of the rule.
         * </p>
         * 
         * @param description
         *        A description of the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role associated with the rule.
         * </p>
         * <p>
         * If you're setting an event bus in another account as the target and that account granted permission to your
         * account through an organization instead of directly by the account ID, you must specify a
         * <code>RoleArn</code> with proper permissions in the <code>Target</code> structure, instead of here in this
         * parameter.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role associated with the rule.</p>
         *        <p>
         *        If you're setting an event bus in another account as the target and that account granted permission to
         *        your account through an organization instead of directly by the account ID, you must specify a
         *        <code>RoleArn</code> with proper permissions in the <code>Target</code> structure, instead of here in
         *        this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The list of key-value pairs to associate with the rule.
         * </p>
         * 
         * @param tags
         *        The list of key-value pairs to associate with the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The list of key-value pairs to associate with the rule.
         * </p>
         * 
         * @param tags
         *        The list of key-value pairs to associate with the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The list of key-value pairs to associate with the rule.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.eventbridge.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.eventbridge.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.eventbridge.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.eventbridge.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The name or ARN of the event bus to associate with this rule. If you omit this, the default event bus is
         * used.
         * </p>
         * 
         * @param eventBusName
         *        The name or ARN of the event bus to associate with this rule. If you omit this, the default event bus
         *        is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventBusName(String eventBusName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EventBridgeRequest.BuilderImpl implements Builder {
        private String name;

        private String scheduleExpression;

        private String eventPattern;

        private String state;

        private String description;

        private String roleArn;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String eventBusName;

        private BuilderImpl() {
        }

        private BuilderImpl(PutRuleRequest model) {
            super(model);
            name(model.name);
            scheduleExpression(model.scheduleExpression);
            eventPattern(model.eventPattern);
            state(model.state);
            description(model.description);
            roleArn(model.roleArn);
            tags(model.tags);
            eventBusName(model.eventBusName);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final String getEventPattern() {
            return eventPattern;
        }

        public final void setEventPattern(String eventPattern) {
            this.eventPattern = eventPattern;
        }

        @Override
        public final Builder eventPattern(String eventPattern) {
            this.eventPattern = eventPattern;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(RuleState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getEventBusName() {
            return eventBusName;
        }

        public final void setEventBusName(String eventBusName) {
            this.eventBusName = eventBusName;
        }

        @Override
        public final Builder eventBusName(String eventBusName) {
            this.eventBusName = eventBusName;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutRuleRequest build() {
            return new PutRuleRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
