/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This entity represents an instance group, which is a group of instances that have common purpose. For example, CORE
 * instance group is used for HDFS.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceGroup implements SdkPojo, Serializable, ToCopyableBuilder<InstanceGroup.Builder, InstanceGroup> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(InstanceGroup::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(InstanceGroup::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> MARKET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Market")
            .getter(getter(InstanceGroup::marketAsString)).setter(setter(Builder::market))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Market").build()).build();

    private static final SdkField<String> INSTANCE_GROUP_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceGroupType").getter(getter(InstanceGroup::instanceGroupTypeAsString))
            .setter(setter(Builder::instanceGroupType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceGroupType").build()).build();

    private static final SdkField<String> BID_PRICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BidPrice").getter(getter(InstanceGroup::bidPrice)).setter(setter(Builder::bidPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BidPrice").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(InstanceGroup::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> REQUESTED_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RequestedInstanceCount").getter(getter(InstanceGroup::requestedInstanceCount))
            .setter(setter(Builder::requestedInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedInstanceCount").build())
            .build();

    private static final SdkField<Integer> RUNNING_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RunningInstanceCount").getter(getter(InstanceGroup::runningInstanceCount))
            .setter(setter(Builder::runningInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RunningInstanceCount").build())
            .build();

    private static final SdkField<InstanceGroupStatus> STATUS_FIELD = SdkField
            .<InstanceGroupStatus> builder(MarshallingType.SDK_POJO).memberName("Status").getter(getter(InstanceGroup::status))
            .setter(setter(Builder::status)).constructor(InstanceGroupStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<Configuration>> CONFIGURATIONS_FIELD = SdkField
            .<List<Configuration>> builder(MarshallingType.LIST)
            .memberName("Configurations")
            .getter(getter(InstanceGroup::configurations))
            .setter(setter(Builder::configurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> CONFIGURATIONS_VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ConfigurationsVersion").getter(getter(InstanceGroup::configurationsVersion))
            .setter(setter(Builder::configurationsVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationsVersion").build())
            .build();

    private static final SdkField<List<Configuration>> LAST_SUCCESSFULLY_APPLIED_CONFIGURATIONS_FIELD = SdkField
            .<List<Configuration>> builder(MarshallingType.LIST)
            .memberName("LastSuccessfullyAppliedConfigurations")
            .getter(getter(InstanceGroup::lastSuccessfullyAppliedConfigurations))
            .setter(setter(Builder::lastSuccessfullyAppliedConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("LastSuccessfullyAppliedConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> LAST_SUCCESSFULLY_APPLIED_CONFIGURATIONS_VERSION_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("LastSuccessfullyAppliedConfigurationsVersion")
            .getter(getter(InstanceGroup::lastSuccessfullyAppliedConfigurationsVersion))
            .setter(setter(Builder::lastSuccessfullyAppliedConfigurationsVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("LastSuccessfullyAppliedConfigurationsVersion").build()).build();

    private static final SdkField<List<EbsBlockDevice>> EBS_BLOCK_DEVICES_FIELD = SdkField
            .<List<EbsBlockDevice>> builder(MarshallingType.LIST)
            .memberName("EbsBlockDevices")
            .getter(getter(InstanceGroup::ebsBlockDevices))
            .setter(setter(Builder::ebsBlockDevices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsBlockDevices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EbsBlockDevice> builder(MarshallingType.SDK_POJO)
                                            .constructor(EbsBlockDevice::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> EBS_OPTIMIZED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EbsOptimized").getter(getter(InstanceGroup::ebsOptimized)).setter(setter(Builder::ebsOptimized))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsOptimized").build()).build();

    private static final SdkField<ShrinkPolicy> SHRINK_POLICY_FIELD = SdkField.<ShrinkPolicy> builder(MarshallingType.SDK_POJO)
            .memberName("ShrinkPolicy").getter(getter(InstanceGroup::shrinkPolicy)).setter(setter(Builder::shrinkPolicy))
            .constructor(ShrinkPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShrinkPolicy").build()).build();

    private static final SdkField<AutoScalingPolicyDescription> AUTO_SCALING_POLICY_FIELD = SdkField
            .<AutoScalingPolicyDescription> builder(MarshallingType.SDK_POJO).memberName("AutoScalingPolicy")
            .getter(getter(InstanceGroup::autoScalingPolicy)).setter(setter(Builder::autoScalingPolicy))
            .constructor(AutoScalingPolicyDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingPolicy").build()).build();

    private static final SdkField<String> CUSTOM_AMI_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomAmiId").getter(getter(InstanceGroup::customAmiId)).setter(setter(Builder::customAmiId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomAmiId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            MARKET_FIELD, INSTANCE_GROUP_TYPE_FIELD, BID_PRICE_FIELD, INSTANCE_TYPE_FIELD, REQUESTED_INSTANCE_COUNT_FIELD,
            RUNNING_INSTANCE_COUNT_FIELD, STATUS_FIELD, CONFIGURATIONS_FIELD, CONFIGURATIONS_VERSION_FIELD,
            LAST_SUCCESSFULLY_APPLIED_CONFIGURATIONS_FIELD, LAST_SUCCESSFULLY_APPLIED_CONFIGURATIONS_VERSION_FIELD,
            EBS_BLOCK_DEVICES_FIELD, EBS_OPTIMIZED_FIELD, SHRINK_POLICY_FIELD, AUTO_SCALING_POLICY_FIELD, CUSTOM_AMI_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String market;

    private final String instanceGroupType;

    private final String bidPrice;

    private final String instanceType;

    private final Integer requestedInstanceCount;

    private final Integer runningInstanceCount;

    private final InstanceGroupStatus status;

    private final List<Configuration> configurations;

    private final Long configurationsVersion;

    private final List<Configuration> lastSuccessfullyAppliedConfigurations;

    private final Long lastSuccessfullyAppliedConfigurationsVersion;

    private final List<EbsBlockDevice> ebsBlockDevices;

    private final Boolean ebsOptimized;

    private final ShrinkPolicy shrinkPolicy;

    private final AutoScalingPolicyDescription autoScalingPolicy;

    private final String customAmiId;

    private InstanceGroup(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.market = builder.market;
        this.instanceGroupType = builder.instanceGroupType;
        this.bidPrice = builder.bidPrice;
        this.instanceType = builder.instanceType;
        this.requestedInstanceCount = builder.requestedInstanceCount;
        this.runningInstanceCount = builder.runningInstanceCount;
        this.status = builder.status;
        this.configurations = builder.configurations;
        this.configurationsVersion = builder.configurationsVersion;
        this.lastSuccessfullyAppliedConfigurations = builder.lastSuccessfullyAppliedConfigurations;
        this.lastSuccessfullyAppliedConfigurationsVersion = builder.lastSuccessfullyAppliedConfigurationsVersion;
        this.ebsBlockDevices = builder.ebsBlockDevices;
        this.ebsOptimized = builder.ebsOptimized;
        this.shrinkPolicy = builder.shrinkPolicy;
        this.autoScalingPolicy = builder.autoScalingPolicy;
        this.customAmiId = builder.customAmiId;
    }

    /**
     * <p>
     * The identifier of the instance group.
     * </p>
     * 
     * @return The identifier of the instance group.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the instance group.
     * </p>
     * 
     * @return The name of the instance group.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #market} will
     * return {@link MarketType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #marketAsString}.
     * </p>
     * 
     * @return The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
     * @see MarketType
     */
    public final MarketType market() {
        return MarketType.fromValue(market);
    }

    /**
     * <p>
     * The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #market} will
     * return {@link MarketType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #marketAsString}.
     * </p>
     * 
     * @return The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
     * @see MarketType
     */
    public final String marketAsString() {
        return market;
    }

    /**
     * <p>
     * The type of the instance group. Valid values are MASTER, CORE or TASK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceGroupType}
     * will return {@link InstanceGroupType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceGroupTypeAsString}.
     * </p>
     * 
     * @return The type of the instance group. Valid values are MASTER, CORE or TASK.
     * @see InstanceGroupType
     */
    public final InstanceGroupType instanceGroupType() {
        return InstanceGroupType.fromValue(instanceGroupType);
    }

    /**
     * <p>
     * The type of the instance group. Valid values are MASTER, CORE or TASK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceGroupType}
     * will return {@link InstanceGroupType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceGroupTypeAsString}.
     * </p>
     * 
     * @return The type of the instance group. Valid values are MASTER, CORE or TASK.
     * @see InstanceGroupType
     */
    public final String instanceGroupTypeAsString() {
        return instanceGroupType;
    }

    /**
     * <p>
     * If specified, indicates that the instance group uses Spot Instances. This is the maximum price you are willing to
     * pay for Spot Instances. Specify <code>OnDemandPrice</code> to set the amount equal to the On-Demand price, or
     * specify an amount in USD.
     * </p>
     * 
     * @return If specified, indicates that the instance group uses Spot Instances. This is the maximum price you are
     *         willing to pay for Spot Instances. Specify <code>OnDemandPrice</code> to set the amount equal to the
     *         On-Demand price, or specify an amount in USD.
     */
    public final String bidPrice() {
        return bidPrice;
    }

    /**
     * <p>
     * The Amazon EC2 instance type for all instances in the instance group.
     * </p>
     * 
     * @return The Amazon EC2 instance type for all instances in the instance group.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The target number of instances for the instance group.
     * </p>
     * 
     * @return The target number of instances for the instance group.
     */
    public final Integer requestedInstanceCount() {
        return requestedInstanceCount;
    }

    /**
     * <p>
     * The number of instances currently running in this instance group.
     * </p>
     * 
     * @return The number of instances currently running in this instance group.
     */
    public final Integer runningInstanceCount() {
        return runningInstanceCount;
    }

    /**
     * <p>
     * The current status of the instance group.
     * </p>
     * 
     * @return The current status of the instance group.
     */
    public final InstanceGroupStatus status() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Configurations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConfigurations() {
        return configurations != null && !(configurations instanceof SdkAutoConstructList);
    }

    /**
     * <note>
     * <p>
     * Amazon EMR releases 4.x or later.
     * </p>
     * </note>
     * <p>
     * The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
     * configuration for each instance group (master, core, and task).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConfigurations} method.
     * </p>
     * 
     * @return <p>
     *         Amazon EMR releases 4.x or later.
     *         </p>
     *         </note>
     *         <p>
     *         The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
     *         configuration for each instance group (master, core, and task).
     */
    public final List<Configuration> configurations() {
        return configurations;
    }

    /**
     * <p>
     * The version number of the requested configuration specification for this instance group.
     * </p>
     * 
     * @return The version number of the requested configuration specification for this instance group.
     */
    public final Long configurationsVersion() {
        return configurationsVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the LastSuccessfullyAppliedConfigurations
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasLastSuccessfullyAppliedConfigurations() {
        return lastSuccessfullyAppliedConfigurations != null
                && !(lastSuccessfullyAppliedConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of configurations that were successfully applied for an instance group last time.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLastSuccessfullyAppliedConfigurations}
     * method.
     * </p>
     * 
     * @return A list of configurations that were successfully applied for an instance group last time.
     */
    public final List<Configuration> lastSuccessfullyAppliedConfigurations() {
        return lastSuccessfullyAppliedConfigurations;
    }

    /**
     * <p>
     * The version number of a configuration specification that was successfully applied for an instance group last
     * time.
     * </p>
     * 
     * @return The version number of a configuration specification that was successfully applied for an instance group
     *         last time.
     */
    public final Long lastSuccessfullyAppliedConfigurationsVersion() {
        return lastSuccessfullyAppliedConfigurationsVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the EbsBlockDevices property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEbsBlockDevices() {
        return ebsBlockDevices != null && !(ebsBlockDevices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The EBS block devices that are mapped to this instance group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEbsBlockDevices} method.
     * </p>
     * 
     * @return The EBS block devices that are mapped to this instance group.
     */
    public final List<EbsBlockDevice> ebsBlockDevices() {
        return ebsBlockDevices;
    }

    /**
     * <p>
     * If the instance group is EBS-optimized. An Amazon EBS-optimized instance uses an optimized configuration stack
     * and provides additional, dedicated capacity for Amazon EBS I/O.
     * </p>
     * 
     * @return If the instance group is EBS-optimized. An Amazon EBS-optimized instance uses an optimized configuration
     *         stack and provides additional, dedicated capacity for Amazon EBS I/O.
     */
    public final Boolean ebsOptimized() {
        return ebsOptimized;
    }

    /**
     * <p>
     * Policy for customizing shrink operations.
     * </p>
     * 
     * @return Policy for customizing shrink operations.
     */
    public final ShrinkPolicy shrinkPolicy() {
        return shrinkPolicy;
    }

    /**
     * <p>
     * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
     * automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2 instances in
     * response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
     * </p>
     * 
     * @return An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.
     *         The automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2
     *         instances in response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
     */
    public final AutoScalingPolicyDescription autoScalingPolicy() {
        return autoScalingPolicy;
    }

    /**
     * <p>
     * The custom AMI ID to use for the provisioned instance group.
     * </p>
     * 
     * @return The custom AMI ID to use for the provisioned instance group.
     */
    public final String customAmiId() {
        return customAmiId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(marketAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceGroupTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bidPrice());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(requestedInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(runningInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfigurations() ? configurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(configurationsVersion());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasLastSuccessfullyAppliedConfigurations() ? lastSuccessfullyAppliedConfigurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastSuccessfullyAppliedConfigurationsVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasEbsBlockDevices() ? ebsBlockDevices() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ebsOptimized());
        hashCode = 31 * hashCode + Objects.hashCode(shrinkPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(customAmiId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceGroup)) {
            return false;
        }
        InstanceGroup other = (InstanceGroup) obj;
        return Objects.equals(id(), other.id())
                && Objects.equals(name(), other.name())
                && Objects.equals(marketAsString(), other.marketAsString())
                && Objects.equals(instanceGroupTypeAsString(), other.instanceGroupTypeAsString())
                && Objects.equals(bidPrice(), other.bidPrice())
                && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(requestedInstanceCount(), other.requestedInstanceCount())
                && Objects.equals(runningInstanceCount(), other.runningInstanceCount())
                && Objects.equals(status(), other.status())
                && hasConfigurations() == other.hasConfigurations()
                && Objects.equals(configurations(), other.configurations())
                && Objects.equals(configurationsVersion(), other.configurationsVersion())
                && hasLastSuccessfullyAppliedConfigurations() == other.hasLastSuccessfullyAppliedConfigurations()
                && Objects.equals(lastSuccessfullyAppliedConfigurations(), other.lastSuccessfullyAppliedConfigurations())
                && Objects.equals(lastSuccessfullyAppliedConfigurationsVersion(),
                        other.lastSuccessfullyAppliedConfigurationsVersion())
                && hasEbsBlockDevices() == other.hasEbsBlockDevices()
                && Objects.equals(ebsBlockDevices(), other.ebsBlockDevices())
                && Objects.equals(ebsOptimized(), other.ebsOptimized()) && Objects.equals(shrinkPolicy(), other.shrinkPolicy())
                && Objects.equals(autoScalingPolicy(), other.autoScalingPolicy())
                && Objects.equals(customAmiId(), other.customAmiId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("InstanceGroup")
                .add("Id", id())
                .add("Name", name())
                .add("Market", marketAsString())
                .add("InstanceGroupType", instanceGroupTypeAsString())
                .add("BidPrice", bidPrice())
                .add("InstanceType", instanceType())
                .add("RequestedInstanceCount", requestedInstanceCount())
                .add("RunningInstanceCount", runningInstanceCount())
                .add("Status", status())
                .add("Configurations", hasConfigurations() ? configurations() : null)
                .add("ConfigurationsVersion", configurationsVersion())
                .add("LastSuccessfullyAppliedConfigurations",
                        hasLastSuccessfullyAppliedConfigurations() ? lastSuccessfullyAppliedConfigurations() : null)
                .add("LastSuccessfullyAppliedConfigurationsVersion", lastSuccessfullyAppliedConfigurationsVersion())
                .add("EbsBlockDevices", hasEbsBlockDevices() ? ebsBlockDevices() : null).add("EbsOptimized", ebsOptimized())
                .add("ShrinkPolicy", shrinkPolicy()).add("AutoScalingPolicy", autoScalingPolicy())
                .add("CustomAmiId", customAmiId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Market":
            return Optional.ofNullable(clazz.cast(marketAsString()));
        case "InstanceGroupType":
            return Optional.ofNullable(clazz.cast(instanceGroupTypeAsString()));
        case "BidPrice":
            return Optional.ofNullable(clazz.cast(bidPrice()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "RequestedInstanceCount":
            return Optional.ofNullable(clazz.cast(requestedInstanceCount()));
        case "RunningInstanceCount":
            return Optional.ofNullable(clazz.cast(runningInstanceCount()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Configurations":
            return Optional.ofNullable(clazz.cast(configurations()));
        case "ConfigurationsVersion":
            return Optional.ofNullable(clazz.cast(configurationsVersion()));
        case "LastSuccessfullyAppliedConfigurations":
            return Optional.ofNullable(clazz.cast(lastSuccessfullyAppliedConfigurations()));
        case "LastSuccessfullyAppliedConfigurationsVersion":
            return Optional.ofNullable(clazz.cast(lastSuccessfullyAppliedConfigurationsVersion()));
        case "EbsBlockDevices":
            return Optional.ofNullable(clazz.cast(ebsBlockDevices()));
        case "EbsOptimized":
            return Optional.ofNullable(clazz.cast(ebsOptimized()));
        case "ShrinkPolicy":
            return Optional.ofNullable(clazz.cast(shrinkPolicy()));
        case "AutoScalingPolicy":
            return Optional.ofNullable(clazz.cast(autoScalingPolicy()));
        case "CustomAmiId":
            return Optional.ofNullable(clazz.cast(customAmiId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceGroup, T> g) {
        return obj -> g.apply((InstanceGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceGroup> {
        /**
         * <p>
         * The identifier of the instance group.
         * </p>
         * 
         * @param id
         *        The identifier of the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the instance group.
         * </p>
         * 
         * @param name
         *        The name of the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
         * </p>
         * 
         * @param market
         *        The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
         * @see MarketType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MarketType
         */
        Builder market(String market);

        /**
         * <p>
         * The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
         * </p>
         * 
         * @param market
         *        The marketplace to provision instances for this group. Valid values are ON_DEMAND or SPOT.
         * @see MarketType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MarketType
         */
        Builder market(MarketType market);

        /**
         * <p>
         * The type of the instance group. Valid values are MASTER, CORE or TASK.
         * </p>
         * 
         * @param instanceGroupType
         *        The type of the instance group. Valid values are MASTER, CORE or TASK.
         * @see InstanceGroupType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceGroupType
         */
        Builder instanceGroupType(String instanceGroupType);

        /**
         * <p>
         * The type of the instance group. Valid values are MASTER, CORE or TASK.
         * </p>
         * 
         * @param instanceGroupType
         *        The type of the instance group. Valid values are MASTER, CORE or TASK.
         * @see InstanceGroupType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceGroupType
         */
        Builder instanceGroupType(InstanceGroupType instanceGroupType);

        /**
         * <p>
         * If specified, indicates that the instance group uses Spot Instances. This is the maximum price you are
         * willing to pay for Spot Instances. Specify <code>OnDemandPrice</code> to set the amount equal to the
         * On-Demand price, or specify an amount in USD.
         * </p>
         * 
         * @param bidPrice
         *        If specified, indicates that the instance group uses Spot Instances. This is the maximum price you are
         *        willing to pay for Spot Instances. Specify <code>OnDemandPrice</code> to set the amount equal to the
         *        On-Demand price, or specify an amount in USD.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPrice(String bidPrice);

        /**
         * <p>
         * The Amazon EC2 instance type for all instances in the instance group.
         * </p>
         * 
         * @param instanceType
         *        The Amazon EC2 instance type for all instances in the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The target number of instances for the instance group.
         * </p>
         * 
         * @param requestedInstanceCount
         *        The target number of instances for the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedInstanceCount(Integer requestedInstanceCount);

        /**
         * <p>
         * The number of instances currently running in this instance group.
         * </p>
         * 
         * @param runningInstanceCount
         *        The number of instances currently running in this instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runningInstanceCount(Integer runningInstanceCount);

        /**
         * <p>
         * The current status of the instance group.
         * </p>
         * 
         * @param status
         *        The current status of the instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(InstanceGroupStatus status);

        /**
         * <p>
         * The current status of the instance group.
         * </p>
         * This is a convenience method that creates an instance of the {@link InstanceGroupStatus.Builder} avoiding the
         * need to create one manually via {@link InstanceGroupStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceGroupStatus.Builder#build()} is called immediately and
         * its result is passed to {@link #status(InstanceGroupStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link InstanceGroupStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(InstanceGroupStatus)
         */
        default Builder status(Consumer<InstanceGroupStatus.Builder> status) {
            return status(InstanceGroupStatus.builder().applyMutation(status).build());
        }

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or later.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * 
         * @param configurations
         *        <p>
         *        Amazon EMR releases 4.x or later.
         *        </p>
         *        </note>
         *        <p>
         *        The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a
         *        separate configuration for each instance group (master, core, and task).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Collection<Configuration> configurations);

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or later.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * 
         * @param configurations
         *        <p>
         *        Amazon EMR releases 4.x or later.
         *        </p>
         *        </note>
         *        <p>
         *        The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a
         *        separate configuration for each instance group (master, core, and task).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Configuration... configurations);

        /**
         * <note>
         * <p>
         * Amazon EMR releases 4.x or later.
         * </p>
         * </note>
         * <p>
         * The list of configurations supplied for an Amazon EMR cluster instance group. You can specify a separate
         * configuration for each instance group (master, core, and task).
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.emr.model.Configuration.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.emr.model.Configuration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.emr.model.Configuration.Builder#build()} is called immediately and its
         * result is passed to {@link #configurations(List<Configuration>)}.
         * 
         * @param configurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.emr.model.Configuration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurations(java.util.Collection<Configuration>)
         */
        Builder configurations(Consumer<Configuration.Builder>... configurations);

        /**
         * <p>
         * The version number of the requested configuration specification for this instance group.
         * </p>
         * 
         * @param configurationsVersion
         *        The version number of the requested configuration specification for this instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationsVersion(Long configurationsVersion);

        /**
         * <p>
         * A list of configurations that were successfully applied for an instance group last time.
         * </p>
         * 
         * @param lastSuccessfullyAppliedConfigurations
         *        A list of configurations that were successfully applied for an instance group last time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfullyAppliedConfigurations(Collection<Configuration> lastSuccessfullyAppliedConfigurations);

        /**
         * <p>
         * A list of configurations that were successfully applied for an instance group last time.
         * </p>
         * 
         * @param lastSuccessfullyAppliedConfigurations
         *        A list of configurations that were successfully applied for an instance group last time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfullyAppliedConfigurations(Configuration... lastSuccessfullyAppliedConfigurations);

        /**
         * <p>
         * A list of configurations that were successfully applied for an instance group last time.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.emr.model.Configuration.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.emr.model.Configuration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.emr.model.Configuration.Builder#build()} is called immediately and its
         * result is passed to {@link #lastSuccessfullyAppliedConfigurations(List<Configuration>)}.
         * 
         * @param lastSuccessfullyAppliedConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.emr.model.Configuration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lastSuccessfullyAppliedConfigurations(java.util.Collection<Configuration>)
         */
        Builder lastSuccessfullyAppliedConfigurations(Consumer<Configuration.Builder>... lastSuccessfullyAppliedConfigurations);

        /**
         * <p>
         * The version number of a configuration specification that was successfully applied for an instance group last
         * time.
         * </p>
         * 
         * @param lastSuccessfullyAppliedConfigurationsVersion
         *        The version number of a configuration specification that was successfully applied for an instance
         *        group last time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSuccessfullyAppliedConfigurationsVersion(Long lastSuccessfullyAppliedConfigurationsVersion);

        /**
         * <p>
         * The EBS block devices that are mapped to this instance group.
         * </p>
         * 
         * @param ebsBlockDevices
         *        The EBS block devices that are mapped to this instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsBlockDevices(Collection<EbsBlockDevice> ebsBlockDevices);

        /**
         * <p>
         * The EBS block devices that are mapped to this instance group.
         * </p>
         * 
         * @param ebsBlockDevices
         *        The EBS block devices that are mapped to this instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsBlockDevices(EbsBlockDevice... ebsBlockDevices);

        /**
         * <p>
         * The EBS block devices that are mapped to this instance group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.emr.model.EbsBlockDevice.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.emr.model.EbsBlockDevice#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.emr.model.EbsBlockDevice.Builder#build()} is called immediately and
         * its result is passed to {@link #ebsBlockDevices(List<EbsBlockDevice>)}.
         * 
         * @param ebsBlockDevices
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.emr.model.EbsBlockDevice.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsBlockDevices(java.util.Collection<EbsBlockDevice>)
         */
        Builder ebsBlockDevices(Consumer<EbsBlockDevice.Builder>... ebsBlockDevices);

        /**
         * <p>
         * If the instance group is EBS-optimized. An Amazon EBS-optimized instance uses an optimized configuration
         * stack and provides additional, dedicated capacity for Amazon EBS I/O.
         * </p>
         * 
         * @param ebsOptimized
         *        If the instance group is EBS-optimized. An Amazon EBS-optimized instance uses an optimized
         *        configuration stack and provides additional, dedicated capacity for Amazon EBS I/O.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsOptimized(Boolean ebsOptimized);

        /**
         * <p>
         * Policy for customizing shrink operations.
         * </p>
         * 
         * @param shrinkPolicy
         *        Policy for customizing shrink operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shrinkPolicy(ShrinkPolicy shrinkPolicy);

        /**
         * <p>
         * Policy for customizing shrink operations.
         * </p>
         * This is a convenience method that creates an instance of the {@link ShrinkPolicy.Builder} avoiding the need
         * to create one manually via {@link ShrinkPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ShrinkPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #shrinkPolicy(ShrinkPolicy)}.
         * 
         * @param shrinkPolicy
         *        a consumer that will call methods on {@link ShrinkPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #shrinkPolicy(ShrinkPolicy)
         */
        default Builder shrinkPolicy(Consumer<ShrinkPolicy.Builder> shrinkPolicy) {
            return shrinkPolicy(ShrinkPolicy.builder().applyMutation(shrinkPolicy).build());
        }

        /**
         * <p>
         * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
         * automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2 instances
         * in response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
         * </p>
         * 
         * @param autoScalingPolicy
         *        An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.
         *        The automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2
         *        instances in response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingPolicy(AutoScalingPolicyDescription autoScalingPolicy);

        /**
         * <p>
         * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The
         * automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2 instances
         * in response to the value of a CloudWatch metric. See PutAutoScalingPolicy.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoScalingPolicyDescription.Builder}
         * avoiding the need to create one manually via {@link AutoScalingPolicyDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoScalingPolicyDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #autoScalingPolicy(AutoScalingPolicyDescription)}.
         * 
         * @param autoScalingPolicy
         *        a consumer that will call methods on {@link AutoScalingPolicyDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoScalingPolicy(AutoScalingPolicyDescription)
         */
        default Builder autoScalingPolicy(Consumer<AutoScalingPolicyDescription.Builder> autoScalingPolicy) {
            return autoScalingPolicy(AutoScalingPolicyDescription.builder().applyMutation(autoScalingPolicy).build());
        }

        /**
         * <p>
         * The custom AMI ID to use for the provisioned instance group.
         * </p>
         * 
         * @param customAmiId
         *        The custom AMI ID to use for the provisioned instance group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customAmiId(String customAmiId);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String market;

        private String instanceGroupType;

        private String bidPrice;

        private String instanceType;

        private Integer requestedInstanceCount;

        private Integer runningInstanceCount;

        private InstanceGroupStatus status;

        private List<Configuration> configurations = DefaultSdkAutoConstructList.getInstance();

        private Long configurationsVersion;

        private List<Configuration> lastSuccessfullyAppliedConfigurations = DefaultSdkAutoConstructList.getInstance();

        private Long lastSuccessfullyAppliedConfigurationsVersion;

        private List<EbsBlockDevice> ebsBlockDevices = DefaultSdkAutoConstructList.getInstance();

        private Boolean ebsOptimized;

        private ShrinkPolicy shrinkPolicy;

        private AutoScalingPolicyDescription autoScalingPolicy;

        private String customAmiId;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceGroup model) {
            id(model.id);
            name(model.name);
            market(model.market);
            instanceGroupType(model.instanceGroupType);
            bidPrice(model.bidPrice);
            instanceType(model.instanceType);
            requestedInstanceCount(model.requestedInstanceCount);
            runningInstanceCount(model.runningInstanceCount);
            status(model.status);
            configurations(model.configurations);
            configurationsVersion(model.configurationsVersion);
            lastSuccessfullyAppliedConfigurations(model.lastSuccessfullyAppliedConfigurations);
            lastSuccessfullyAppliedConfigurationsVersion(model.lastSuccessfullyAppliedConfigurationsVersion);
            ebsBlockDevices(model.ebsBlockDevices);
            ebsOptimized(model.ebsOptimized);
            shrinkPolicy(model.shrinkPolicy);
            autoScalingPolicy(model.autoScalingPolicy);
            customAmiId(model.customAmiId);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getMarket() {
            return market;
        }

        public final void setMarket(String market) {
            this.market = market;
        }

        @Override
        public final Builder market(String market) {
            this.market = market;
            return this;
        }

        @Override
        public final Builder market(MarketType market) {
            this.market(market == null ? null : market.toString());
            return this;
        }

        public final String getInstanceGroupType() {
            return instanceGroupType;
        }

        public final void setInstanceGroupType(String instanceGroupType) {
            this.instanceGroupType = instanceGroupType;
        }

        @Override
        public final Builder instanceGroupType(String instanceGroupType) {
            this.instanceGroupType = instanceGroupType;
            return this;
        }

        @Override
        public final Builder instanceGroupType(InstanceGroupType instanceGroupType) {
            this.instanceGroupType(instanceGroupType == null ? null : instanceGroupType.toString());
            return this;
        }

        public final String getBidPrice() {
            return bidPrice;
        }

        public final void setBidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
        }

        @Override
        public final Builder bidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final Integer getRequestedInstanceCount() {
            return requestedInstanceCount;
        }

        public final void setRequestedInstanceCount(Integer requestedInstanceCount) {
            this.requestedInstanceCount = requestedInstanceCount;
        }

        @Override
        public final Builder requestedInstanceCount(Integer requestedInstanceCount) {
            this.requestedInstanceCount = requestedInstanceCount;
            return this;
        }

        public final Integer getRunningInstanceCount() {
            return runningInstanceCount;
        }

        public final void setRunningInstanceCount(Integer runningInstanceCount) {
            this.runningInstanceCount = runningInstanceCount;
        }

        @Override
        public final Builder runningInstanceCount(Integer runningInstanceCount) {
            this.runningInstanceCount = runningInstanceCount;
            return this;
        }

        public final InstanceGroupStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(InstanceGroupStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(InstanceGroupStatus status) {
            this.status = status;
            return this;
        }

        public final List<Configuration.Builder> getConfigurations() {
            List<Configuration.Builder> result = ConfigurationListCopier.copyToBuilder(this.configurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setConfigurations(Collection<Configuration.BuilderImpl> configurations) {
            this.configurations = ConfigurationListCopier.copyFromBuilder(configurations);
        }

        @Override
        public final Builder configurations(Collection<Configuration> configurations) {
            this.configurations = ConfigurationListCopier.copy(configurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Configuration... configurations) {
            configurations(Arrays.asList(configurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Consumer<Configuration.Builder>... configurations) {
            configurations(Stream.of(configurations).map(c -> Configuration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Long getConfigurationsVersion() {
            return configurationsVersion;
        }

        public final void setConfigurationsVersion(Long configurationsVersion) {
            this.configurationsVersion = configurationsVersion;
        }

        @Override
        public final Builder configurationsVersion(Long configurationsVersion) {
            this.configurationsVersion = configurationsVersion;
            return this;
        }

        public final List<Configuration.Builder> getLastSuccessfullyAppliedConfigurations() {
            List<Configuration.Builder> result = ConfigurationListCopier
                    .copyToBuilder(this.lastSuccessfullyAppliedConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLastSuccessfullyAppliedConfigurations(
                Collection<Configuration.BuilderImpl> lastSuccessfullyAppliedConfigurations) {
            this.lastSuccessfullyAppliedConfigurations = ConfigurationListCopier
                    .copyFromBuilder(lastSuccessfullyAppliedConfigurations);
        }

        @Override
        public final Builder lastSuccessfullyAppliedConfigurations(Collection<Configuration> lastSuccessfullyAppliedConfigurations) {
            this.lastSuccessfullyAppliedConfigurations = ConfigurationListCopier.copy(lastSuccessfullyAppliedConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lastSuccessfullyAppliedConfigurations(Configuration... lastSuccessfullyAppliedConfigurations) {
            lastSuccessfullyAppliedConfigurations(Arrays.asList(lastSuccessfullyAppliedConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lastSuccessfullyAppliedConfigurations(
                Consumer<Configuration.Builder>... lastSuccessfullyAppliedConfigurations) {
            lastSuccessfullyAppliedConfigurations(Stream.of(lastSuccessfullyAppliedConfigurations)
                    .map(c -> Configuration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Long getLastSuccessfullyAppliedConfigurationsVersion() {
            return lastSuccessfullyAppliedConfigurationsVersion;
        }

        public final void setLastSuccessfullyAppliedConfigurationsVersion(Long lastSuccessfullyAppliedConfigurationsVersion) {
            this.lastSuccessfullyAppliedConfigurationsVersion = lastSuccessfullyAppliedConfigurationsVersion;
        }

        @Override
        public final Builder lastSuccessfullyAppliedConfigurationsVersion(Long lastSuccessfullyAppliedConfigurationsVersion) {
            this.lastSuccessfullyAppliedConfigurationsVersion = lastSuccessfullyAppliedConfigurationsVersion;
            return this;
        }

        public final List<EbsBlockDevice.Builder> getEbsBlockDevices() {
            List<EbsBlockDevice.Builder> result = EbsBlockDeviceListCopier.copyToBuilder(this.ebsBlockDevices);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEbsBlockDevices(Collection<EbsBlockDevice.BuilderImpl> ebsBlockDevices) {
            this.ebsBlockDevices = EbsBlockDeviceListCopier.copyFromBuilder(ebsBlockDevices);
        }

        @Override
        public final Builder ebsBlockDevices(Collection<EbsBlockDevice> ebsBlockDevices) {
            this.ebsBlockDevices = EbsBlockDeviceListCopier.copy(ebsBlockDevices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsBlockDevices(EbsBlockDevice... ebsBlockDevices) {
            ebsBlockDevices(Arrays.asList(ebsBlockDevices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsBlockDevices(Consumer<EbsBlockDevice.Builder>... ebsBlockDevices) {
            ebsBlockDevices(Stream.of(ebsBlockDevices).map(c -> EbsBlockDevice.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getEbsOptimized() {
            return ebsOptimized;
        }

        public final void setEbsOptimized(Boolean ebsOptimized) {
            this.ebsOptimized = ebsOptimized;
        }

        @Override
        public final Builder ebsOptimized(Boolean ebsOptimized) {
            this.ebsOptimized = ebsOptimized;
            return this;
        }

        public final ShrinkPolicy.Builder getShrinkPolicy() {
            return shrinkPolicy != null ? shrinkPolicy.toBuilder() : null;
        }

        public final void setShrinkPolicy(ShrinkPolicy.BuilderImpl shrinkPolicy) {
            this.shrinkPolicy = shrinkPolicy != null ? shrinkPolicy.build() : null;
        }

        @Override
        public final Builder shrinkPolicy(ShrinkPolicy shrinkPolicy) {
            this.shrinkPolicy = shrinkPolicy;
            return this;
        }

        public final AutoScalingPolicyDescription.Builder getAutoScalingPolicy() {
            return autoScalingPolicy != null ? autoScalingPolicy.toBuilder() : null;
        }

        public final void setAutoScalingPolicy(AutoScalingPolicyDescription.BuilderImpl autoScalingPolicy) {
            this.autoScalingPolicy = autoScalingPolicy != null ? autoScalingPolicy.build() : null;
        }

        @Override
        public final Builder autoScalingPolicy(AutoScalingPolicyDescription autoScalingPolicy) {
            this.autoScalingPolicy = autoScalingPolicy;
            return this;
        }

        public final String getCustomAmiId() {
            return customAmiId;
        }

        public final void setCustomAmiId(String customAmiId) {
            this.customAmiId = customAmiId;
        }

        @Override
        public final Builder customAmiId(String customAmiId) {
            this.customAmiId = customAmiId;
            return this;
        }

        @Override
        public InstanceGroup build() {
            return new InstanceGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
