/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the Amazon EC2 instances in a cluster grouped by category. For example, key name, subnet
 * ID, IAM instance profile, and so on.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Ec2InstanceAttributes implements SdkPojo, Serializable,
        ToCopyableBuilder<Ec2InstanceAttributes.Builder, Ec2InstanceAttributes> {
    private static final SdkField<String> EC2_KEY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Ec2KeyName").getter(getter(Ec2InstanceAttributes::ec2KeyName)).setter(setter(Builder::ec2KeyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ec2KeyName").build()).build();

    private static final SdkField<String> EC2_SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Ec2SubnetId").getter(getter(Ec2InstanceAttributes::ec2SubnetId)).setter(setter(Builder::ec2SubnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ec2SubnetId").build()).build();

    private static final SdkField<List<String>> REQUESTED_EC2_SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RequestedEc2SubnetIds")
            .getter(getter(Ec2InstanceAttributes::requestedEc2SubnetIds))
            .setter(setter(Builder::requestedEc2SubnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedEc2SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> EC2_AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Ec2AvailabilityZone").getter(getter(Ec2InstanceAttributes::ec2AvailabilityZone))
            .setter(setter(Builder::ec2AvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ec2AvailabilityZone").build())
            .build();

    private static final SdkField<List<String>> REQUESTED_EC2_AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RequestedEc2AvailabilityZones")
            .getter(getter(Ec2InstanceAttributes::requestedEc2AvailabilityZones))
            .setter(setter(Builder::requestedEc2AvailabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedEc2AvailabilityZones")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IAM_INSTANCE_PROFILE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamInstanceProfile").getter(getter(Ec2InstanceAttributes::iamInstanceProfile))
            .setter(setter(Builder::iamInstanceProfile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamInstanceProfile").build())
            .build();

    private static final SdkField<String> EMR_MANAGED_MASTER_SECURITY_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EmrManagedMasterSecurityGroup")
            .getter(getter(Ec2InstanceAttributes::emrManagedMasterSecurityGroup))
            .setter(setter(Builder::emrManagedMasterSecurityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmrManagedMasterSecurityGroup")
                    .build()).build();

    private static final SdkField<String> EMR_MANAGED_SLAVE_SECURITY_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EmrManagedSlaveSecurityGroup")
            .getter(getter(Ec2InstanceAttributes::emrManagedSlaveSecurityGroup))
            .setter(setter(Builder::emrManagedSlaveSecurityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmrManagedSlaveSecurityGroup")
                    .build()).build();

    private static final SdkField<String> SERVICE_ACCESS_SECURITY_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ServiceAccessSecurityGroup")
            .getter(getter(Ec2InstanceAttributes::serviceAccessSecurityGroup))
            .setter(setter(Builder::serviceAccessSecurityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceAccessSecurityGroup").build())
            .build();

    private static final SdkField<List<String>> ADDITIONAL_MASTER_SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AdditionalMasterSecurityGroups")
            .getter(getter(Ec2InstanceAttributes::additionalMasterSecurityGroups))
            .setter(setter(Builder::additionalMasterSecurityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalMasterSecurityGroups")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ADDITIONAL_SLAVE_SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AdditionalSlaveSecurityGroups")
            .getter(getter(Ec2InstanceAttributes::additionalSlaveSecurityGroups))
            .setter(setter(Builder::additionalSlaveSecurityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalSlaveSecurityGroups")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EC2_KEY_NAME_FIELD,
            EC2_SUBNET_ID_FIELD, REQUESTED_EC2_SUBNET_IDS_FIELD, EC2_AVAILABILITY_ZONE_FIELD,
            REQUESTED_EC2_AVAILABILITY_ZONES_FIELD, IAM_INSTANCE_PROFILE_FIELD, EMR_MANAGED_MASTER_SECURITY_GROUP_FIELD,
            EMR_MANAGED_SLAVE_SECURITY_GROUP_FIELD, SERVICE_ACCESS_SECURITY_GROUP_FIELD, ADDITIONAL_MASTER_SECURITY_GROUPS_FIELD,
            ADDITIONAL_SLAVE_SECURITY_GROUPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ec2KeyName;

    private final String ec2SubnetId;

    private final List<String> requestedEc2SubnetIds;

    private final String ec2AvailabilityZone;

    private final List<String> requestedEc2AvailabilityZones;

    private final String iamInstanceProfile;

    private final String emrManagedMasterSecurityGroup;

    private final String emrManagedSlaveSecurityGroup;

    private final String serviceAccessSecurityGroup;

    private final List<String> additionalMasterSecurityGroups;

    private final List<String> additionalSlaveSecurityGroups;

    private Ec2InstanceAttributes(BuilderImpl builder) {
        this.ec2KeyName = builder.ec2KeyName;
        this.ec2SubnetId = builder.ec2SubnetId;
        this.requestedEc2SubnetIds = builder.requestedEc2SubnetIds;
        this.ec2AvailabilityZone = builder.ec2AvailabilityZone;
        this.requestedEc2AvailabilityZones = builder.requestedEc2AvailabilityZones;
        this.iamInstanceProfile = builder.iamInstanceProfile;
        this.emrManagedMasterSecurityGroup = builder.emrManagedMasterSecurityGroup;
        this.emrManagedSlaveSecurityGroup = builder.emrManagedSlaveSecurityGroup;
        this.serviceAccessSecurityGroup = builder.serviceAccessSecurityGroup;
        this.additionalMasterSecurityGroups = builder.additionalMasterSecurityGroups;
        this.additionalSlaveSecurityGroups = builder.additionalSlaveSecurityGroups;
    }

    /**
     * <p>
     * The name of the Amazon EC2 key pair to use when connecting with SSH into the master node as a user named
     * "hadoop".
     * </p>
     * 
     * @return The name of the Amazon EC2 key pair to use when connecting with SSH into the master node as a user named
     *         "hadoop".
     */
    public final String ec2KeyName() {
        return ec2KeyName;
    }

    /**
     * <p>
     * Set this parameter to the identifier of the Amazon VPC subnet where you want the cluster to launch. If you do not
     * specify this value, and your account supports EC2-Classic, the cluster launches in EC2-Classic.
     * </p>
     * 
     * @return Set this parameter to the identifier of the Amazon VPC subnet where you want the cluster to launch. If
     *         you do not specify this value, and your account supports EC2-Classic, the cluster launches in
     *         EC2-Classic.
     */
    public final String ec2SubnetId() {
        return ec2SubnetId;
    }

    /**
     * For responses, this returns true if the service returned a value for the RequestedEc2SubnetIds property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRequestedEc2SubnetIds() {
        return requestedEc2SubnetIds != null && !(requestedEc2SubnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Applies to clusters configured with the instance fleets option. Specifies the unique identifier of one or more
     * Amazon EC2 subnets in which to launch Amazon EC2 cluster instances. Subnets must exist within the same VPC.
     * Amazon EMR chooses the Amazon EC2 subnet with the best fit from among the list of
     * <code>RequestedEc2SubnetIds</code>, and then launches all cluster instances within that Subnet. If this value is
     * not specified, and the account and Region support EC2-Classic networks, the cluster launches instances in the
     * EC2-Classic network and uses <code>RequestedEc2AvailabilityZones</code> instead of this setting. If EC2-Classic
     * is not supported, and no Subnet is specified, Amazon EMR chooses the subnet for you.
     * <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be specified together.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequestedEc2SubnetIds} method.
     * </p>
     * 
     * @return Applies to clusters configured with the instance fleets option. Specifies the unique identifier of one or
     *         more Amazon EC2 subnets in which to launch Amazon EC2 cluster instances. Subnets must exist within the
     *         same VPC. Amazon EMR chooses the Amazon EC2 subnet with the best fit from among the list of
     *         <code>RequestedEc2SubnetIds</code>, and then launches all cluster instances within that Subnet. If this
     *         value is not specified, and the account and Region support EC2-Classic networks, the cluster launches
     *         instances in the EC2-Classic network and uses <code>RequestedEc2AvailabilityZones</code> instead of this
     *         setting. If EC2-Classic is not supported, and no Subnet is specified, Amazon EMR chooses the subnet for
     *         you. <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be
     *         specified together.
     */
    public final List<String> requestedEc2SubnetIds() {
        return requestedEc2SubnetIds;
    }

    /**
     * <p>
     * The Availability Zone in which the cluster will run.
     * </p>
     * 
     * @return The Availability Zone in which the cluster will run.
     */
    public final String ec2AvailabilityZone() {
        return ec2AvailabilityZone;
    }

    /**
     * For responses, this returns true if the service returned a value for the RequestedEc2AvailabilityZones property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRequestedEc2AvailabilityZones() {
        return requestedEc2AvailabilityZones != null && !(requestedEc2AvailabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Applies to clusters configured with the instance fleets option. Specifies one or more Availability Zones in which
     * to launch Amazon EC2 cluster instances when the EC2-Classic network configuration is supported. Amazon EMR
     * chooses the Availability Zone with the best fit from among the list of <code>RequestedEc2AvailabilityZones</code>
     * , and then launches all cluster instances within that Availability Zone. If you do not specify this value, Amazon
     * EMR chooses the Availability Zone for you. <code>RequestedEc2SubnetIDs</code> and
     * <code>RequestedEc2AvailabilityZones</code> cannot be specified together.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequestedEc2AvailabilityZones} method.
     * </p>
     * 
     * @return Applies to clusters configured with the instance fleets option. Specifies one or more Availability Zones
     *         in which to launch Amazon EC2 cluster instances when the EC2-Classic network configuration is supported.
     *         Amazon EMR chooses the Availability Zone with the best fit from among the list of
     *         <code>RequestedEc2AvailabilityZones</code>, and then launches all cluster instances within that
     *         Availability Zone. If you do not specify this value, Amazon EMR chooses the Availability Zone for you.
     *         <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be specified
     *         together.
     */
    public final List<String> requestedEc2AvailabilityZones() {
        return requestedEc2AvailabilityZones;
    }

    /**
     * <p>
     * The IAM role that was specified when the cluster was launched. The Amazon EC2 instances of the cluster assume
     * this role.
     * </p>
     * 
     * @return The IAM role that was specified when the cluster was launched. The Amazon EC2 instances of the cluster
     *         assume this role.
     */
    public final String iamInstanceProfile() {
        return iamInstanceProfile;
    }

    /**
     * <p>
     * The identifier of the Amazon EC2 security group for the master node.
     * </p>
     * 
     * @return The identifier of the Amazon EC2 security group for the master node.
     */
    public final String emrManagedMasterSecurityGroup() {
        return emrManagedMasterSecurityGroup;
    }

    /**
     * <p>
     * The identifier of the Amazon EC2 security group for the core and task nodes.
     * </p>
     * 
     * @return The identifier of the Amazon EC2 security group for the core and task nodes.
     */
    public final String emrManagedSlaveSecurityGroup() {
        return emrManagedSlaveSecurityGroup;
    }

    /**
     * <p>
     * The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC private
     * subnets.
     * </p>
     * 
     * @return The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC
     *         private subnets.
     */
    public final String serviceAccessSecurityGroup() {
        return serviceAccessSecurityGroup;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalMasterSecurityGroups property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAdditionalMasterSecurityGroups() {
        return additionalMasterSecurityGroups != null && !(additionalMasterSecurityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of additional Amazon EC2 security group IDs for the master node.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalMasterSecurityGroups} method.
     * </p>
     * 
     * @return A list of additional Amazon EC2 security group IDs for the master node.
     */
    public final List<String> additionalMasterSecurityGroups() {
        return additionalMasterSecurityGroups;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalSlaveSecurityGroups property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAdditionalSlaveSecurityGroups() {
        return additionalSlaveSecurityGroups != null && !(additionalSlaveSecurityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of additional Amazon EC2 security group IDs for the core and task nodes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalSlaveSecurityGroups} method.
     * </p>
     * 
     * @return A list of additional Amazon EC2 security group IDs for the core and task nodes.
     */
    public final List<String> additionalSlaveSecurityGroups() {
        return additionalSlaveSecurityGroups;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ec2KeyName());
        hashCode = 31 * hashCode + Objects.hashCode(ec2SubnetId());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequestedEc2SubnetIds() ? requestedEc2SubnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ec2AvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequestedEc2AvailabilityZones() ? requestedEc2AvailabilityZones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(iamInstanceProfile());
        hashCode = 31 * hashCode + Objects.hashCode(emrManagedMasterSecurityGroup());
        hashCode = 31 * hashCode + Objects.hashCode(emrManagedSlaveSecurityGroup());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAccessSecurityGroup());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasAdditionalMasterSecurityGroups() ? additionalMasterSecurityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAdditionalSlaveSecurityGroups() ? additionalSlaveSecurityGroups() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Ec2InstanceAttributes)) {
            return false;
        }
        Ec2InstanceAttributes other = (Ec2InstanceAttributes) obj;
        return Objects.equals(ec2KeyName(), other.ec2KeyName()) && Objects.equals(ec2SubnetId(), other.ec2SubnetId())
                && hasRequestedEc2SubnetIds() == other.hasRequestedEc2SubnetIds()
                && Objects.equals(requestedEc2SubnetIds(), other.requestedEc2SubnetIds())
                && Objects.equals(ec2AvailabilityZone(), other.ec2AvailabilityZone())
                && hasRequestedEc2AvailabilityZones() == other.hasRequestedEc2AvailabilityZones()
                && Objects.equals(requestedEc2AvailabilityZones(), other.requestedEc2AvailabilityZones())
                && Objects.equals(iamInstanceProfile(), other.iamInstanceProfile())
                && Objects.equals(emrManagedMasterSecurityGroup(), other.emrManagedMasterSecurityGroup())
                && Objects.equals(emrManagedSlaveSecurityGroup(), other.emrManagedSlaveSecurityGroup())
                && Objects.equals(serviceAccessSecurityGroup(), other.serviceAccessSecurityGroup())
                && hasAdditionalMasterSecurityGroups() == other.hasAdditionalMasterSecurityGroups()
                && Objects.equals(additionalMasterSecurityGroups(), other.additionalMasterSecurityGroups())
                && hasAdditionalSlaveSecurityGroups() == other.hasAdditionalSlaveSecurityGroups()
                && Objects.equals(additionalSlaveSecurityGroups(), other.additionalSlaveSecurityGroups());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("Ec2InstanceAttributes")
                .add("Ec2KeyName", ec2KeyName())
                .add("Ec2SubnetId", ec2SubnetId())
                .add("RequestedEc2SubnetIds", hasRequestedEc2SubnetIds() ? requestedEc2SubnetIds() : null)
                .add("Ec2AvailabilityZone", ec2AvailabilityZone())
                .add("RequestedEc2AvailabilityZones", hasRequestedEc2AvailabilityZones() ? requestedEc2AvailabilityZones() : null)
                .add("IamInstanceProfile", iamInstanceProfile())
                .add("EmrManagedMasterSecurityGroup", emrManagedMasterSecurityGroup())
                .add("EmrManagedSlaveSecurityGroup", emrManagedSlaveSecurityGroup())
                .add("ServiceAccessSecurityGroup", serviceAccessSecurityGroup())
                .add("AdditionalMasterSecurityGroups",
                        hasAdditionalMasterSecurityGroups() ? additionalMasterSecurityGroups() : null)
                .add("AdditionalSlaveSecurityGroups", hasAdditionalSlaveSecurityGroups() ? additionalSlaveSecurityGroups() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Ec2KeyName":
            return Optional.ofNullable(clazz.cast(ec2KeyName()));
        case "Ec2SubnetId":
            return Optional.ofNullable(clazz.cast(ec2SubnetId()));
        case "RequestedEc2SubnetIds":
            return Optional.ofNullable(clazz.cast(requestedEc2SubnetIds()));
        case "Ec2AvailabilityZone":
            return Optional.ofNullable(clazz.cast(ec2AvailabilityZone()));
        case "RequestedEc2AvailabilityZones":
            return Optional.ofNullable(clazz.cast(requestedEc2AvailabilityZones()));
        case "IamInstanceProfile":
            return Optional.ofNullable(clazz.cast(iamInstanceProfile()));
        case "EmrManagedMasterSecurityGroup":
            return Optional.ofNullable(clazz.cast(emrManagedMasterSecurityGroup()));
        case "EmrManagedSlaveSecurityGroup":
            return Optional.ofNullable(clazz.cast(emrManagedSlaveSecurityGroup()));
        case "ServiceAccessSecurityGroup":
            return Optional.ofNullable(clazz.cast(serviceAccessSecurityGroup()));
        case "AdditionalMasterSecurityGroups":
            return Optional.ofNullable(clazz.cast(additionalMasterSecurityGroups()));
        case "AdditionalSlaveSecurityGroups":
            return Optional.ofNullable(clazz.cast(additionalSlaveSecurityGroups()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Ec2InstanceAttributes, T> g) {
        return obj -> g.apply((Ec2InstanceAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Ec2InstanceAttributes> {
        /**
         * <p>
         * The name of the Amazon EC2 key pair to use when connecting with SSH into the master node as a user named
         * "hadoop".
         * </p>
         * 
         * @param ec2KeyName
         *        The name of the Amazon EC2 key pair to use when connecting with SSH into the master node as a user
         *        named "hadoop".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2KeyName(String ec2KeyName);

        /**
         * <p>
         * Set this parameter to the identifier of the Amazon VPC subnet where you want the cluster to launch. If you do
         * not specify this value, and your account supports EC2-Classic, the cluster launches in EC2-Classic.
         * </p>
         * 
         * @param ec2SubnetId
         *        Set this parameter to the identifier of the Amazon VPC subnet where you want the cluster to launch. If
         *        you do not specify this value, and your account supports EC2-Classic, the cluster launches in
         *        EC2-Classic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2SubnetId(String ec2SubnetId);

        /**
         * <p>
         * Applies to clusters configured with the instance fleets option. Specifies the unique identifier of one or
         * more Amazon EC2 subnets in which to launch Amazon EC2 cluster instances. Subnets must exist within the same
         * VPC. Amazon EMR chooses the Amazon EC2 subnet with the best fit from among the list of
         * <code>RequestedEc2SubnetIds</code>, and then launches all cluster instances within that Subnet. If this value
         * is not specified, and the account and Region support EC2-Classic networks, the cluster launches instances in
         * the EC2-Classic network and uses <code>RequestedEc2AvailabilityZones</code> instead of this setting. If
         * EC2-Classic is not supported, and no Subnet is specified, Amazon EMR chooses the subnet for you.
         * <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be specified
         * together.
         * </p>
         * 
         * @param requestedEc2SubnetIds
         *        Applies to clusters configured with the instance fleets option. Specifies the unique identifier of one
         *        or more Amazon EC2 subnets in which to launch Amazon EC2 cluster instances. Subnets must exist within
         *        the same VPC. Amazon EMR chooses the Amazon EC2 subnet with the best fit from among the list of
         *        <code>RequestedEc2SubnetIds</code>, and then launches all cluster instances within that Subnet. If
         *        this value is not specified, and the account and Region support EC2-Classic networks, the cluster
         *        launches instances in the EC2-Classic network and uses <code>RequestedEc2AvailabilityZones</code>
         *        instead of this setting. If EC2-Classic is not supported, and no Subnet is specified, Amazon EMR
         *        chooses the subnet for you. <code>RequestedEc2SubnetIDs</code> and
         *        <code>RequestedEc2AvailabilityZones</code> cannot be specified together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedEc2SubnetIds(Collection<String> requestedEc2SubnetIds);

        /**
         * <p>
         * Applies to clusters configured with the instance fleets option. Specifies the unique identifier of one or
         * more Amazon EC2 subnets in which to launch Amazon EC2 cluster instances. Subnets must exist within the same
         * VPC. Amazon EMR chooses the Amazon EC2 subnet with the best fit from among the list of
         * <code>RequestedEc2SubnetIds</code>, and then launches all cluster instances within that Subnet. If this value
         * is not specified, and the account and Region support EC2-Classic networks, the cluster launches instances in
         * the EC2-Classic network and uses <code>RequestedEc2AvailabilityZones</code> instead of this setting. If
         * EC2-Classic is not supported, and no Subnet is specified, Amazon EMR chooses the subnet for you.
         * <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be specified
         * together.
         * </p>
         * 
         * @param requestedEc2SubnetIds
         *        Applies to clusters configured with the instance fleets option. Specifies the unique identifier of one
         *        or more Amazon EC2 subnets in which to launch Amazon EC2 cluster instances. Subnets must exist within
         *        the same VPC. Amazon EMR chooses the Amazon EC2 subnet with the best fit from among the list of
         *        <code>RequestedEc2SubnetIds</code>, and then launches all cluster instances within that Subnet. If
         *        this value is not specified, and the account and Region support EC2-Classic networks, the cluster
         *        launches instances in the EC2-Classic network and uses <code>RequestedEc2AvailabilityZones</code>
         *        instead of this setting. If EC2-Classic is not supported, and no Subnet is specified, Amazon EMR
         *        chooses the subnet for you. <code>RequestedEc2SubnetIDs</code> and
         *        <code>RequestedEc2AvailabilityZones</code> cannot be specified together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedEc2SubnetIds(String... requestedEc2SubnetIds);

        /**
         * <p>
         * The Availability Zone in which the cluster will run.
         * </p>
         * 
         * @param ec2AvailabilityZone
         *        The Availability Zone in which the cluster will run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2AvailabilityZone(String ec2AvailabilityZone);

        /**
         * <p>
         * Applies to clusters configured with the instance fleets option. Specifies one or more Availability Zones in
         * which to launch Amazon EC2 cluster instances when the EC2-Classic network configuration is supported. Amazon
         * EMR chooses the Availability Zone with the best fit from among the list of
         * <code>RequestedEc2AvailabilityZones</code>, and then launches all cluster instances within that Availability
         * Zone. If you do not specify this value, Amazon EMR chooses the Availability Zone for you.
         * <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be specified
         * together.
         * </p>
         * 
         * @param requestedEc2AvailabilityZones
         *        Applies to clusters configured with the instance fleets option. Specifies one or more Availability
         *        Zones in which to launch Amazon EC2 cluster instances when the EC2-Classic network configuration is
         *        supported. Amazon EMR chooses the Availability Zone with the best fit from among the list of
         *        <code>RequestedEc2AvailabilityZones</code>, and then launches all cluster instances within that
         *        Availability Zone. If you do not specify this value, Amazon EMR chooses the Availability Zone for you.
         *        <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be specified
         *        together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedEc2AvailabilityZones(Collection<String> requestedEc2AvailabilityZones);

        /**
         * <p>
         * Applies to clusters configured with the instance fleets option. Specifies one or more Availability Zones in
         * which to launch Amazon EC2 cluster instances when the EC2-Classic network configuration is supported. Amazon
         * EMR chooses the Availability Zone with the best fit from among the list of
         * <code>RequestedEc2AvailabilityZones</code>, and then launches all cluster instances within that Availability
         * Zone. If you do not specify this value, Amazon EMR chooses the Availability Zone for you.
         * <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be specified
         * together.
         * </p>
         * 
         * @param requestedEc2AvailabilityZones
         *        Applies to clusters configured with the instance fleets option. Specifies one or more Availability
         *        Zones in which to launch Amazon EC2 cluster instances when the EC2-Classic network configuration is
         *        supported. Amazon EMR chooses the Availability Zone with the best fit from among the list of
         *        <code>RequestedEc2AvailabilityZones</code>, and then launches all cluster instances within that
         *        Availability Zone. If you do not specify this value, Amazon EMR chooses the Availability Zone for you.
         *        <code>RequestedEc2SubnetIDs</code> and <code>RequestedEc2AvailabilityZones</code> cannot be specified
         *        together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedEc2AvailabilityZones(String... requestedEc2AvailabilityZones);

        /**
         * <p>
         * The IAM role that was specified when the cluster was launched. The Amazon EC2 instances of the cluster assume
         * this role.
         * </p>
         * 
         * @param iamInstanceProfile
         *        The IAM role that was specified when the cluster was launched. The Amazon EC2 instances of the cluster
         *        assume this role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamInstanceProfile(String iamInstanceProfile);

        /**
         * <p>
         * The identifier of the Amazon EC2 security group for the master node.
         * </p>
         * 
         * @param emrManagedMasterSecurityGroup
         *        The identifier of the Amazon EC2 security group for the master node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emrManagedMasterSecurityGroup(String emrManagedMasterSecurityGroup);

        /**
         * <p>
         * The identifier of the Amazon EC2 security group for the core and task nodes.
         * </p>
         * 
         * @param emrManagedSlaveSecurityGroup
         *        The identifier of the Amazon EC2 security group for the core and task nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emrManagedSlaveSecurityGroup(String emrManagedSlaveSecurityGroup);

        /**
         * <p>
         * The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC private
         * subnets.
         * </p>
         * 
         * @param serviceAccessSecurityGroup
         *        The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC
         *        private subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceAccessSecurityGroup(String serviceAccessSecurityGroup);

        /**
         * <p>
         * A list of additional Amazon EC2 security group IDs for the master node.
         * </p>
         * 
         * @param additionalMasterSecurityGroups
         *        A list of additional Amazon EC2 security group IDs for the master node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalMasterSecurityGroups(Collection<String> additionalMasterSecurityGroups);

        /**
         * <p>
         * A list of additional Amazon EC2 security group IDs for the master node.
         * </p>
         * 
         * @param additionalMasterSecurityGroups
         *        A list of additional Amazon EC2 security group IDs for the master node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalMasterSecurityGroups(String... additionalMasterSecurityGroups);

        /**
         * <p>
         * A list of additional Amazon EC2 security group IDs for the core and task nodes.
         * </p>
         * 
         * @param additionalSlaveSecurityGroups
         *        A list of additional Amazon EC2 security group IDs for the core and task nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalSlaveSecurityGroups(Collection<String> additionalSlaveSecurityGroups);

        /**
         * <p>
         * A list of additional Amazon EC2 security group IDs for the core and task nodes.
         * </p>
         * 
         * @param additionalSlaveSecurityGroups
         *        A list of additional Amazon EC2 security group IDs for the core and task nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalSlaveSecurityGroups(String... additionalSlaveSecurityGroups);
    }

    static final class BuilderImpl implements Builder {
        private String ec2KeyName;

        private String ec2SubnetId;

        private List<String> requestedEc2SubnetIds = DefaultSdkAutoConstructList.getInstance();

        private String ec2AvailabilityZone;

        private List<String> requestedEc2AvailabilityZones = DefaultSdkAutoConstructList.getInstance();

        private String iamInstanceProfile;

        private String emrManagedMasterSecurityGroup;

        private String emrManagedSlaveSecurityGroup;

        private String serviceAccessSecurityGroup;

        private List<String> additionalMasterSecurityGroups = DefaultSdkAutoConstructList.getInstance();

        private List<String> additionalSlaveSecurityGroups = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Ec2InstanceAttributes model) {
            ec2KeyName(model.ec2KeyName);
            ec2SubnetId(model.ec2SubnetId);
            requestedEc2SubnetIds(model.requestedEc2SubnetIds);
            ec2AvailabilityZone(model.ec2AvailabilityZone);
            requestedEc2AvailabilityZones(model.requestedEc2AvailabilityZones);
            iamInstanceProfile(model.iamInstanceProfile);
            emrManagedMasterSecurityGroup(model.emrManagedMasterSecurityGroup);
            emrManagedSlaveSecurityGroup(model.emrManagedSlaveSecurityGroup);
            serviceAccessSecurityGroup(model.serviceAccessSecurityGroup);
            additionalMasterSecurityGroups(model.additionalMasterSecurityGroups);
            additionalSlaveSecurityGroups(model.additionalSlaveSecurityGroups);
        }

        public final String getEc2KeyName() {
            return ec2KeyName;
        }

        public final void setEc2KeyName(String ec2KeyName) {
            this.ec2KeyName = ec2KeyName;
        }

        @Override
        public final Builder ec2KeyName(String ec2KeyName) {
            this.ec2KeyName = ec2KeyName;
            return this;
        }

        public final String getEc2SubnetId() {
            return ec2SubnetId;
        }

        public final void setEc2SubnetId(String ec2SubnetId) {
            this.ec2SubnetId = ec2SubnetId;
        }

        @Override
        public final Builder ec2SubnetId(String ec2SubnetId) {
            this.ec2SubnetId = ec2SubnetId;
            return this;
        }

        public final Collection<String> getRequestedEc2SubnetIds() {
            if (requestedEc2SubnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return requestedEc2SubnetIds;
        }

        public final void setRequestedEc2SubnetIds(Collection<String> requestedEc2SubnetIds) {
            this.requestedEc2SubnetIds = XmlStringMaxLen256ListCopier.copy(requestedEc2SubnetIds);
        }

        @Override
        public final Builder requestedEc2SubnetIds(Collection<String> requestedEc2SubnetIds) {
            this.requestedEc2SubnetIds = XmlStringMaxLen256ListCopier.copy(requestedEc2SubnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requestedEc2SubnetIds(String... requestedEc2SubnetIds) {
            requestedEc2SubnetIds(Arrays.asList(requestedEc2SubnetIds));
            return this;
        }

        public final String getEc2AvailabilityZone() {
            return ec2AvailabilityZone;
        }

        public final void setEc2AvailabilityZone(String ec2AvailabilityZone) {
            this.ec2AvailabilityZone = ec2AvailabilityZone;
        }

        @Override
        public final Builder ec2AvailabilityZone(String ec2AvailabilityZone) {
            this.ec2AvailabilityZone = ec2AvailabilityZone;
            return this;
        }

        public final Collection<String> getRequestedEc2AvailabilityZones() {
            if (requestedEc2AvailabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return requestedEc2AvailabilityZones;
        }

        public final void setRequestedEc2AvailabilityZones(Collection<String> requestedEc2AvailabilityZones) {
            this.requestedEc2AvailabilityZones = XmlStringMaxLen256ListCopier.copy(requestedEc2AvailabilityZones);
        }

        @Override
        public final Builder requestedEc2AvailabilityZones(Collection<String> requestedEc2AvailabilityZones) {
            this.requestedEc2AvailabilityZones = XmlStringMaxLen256ListCopier.copy(requestedEc2AvailabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requestedEc2AvailabilityZones(String... requestedEc2AvailabilityZones) {
            requestedEc2AvailabilityZones(Arrays.asList(requestedEc2AvailabilityZones));
            return this;
        }

        public final String getIamInstanceProfile() {
            return iamInstanceProfile;
        }

        public final void setIamInstanceProfile(String iamInstanceProfile) {
            this.iamInstanceProfile = iamInstanceProfile;
        }

        @Override
        public final Builder iamInstanceProfile(String iamInstanceProfile) {
            this.iamInstanceProfile = iamInstanceProfile;
            return this;
        }

        public final String getEmrManagedMasterSecurityGroup() {
            return emrManagedMasterSecurityGroup;
        }

        public final void setEmrManagedMasterSecurityGroup(String emrManagedMasterSecurityGroup) {
            this.emrManagedMasterSecurityGroup = emrManagedMasterSecurityGroup;
        }

        @Override
        public final Builder emrManagedMasterSecurityGroup(String emrManagedMasterSecurityGroup) {
            this.emrManagedMasterSecurityGroup = emrManagedMasterSecurityGroup;
            return this;
        }

        public final String getEmrManagedSlaveSecurityGroup() {
            return emrManagedSlaveSecurityGroup;
        }

        public final void setEmrManagedSlaveSecurityGroup(String emrManagedSlaveSecurityGroup) {
            this.emrManagedSlaveSecurityGroup = emrManagedSlaveSecurityGroup;
        }

        @Override
        public final Builder emrManagedSlaveSecurityGroup(String emrManagedSlaveSecurityGroup) {
            this.emrManagedSlaveSecurityGroup = emrManagedSlaveSecurityGroup;
            return this;
        }

        public final String getServiceAccessSecurityGroup() {
            return serviceAccessSecurityGroup;
        }

        public final void setServiceAccessSecurityGroup(String serviceAccessSecurityGroup) {
            this.serviceAccessSecurityGroup = serviceAccessSecurityGroup;
        }

        @Override
        public final Builder serviceAccessSecurityGroup(String serviceAccessSecurityGroup) {
            this.serviceAccessSecurityGroup = serviceAccessSecurityGroup;
            return this;
        }

        public final Collection<String> getAdditionalMasterSecurityGroups() {
            if (additionalMasterSecurityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return additionalMasterSecurityGroups;
        }

        public final void setAdditionalMasterSecurityGroups(Collection<String> additionalMasterSecurityGroups) {
            this.additionalMasterSecurityGroups = StringListCopier.copy(additionalMasterSecurityGroups);
        }

        @Override
        public final Builder additionalMasterSecurityGroups(Collection<String> additionalMasterSecurityGroups) {
            this.additionalMasterSecurityGroups = StringListCopier.copy(additionalMasterSecurityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder additionalMasterSecurityGroups(String... additionalMasterSecurityGroups) {
            additionalMasterSecurityGroups(Arrays.asList(additionalMasterSecurityGroups));
            return this;
        }

        public final Collection<String> getAdditionalSlaveSecurityGroups() {
            if (additionalSlaveSecurityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return additionalSlaveSecurityGroups;
        }

        public final void setAdditionalSlaveSecurityGroups(Collection<String> additionalSlaveSecurityGroups) {
            this.additionalSlaveSecurityGroups = StringListCopier.copy(additionalSlaveSecurityGroups);
        }

        @Override
        public final Builder additionalSlaveSecurityGroups(Collection<String> additionalSlaveSecurityGroups) {
            this.additionalSlaveSecurityGroups = StringListCopier.copy(additionalSlaveSecurityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder additionalSlaveSecurityGroups(String... additionalSlaveSecurityGroups) {
            additionalSlaveSecurityGroups(Arrays.asList(additionalSlaveSecurityGroups));
            return this;
        }

        @Override
        public Ec2InstanceAttributes build() {
            return new Ec2InstanceAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
