/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticloadbalancingv2.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.elasticloadbalancingv2.ElasticLoadBalancingV2Client;
import software.amazon.awssdk.services.elasticloadbalancingv2.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeLoadBalancersRequest;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeLoadBalancersResponse;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeTargetHealthRequest;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeTargetHealthResponse;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.ElasticLoadBalancingV2Request;
import software.amazon.awssdk.services.elasticloadbalancingv2.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultElasticLoadBalancingV2Waiter implements ElasticLoadBalancingV2Waiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final ElasticLoadBalancingV2Client client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeLoadBalancersResponse> loadBalancerExistsWaiter;

    private final Waiter<DescribeLoadBalancersResponse> loadBalancerAvailableWaiter;

    private final Waiter<DescribeLoadBalancersResponse> loadBalancersDeletedWaiter;

    private final Waiter<DescribeTargetHealthResponse> targetInServiceWaiter;

    private final Waiter<DescribeTargetHealthResponse> targetDeregisteredWaiter;

    private DefaultElasticLoadBalancingV2Waiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ElasticLoadBalancingV2Client.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.loadBalancerExistsWaiter = Waiter.builder(DescribeLoadBalancersResponse.class)
                .acceptors(loadBalancerExistsWaiterAcceptors())
                .overrideConfiguration(loadBalancerExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.loadBalancerAvailableWaiter = Waiter.builder(DescribeLoadBalancersResponse.class)
                .acceptors(loadBalancerAvailableWaiterAcceptors())
                .overrideConfiguration(loadBalancerAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.loadBalancersDeletedWaiter = Waiter.builder(DescribeLoadBalancersResponse.class)
                .acceptors(loadBalancersDeletedWaiterAcceptors())
                .overrideConfiguration(loadBalancersDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.targetInServiceWaiter = Waiter.builder(DescribeTargetHealthResponse.class)
                .acceptors(targetInServiceWaiterAcceptors())
                .overrideConfiguration(targetInServiceWaiterConfig(builder.overrideConfiguration)).build();
        this.targetDeregisteredWaiter = Waiter.builder(DescribeTargetHealthResponse.class)
                .acceptors(targetDeregisteredWaiterAcceptors())
                .overrideConfiguration(targetDeregisteredWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerAvailable(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        return loadBalancerAvailableWaiter.run(() -> client
                .describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)));
    }

    @Override
    public WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerAvailable(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        return loadBalancerAvailableWaiter.run(
                () -> client.describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)),
                loadBalancerAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerExists(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        return loadBalancerExistsWaiter.run(() -> client
                .describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)));
    }

    @Override
    public WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerExists(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        return loadBalancerExistsWaiter.run(
                () -> client.describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)),
                loadBalancerExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancersDeleted(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        return loadBalancersDeletedWaiter.run(() -> client
                .describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)));
    }

    @Override
    public WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancersDeleted(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        return loadBalancersDeletedWaiter.run(
                () -> client.describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)),
                loadBalancersDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetDeregistered(
            DescribeTargetHealthRequest describeTargetHealthRequest) {
        return targetDeregisteredWaiter
                .run(() -> client.describeTargetHealth(applyWaitersUserAgent(describeTargetHealthRequest)));
    }

    @Override
    public WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetDeregistered(
            DescribeTargetHealthRequest describeTargetHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        return targetDeregisteredWaiter.run(
                () -> client.describeTargetHealth(applyWaitersUserAgent(describeTargetHealthRequest)),
                targetDeregisteredWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetInService(
            DescribeTargetHealthRequest describeTargetHealthRequest) {
        return targetInServiceWaiter.run(() -> client.describeTargetHealth(applyWaitersUserAgent(describeTargetHealthRequest)));
    }

    @Override
    public WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetInService(
            DescribeTargetHealthRequest describeTargetHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        return targetInServiceWaiter.run(() -> client.describeTargetHealth(applyWaitersUserAgent(describeTargetHealthRequest)),
                targetInServiceWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> loadBalancerExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "LoadBalancerNotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> loadBalancerAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("LoadBalancers").flatten().field("State").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "active"));
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("LoadBalancers").flatten().field("State").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "provisioning"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "LoadBalancerNotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> loadBalancersDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("LoadBalancers").flatten().field("State").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "active"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "LoadBalancerNotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTargetHealthResponse>> targetInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTargetHealthResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("TargetHealthDescriptions").flatten().field("TargetHealth").field("State")
                    .values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "healthy"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidInstance")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTargetHealthResponse>> targetDeregisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTargetHealthResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidTarget")));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("TargetHealthDescriptions").flatten().field("TargetHealth").field("State")
                    .values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "unused"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration loadBalancerExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration loadBalancerAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration loadBalancersDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration targetInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration targetDeregisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ElasticLoadBalancingV2Waiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ElasticLoadBalancingV2Request> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ElasticLoadBalancingV2Waiter.Builder {
        private ElasticLoadBalancingV2Client client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public ElasticLoadBalancingV2Waiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ElasticLoadBalancingV2Waiter.Builder client(ElasticLoadBalancingV2Client client) {
            this.client = client;
            return this;
        }

        public ElasticLoadBalancingV2Waiter build() {
            return new DefaultElasticLoadBalancingV2Waiter(this);
        }
    }
}
