/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to describe one or more environments.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeEnvironmentsRequest extends ElasticBeanstalkRequest implements
        ToCopyableBuilder<DescribeEnvironmentsRequest.Builder, DescribeEnvironmentsRequest> {
    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationName").getter(getter(DescribeEnvironmentsRequest::applicationName))
            .setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationName").build()).build();

    private static final SdkField<String> VERSION_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionLabel").getter(getter(DescribeEnvironmentsRequest::versionLabel))
            .setter(setter(Builder::versionLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionLabel").build()).build();

    private static final SdkField<List<String>> ENVIRONMENT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EnvironmentIds")
            .getter(getter(DescribeEnvironmentsRequest::environmentIds))
            .setter(setter(Builder::environmentIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ENVIRONMENT_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EnvironmentNames")
            .getter(getter(DescribeEnvironmentsRequest::environmentNames))
            .setter(setter(Builder::environmentNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> INCLUDE_DELETED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IncludeDeleted").getter(getter(DescribeEnvironmentsRequest::includeDeleted))
            .setter(setter(Builder::includeDeleted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeDeleted").build()).build();

    private static final SdkField<Instant> INCLUDED_DELETED_BACK_TO_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("IncludedDeletedBackTo").getter(getter(DescribeEnvironmentsRequest::includedDeletedBackTo))
            .setter(setter(Builder::includedDeletedBackTo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludedDeletedBackTo").build())
            .build();

    private static final SdkField<Integer> MAX_RECORDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxRecords").getter(getter(DescribeEnvironmentsRequest::maxRecords)).setter(setter(Builder::maxRecords))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRecords").build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextToken").getter(getter(DescribeEnvironmentsRequest::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_NAME_FIELD,
            VERSION_LABEL_FIELD, ENVIRONMENT_IDS_FIELD, ENVIRONMENT_NAMES_FIELD, INCLUDE_DELETED_FIELD,
            INCLUDED_DELETED_BACK_TO_FIELD, MAX_RECORDS_FIELD, NEXT_TOKEN_FIELD));

    private final String applicationName;

    private final String versionLabel;

    private final List<String> environmentIds;

    private final List<String> environmentNames;

    private final Boolean includeDeleted;

    private final Instant includedDeletedBackTo;

    private final Integer maxRecords;

    private final String nextToken;

    private DescribeEnvironmentsRequest(BuilderImpl builder) {
        super(builder);
        this.applicationName = builder.applicationName;
        this.versionLabel = builder.versionLabel;
        this.environmentIds = builder.environmentIds;
        this.environmentNames = builder.environmentNames;
        this.includeDeleted = builder.includeDeleted;
        this.includedDeletedBackTo = builder.includedDeletedBackTo;
        this.maxRecords = builder.maxRecords;
        this.nextToken = builder.nextToken;
    }

    /**
     * <p>
     * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that are associated
     * with this application.
     * </p>
     * 
     * @return If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that are
     *         associated with this application.
     */
    public final String applicationName() {
        return applicationName;
    }

    /**
     * <p>
     * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that are associated
     * with this application version.
     * </p>
     * 
     * @return If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that are
     *         associated with this application version.
     */
    public final String versionLabel() {
        return versionLabel;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnvironmentIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironmentIds() {
        return environmentIds != null && !(environmentIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that have the
     * specified IDs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironmentIds} method.
     * </p>
     * 
     * @return If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that have
     *         the specified IDs.
     */
    public final List<String> environmentIds() {
        return environmentIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnvironmentNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironmentNames() {
        return environmentNames != null && !(environmentNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that have the
     * specified names.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironmentNames} method.
     * </p>
     * 
     * @return If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that have
     *         the specified names.
     */
    public final List<String> environmentNames() {
        return environmentNames;
    }

    /**
     * <p>
     * Indicates whether to include deleted environments:
     * </p>
     * <p>
     * <code>true</code>: Environments that have been deleted after <code>IncludedDeletedBackTo</code> are displayed.
     * </p>
     * <p>
     * <code>false</code>: Do not include deleted environments.
     * </p>
     * 
     * @return Indicates whether to include deleted environments:</p>
     *         <p>
     *         <code>true</code>: Environments that have been deleted after <code>IncludedDeletedBackTo</code> are
     *         displayed.
     *         </p>
     *         <p>
     *         <code>false</code>: Do not include deleted environments.
     */
    public final Boolean includeDeleted() {
        return includeDeleted;
    }

    /**
     * <p>
     * If specified when <code>IncludeDeleted</code> is set to <code>true</code>, then environments deleted after this
     * date are displayed.
     * </p>
     * 
     * @return If specified when <code>IncludeDeleted</code> is set to <code>true</code>, then environments deleted
     *         after this date are displayed.
     */
    public final Instant includedDeletedBackTo() {
        return includedDeletedBackTo;
    }

    /**
     * <p>
     * For a paginated request. Specify a maximum number of environments to include in each response.
     * </p>
     * <p>
     * If no <code>MaxRecords</code> is specified, all available environments are retrieved in a single response.
     * </p>
     * 
     * @return For a paginated request. Specify a maximum number of environments to include in each response.</p>
     *         <p>
     *         If no <code>MaxRecords</code> is specified, all available environments are retrieved in a single
     *         response.
     */
    public final Integer maxRecords() {
        return maxRecords;
    }

    /**
     * <p>
     * For a paginated request. Specify a token from a previous response page to retrieve the next response page. All
     * other parameter values must be identical to the ones specified in the initial request.
     * </p>
     * <p>
     * If no <code>NextToken</code> is specified, the first page is retrieved.
     * </p>
     * 
     * @return For a paginated request. Specify a token from a previous response page to retrieve the next response
     *         page. All other parameter values must be identical to the ones specified in the initial request.</p>
     *         <p>
     *         If no <code>NextToken</code> is specified, the first page is retrieved.
     */
    public final String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(versionLabel());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentIds() ? environmentIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentNames() ? environmentNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(includeDeleted());
        hashCode = 31 * hashCode + Objects.hashCode(includedDeletedBackTo());
        hashCode = 31 * hashCode + Objects.hashCode(maxRecords());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeEnvironmentsRequest)) {
            return false;
        }
        DescribeEnvironmentsRequest other = (DescribeEnvironmentsRequest) obj;
        return Objects.equals(applicationName(), other.applicationName()) && Objects.equals(versionLabel(), other.versionLabel())
                && hasEnvironmentIds() == other.hasEnvironmentIds() && Objects.equals(environmentIds(), other.environmentIds())
                && hasEnvironmentNames() == other.hasEnvironmentNames()
                && Objects.equals(environmentNames(), other.environmentNames())
                && Objects.equals(includeDeleted(), other.includeDeleted())
                && Objects.equals(includedDeletedBackTo(), other.includedDeletedBackTo())
                && Objects.equals(maxRecords(), other.maxRecords()) && Objects.equals(nextToken(), other.nextToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribeEnvironmentsRequest").add("ApplicationName", applicationName())
                .add("VersionLabel", versionLabel()).add("EnvironmentIds", hasEnvironmentIds() ? environmentIds() : null)
                .add("EnvironmentNames", hasEnvironmentNames() ? environmentNames() : null)
                .add("IncludeDeleted", includeDeleted()).add("IncludedDeletedBackTo", includedDeletedBackTo())
                .add("MaxRecords", maxRecords()).add("NextToken", nextToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "VersionLabel":
            return Optional.ofNullable(clazz.cast(versionLabel()));
        case "EnvironmentIds":
            return Optional.ofNullable(clazz.cast(environmentIds()));
        case "EnvironmentNames":
            return Optional.ofNullable(clazz.cast(environmentNames()));
        case "IncludeDeleted":
            return Optional.ofNullable(clazz.cast(includeDeleted()));
        case "IncludedDeletedBackTo":
            return Optional.ofNullable(clazz.cast(includedDeletedBackTo()));
        case "MaxRecords":
            return Optional.ofNullable(clazz.cast(maxRecords()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeEnvironmentsRequest, T> g) {
        return obj -> g.apply((DescribeEnvironmentsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ElasticBeanstalkRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, DescribeEnvironmentsRequest> {
        /**
         * <p>
         * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that are
         * associated with this application.
         * </p>
         * 
         * @param applicationName
         *        If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that are
         *        associated with this application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that are
         * associated with this application version.
         * </p>
         * 
         * @param versionLabel
         *        If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that are
         *        associated with this application version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionLabel(String versionLabel);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that have the
         * specified IDs.
         * </p>
         * 
         * @param environmentIds
         *        If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that
         *        have the specified IDs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentIds(Collection<String> environmentIds);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that have the
         * specified IDs.
         * </p>
         * 
         * @param environmentIds
         *        If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that
         *        have the specified IDs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentIds(String... environmentIds);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that have the
         * specified names.
         * </p>
         * 
         * @param environmentNames
         *        If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that
         *        have the specified names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentNames(Collection<String> environmentNames);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that have the
         * specified names.
         * </p>
         * 
         * @param environmentNames
         *        If specified, AWS Elastic Beanstalk restricts the returned descriptions to include only those that
         *        have the specified names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentNames(String... environmentNames);

        /**
         * <p>
         * Indicates whether to include deleted environments:
         * </p>
         * <p>
         * <code>true</code>: Environments that have been deleted after <code>IncludedDeletedBackTo</code> are
         * displayed.
         * </p>
         * <p>
         * <code>false</code>: Do not include deleted environments.
         * </p>
         * 
         * @param includeDeleted
         *        Indicates whether to include deleted environments:</p>
         *        <p>
         *        <code>true</code>: Environments that have been deleted after <code>IncludedDeletedBackTo</code> are
         *        displayed.
         *        </p>
         *        <p>
         *        <code>false</code>: Do not include deleted environments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeDeleted(Boolean includeDeleted);

        /**
         * <p>
         * If specified when <code>IncludeDeleted</code> is set to <code>true</code>, then environments deleted after
         * this date are displayed.
         * </p>
         * 
         * @param includedDeletedBackTo
         *        If specified when <code>IncludeDeleted</code> is set to <code>true</code>, then environments deleted
         *        after this date are displayed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includedDeletedBackTo(Instant includedDeletedBackTo);

        /**
         * <p>
         * For a paginated request. Specify a maximum number of environments to include in each response.
         * </p>
         * <p>
         * If no <code>MaxRecords</code> is specified, all available environments are retrieved in a single response.
         * </p>
         * 
         * @param maxRecords
         *        For a paginated request. Specify a maximum number of environments to include in each response.</p>
         *        <p>
         *        If no <code>MaxRecords</code> is specified, all available environments are retrieved in a single
         *        response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRecords(Integer maxRecords);

        /**
         * <p>
         * For a paginated request. Specify a token from a previous response page to retrieve the next response page.
         * All other parameter values must be identical to the ones specified in the initial request.
         * </p>
         * <p>
         * If no <code>NextToken</code> is specified, the first page is retrieved.
         * </p>
         * 
         * @param nextToken
         *        For a paginated request. Specify a token from a previous response page to retrieve the next response
         *        page. All other parameter values must be identical to the ones specified in the initial request.</p>
         *        <p>
         *        If no <code>NextToken</code> is specified, the first page is retrieved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ElasticBeanstalkRequest.BuilderImpl implements Builder {
        private String applicationName;

        private String versionLabel;

        private List<String> environmentIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> environmentNames = DefaultSdkAutoConstructList.getInstance();

        private Boolean includeDeleted;

        private Instant includedDeletedBackTo;

        private Integer maxRecords;

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeEnvironmentsRequest model) {
            super(model);
            applicationName(model.applicationName);
            versionLabel(model.versionLabel);
            environmentIds(model.environmentIds);
            environmentNames(model.environmentNames);
            includeDeleted(model.includeDeleted);
            includedDeletedBackTo(model.includedDeletedBackTo);
            maxRecords(model.maxRecords);
            nextToken(model.nextToken);
        }

        public final String getApplicationName() {
            return applicationName;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final String getVersionLabel() {
            return versionLabel;
        }

        public final void setVersionLabel(String versionLabel) {
            this.versionLabel = versionLabel;
        }

        @Override
        public final Builder versionLabel(String versionLabel) {
            this.versionLabel = versionLabel;
            return this;
        }

        public final Collection<String> getEnvironmentIds() {
            if (environmentIds instanceof SdkAutoConstructList) {
                return null;
            }
            return environmentIds;
        }

        public final void setEnvironmentIds(Collection<String> environmentIds) {
            this.environmentIds = EnvironmentIdListCopier.copy(environmentIds);
        }

        @Override
        public final Builder environmentIds(Collection<String> environmentIds) {
            this.environmentIds = EnvironmentIdListCopier.copy(environmentIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentIds(String... environmentIds) {
            environmentIds(Arrays.asList(environmentIds));
            return this;
        }

        public final Collection<String> getEnvironmentNames() {
            if (environmentNames instanceof SdkAutoConstructList) {
                return null;
            }
            return environmentNames;
        }

        public final void setEnvironmentNames(Collection<String> environmentNames) {
            this.environmentNames = EnvironmentNamesListCopier.copy(environmentNames);
        }

        @Override
        public final Builder environmentNames(Collection<String> environmentNames) {
            this.environmentNames = EnvironmentNamesListCopier.copy(environmentNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentNames(String... environmentNames) {
            environmentNames(Arrays.asList(environmentNames));
            return this;
        }

        public final Boolean getIncludeDeleted() {
            return includeDeleted;
        }

        public final void setIncludeDeleted(Boolean includeDeleted) {
            this.includeDeleted = includeDeleted;
        }

        @Override
        public final Builder includeDeleted(Boolean includeDeleted) {
            this.includeDeleted = includeDeleted;
            return this;
        }

        public final Instant getIncludedDeletedBackTo() {
            return includedDeletedBackTo;
        }

        public final void setIncludedDeletedBackTo(Instant includedDeletedBackTo) {
            this.includedDeletedBackTo = includedDeletedBackTo;
        }

        @Override
        public final Builder includedDeletedBackTo(Instant includedDeletedBackTo) {
            this.includedDeletedBackTo = includedDeletedBackTo;
            return this;
        }

        public final Integer getMaxRecords() {
            return maxRecords;
        }

        public final void setMaxRecords(Integer maxRecords) {
            this.maxRecords = maxRecords;
        }

        @Override
        public final Builder maxRecords(Integer maxRecords) {
            this.maxRecords = maxRecords;
            return this;
        }

        public final String getNextToken() {
            return nextToken;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DescribeEnvironmentsRequest build() {
            return new DescribeEnvironmentsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
