/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing a managed capability in an Amazon EKS cluster. This includes all configuration, status, and
 * health information for the capability.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Capability implements SdkPojo, Serializable, ToCopyableBuilder<Capability.Builder, Capability> {
    private static final SdkField<String> CAPABILITY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("capabilityName").getter(getter(Capability::capabilityName)).setter(setter(Builder::capabilityName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capabilityName").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Capability::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> CLUSTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clusterName").getter(getter(Capability::clusterName)).setter(setter(Builder::clusterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusterName").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Capability::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(Capability::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Capability::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(Capability::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<CapabilityConfigurationResponse> CONFIGURATION_FIELD = SdkField
            .<CapabilityConfigurationResponse> builder(MarshallingType.SDK_POJO).memberName("configuration")
            .getter(getter(Capability::configuration)).setter(setter(Builder::configuration))
            .constructor(CapabilityConfigurationResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Capability::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<CapabilityHealth> HEALTH_FIELD = SdkField.<CapabilityHealth> builder(MarshallingType.SDK_POJO)
            .memberName("health").getter(getter(Capability::health)).setter(setter(Builder::health))
            .constructor(CapabilityHealth::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("health").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Capability::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt").getter(getter(Capability::modifiedAt)).setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build()).build();

    private static final SdkField<String> DELETE_PROPAGATION_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deletePropagationPolicy").getter(getter(Capability::deletePropagationPolicyAsString))
            .setter(setter(Builder::deletePropagationPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deletePropagationPolicy").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CAPABILITY_NAME_FIELD,
            ARN_FIELD, CLUSTER_NAME_FIELD, TYPE_FIELD, ROLE_ARN_FIELD, STATUS_FIELD, VERSION_FIELD, CONFIGURATION_FIELD,
            TAGS_FIELD, HEALTH_FIELD, CREATED_AT_FIELD, MODIFIED_AT_FIELD, DELETE_PROPAGATION_POLICY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String capabilityName;

    private final String arn;

    private final String clusterName;

    private final String type;

    private final String roleArn;

    private final String status;

    private final String version;

    private final CapabilityConfigurationResponse configuration;

    private final Map<String, String> tags;

    private final CapabilityHealth health;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private final String deletePropagationPolicy;

    private Capability(BuilderImpl builder) {
        this.capabilityName = builder.capabilityName;
        this.arn = builder.arn;
        this.clusterName = builder.clusterName;
        this.type = builder.type;
        this.roleArn = builder.roleArn;
        this.status = builder.status;
        this.version = builder.version;
        this.configuration = builder.configuration;
        this.tags = builder.tags;
        this.health = builder.health;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
        this.deletePropagationPolicy = builder.deletePropagationPolicy;
    }

    /**
     * <p>
     * The unique name of the capability within the cluster.
     * </p>
     * 
     * @return The unique name of the capability within the cluster.
     */
    public final String capabilityName() {
        return capabilityName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the capability.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the capability.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name of the Amazon EKS cluster that contains this capability.
     * </p>
     * 
     * @return The name of the Amazon EKS cluster that contains this capability.
     */
    public final String clusterName() {
        return clusterName;
    }

    /**
     * <p>
     * The type of capability. Valid values are <code>ACK</code>, <code>ARGOCD</code>, or <code>KRO</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CapabilityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of capability. Valid values are <code>ACK</code>, <code>ARGOCD</code>, or <code>KRO</code>.
     * @see CapabilityType
     */
    public final CapabilityType type() {
        return CapabilityType.fromValue(type);
    }

    /**
     * <p>
     * The type of capability. Valid values are <code>ACK</code>, <code>ARGOCD</code>, or <code>KRO</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CapabilityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of capability. Valid values are <code>ACK</code>, <code>ARGOCD</code>, or <code>KRO</code>.
     * @see CapabilityType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that the capability uses to interact with Amazon Web Services
     * services.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that the capability uses to interact with Amazon Web
     *         Services services.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The current status of the capability. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> – The capability is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> – The capability is running and available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> – The capability is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> – The capability is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code> – The capability creation failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATE_FAILED</code> – The capability update failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_FAILED</code> – The capability deletion failed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CapabilityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the capability. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> – The capability is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> – The capability is running and available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> – The capability is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> – The capability is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code> – The capability creation failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATE_FAILED</code> – The capability update failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_FAILED</code> – The capability deletion failed.
     *         </p>
     *         </li>
     * @see CapabilityStatus
     */
    public final CapabilityStatus status() {
        return CapabilityStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the capability. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> – The capability is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> – The capability is running and available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> – The capability is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> – The capability is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATE_FAILED</code> – The capability creation failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATE_FAILED</code> – The capability update failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE_FAILED</code> – The capability deletion failed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CapabilityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the capability. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> – The capability is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> – The capability is running and available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> – The capability is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> – The capability is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATE_FAILED</code> – The capability creation failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATE_FAILED</code> – The capability update failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE_FAILED</code> – The capability deletion failed.
     *         </p>
     *         </li>
     * @see CapabilityStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The version of the capability software that is currently running.
     * </p>
     * 
     * @return The version of the capability software that is currently running.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The configuration settings for the capability. The structure varies depending on the capability type.
     * </p>
     * 
     * @return The configuration settings for the capability. The structure varies depending on the capability type.
     */
    public final CapabilityConfigurationResponse configuration() {
        return configuration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * Returns the value of the Tags property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The value of the Tags property for this object.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * Health information for the capability, including any issues that may be affecting its operation.
     * </p>
     * 
     * @return Health information for the capability, including any issues that may be affecting its operation.
     */
    public final CapabilityHealth health() {
        return health;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the capability was created.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the capability was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the capability was last modified.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the capability was last modified.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * The delete propagation policy for the capability. Currently, the only supported value is <code>RETAIN</code>,
     * which keeps all resources managed by the capability when the capability is deleted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deletePropagationPolicy} will return {@link CapabilityDeletePropagationPolicy#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #deletePropagationPolicyAsString}.
     * </p>
     * 
     * @return The delete propagation policy for the capability. Currently, the only supported value is
     *         <code>RETAIN</code>, which keeps all resources managed by the capability when the capability is deleted.
     * @see CapabilityDeletePropagationPolicy
     */
    public final CapabilityDeletePropagationPolicy deletePropagationPolicy() {
        return CapabilityDeletePropagationPolicy.fromValue(deletePropagationPolicy);
    }

    /**
     * <p>
     * The delete propagation policy for the capability. Currently, the only supported value is <code>RETAIN</code>,
     * which keeps all resources managed by the capability when the capability is deleted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deletePropagationPolicy} will return {@link CapabilityDeletePropagationPolicy#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #deletePropagationPolicyAsString}.
     * </p>
     * 
     * @return The delete propagation policy for the capability. Currently, the only supported value is
     *         <code>RETAIN</code>, which keeps all resources managed by the capability when the capability is deleted.
     * @see CapabilityDeletePropagationPolicy
     */
    public final String deletePropagationPolicyAsString() {
        return deletePropagationPolicy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(capabilityName());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(clusterName());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(health());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(deletePropagationPolicyAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Capability)) {
            return false;
        }
        Capability other = (Capability) obj;
        return Objects.equals(capabilityName(), other.capabilityName()) && Objects.equals(arn(), other.arn())
                && Objects.equals(clusterName(), other.clusterName()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(version(), other.version()) && Objects.equals(configuration(), other.configuration())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(health(), other.health()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt())
                && Objects.equals(deletePropagationPolicyAsString(), other.deletePropagationPolicyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Capability").add("CapabilityName", capabilityName()).add("Arn", arn())
                .add("ClusterName", clusterName()).add("Type", typeAsString()).add("RoleArn", roleArn())
                .add("Status", statusAsString()).add("Version", version()).add("Configuration", configuration())
                .add("Tags", hasTags() ? tags() : null).add("Health", health()).add("CreatedAt", createdAt())
                .add("ModifiedAt", modifiedAt()).add("DeletePropagationPolicy", deletePropagationPolicyAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "capabilityName":
            return Optional.ofNullable(clazz.cast(capabilityName()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "clusterName":
            return Optional.ofNullable(clazz.cast(clusterName()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "health":
            return Optional.ofNullable(clazz.cast(health()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "deletePropagationPolicy":
            return Optional.ofNullable(clazz.cast(deletePropagationPolicyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("capabilityName", CAPABILITY_NAME_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("clusterName", CLUSTER_NAME_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("roleArn", ROLE_ARN_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("version", VERSION_FIELD);
        map.put("configuration", CONFIGURATION_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("health", HEALTH_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("modifiedAt", MODIFIED_AT_FIELD);
        map.put("deletePropagationPolicy", DELETE_PROPAGATION_POLICY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Capability, T> g) {
        return obj -> g.apply((Capability) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Capability> {
        /**
         * <p>
         * The unique name of the capability within the cluster.
         * </p>
         * 
         * @param capabilityName
         *        The unique name of the capability within the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilityName(String capabilityName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the capability.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the capability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name of the Amazon EKS cluster that contains this capability.
         * </p>
         * 
         * @param clusterName
         *        The name of the Amazon EKS cluster that contains this capability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterName(String clusterName);

        /**
         * <p>
         * The type of capability. Valid values are <code>ACK</code>, <code>ARGOCD</code>, or <code>KRO</code>.
         * </p>
         * 
         * @param type
         *        The type of capability. Valid values are <code>ACK</code>, <code>ARGOCD</code>, or <code>KRO</code>.
         * @see CapabilityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of capability. Valid values are <code>ACK</code>, <code>ARGOCD</code>, or <code>KRO</code>.
         * </p>
         * 
         * @param type
         *        The type of capability. Valid values are <code>ACK</code>, <code>ARGOCD</code>, or <code>KRO</code>.
         * @see CapabilityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityType
         */
        Builder type(CapabilityType type);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that the capability uses to interact with Amazon Web Services
         * services.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role that the capability uses to interact with Amazon Web
         *        Services services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The current status of the capability. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> – The capability is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> – The capability is running and available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> – The capability is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> – The capability is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code> – The capability creation failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATE_FAILED</code> – The capability update failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_FAILED</code> – The capability deletion failed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the capability. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> – The capability is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> – The capability is running and available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> – The capability is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> – The capability is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code> – The capability creation failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATE_FAILED</code> – The capability update failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_FAILED</code> – The capability deletion failed.
         *        </p>
         *        </li>
         * @see CapabilityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the capability. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> – The capability is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> – The capability is running and available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> – The capability is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> – The capability is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATE_FAILED</code> – The capability creation failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATE_FAILED</code> – The capability update failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE_FAILED</code> – The capability deletion failed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the capability. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> – The capability is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> – The capability is running and available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> – The capability is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> – The capability is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATE_FAILED</code> – The capability creation failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATE_FAILED</code> – The capability update failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE_FAILED</code> – The capability deletion failed.
         *        </p>
         *        </li>
         * @see CapabilityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityStatus
         */
        Builder status(CapabilityStatus status);

        /**
         * <p>
         * The version of the capability software that is currently running.
         * </p>
         * 
         * @param version
         *        The version of the capability software that is currently running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The configuration settings for the capability. The structure varies depending on the capability type.
         * </p>
         * 
         * @param configuration
         *        The configuration settings for the capability. The structure varies depending on the capability type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(CapabilityConfigurationResponse configuration);

        /**
         * <p>
         * The configuration settings for the capability. The structure varies depending on the capability type.
         * </p>
         * This is a convenience method that creates an instance of the {@link CapabilityConfigurationResponse.Builder}
         * avoiding the need to create one manually via {@link CapabilityConfigurationResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CapabilityConfigurationResponse.Builder#build()} is called
         * immediately and its result is passed to {@link #configuration(CapabilityConfigurationResponse)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link CapabilityConfigurationResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(CapabilityConfigurationResponse)
         */
        default Builder configuration(Consumer<CapabilityConfigurationResponse.Builder> configuration) {
            return configuration(CapabilityConfigurationResponse.builder().applyMutation(configuration).build());
        }

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * Health information for the capability, including any issues that may be affecting its operation.
         * </p>
         * 
         * @param health
         *        Health information for the capability, including any issues that may be affecting its operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder health(CapabilityHealth health);

        /**
         * <p>
         * Health information for the capability, including any issues that may be affecting its operation.
         * </p>
         * This is a convenience method that creates an instance of the {@link CapabilityHealth.Builder} avoiding the
         * need to create one manually via {@link CapabilityHealth#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CapabilityHealth.Builder#build()} is called immediately and its
         * result is passed to {@link #health(CapabilityHealth)}.
         * 
         * @param health
         *        a consumer that will call methods on {@link CapabilityHealth.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #health(CapabilityHealth)
         */
        default Builder health(Consumer<CapabilityHealth.Builder> health) {
            return health(CapabilityHealth.builder().applyMutation(health).build());
        }

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the capability was created.
         * </p>
         * 
         * @param createdAt
         *        The Unix epoch timestamp in seconds for when the capability was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the capability was last modified.
         * </p>
         * 
         * @param modifiedAt
         *        The Unix epoch timestamp in seconds for when the capability was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * The delete propagation policy for the capability. Currently, the only supported value is <code>RETAIN</code>,
         * which keeps all resources managed by the capability when the capability is deleted.
         * </p>
         * 
         * @param deletePropagationPolicy
         *        The delete propagation policy for the capability. Currently, the only supported value is
         *        <code>RETAIN</code>, which keeps all resources managed by the capability when the capability is
         *        deleted.
         * @see CapabilityDeletePropagationPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityDeletePropagationPolicy
         */
        Builder deletePropagationPolicy(String deletePropagationPolicy);

        /**
         * <p>
         * The delete propagation policy for the capability. Currently, the only supported value is <code>RETAIN</code>,
         * which keeps all resources managed by the capability when the capability is deleted.
         * </p>
         * 
         * @param deletePropagationPolicy
         *        The delete propagation policy for the capability. Currently, the only supported value is
         *        <code>RETAIN</code>, which keeps all resources managed by the capability when the capability is
         *        deleted.
         * @see CapabilityDeletePropagationPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityDeletePropagationPolicy
         */
        Builder deletePropagationPolicy(CapabilityDeletePropagationPolicy deletePropagationPolicy);
    }

    static final class BuilderImpl implements Builder {
        private String capabilityName;

        private String arn;

        private String clusterName;

        private String type;

        private String roleArn;

        private String status;

        private String version;

        private CapabilityConfigurationResponse configuration;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private CapabilityHealth health;

        private Instant createdAt;

        private Instant modifiedAt;

        private String deletePropagationPolicy;

        private BuilderImpl() {
        }

        private BuilderImpl(Capability model) {
            capabilityName(model.capabilityName);
            arn(model.arn);
            clusterName(model.clusterName);
            type(model.type);
            roleArn(model.roleArn);
            status(model.status);
            version(model.version);
            configuration(model.configuration);
            tags(model.tags);
            health(model.health);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
            deletePropagationPolicy(model.deletePropagationPolicy);
        }

        public final String getCapabilityName() {
            return capabilityName;
        }

        public final void setCapabilityName(String capabilityName) {
            this.capabilityName = capabilityName;
        }

        @Override
        public final Builder capabilityName(String capabilityName) {
            this.capabilityName = capabilityName;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getClusterName() {
            return clusterName;
        }

        public final void setClusterName(String clusterName) {
            this.clusterName = clusterName;
        }

        @Override
        public final Builder clusterName(String clusterName) {
            this.clusterName = clusterName;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CapabilityType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CapabilityStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final CapabilityConfigurationResponse.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        public final void setConfiguration(CapabilityConfigurationResponse.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        @Override
        public final Builder configuration(CapabilityConfigurationResponse configuration) {
            this.configuration = configuration;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final CapabilityHealth.Builder getHealth() {
            return health != null ? health.toBuilder() : null;
        }

        public final void setHealth(CapabilityHealth.BuilderImpl health) {
            this.health = health != null ? health.build() : null;
        }

        @Override
        public final Builder health(CapabilityHealth health) {
            this.health = health;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final String getDeletePropagationPolicy() {
            return deletePropagationPolicy;
        }

        public final void setDeletePropagationPolicy(String deletePropagationPolicy) {
            this.deletePropagationPolicy = deletePropagationPolicy;
        }

        @Override
        public final Builder deletePropagationPolicy(String deletePropagationPolicy) {
            this.deletePropagationPolicy = deletePropagationPolicy;
            return this;
        }

        @Override
        public final Builder deletePropagationPolicy(CapabilityDeletePropagationPolicy deletePropagationPolicy) {
            this.deletePropagationPolicy(deletePropagationPolicy == null ? null : deletePropagationPolicy.toString());
            return this;
        }

        @Override
        public Capability build() {
            return new Capability(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
