/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing an AWS Fargate profile selector.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FargateProfileSelector implements SdkPojo, Serializable,
        ToCopyableBuilder<FargateProfileSelector.Builder, FargateProfileSelector> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespace").getter(getter(FargateProfileSelector::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build()).build();

    private static final SdkField<Map<String, String>> LABELS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("labels")
            .getter(getter(FargateProfileSelector::labels))
            .setter(setter(Builder::labels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("labels").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(NAMESPACE_FIELD, LABELS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String namespace;

    private final Map<String, String> labels;

    private FargateProfileSelector(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.labels = builder.labels;
    }

    /**
     * <p>
     * The Kubernetes namespace that the selector should match.
     * </p>
     * 
     * @return The Kubernetes namespace that the selector should match.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * Returns true if the Labels property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasLabels() {
        return labels != null && !(labels instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The Kubernetes labels that the selector should match. A pod must contain all of the labels that are specified in
     * the selector for it to be considered a match.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLabels()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Kubernetes labels that the selector should match. A pod must contain all of the labels that are
     *         specified in the selector for it to be considered a match.
     */
    public final Map<String, String> labels() {
        return labels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(hasLabels() ? labels() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FargateProfileSelector)) {
            return false;
        }
        FargateProfileSelector other = (FargateProfileSelector) obj;
        return Objects.equals(namespace(), other.namespace()) && hasLabels() == other.hasLabels()
                && Objects.equals(labels(), other.labels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FargateProfileSelector").add("Namespace", namespace())
                .add("Labels", hasLabels() ? labels() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "labels":
            return Optional.ofNullable(clazz.cast(labels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FargateProfileSelector, T> g) {
        return obj -> g.apply((FargateProfileSelector) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FargateProfileSelector> {
        /**
         * <p>
         * The Kubernetes namespace that the selector should match.
         * </p>
         * 
         * @param namespace
         *        The Kubernetes namespace that the selector should match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The Kubernetes labels that the selector should match. A pod must contain all of the labels that are specified
         * in the selector for it to be considered a match.
         * </p>
         * 
         * @param labels
         *        The Kubernetes labels that the selector should match. A pod must contain all of the labels that are
         *        specified in the selector for it to be considered a match.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labels(Map<String, String> labels);
    }

    static final class BuilderImpl implements Builder {
        private String namespace;

        private Map<String, String> labels = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FargateProfileSelector model) {
            namespace(model.namespace);
            labels(model.labels);
        }

        public final String getNamespace() {
            return namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        public final Map<String, String> getLabels() {
            if (labels instanceof SdkAutoConstructMap) {
                return null;
            }
            return labels;
        }

        @Override
        public final Builder labels(Map<String, String> labels) {
            this.labels = FargateProfileLabelCopier.copy(labels);
            return this;
        }

        public final void setLabels(Map<String, String> labels) {
            this.labels = FargateProfileLabelCopier.copy(labels);
        }

        @Override
        public FargateProfileSelector build() {
            return new FargateProfileSelector(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
