/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.eks.model.BadRequestException;
import software.amazon.awssdk.services.eks.model.ClientException;
import software.amazon.awssdk.services.eks.model.CreateClusterRequest;
import software.amazon.awssdk.services.eks.model.CreateClusterResponse;
import software.amazon.awssdk.services.eks.model.CreateFargateProfileRequest;
import software.amazon.awssdk.services.eks.model.CreateFargateProfileResponse;
import software.amazon.awssdk.services.eks.model.CreateNodegroupRequest;
import software.amazon.awssdk.services.eks.model.CreateNodegroupResponse;
import software.amazon.awssdk.services.eks.model.DeleteClusterRequest;
import software.amazon.awssdk.services.eks.model.DeleteClusterResponse;
import software.amazon.awssdk.services.eks.model.DeleteFargateProfileRequest;
import software.amazon.awssdk.services.eks.model.DeleteFargateProfileResponse;
import software.amazon.awssdk.services.eks.model.DeleteNodegroupRequest;
import software.amazon.awssdk.services.eks.model.DeleteNodegroupResponse;
import software.amazon.awssdk.services.eks.model.DescribeClusterRequest;
import software.amazon.awssdk.services.eks.model.DescribeClusterResponse;
import software.amazon.awssdk.services.eks.model.DescribeFargateProfileRequest;
import software.amazon.awssdk.services.eks.model.DescribeFargateProfileResponse;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupRequest;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupResponse;
import software.amazon.awssdk.services.eks.model.DescribeUpdateRequest;
import software.amazon.awssdk.services.eks.model.DescribeUpdateResponse;
import software.amazon.awssdk.services.eks.model.EksException;
import software.amazon.awssdk.services.eks.model.EksRequest;
import software.amazon.awssdk.services.eks.model.InvalidParameterException;
import software.amazon.awssdk.services.eks.model.InvalidRequestException;
import software.amazon.awssdk.services.eks.model.ListClustersRequest;
import software.amazon.awssdk.services.eks.model.ListClustersResponse;
import software.amazon.awssdk.services.eks.model.ListFargateProfilesRequest;
import software.amazon.awssdk.services.eks.model.ListFargateProfilesResponse;
import software.amazon.awssdk.services.eks.model.ListNodegroupsRequest;
import software.amazon.awssdk.services.eks.model.ListNodegroupsResponse;
import software.amazon.awssdk.services.eks.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.eks.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.eks.model.ListUpdatesRequest;
import software.amazon.awssdk.services.eks.model.ListUpdatesResponse;
import software.amazon.awssdk.services.eks.model.NotFoundException;
import software.amazon.awssdk.services.eks.model.ResourceInUseException;
import software.amazon.awssdk.services.eks.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.eks.model.ResourceNotFoundException;
import software.amazon.awssdk.services.eks.model.ServerException;
import software.amazon.awssdk.services.eks.model.ServiceUnavailableException;
import software.amazon.awssdk.services.eks.model.TagResourceRequest;
import software.amazon.awssdk.services.eks.model.TagResourceResponse;
import software.amazon.awssdk.services.eks.model.UnsupportedAvailabilityZoneException;
import software.amazon.awssdk.services.eks.model.UntagResourceRequest;
import software.amazon.awssdk.services.eks.model.UntagResourceResponse;
import software.amazon.awssdk.services.eks.model.UpdateClusterConfigRequest;
import software.amazon.awssdk.services.eks.model.UpdateClusterConfigResponse;
import software.amazon.awssdk.services.eks.model.UpdateClusterVersionRequest;
import software.amazon.awssdk.services.eks.model.UpdateClusterVersionResponse;
import software.amazon.awssdk.services.eks.model.UpdateNodegroupConfigRequest;
import software.amazon.awssdk.services.eks.model.UpdateNodegroupConfigResponse;
import software.amazon.awssdk.services.eks.model.UpdateNodegroupVersionRequest;
import software.amazon.awssdk.services.eks.model.UpdateNodegroupVersionResponse;
import software.amazon.awssdk.services.eks.paginators.ListClustersIterable;
import software.amazon.awssdk.services.eks.paginators.ListFargateProfilesIterable;
import software.amazon.awssdk.services.eks.paginators.ListNodegroupsIterable;
import software.amazon.awssdk.services.eks.paginators.ListUpdatesIterable;
import software.amazon.awssdk.services.eks.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreateFargateProfileRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.CreateNodegroupRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteFargateProfileRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DeleteNodegroupRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeFargateProfileRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeNodegroupRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.DescribeUpdateRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListFargateProfilesRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListNodegroupsRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.ListUpdatesRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateClusterConfigRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateClusterVersionRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateNodegroupConfigRequestMarshaller;
import software.amazon.awssdk.services.eks.transform.UpdateNodegroupVersionRequestMarshaller;

/**
 * Internal implementation of {@link EksClient}.
 *
 * @see EksClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEksClient implements EksClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEksClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Amazon EKS control plane.
     * </p>
     * <p>
     * The Amazon EKS control plane consists of control plane instances that run the Kubernetes software, such as
     * <code>etcd</code> and the API server. The control plane runs in an account managed by AWS, and the Kubernetes API
     * is exposed via the Amazon EKS API server endpoint. Each Amazon EKS cluster control plane is single-tenant and
     * unique and runs on its own set of Amazon EC2 instances.
     * </p>
     * <p>
     * The cluster control plane is provisioned across multiple Availability Zones and fronted by an Elastic Load
     * Balancing Network Load Balancer. Amazon EKS also provisions elastic network interfaces in your VPC subnets to
     * provide connectivity from the control plane instances to the worker nodes (for example, to support
     * <code>kubectl exec</code>, <code>logs</code>, and <code>proxy</code> data flows).
     * </p>
     * <p>
     * Amazon EKS worker nodes run in your AWS account and connect to your cluster's control plane via the Kubernetes
     * API server endpoint and a certificate file that is created for your cluster.
     * </p>
     * <p>
     * You can use the <code>endpointPublicAccess</code> and <code>endpointPrivateAccess</code> parameters to enable or
     * disable public and private access to your cluster's Kubernetes API server endpoint. By default, public access is
     * enabled, and private access is disabled. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">Amazon EKS Cluster Endpoint Access
     * Control</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <p>
     * You can use the <code>logging</code> parameter to enable or disable exporting the Kubernetes control plane logs
     * for your cluster to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html">Amazon EKS Cluster Control Plane
     * Logs</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For
     * more information, see <a href="http://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * </note>
     * <p>
     * Cluster creation typically takes between 10 and 15 minutes. After you create an Amazon EKS cluster, you must
     * configure your Kubernetes tooling to communicate with the API server and launch worker nodes into your cluster.
     * For more information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/managing-auth.html">Managing
     * Cluster Authentication</a> and <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html">Launching Amazon EKS Worker Nodes</a>
     * in the <i>Amazon EKS User Guide</i>.
     * </p>
     *
     * @param createClusterRequest
     * @return Result of the CreateCluster operation returned by the service.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws ResourceLimitExceededException
     *         You have encountered a service limit on the specified resource.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws UnsupportedAvailabilityZoneException
     *         At least one of your specified cluster subnets is in an Availability Zone that does not support Amazon
     *         EKS. The exception output specifies the supported Availability Zones for your account, from which you can
     *         choose subnets for your cluster.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.CreateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws ResourceInUseException,
            ResourceLimitExceededException, InvalidParameterException, ClientException, ServerException,
            ServiceUnavailableException, UnsupportedAvailabilityZoneException, AwsServiceException, SdkClientException,
            EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                .withOperationName("CreateCluster").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createClusterRequest)
                .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an AWS Fargate profile for your Amazon EKS cluster. You must have at least one Fargate profile in a
     * cluster to be able to run pods on Fargate.
     * </p>
     * <p>
     * The Fargate profile allows an administrator to declare which pods run on Fargate and specify which pods run on
     * which Fargate profile. This declaration is done through the profile’s selectors. Each profile can have up to five
     * selectors that contain a namespace and labels. A namespace is required for every selector. The label field
     * consists of multiple optional key-value pairs. Pods that match the selectors are scheduled on Fargate. If a
     * to-be-scheduled pod matches any of the selectors in the Fargate profile, then that pod is run on Fargate.
     * </p>
     * <p>
     * When you create a Fargate profile, you must specify a pod execution role to use with the pods that are scheduled
     * with the profile. This role is added to the cluster's Kubernetes <a
     * href="https://kubernetes.io/docs/admin/authorization/rbac/">Role Based Access Control</a> (RBAC) for
     * authorization so that the <code>kubelet</code> that is running on the Fargate infrastructure can register with
     * your Amazon EKS cluster so that it can appear in your cluster as a node. The pod execution role also provides IAM
     * permissions to the Fargate infrastructure to allow read access to Amazon ECR image repositories. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/pod-execution-role.html">Pod Execution
     * Role</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     * <p>
     * Fargate profiles are immutable. However, you can create a new updated profile to replace an existing profile and
     * then delete the original after the updated profile has finished creating.
     * </p>
     * <p>
     * If any Fargate profiles in a cluster are in the <code>DELETING</code> status, you must wait for that Fargate
     * profile to finish deleting before you can create any other profiles in that cluster.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html">AWS
     * Fargate Profile</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     *
     * @param createFargateProfileRequest
     * @return Result of the CreateFargateProfile operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidRequestException
     *         The request is invalid given the state of the cluster. Check the state of the cluster and the associated
     *         operations.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceLimitExceededException
     *         You have encountered a service limit on the specified resource.
     * @throws UnsupportedAvailabilityZoneException
     *         At least one of your specified cluster subnets is in an Availability Zone that does not support Amazon
     *         EKS. The exception output specifies the supported Availability Zones for your account, from which you can
     *         choose subnets for your cluster.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.CreateFargateProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateFargateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFargateProfileResponse createFargateProfile(CreateFargateProfileRequest createFargateProfileRequest)
            throws InvalidParameterException, InvalidRequestException, ClientException, ServerException,
            ResourceLimitExceededException, UnsupportedAvailabilityZoneException, AwsServiceException, SdkClientException,
            EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFargateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateFargateProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateFargateProfileRequest, CreateFargateProfileResponse>()
                .withOperationName("CreateFargateProfile").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createFargateProfileRequest)
                .withMarshaller(new CreateFargateProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a managed worker node group for an Amazon EKS cluster. You can only create a node group for your cluster
     * that is equal to the current Kubernetes version for the cluster. All node groups are created with the latest AMI
     * release version for the respective minor Kubernetes version of the cluster.
     * </p>
     * <p>
     * An Amazon EKS managed node group is an Amazon EC2 Auto Scaling group and associated Amazon EC2 instances that are
     * managed by AWS for an Amazon EKS cluster. Each node group uses a version of the Amazon EKS-optimized Amazon Linux
     * 2 AMI. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html">Managed Node Groups</a> in the
     * <i>Amazon EKS User Guide</i>.
     * </p>
     *
     * @param createNodegroupRequest
     * @return Result of the CreateNodegroup operation returned by the service.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws ResourceLimitExceededException
     *         You have encountered a service limit on the specified resource.
     * @throws InvalidRequestException
     *         The request is invalid given the state of the cluster. Check the state of the cluster and the associated
     *         operations.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.CreateNodegroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateNodegroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateNodegroupResponse createNodegroup(CreateNodegroupRequest createNodegroupRequest) throws ResourceInUseException,
            ResourceLimitExceededException, InvalidRequestException, InvalidParameterException, ClientException, ServerException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNodegroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateNodegroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateNodegroupRequest, CreateNodegroupResponse>()
                .withOperationName("CreateNodegroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createNodegroupRequest)
                .withMarshaller(new CreateNodegroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the Amazon EKS cluster control plane.
     * </p>
     * <p>
     * If you have active services in your cluster that are associated with a load balancer, you must delete those
     * services before deleting the cluster so that the load balancers are deleted properly. Otherwise, you can have
     * orphaned resources in your VPC that prevent you from being able to delete the VPC. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/delete-cluster.html">Deleting a Cluster</a> in the
     * <i>Amazon EKS User Guide</i>.
     * </p>
     * <p>
     * If you have managed node groups or Fargate profiles attached to the cluster, you must delete them first. For more
     * information, see <a>DeleteNodegroup</a> and <a>DeleteFargateProfile</a>.
     * </p>
     *
     * @param deleteClusterRequest
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.DeleteCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteClusterResponse deleteCluster(DeleteClusterRequest deleteClusterRequest) throws ResourceInUseException,
            ResourceNotFoundException, ClientException, ServerException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                .withOperationName("DeleteCluster").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteClusterRequest)
                .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes an AWS Fargate profile.
     * </p>
     * <p>
     * When you delete a Fargate profile, any pods running on Fargate that were created with the profile are deleted. If
     * those pods match another Fargate profile, then they are scheduled on Fargate with that profile. If they no longer
     * match any Fargate profiles, then they are not scheduled on Fargate and they may remain in a pending state.
     * </p>
     * <p>
     * Only one Fargate profile in a cluster can be in the <code>DELETING</code> status at a time. You must wait for a
     * Fargate profile to finish deleting before you can delete any other profiles in that cluster.
     * </p>
     *
     * @param deleteFargateProfileRequest
     * @return Result of the DeleteFargateProfile operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.DeleteFargateProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteFargateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFargateProfileResponse deleteFargateProfile(DeleteFargateProfileRequest deleteFargateProfileRequest)
            throws InvalidParameterException, ClientException, ServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFargateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteFargateProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteFargateProfileRequest, DeleteFargateProfileResponse>()
                .withOperationName("DeleteFargateProfile").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteFargateProfileRequest)
                .withMarshaller(new DeleteFargateProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes an Amazon EKS node group for a cluster.
     * </p>
     *
     * @param deleteNodegroupRequest
     * @return Result of the DeleteNodegroup operation returned by the service.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.DeleteNodegroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteNodegroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteNodegroupResponse deleteNodegroup(DeleteNodegroupRequest deleteNodegroupRequest) throws ResourceInUseException,
            ResourceNotFoundException, InvalidParameterException, ClientException, ServerException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNodegroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteNodegroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteNodegroupRequest, DeleteNodegroupResponse>()
                .withOperationName("DeleteNodegroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteNodegroupRequest)
                .withMarshaller(new DeleteNodegroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns descriptive information about an Amazon EKS cluster.
     * </p>
     * <p>
     * The API server endpoint and certificate authority data returned by this operation are required for
     * <code>kubelet</code> and <code>kubectl</code> to communicate with your Kubernetes API server. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/create-kubeconfig.html">Create a
     * kubeconfig for Amazon EKS</a>.
     * </p>
     * <note>
     * <p>
     * The API server endpoint and certificate authority data aren't available until the cluster reaches the
     * <code>ACTIVE</code> state.
     * </p>
     * </note>
     *
     * @param describeClusterRequest
     * @return Result of the DescribeCluster operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.DescribeCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeClusterResponse describeCluster(DescribeClusterRequest describeClusterRequest)
            throws ResourceNotFoundException, ClientException, ServerException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                .withOperationName("DescribeCluster").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeClusterRequest)
                .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns descriptive information about an AWS Fargate profile.
     * </p>
     *
     * @param describeFargateProfileRequest
     * @return Result of the DescribeFargateProfile operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.DescribeFargateProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeFargateProfile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFargateProfileResponse describeFargateProfile(DescribeFargateProfileRequest describeFargateProfileRequest)
            throws InvalidParameterException, ClientException, ServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFargateProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFargateProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeFargateProfileRequest, DescribeFargateProfileResponse>()
                .withOperationName("DescribeFargateProfile").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeFargateProfileRequest)
                .withMarshaller(new DescribeFargateProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns descriptive information about an Amazon EKS node group.
     * </p>
     *
     * @param describeNodegroupRequest
     * @return Result of the DescribeNodegroup operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.DescribeNodegroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeNodegroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeNodegroupResponse describeNodegroup(DescribeNodegroupRequest describeNodegroupRequest)
            throws InvalidParameterException, ResourceNotFoundException, ClientException, ServerException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNodegroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeNodegroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeNodegroupRequest, DescribeNodegroupResponse>()
                .withOperationName("DescribeNodegroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeNodegroupRequest)
                .withMarshaller(new DescribeNodegroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns descriptive information about an update against your Amazon EKS cluster or associated managed node group.
     * </p>
     * <p>
     * When the status of the update is <code>Succeeded</code>, the update is complete. If an update fails, the status
     * is <code>Failed</code>, and an error detail explains the reason for the failure.
     * </p>
     *
     * @param describeUpdateRequest
     * @return Result of the DescribeUpdate operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.DescribeUpdate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeUpdate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeUpdateResponse describeUpdate(DescribeUpdateRequest describeUpdateRequest) throws InvalidParameterException,
            ClientException, ServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUpdateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUpdateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeUpdateRequest, DescribeUpdateResponse>()
                .withOperationName("DescribeUpdate").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeUpdateRequest)
                .withMarshaller(new DescribeUpdateRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the Amazon EKS clusters in your AWS account in the specified Region.
     * </p>
     *
     * @param listClustersRequest
     * @return Result of the ListClusters operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws InvalidParameterException,
            ClientException, ServerException, ServiceUnavailableException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                .withOperationName("ListClusters").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listClustersRequest)
                .withMarshaller(new ListClustersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the Amazon EKS clusters in your AWS account in the specified Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listClusters(software.amazon.awssdk.services.eks.model.ListClustersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.eks.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.eks.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     *     for (software.amazon.awssdk.services.eks.model.ListClustersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.eks.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.eks.model.ListClustersRequest)} operation.</b>
     * </p>
     *
     * @param listClustersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListClustersIterable listClustersPaginator(ListClustersRequest listClustersRequest) throws InvalidParameterException,
            ClientException, ServerException, ServiceUnavailableException, AwsServiceException, SdkClientException, EksException {
        return new ListClustersIterable(this, applyPaginatorUserAgent(listClustersRequest));
    }

    /**
     * <p>
     * Lists the AWS Fargate profiles associated with the specified cluster in your AWS account in the specified Region.
     * </p>
     *
     * @param listFargateProfilesRequest
     * @return Result of the ListFargateProfiles operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListFargateProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListFargateProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFargateProfilesResponse listFargateProfiles(ListFargateProfilesRequest listFargateProfilesRequest)
            throws InvalidParameterException, ResourceNotFoundException, ClientException, ServerException, AwsServiceException,
            SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFargateProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFargateProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListFargateProfilesRequest, ListFargateProfilesResponse>()
                .withOperationName("ListFargateProfiles").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listFargateProfilesRequest)
                .withMarshaller(new ListFargateProfilesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the AWS Fargate profiles associated with the specified cluster in your AWS account in the specified Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFargateProfiles(software.amazon.awssdk.services.eks.model.ListFargateProfilesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.eks.paginators.ListFargateProfilesIterable responses = client.listFargateProfilesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.eks.paginators.ListFargateProfilesIterable responses = client
     *             .listFargateProfilesPaginator(request);
     *     for (software.amazon.awssdk.services.eks.model.ListFargateProfilesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.eks.paginators.ListFargateProfilesIterable responses = client.listFargateProfilesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFargateProfiles(software.amazon.awssdk.services.eks.model.ListFargateProfilesRequest)} operation.</b>
     * </p>
     *
     * @param listFargateProfilesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListFargateProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListFargateProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFargateProfilesIterable listFargateProfilesPaginator(ListFargateProfilesRequest listFargateProfilesRequest)
            throws InvalidParameterException, ResourceNotFoundException, ClientException, ServerException, AwsServiceException,
            SdkClientException, EksException {
        return new ListFargateProfilesIterable(this, applyPaginatorUserAgent(listFargateProfilesRequest));
    }

    /**
     * <p>
     * Lists the Amazon EKS managed node groups associated with the specified cluster in your AWS account in the
     * specified Region. Self-managed node groups are not listed.
     * </p>
     *
     * @param listNodegroupsRequest
     * @return Result of the ListNodegroups operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListNodegroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListNodegroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListNodegroupsResponse listNodegroups(ListNodegroupsRequest listNodegroupsRequest) throws InvalidParameterException,
            ClientException, ServerException, ServiceUnavailableException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNodegroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNodegroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListNodegroupsRequest, ListNodegroupsResponse>()
                .withOperationName("ListNodegroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listNodegroupsRequest)
                .withMarshaller(new ListNodegroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the Amazon EKS managed node groups associated with the specified cluster in your AWS account in the
     * specified Region. Self-managed node groups are not listed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listNodegroups(software.amazon.awssdk.services.eks.model.ListNodegroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.eks.paginators.ListNodegroupsIterable responses = client.listNodegroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.eks.paginators.ListNodegroupsIterable responses = client.listNodegroupsPaginator(request);
     *     for (software.amazon.awssdk.services.eks.model.ListNodegroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.eks.paginators.ListNodegroupsIterable responses = client.listNodegroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNodegroups(software.amazon.awssdk.services.eks.model.ListNodegroupsRequest)} operation.</b>
     * </p>
     *
     * @param listNodegroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListNodegroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListNodegroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListNodegroupsIterable listNodegroupsPaginator(ListNodegroupsRequest listNodegroupsRequest)
            throws InvalidParameterException, ClientException, ServerException, ServiceUnavailableException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EksException {
        return new ListNodegroupsIterable(this, applyPaginatorUserAgent(listNodegroupsRequest));
    }

    /**
     * <p>
     * List the tags for an Amazon EKS resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         This exception is thrown if the request contains a semantic error. The precise meaning will depend on the
     *         API, and will be documented in the error message.
     * @throws NotFoundException
     *         A service resource associated with the request could not be found. Clients should not retry such
     *         requests.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, NotFoundException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the updates associated with an Amazon EKS cluster or managed node group in your AWS account, in the
     * specified Region.
     * </p>
     *
     * @param listUpdatesRequest
     * @return Result of the ListUpdates operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListUpdates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListUpdates" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUpdatesResponse listUpdates(ListUpdatesRequest listUpdatesRequest) throws InvalidParameterException,
            ClientException, ServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUpdatesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUpdatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListUpdatesRequest, ListUpdatesResponse>()
                .withOperationName("ListUpdates").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listUpdatesRequest)
                .withMarshaller(new ListUpdatesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the updates associated with an Amazon EKS cluster or managed node group in your AWS account, in the
     * specified Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUpdates(software.amazon.awssdk.services.eks.model.ListUpdatesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.eks.paginators.ListUpdatesIterable responses = client.listUpdatesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.eks.paginators.ListUpdatesIterable responses = client.listUpdatesPaginator(request);
     *     for (software.amazon.awssdk.services.eks.model.ListUpdatesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.eks.paginators.ListUpdatesIterable responses = client.listUpdatesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUpdates(software.amazon.awssdk.services.eks.model.ListUpdatesRequest)} operation.</b>
     * </p>
     *
     * @param listUpdatesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.ListUpdates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListUpdates" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUpdatesIterable listUpdatesPaginator(ListUpdatesRequest listUpdatesRequest) throws InvalidParameterException,
            ClientException, ServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, EksException {
        return new ListUpdatesIterable(this, applyPaginatorUserAgent(listUpdatesRequest));
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource are not specified in the request parameters, they are not changed. When a resource is deleted, the tags
     * associated with that resource are deleted as well. Tags that you create for Amazon EKS resources do not propagate
     * to any other resources associated with the cluster. For example, if you tag a cluster with this operation, that
     * tag does not automatically propagate to the subnets and worker nodes associated with the cluster.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         This exception is thrown if the request contains a semantic error. The precise meaning will depend on the
     *         API, and will be documented in the error message.
     * @throws NotFoundException
     *         A service resource associated with the request could not be found. Clients should not retry such
     *         requests.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException, NotFoundException,
            AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         This exception is thrown if the request contains a semantic error. The precise meaning will depend on the
     *         API, and will be documented in the error message.
     * @throws NotFoundException
     *         A service resource associated with the request could not be found. Clients should not retry such
     *         requests.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            NotFoundException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an Amazon EKS cluster configuration. Your cluster continues to function during the update. The response
     * output includes an update ID that you can use to track the status of your cluster update with the
     * <a>DescribeUpdate</a> API operation.
     * </p>
     * <p>
     * You can use this API operation to enable or disable exporting the Kubernetes control plane logs for your cluster
     * to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html">Amazon EKS
     * Cluster Control Plane Logs</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For
     * more information, see <a href="http://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * </note>
     * <p>
     * You can also use this API operation to enable or disable public and private access to your cluster's Kubernetes
     * API server endpoint. By default, public access is enabled, and private access is disabled. For more information,
     * see <a href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">Amazon EKS Cluster Endpoint
     * Access Control</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <important>
     * <p>
     * At this time, you can not update the subnets or security group IDs for an existing cluster.
     * </p>
     * </important>
     * <p>
     * Cluster updates are asynchronous, and they should finish within a few minutes. During an update, the cluster
     * status moves to <code>UPDATING</code> (this status transition is eventually consistent). When the update is
     * complete (either <code>Failed</code> or <code>Successful</code>), the cluster status moves to <code>Active</code>
     * .
     * </p>
     *
     * @param updateClusterConfigRequest
     * @return Result of the UpdateClusterConfig operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws InvalidRequestException
     *         The request is invalid given the state of the cluster. Check the state of the cluster and the associated
     *         operations.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.UpdateClusterConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateClusterConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateClusterConfigResponse updateClusterConfig(UpdateClusterConfigRequest updateClusterConfigRequest)
            throws InvalidParameterException, ClientException, ServerException, ResourceInUseException,
            ResourceNotFoundException, InvalidRequestException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateClusterConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateClusterConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateClusterConfigRequest, UpdateClusterConfigResponse>()
                .withOperationName("UpdateClusterConfig").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateClusterConfigRequest)
                .withMarshaller(new UpdateClusterConfigRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an Amazon EKS cluster to the specified Kubernetes version. Your cluster continues to function during the
     * update. The response output includes an update ID that you can use to track the status of your cluster update
     * with the <a>DescribeUpdate</a> API operation.
     * </p>
     * <p>
     * Cluster updates are asynchronous, and they should finish within a few minutes. During an update, the cluster
     * status moves to <code>UPDATING</code> (this status transition is eventually consistent). When the update is
     * complete (either <code>Failed</code> or <code>Successful</code>), the cluster status moves to <code>Active</code>
     * .
     * </p>
     * <p>
     * If your cluster has managed node groups attached to it, all of your node groups’ Kubernetes versions must match
     * the cluster’s Kubernetes version in order to update the cluster to a new Kubernetes version.
     * </p>
     *
     * @param updateClusterVersionRequest
     * @return Result of the UpdateClusterVersion operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws InvalidRequestException
     *         The request is invalid given the state of the cluster. Check the state of the cluster and the associated
     *         operations.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.UpdateClusterVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateClusterVersion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateClusterVersionResponse updateClusterVersion(UpdateClusterVersionRequest updateClusterVersionRequest)
            throws InvalidParameterException, ClientException, ServerException, ResourceInUseException,
            ResourceNotFoundException, InvalidRequestException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateClusterVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateClusterVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateClusterVersionRequest, UpdateClusterVersionResponse>()
                .withOperationName("UpdateClusterVersion").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateClusterVersionRequest)
                .withMarshaller(new UpdateClusterVersionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an Amazon EKS managed node group configuration. Your node group continues to function during the update.
     * The response output includes an update ID that you can use to track the status of your node group update with the
     * <a>DescribeUpdate</a> API operation. Currently you can update the Kubernetes labels for a node group or the
     * scaling configuration.
     * </p>
     *
     * @param updateNodegroupConfigRequest
     * @return Result of the UpdateNodegroupConfig operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws InvalidRequestException
     *         The request is invalid given the state of the cluster. Check the state of the cluster and the associated
     *         operations.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.UpdateNodegroupConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateNodegroupConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateNodegroupConfigResponse updateNodegroupConfig(UpdateNodegroupConfigRequest updateNodegroupConfigRequest)
            throws InvalidParameterException, ClientException, ServerException, ResourceInUseException,
            ResourceNotFoundException, InvalidRequestException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNodegroupConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNodegroupConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateNodegroupConfigRequest, UpdateNodegroupConfigResponse>()
                .withOperationName("UpdateNodegroupConfig").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateNodegroupConfigRequest)
                .withMarshaller(new UpdateNodegroupConfigRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the Kubernetes version or AMI version of an Amazon EKS managed node group.
     * </p>
     * <p>
     * You can update to the latest available AMI version of a node group's current Kubernetes version by not specifying
     * a Kubernetes version in the request. You can update to the latest AMI version of your cluster's current
     * Kubernetes version by specifying your cluster's Kubernetes version in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html">Amazon EKS-Optimized Linux
     * AMI Versions</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     * <p>
     * You cannot roll back a node group to an earlier Kubernetes version or AMI version.
     * </p>
     * <p>
     * When a node in a managed node group is terminated due to a scaling action or update, the pods in that node are
     * drained first. Amazon EKS attempts to drain the nodes gracefully and will fail if it is unable to do so. You can
     * <code>force</code> the update if Amazon EKS is unable to drain the nodes as a result of a pod disruption budget
     * issue.
     * </p>
     *
     * @param updateNodegroupVersionRequest
     * @return Result of the UpdateNodegroupVersion operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ClientException
     *         These errors are usually caused by a client action. Actions can include using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource or specifying an identifier
     *         that is not valid.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ResourceInUseException
     *         The specified resource is in use.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found. You can view your available clusters with <a>ListClusters</a>.
     *         You can view your available managed node groups with <a>ListNodegroups</a>. Amazon EKS clusters and node
     *         groups are Region-specific.
     * @throws InvalidRequestException
     *         The request is invalid given the state of the cluster. Check the state of the cluster and the associated
     *         operations.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EksClient.UpdateNodegroupVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateNodegroupVersion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateNodegroupVersionResponse updateNodegroupVersion(UpdateNodegroupVersionRequest updateNodegroupVersionRequest)
            throws InvalidParameterException, ClientException, ServerException, ResourceInUseException,
            ResourceNotFoundException, InvalidRequestException, AwsServiceException, SdkClientException, EksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNodegroupVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNodegroupVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateNodegroupVersionRequest, UpdateNodegroupVersionResponse>()
                .withOperationName("UpdateNodegroupVersion").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateNodegroupVersionRequest)
                .withMarshaller(new UpdateNodegroupVersionRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EksException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedAvailabilityZoneException")
                                .exceptionBuilderSupplier(UnsupportedAvailabilityZoneException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerException")
                                .exceptionBuilderSupplier(ServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientException")
                                .exceptionBuilderSupplier(ClientException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends EksRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
