/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the resources available for a container instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Resource implements SdkPojo, Serializable, ToCopyableBuilder<Resource.Builder, Resource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Resource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Resource::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<Double> DOUBLE_VALUE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("doubleValue").getter(getter(Resource::doubleValue)).setter(setter(Builder::doubleValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("doubleValue").build()).build();

    private static final SdkField<Long> LONG_VALUE_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("longValue")
            .getter(getter(Resource::longValue)).setter(setter(Builder::longValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("longValue").build()).build();

    private static final SdkField<Integer> INTEGER_VALUE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("integerValue").getter(getter(Resource::integerValue)).setter(setter(Builder::integerValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("integerValue").build()).build();

    private static final SdkField<List<String>> STRING_SET_VALUE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("stringSetValue")
            .getter(getter(Resource::stringSetValue))
            .setter(setter(Builder::stringSetValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stringSetValue").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TYPE_FIELD,
            DOUBLE_VALUE_FIELD, LONG_VALUE_FIELD, INTEGER_VALUE_FIELD, STRING_SET_VALUE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String type;

    private final Double doubleValue;

    private final Long longValue;

    private final Integer integerValue;

    private final List<String> stringSetValue;

    private Resource(BuilderImpl builder) {
        this.name = builder.name;
        this.type = builder.type;
        this.doubleValue = builder.doubleValue;
        this.longValue = builder.longValue;
        this.integerValue = builder.integerValue;
        this.stringSetValue = builder.stringSetValue;
    }

    /**
     * <p>
     * The name of the resource, such as <code>CPU</code>, <code>MEMORY</code>, <code>PORTS</code>,
     * <code>PORTS_UDP</code>, or a user-defined resource.
     * </p>
     * 
     * @return The name of the resource, such as <code>CPU</code>, <code>MEMORY</code>, <code>PORTS</code>,
     *         <code>PORTS_UDP</code>, or a user-defined resource.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of the resource. Valid values: <code>INTEGER</code>, <code>DOUBLE</code>, <code>LONG</code>, or
     * <code>STRINGSET</code>.
     * </p>
     * 
     * @return The type of the resource. Valid values: <code>INTEGER</code>, <code>DOUBLE</code>, <code>LONG</code>, or
     *         <code>STRINGSET</code>.
     */
    public final String type() {
        return type;
    }

    /**
     * <p>
     * When the <code>doubleValue</code> type is set, the value of the resource must be a double precision
     * floating-point type.
     * </p>
     * 
     * @return When the <code>doubleValue</code> type is set, the value of the resource must be a double precision
     *         floating-point type.
     */
    public final Double doubleValue() {
        return doubleValue;
    }

    /**
     * <p>
     * When the <code>longValue</code> type is set, the value of the resource must be an extended precision
     * floating-point type.
     * </p>
     * 
     * @return When the <code>longValue</code> type is set, the value of the resource must be an extended precision
     *         floating-point type.
     */
    public final Long longValue() {
        return longValue;
    }

    /**
     * <p>
     * When the <code>integerValue</code> type is set, the value of the resource must be an integer.
     * </p>
     * 
     * @return When the <code>integerValue</code> type is set, the value of the resource must be an integer.
     */
    public final Integer integerValue() {
        return integerValue;
    }

    /**
     * For responses, this returns true if the service returned a value for the StringSetValue property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStringSetValue() {
        return stringSetValue != null && !(stringSetValue instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStringSetValue} method.
     * </p>
     * 
     * @return When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
     */
    public final List<String> stringSetValue() {
        return stringSetValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(doubleValue());
        hashCode = 31 * hashCode + Objects.hashCode(longValue());
        hashCode = 31 * hashCode + Objects.hashCode(integerValue());
        hashCode = 31 * hashCode + Objects.hashCode(hasStringSetValue() ? stringSetValue() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Resource)) {
            return false;
        }
        Resource other = (Resource) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(type(), other.type())
                && Objects.equals(doubleValue(), other.doubleValue()) && Objects.equals(longValue(), other.longValue())
                && Objects.equals(integerValue(), other.integerValue()) && hasStringSetValue() == other.hasStringSetValue()
                && Objects.equals(stringSetValue(), other.stringSetValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Resource").add("Name", name()).add("Type", type()).add("DoubleValue", doubleValue())
                .add("LongValue", longValue()).add("IntegerValue", integerValue())
                .add("StringSetValue", hasStringSetValue() ? stringSetValue() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "type":
            return Optional.ofNullable(clazz.cast(type()));
        case "doubleValue":
            return Optional.ofNullable(clazz.cast(doubleValue()));
        case "longValue":
            return Optional.ofNullable(clazz.cast(longValue()));
        case "integerValue":
            return Optional.ofNullable(clazz.cast(integerValue()));
        case "stringSetValue":
            return Optional.ofNullable(clazz.cast(stringSetValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("doubleValue", DOUBLE_VALUE_FIELD);
        map.put("longValue", LONG_VALUE_FIELD);
        map.put("integerValue", INTEGER_VALUE_FIELD);
        map.put("stringSetValue", STRING_SET_VALUE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Resource, T> g) {
        return obj -> g.apply((Resource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Resource> {
        /**
         * <p>
         * The name of the resource, such as <code>CPU</code>, <code>MEMORY</code>, <code>PORTS</code>,
         * <code>PORTS_UDP</code>, or a user-defined resource.
         * </p>
         * 
         * @param name
         *        The name of the resource, such as <code>CPU</code>, <code>MEMORY</code>, <code>PORTS</code>,
         *        <code>PORTS_UDP</code>, or a user-defined resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of the resource. Valid values: <code>INTEGER</code>, <code>DOUBLE</code>, <code>LONG</code>, or
         * <code>STRINGSET</code>.
         * </p>
         * 
         * @param type
         *        The type of the resource. Valid values: <code>INTEGER</code>, <code>DOUBLE</code>, <code>LONG</code>,
         *        or <code>STRINGSET</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * When the <code>doubleValue</code> type is set, the value of the resource must be a double precision
         * floating-point type.
         * </p>
         * 
         * @param doubleValue
         *        When the <code>doubleValue</code> type is set, the value of the resource must be a double precision
         *        floating-point type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder doubleValue(Double doubleValue);

        /**
         * <p>
         * When the <code>longValue</code> type is set, the value of the resource must be an extended precision
         * floating-point type.
         * </p>
         * 
         * @param longValue
         *        When the <code>longValue</code> type is set, the value of the resource must be an extended precision
         *        floating-point type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder longValue(Long longValue);

        /**
         * <p>
         * When the <code>integerValue</code> type is set, the value of the resource must be an integer.
         * </p>
         * 
         * @param integerValue
         *        When the <code>integerValue</code> type is set, the value of the resource must be an integer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integerValue(Integer integerValue);

        /**
         * <p>
         * When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
         * </p>
         * 
         * @param stringSetValue
         *        When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringSetValue(Collection<String> stringSetValue);

        /**
         * <p>
         * When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
         * </p>
         * 
         * @param stringSetValue
         *        When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringSetValue(String... stringSetValue);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String type;

        private Double doubleValue;

        private Long longValue;

        private Integer integerValue;

        private List<String> stringSetValue = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Resource model) {
            name(model.name);
            type(model.type);
            doubleValue(model.doubleValue);
            longValue(model.longValue);
            integerValue(model.integerValue);
            stringSetValue(model.stringSetValue);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final Double getDoubleValue() {
            return doubleValue;
        }

        public final void setDoubleValue(Double doubleValue) {
            this.doubleValue = doubleValue;
        }

        @Override
        public final Builder doubleValue(Double doubleValue) {
            this.doubleValue = doubleValue;
            return this;
        }

        public final Long getLongValue() {
            return longValue;
        }

        public final void setLongValue(Long longValue) {
            this.longValue = longValue;
        }

        @Override
        public final Builder longValue(Long longValue) {
            this.longValue = longValue;
            return this;
        }

        public final Integer getIntegerValue() {
            return integerValue;
        }

        public final void setIntegerValue(Integer integerValue) {
            this.integerValue = integerValue;
        }

        @Override
        public final Builder integerValue(Integer integerValue) {
            this.integerValue = integerValue;
            return this;
        }

        public final Collection<String> getStringSetValue() {
            if (stringSetValue instanceof SdkAutoConstructList) {
                return null;
            }
            return stringSetValue;
        }

        public final void setStringSetValue(Collection<String> stringSetValue) {
            this.stringSetValue = StringListCopier.copy(stringSetValue);
        }

        @Override
        public final Builder stringSetValue(Collection<String> stringSetValue) {
            this.stringSetValue = StringListCopier.copy(stringSetValue);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stringSetValue(String... stringSetValue) {
            stringSetValue(Arrays.asList(stringSetValue));
            return this;
        }

        @Override
        public Resource build() {
            return new Resource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
