/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.document.Document;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A deployment lifecycle hook runs custom logic at specific stages of the deployment process. Currently, you can use
 * Lambda functions as hook targets.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-lifecycle-hooks.html">Lifecycle hooks
 * for Amazon ECS service deployments</a> in the <i> Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentLifecycleHook implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentLifecycleHook.Builder, DeploymentLifecycleHook> {
    private static final SdkField<String> HOOK_TARGET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hookTargetArn").getter(getter(DeploymentLifecycleHook::hookTargetArn))
            .setter(setter(Builder::hookTargetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hookTargetArn").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(DeploymentLifecycleHook::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<List<String>> LIFECYCLE_STAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("lifecycleStages")
            .getter(getter(DeploymentLifecycleHook::lifecycleStagesAsStrings))
            .setter(setter(Builder::lifecycleStagesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecycleStages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Document> HOOK_DETAILS_FIELD = SdkField.<Document> builder(MarshallingType.DOCUMENT)
            .memberName("hookDetails").getter(getter(DeploymentLifecycleHook::hookDetails)).setter(setter(Builder::hookDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hookDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOOK_TARGET_ARN_FIELD,
            ROLE_ARN_FIELD, LIFECYCLE_STAGES_FIELD, HOOK_DETAILS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String hookTargetArn;

    private final String roleArn;

    private final List<String> lifecycleStages;

    private final Document hookDetails;

    private DeploymentLifecycleHook(BuilderImpl builder) {
        this.hookTargetArn = builder.hookTargetArn;
        this.roleArn = builder.roleArn;
        this.lifecycleStages = builder.lifecycleStages;
        this.hookDetails = builder.hookDetails;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the hook target. Currently, only Lambda function ARNs are supported.
     * </p>
     * <p>
     * You must provide this parameter when configuring a deployment lifecycle hook.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the hook target. Currently, only Lambda function ARNs are
     *         supported.</p>
     *         <p>
     *         You must provide this parameter when configuring a deployment lifecycle hook.
     */
    public final String hookTargetArn() {
        return hookTargetArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that grants Amazon ECS permission to call Lambda functions on your
     * behalf.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/blue-green-permissions.html">Permissions
     * required for Lambda functions in Amazon ECS blue/green deployments</a> in the <i> Amazon Elastic Container
     * Service Developer Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that grants Amazon ECS permission to call Lambda functions
     *         on your behalf.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/blue-green-permissions.html"
     *         >Permissions required for Lambda functions in Amazon ECS blue/green deployments</a> in the <i> Amazon
     *         Elastic Container Service Developer Guide</i>.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The lifecycle stages at which to run the hook. Choose from these valid values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * RECONCILE_SERVICE
     * </p>
     * <p>
     * The reconciliation stage that only happens when you start a new service deployment with more than 1 service
     * revision in an ACTIVE state.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * PRE_SCALE_UP
     * </p>
     * <p>
     * The green service revision has not started. The blue service revision is handling 100% of the production traffic.
     * There is no test traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * POST_SCALE_UP
     * </p>
     * <p>
     * The green service revision has started. The blue service revision is handling 100% of the production traffic.
     * There is no test traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * TEST_TRAFFIC_SHIFT
     * </p>
     * <p>
     * The blue and green service revisions are running. The blue service revision handles 100% of the production
     * traffic. The green service revision is migrating from 0% to 100% of test traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * POST_TEST_TRAFFIC_SHIFT
     * </p>
     * <p>
     * The test traffic shift is complete. The green service revision handles 100% of the test traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * PRODUCTION_TRAFFIC_SHIFT
     * </p>
     * <p>
     * Production traffic is shifting to the green service revision. The green service revision is migrating from 0% to
     * 100% of production traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * POST_PRODUCTION_TRAFFIC_SHIFT
     * </p>
     * <p>
     * The production traffic shift is complete.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You must provide this parameter when configuring a deployment lifecycle hook.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLifecycleStages} method.
     * </p>
     * 
     * @return The lifecycle stages at which to run the hook. Choose from these valid values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         RECONCILE_SERVICE
     *         </p>
     *         <p>
     *         The reconciliation stage that only happens when you start a new service deployment with more than 1
     *         service revision in an ACTIVE state.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PRE_SCALE_UP
     *         </p>
     *         <p>
     *         The green service revision has not started. The blue service revision is handling 100% of the production
     *         traffic. There is no test traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         POST_SCALE_UP
     *         </p>
     *         <p>
     *         The green service revision has started. The blue service revision is handling 100% of the production
     *         traffic. There is no test traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         TEST_TRAFFIC_SHIFT
     *         </p>
     *         <p>
     *         The blue and green service revisions are running. The blue service revision handles 100% of the
     *         production traffic. The green service revision is migrating from 0% to 100% of test traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         POST_TEST_TRAFFIC_SHIFT
     *         </p>
     *         <p>
     *         The test traffic shift is complete. The green service revision handles 100% of the test traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PRODUCTION_TRAFFIC_SHIFT
     *         </p>
     *         <p>
     *         Production traffic is shifting to the green service revision. The green service revision is migrating
     *         from 0% to 100% of production traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         POST_PRODUCTION_TRAFFIC_SHIFT
     *         </p>
     *         <p>
     *         The production traffic shift is complete.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You must provide this parameter when configuring a deployment lifecycle hook.
     */
    public final List<DeploymentLifecycleHookStage> lifecycleStages() {
        return DeploymentLifecycleHookStageListCopier.copyStringToEnum(lifecycleStages);
    }

    /**
     * For responses, this returns true if the service returned a value for the LifecycleStages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLifecycleStages() {
        return lifecycleStages != null && !(lifecycleStages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The lifecycle stages at which to run the hook. Choose from these valid values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * RECONCILE_SERVICE
     * </p>
     * <p>
     * The reconciliation stage that only happens when you start a new service deployment with more than 1 service
     * revision in an ACTIVE state.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * PRE_SCALE_UP
     * </p>
     * <p>
     * The green service revision has not started. The blue service revision is handling 100% of the production traffic.
     * There is no test traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * POST_SCALE_UP
     * </p>
     * <p>
     * The green service revision has started. The blue service revision is handling 100% of the production traffic.
     * There is no test traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * TEST_TRAFFIC_SHIFT
     * </p>
     * <p>
     * The blue and green service revisions are running. The blue service revision handles 100% of the production
     * traffic. The green service revision is migrating from 0% to 100% of test traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * POST_TEST_TRAFFIC_SHIFT
     * </p>
     * <p>
     * The test traffic shift is complete. The green service revision handles 100% of the test traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * PRODUCTION_TRAFFIC_SHIFT
     * </p>
     * <p>
     * Production traffic is shifting to the green service revision. The green service revision is migrating from 0% to
     * 100% of production traffic.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * POST_PRODUCTION_TRAFFIC_SHIFT
     * </p>
     * <p>
     * The production traffic shift is complete.
     * </p>
     * <p>
     * You can use a lifecycle hook for this stage.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You must provide this parameter when configuring a deployment lifecycle hook.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLifecycleStages} method.
     * </p>
     * 
     * @return The lifecycle stages at which to run the hook. Choose from these valid values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         RECONCILE_SERVICE
     *         </p>
     *         <p>
     *         The reconciliation stage that only happens when you start a new service deployment with more than 1
     *         service revision in an ACTIVE state.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PRE_SCALE_UP
     *         </p>
     *         <p>
     *         The green service revision has not started. The blue service revision is handling 100% of the production
     *         traffic. There is no test traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         POST_SCALE_UP
     *         </p>
     *         <p>
     *         The green service revision has started. The blue service revision is handling 100% of the production
     *         traffic. There is no test traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         TEST_TRAFFIC_SHIFT
     *         </p>
     *         <p>
     *         The blue and green service revisions are running. The blue service revision handles 100% of the
     *         production traffic. The green service revision is migrating from 0% to 100% of test traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         POST_TEST_TRAFFIC_SHIFT
     *         </p>
     *         <p>
     *         The test traffic shift is complete. The green service revision handles 100% of the test traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PRODUCTION_TRAFFIC_SHIFT
     *         </p>
     *         <p>
     *         Production traffic is shifting to the green service revision. The green service revision is migrating
     *         from 0% to 100% of production traffic.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         POST_PRODUCTION_TRAFFIC_SHIFT
     *         </p>
     *         <p>
     *         The production traffic shift is complete.
     *         </p>
     *         <p>
     *         You can use a lifecycle hook for this stage.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You must provide this parameter when configuring a deployment lifecycle hook.
     */
    public final List<String> lifecycleStagesAsStrings() {
        return lifecycleStages;
    }

    /**
     * <p>
     * Use this field to specify custom parameters that Amazon ECS will pass to your hook target invocations (such as a
     * Lambda function).
     * </p>
     * 
     * @return Use this field to specify custom parameters that Amazon ECS will pass to your hook target invocations
     *         (such as a Lambda function).
     */
    public final Document hookDetails() {
        return hookDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hookTargetArn());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasLifecycleStages() ? lifecycleStagesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hookDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentLifecycleHook)) {
            return false;
        }
        DeploymentLifecycleHook other = (DeploymentLifecycleHook) obj;
        return Objects.equals(hookTargetArn(), other.hookTargetArn()) && Objects.equals(roleArn(), other.roleArn())
                && hasLifecycleStages() == other.hasLifecycleStages()
                && Objects.equals(lifecycleStagesAsStrings(), other.lifecycleStagesAsStrings())
                && Objects.equals(hookDetails(), other.hookDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeploymentLifecycleHook").add("HookTargetArn", hookTargetArn()).add("RoleArn", roleArn())
                .add("LifecycleStages", hasLifecycleStages() ? lifecycleStagesAsStrings() : null)
                .add("HookDetails", hookDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "hookTargetArn":
            return Optional.ofNullable(clazz.cast(hookTargetArn()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "lifecycleStages":
            return Optional.ofNullable(clazz.cast(lifecycleStagesAsStrings()));
        case "hookDetails":
            return Optional.ofNullable(clazz.cast(hookDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("hookTargetArn", HOOK_TARGET_ARN_FIELD);
        map.put("roleArn", ROLE_ARN_FIELD);
        map.put("lifecycleStages", LIFECYCLE_STAGES_FIELD);
        map.put("hookDetails", HOOK_DETAILS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DeploymentLifecycleHook, T> g) {
        return obj -> g.apply((DeploymentLifecycleHook) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentLifecycleHook> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the hook target. Currently, only Lambda function ARNs are supported.
         * </p>
         * <p>
         * You must provide this parameter when configuring a deployment lifecycle hook.
         * </p>
         * 
         * @param hookTargetArn
         *        The Amazon Resource Name (ARN) of the hook target. Currently, only Lambda function ARNs are
         *        supported.</p>
         *        <p>
         *        You must provide this parameter when configuring a deployment lifecycle hook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hookTargetArn(String hookTargetArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that grants Amazon ECS permission to call Lambda functions on
         * your behalf.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/blue-green-permissions.html">Permissions
         * required for Lambda functions in Amazon ECS blue/green deployments</a> in the <i> Amazon Elastic Container
         * Service Developer Guide</i>.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role that grants Amazon ECS permission to call Lambda
         *        functions on your behalf.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/blue-green-permissions.html"
         *        >Permissions required for Lambda functions in Amazon ECS blue/green deployments</a> in the <i> Amazon
         *        Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The lifecycle stages at which to run the hook. Choose from these valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * RECONCILE_SERVICE
         * </p>
         * <p>
         * The reconciliation stage that only happens when you start a new service deployment with more than 1 service
         * revision in an ACTIVE state.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * PRE_SCALE_UP
         * </p>
         * <p>
         * The green service revision has not started. The blue service revision is handling 100% of the production
         * traffic. There is no test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_SCALE_UP
         * </p>
         * <p>
         * The green service revision has started. The blue service revision is handling 100% of the production traffic.
         * There is no test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * TEST_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The blue and green service revisions are running. The blue service revision handles 100% of the production
         * traffic. The green service revision is migrating from 0% to 100% of test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_TEST_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The test traffic shift is complete. The green service revision handles 100% of the test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * PRODUCTION_TRAFFIC_SHIFT
         * </p>
         * <p>
         * Production traffic is shifting to the green service revision. The green service revision is migrating from 0%
         * to 100% of production traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_PRODUCTION_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The production traffic shift is complete.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You must provide this parameter when configuring a deployment lifecycle hook.
         * </p>
         * 
         * @param lifecycleStages
         *        The lifecycle stages at which to run the hook. Choose from these valid values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        RECONCILE_SERVICE
         *        </p>
         *        <p>
         *        The reconciliation stage that only happens when you start a new service deployment with more than 1
         *        service revision in an ACTIVE state.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PRE_SCALE_UP
         *        </p>
         *        <p>
         *        The green service revision has not started. The blue service revision is handling 100% of the
         *        production traffic. There is no test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_SCALE_UP
         *        </p>
         *        <p>
         *        The green service revision has started. The blue service revision is handling 100% of the production
         *        traffic. There is no test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TEST_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The blue and green service revisions are running. The blue service revision handles 100% of the
         *        production traffic. The green service revision is migrating from 0% to 100% of test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_TEST_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The test traffic shift is complete. The green service revision handles 100% of the test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PRODUCTION_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        Production traffic is shifting to the green service revision. The green service revision is migrating
         *        from 0% to 100% of production traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_PRODUCTION_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The production traffic shift is complete.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You must provide this parameter when configuring a deployment lifecycle hook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleStagesWithStrings(Collection<String> lifecycleStages);

        /**
         * <p>
         * The lifecycle stages at which to run the hook. Choose from these valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * RECONCILE_SERVICE
         * </p>
         * <p>
         * The reconciliation stage that only happens when you start a new service deployment with more than 1 service
         * revision in an ACTIVE state.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * PRE_SCALE_UP
         * </p>
         * <p>
         * The green service revision has not started. The blue service revision is handling 100% of the production
         * traffic. There is no test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_SCALE_UP
         * </p>
         * <p>
         * The green service revision has started. The blue service revision is handling 100% of the production traffic.
         * There is no test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * TEST_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The blue and green service revisions are running. The blue service revision handles 100% of the production
         * traffic. The green service revision is migrating from 0% to 100% of test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_TEST_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The test traffic shift is complete. The green service revision handles 100% of the test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * PRODUCTION_TRAFFIC_SHIFT
         * </p>
         * <p>
         * Production traffic is shifting to the green service revision. The green service revision is migrating from 0%
         * to 100% of production traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_PRODUCTION_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The production traffic shift is complete.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You must provide this parameter when configuring a deployment lifecycle hook.
         * </p>
         * 
         * @param lifecycleStages
         *        The lifecycle stages at which to run the hook. Choose from these valid values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        RECONCILE_SERVICE
         *        </p>
         *        <p>
         *        The reconciliation stage that only happens when you start a new service deployment with more than 1
         *        service revision in an ACTIVE state.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PRE_SCALE_UP
         *        </p>
         *        <p>
         *        The green service revision has not started. The blue service revision is handling 100% of the
         *        production traffic. There is no test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_SCALE_UP
         *        </p>
         *        <p>
         *        The green service revision has started. The blue service revision is handling 100% of the production
         *        traffic. There is no test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TEST_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The blue and green service revisions are running. The blue service revision handles 100% of the
         *        production traffic. The green service revision is migrating from 0% to 100% of test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_TEST_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The test traffic shift is complete. The green service revision handles 100% of the test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PRODUCTION_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        Production traffic is shifting to the green service revision. The green service revision is migrating
         *        from 0% to 100% of production traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_PRODUCTION_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The production traffic shift is complete.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You must provide this parameter when configuring a deployment lifecycle hook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleStagesWithStrings(String... lifecycleStages);

        /**
         * <p>
         * The lifecycle stages at which to run the hook. Choose from these valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * RECONCILE_SERVICE
         * </p>
         * <p>
         * The reconciliation stage that only happens when you start a new service deployment with more than 1 service
         * revision in an ACTIVE state.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * PRE_SCALE_UP
         * </p>
         * <p>
         * The green service revision has not started. The blue service revision is handling 100% of the production
         * traffic. There is no test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_SCALE_UP
         * </p>
         * <p>
         * The green service revision has started. The blue service revision is handling 100% of the production traffic.
         * There is no test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * TEST_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The blue and green service revisions are running. The blue service revision handles 100% of the production
         * traffic. The green service revision is migrating from 0% to 100% of test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_TEST_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The test traffic shift is complete. The green service revision handles 100% of the test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * PRODUCTION_TRAFFIC_SHIFT
         * </p>
         * <p>
         * Production traffic is shifting to the green service revision. The green service revision is migrating from 0%
         * to 100% of production traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_PRODUCTION_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The production traffic shift is complete.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You must provide this parameter when configuring a deployment lifecycle hook.
         * </p>
         * 
         * @param lifecycleStages
         *        The lifecycle stages at which to run the hook. Choose from these valid values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        RECONCILE_SERVICE
         *        </p>
         *        <p>
         *        The reconciliation stage that only happens when you start a new service deployment with more than 1
         *        service revision in an ACTIVE state.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PRE_SCALE_UP
         *        </p>
         *        <p>
         *        The green service revision has not started. The blue service revision is handling 100% of the
         *        production traffic. There is no test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_SCALE_UP
         *        </p>
         *        <p>
         *        The green service revision has started. The blue service revision is handling 100% of the production
         *        traffic. There is no test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TEST_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The blue and green service revisions are running. The blue service revision handles 100% of the
         *        production traffic. The green service revision is migrating from 0% to 100% of test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_TEST_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The test traffic shift is complete. The green service revision handles 100% of the test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PRODUCTION_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        Production traffic is shifting to the green service revision. The green service revision is migrating
         *        from 0% to 100% of production traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_PRODUCTION_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The production traffic shift is complete.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You must provide this parameter when configuring a deployment lifecycle hook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleStages(Collection<DeploymentLifecycleHookStage> lifecycleStages);

        /**
         * <p>
         * The lifecycle stages at which to run the hook. Choose from these valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * RECONCILE_SERVICE
         * </p>
         * <p>
         * The reconciliation stage that only happens when you start a new service deployment with more than 1 service
         * revision in an ACTIVE state.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * PRE_SCALE_UP
         * </p>
         * <p>
         * The green service revision has not started. The blue service revision is handling 100% of the production
         * traffic. There is no test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_SCALE_UP
         * </p>
         * <p>
         * The green service revision has started. The blue service revision is handling 100% of the production traffic.
         * There is no test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * TEST_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The blue and green service revisions are running. The blue service revision handles 100% of the production
         * traffic. The green service revision is migrating from 0% to 100% of test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_TEST_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The test traffic shift is complete. The green service revision handles 100% of the test traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * PRODUCTION_TRAFFIC_SHIFT
         * </p>
         * <p>
         * Production traffic is shifting to the green service revision. The green service revision is migrating from 0%
         * to 100% of production traffic.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * <li>
         * <p>
         * POST_PRODUCTION_TRAFFIC_SHIFT
         * </p>
         * <p>
         * The production traffic shift is complete.
         * </p>
         * <p>
         * You can use a lifecycle hook for this stage.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You must provide this parameter when configuring a deployment lifecycle hook.
         * </p>
         * 
         * @param lifecycleStages
         *        The lifecycle stages at which to run the hook. Choose from these valid values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        RECONCILE_SERVICE
         *        </p>
         *        <p>
         *        The reconciliation stage that only happens when you start a new service deployment with more than 1
         *        service revision in an ACTIVE state.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PRE_SCALE_UP
         *        </p>
         *        <p>
         *        The green service revision has not started. The blue service revision is handling 100% of the
         *        production traffic. There is no test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_SCALE_UP
         *        </p>
         *        <p>
         *        The green service revision has started. The blue service revision is handling 100% of the production
         *        traffic. There is no test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TEST_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The blue and green service revisions are running. The blue service revision handles 100% of the
         *        production traffic. The green service revision is migrating from 0% to 100% of test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_TEST_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The test traffic shift is complete. The green service revision handles 100% of the test traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PRODUCTION_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        Production traffic is shifting to the green service revision. The green service revision is migrating
         *        from 0% to 100% of production traffic.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        POST_PRODUCTION_TRAFFIC_SHIFT
         *        </p>
         *        <p>
         *        The production traffic shift is complete.
         *        </p>
         *        <p>
         *        You can use a lifecycle hook for this stage.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You must provide this parameter when configuring a deployment lifecycle hook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleStages(DeploymentLifecycleHookStage... lifecycleStages);

        /**
         * <p>
         * Use this field to specify custom parameters that Amazon ECS will pass to your hook target invocations (such
         * as a Lambda function).
         * </p>
         * 
         * @param hookDetails
         *        Use this field to specify custom parameters that Amazon ECS will pass to your hook target invocations
         *        (such as a Lambda function).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hookDetails(Document hookDetails);
    }

    static final class BuilderImpl implements Builder {
        private String hookTargetArn;

        private String roleArn;

        private List<String> lifecycleStages = DefaultSdkAutoConstructList.getInstance();

        private Document hookDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentLifecycleHook model) {
            hookTargetArn(model.hookTargetArn);
            roleArn(model.roleArn);
            lifecycleStagesWithStrings(model.lifecycleStages);
            hookDetails(model.hookDetails);
        }

        public final String getHookTargetArn() {
            return hookTargetArn;
        }

        public final void setHookTargetArn(String hookTargetArn) {
            this.hookTargetArn = hookTargetArn;
        }

        @Override
        public final Builder hookTargetArn(String hookTargetArn) {
            this.hookTargetArn = hookTargetArn;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final Collection<String> getLifecycleStages() {
            if (lifecycleStages instanceof SdkAutoConstructList) {
                return null;
            }
            return lifecycleStages;
        }

        public final void setLifecycleStages(Collection<String> lifecycleStages) {
            this.lifecycleStages = DeploymentLifecycleHookStageListCopier.copy(lifecycleStages);
        }

        @Override
        public final Builder lifecycleStagesWithStrings(Collection<String> lifecycleStages) {
            this.lifecycleStages = DeploymentLifecycleHookStageListCopier.copy(lifecycleStages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleStagesWithStrings(String... lifecycleStages) {
            lifecycleStagesWithStrings(Arrays.asList(lifecycleStages));
            return this;
        }

        @Override
        public final Builder lifecycleStages(Collection<DeploymentLifecycleHookStage> lifecycleStages) {
            this.lifecycleStages = DeploymentLifecycleHookStageListCopier.copyEnumToString(lifecycleStages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleStages(DeploymentLifecycleHookStage... lifecycleStages) {
            lifecycleStages(Arrays.asList(lifecycleStages));
            return this;
        }

        public final Document getHookDetails() {
            return hookDetails;
        }

        public final void setHookDetails(Document hookDetails) {
            this.hookDetails = hookDetails;
        }

        @Override
        public final Builder hookDetails(Document hookDetails) {
            this.hookDetails = hookDetails;
            return this;
        }

        @Override
        public DeploymentLifecycleHook build() {
            return new DeploymentLifecycleHook(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
