/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for creating a Amazon ECS Managed Instances provider. This specifies how Amazon ECS should manage
 * Amazon EC2 instances, including the infrastructure role, instance launch template, and whether to propagate tags from
 * the capacity provider to the instances.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateManagedInstancesProviderConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CreateManagedInstancesProviderConfiguration.Builder, CreateManagedInstancesProviderConfiguration> {
    private static final SdkField<String> INFRASTRUCTURE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("infrastructureRoleArn")
            .getter(getter(CreateManagedInstancesProviderConfiguration::infrastructureRoleArn))
            .setter(setter(Builder::infrastructureRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("infrastructureRoleArn").build())
            .build();

    private static final SdkField<InstanceLaunchTemplate> INSTANCE_LAUNCH_TEMPLATE_FIELD = SdkField
            .<InstanceLaunchTemplate> builder(MarshallingType.SDK_POJO).memberName("instanceLaunchTemplate")
            .getter(getter(CreateManagedInstancesProviderConfiguration::instanceLaunchTemplate))
            .setter(setter(Builder::instanceLaunchTemplate)).constructor(InstanceLaunchTemplate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceLaunchTemplate").build())
            .build();

    private static final SdkField<String> PROPAGATE_TAGS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("propagateTags").getter(getter(CreateManagedInstancesProviderConfiguration::propagateTagsAsString))
            .setter(setter(Builder::propagateTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propagateTags").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INFRASTRUCTURE_ROLE_ARN_FIELD,
            INSTANCE_LAUNCH_TEMPLATE_FIELD, PROPAGATE_TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String infrastructureRoleArn;

    private final InstanceLaunchTemplate instanceLaunchTemplate;

    private final String propagateTags;

    private CreateManagedInstancesProviderConfiguration(BuilderImpl builder) {
        this.infrastructureRoleArn = builder.infrastructureRoleArn;
        this.instanceLaunchTemplate = builder.instanceLaunchTemplate;
        this.propagateTags = builder.propagateTags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the infrastructure role that Amazon ECS uses to manage instances on your
     * behalf. This role must have permissions to launch, terminate, and manage Amazon EC2 instances, as well as access
     * to other Amazon Web Services services required for Amazon ECS Managed Instances functionality.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/infrastructure_IAM_role.html">Amazon ECS
     * infrastructure IAM role</a> in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the infrastructure role that Amazon ECS uses to manage instances on
     *         your behalf. This role must have permissions to launch, terminate, and manage Amazon EC2 instances, as
     *         well as access to other Amazon Web Services services required for Amazon ECS Managed Instances
     *         functionality.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/infrastructure_IAM_role.html">Amazon
     *         ECS infrastructure IAM role</a> in the <i>Amazon ECS Developer Guide</i>.
     */
    public final String infrastructureRoleArn() {
        return infrastructureRoleArn;
    }

    /**
     * <p>
     * The launch template configuration that specifies how Amazon ECS should launch Amazon EC2 instances. This includes
     * the instance profile, network configuration, storage settings, and instance requirements for attribute-based
     * instance type selection.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">Store instance launch
     * parameters in Amazon EC2 launch templates</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * 
     * @return The launch template configuration that specifies how Amazon ECS should launch Amazon EC2 instances. This
     *         includes the instance profile, network configuration, storage settings, and instance requirements for
     *         attribute-based instance type selection.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">Store instance
     *         launch parameters in Amazon EC2 launch templates</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final InstanceLaunchTemplate instanceLaunchTemplate() {
        return instanceLaunchTemplate;
    }

    /**
     * <p>
     * Specifies whether to propagate tags from the capacity provider to the Amazon ECS Managed Instances. When enabled,
     * tags applied to the capacity provider are automatically applied to all instances launched by this provider.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateMITags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies whether to propagate tags from the capacity provider to the Amazon ECS Managed Instances. When
     *         enabled, tags applied to the capacity provider are automatically applied to all instances launched by
     *         this provider.
     * @see PropagateMITags
     */
    public final PropagateMITags propagateTags() {
        return PropagateMITags.fromValue(propagateTags);
    }

    /**
     * <p>
     * Specifies whether to propagate tags from the capacity provider to the Amazon ECS Managed Instances. When enabled,
     * tags applied to the capacity provider are automatically applied to all instances launched by this provider.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateMITags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies whether to propagate tags from the capacity provider to the Amazon ECS Managed Instances. When
     *         enabled, tags applied to the capacity provider are automatically applied to all instances launched by
     *         this provider.
     * @see PropagateMITags
     */
    public final String propagateTagsAsString() {
        return propagateTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(infrastructureRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(instanceLaunchTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(propagateTagsAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateManagedInstancesProviderConfiguration)) {
            return false;
        }
        CreateManagedInstancesProviderConfiguration other = (CreateManagedInstancesProviderConfiguration) obj;
        return Objects.equals(infrastructureRoleArn(), other.infrastructureRoleArn())
                && Objects.equals(instanceLaunchTemplate(), other.instanceLaunchTemplate())
                && Objects.equals(propagateTagsAsString(), other.propagateTagsAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateManagedInstancesProviderConfiguration")
                .add("InfrastructureRoleArn", infrastructureRoleArn()).add("InstanceLaunchTemplate", instanceLaunchTemplate())
                .add("PropagateTags", propagateTagsAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "infrastructureRoleArn":
            return Optional.ofNullable(clazz.cast(infrastructureRoleArn()));
        case "instanceLaunchTemplate":
            return Optional.ofNullable(clazz.cast(instanceLaunchTemplate()));
        case "propagateTags":
            return Optional.ofNullable(clazz.cast(propagateTagsAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("infrastructureRoleArn", INFRASTRUCTURE_ROLE_ARN_FIELD);
        map.put("instanceLaunchTemplate", INSTANCE_LAUNCH_TEMPLATE_FIELD);
        map.put("propagateTags", PROPAGATE_TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateManagedInstancesProviderConfiguration, T> g) {
        return obj -> g.apply((CreateManagedInstancesProviderConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateManagedInstancesProviderConfiguration> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the infrastructure role that Amazon ECS uses to manage instances on your
         * behalf. This role must have permissions to launch, terminate, and manage Amazon EC2 instances, as well as
         * access to other Amazon Web Services services required for Amazon ECS Managed Instances functionality.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/infrastructure_IAM_role.html">Amazon ECS
         * infrastructure IAM role</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * 
         * @param infrastructureRoleArn
         *        The Amazon Resource Name (ARN) of the infrastructure role that Amazon ECS uses to manage instances on
         *        your behalf. This role must have permissions to launch, terminate, and manage Amazon EC2 instances, as
         *        well as access to other Amazon Web Services services required for Amazon ECS Managed Instances
         *        functionality.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/infrastructure_IAM_role.html">Amazon
         *        ECS infrastructure IAM role</a> in the <i>Amazon ECS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder infrastructureRoleArn(String infrastructureRoleArn);

        /**
         * <p>
         * The launch template configuration that specifies how Amazon ECS should launch Amazon EC2 instances. This
         * includes the instance profile, network configuration, storage settings, and instance requirements for
         * attribute-based instance type selection.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">Store instance launch
         * parameters in Amazon EC2 launch templates</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param instanceLaunchTemplate
         *        The launch template configuration that specifies how Amazon ECS should launch Amazon EC2 instances.
         *        This includes the instance profile, network configuration, storage settings, and instance requirements
         *        for attribute-based instance type selection.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">Store instance
         *        launch parameters in Amazon EC2 launch templates</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceLaunchTemplate(InstanceLaunchTemplate instanceLaunchTemplate);

        /**
         * <p>
         * The launch template configuration that specifies how Amazon ECS should launch Amazon EC2 instances. This
         * includes the instance profile, network configuration, storage settings, and instance requirements for
         * attribute-based instance type selection.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">Store instance launch
         * parameters in Amazon EC2 launch templates</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link InstanceLaunchTemplate.Builder} avoiding
         * the need to create one manually via {@link InstanceLaunchTemplate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceLaunchTemplate.Builder#build()} is called immediately and
         * its result is passed to {@link #instanceLaunchTemplate(InstanceLaunchTemplate)}.
         * 
         * @param instanceLaunchTemplate
         *        a consumer that will call methods on {@link InstanceLaunchTemplate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceLaunchTemplate(InstanceLaunchTemplate)
         */
        default Builder instanceLaunchTemplate(Consumer<InstanceLaunchTemplate.Builder> instanceLaunchTemplate) {
            return instanceLaunchTemplate(InstanceLaunchTemplate.builder().applyMutation(instanceLaunchTemplate).build());
        }

        /**
         * <p>
         * Specifies whether to propagate tags from the capacity provider to the Amazon ECS Managed Instances. When
         * enabled, tags applied to the capacity provider are automatically applied to all instances launched by this
         * provider.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate tags from the capacity provider to the Amazon ECS Managed Instances.
         *        When enabled, tags applied to the capacity provider are automatically applied to all instances
         *        launched by this provider.
         * @see PropagateMITags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateMITags
         */
        Builder propagateTags(String propagateTags);

        /**
         * <p>
         * Specifies whether to propagate tags from the capacity provider to the Amazon ECS Managed Instances. When
         * enabled, tags applied to the capacity provider are automatically applied to all instances launched by this
         * provider.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate tags from the capacity provider to the Amazon ECS Managed Instances.
         *        When enabled, tags applied to the capacity provider are automatically applied to all instances
         *        launched by this provider.
         * @see PropagateMITags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateMITags
         */
        Builder propagateTags(PropagateMITags propagateTags);
    }

    static final class BuilderImpl implements Builder {
        private String infrastructureRoleArn;

        private InstanceLaunchTemplate instanceLaunchTemplate;

        private String propagateTags;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateManagedInstancesProviderConfiguration model) {
            infrastructureRoleArn(model.infrastructureRoleArn);
            instanceLaunchTemplate(model.instanceLaunchTemplate);
            propagateTags(model.propagateTags);
        }

        public final String getInfrastructureRoleArn() {
            return infrastructureRoleArn;
        }

        public final void setInfrastructureRoleArn(String infrastructureRoleArn) {
            this.infrastructureRoleArn = infrastructureRoleArn;
        }

        @Override
        public final Builder infrastructureRoleArn(String infrastructureRoleArn) {
            this.infrastructureRoleArn = infrastructureRoleArn;
            return this;
        }

        public final InstanceLaunchTemplate.Builder getInstanceLaunchTemplate() {
            return instanceLaunchTemplate != null ? instanceLaunchTemplate.toBuilder() : null;
        }

        public final void setInstanceLaunchTemplate(InstanceLaunchTemplate.BuilderImpl instanceLaunchTemplate) {
            this.instanceLaunchTemplate = instanceLaunchTemplate != null ? instanceLaunchTemplate.build() : null;
        }

        @Override
        public final Builder instanceLaunchTemplate(InstanceLaunchTemplate instanceLaunchTemplate) {
            this.instanceLaunchTemplate = instanceLaunchTemplate;
            return this;
        }

        public final String getPropagateTags() {
            return propagateTags;
        }

        public final void setPropagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
        }

        @Override
        public final Builder propagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
            return this;
        }

        @Override
        public final Builder propagateTags(PropagateMITags propagateTags) {
            this.propagateTags(propagateTags == null ? null : propagateTags.toString());
            return this;
        }

        @Override
        public CreateManagedInstancesProviderConfiguration build() {
            return new CreateManagedInstancesProviderConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
