/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for the Amazon EBS volume that Amazon ECS creates and manages on your behalf. These settings are
 * used to create each Amazon EBS volume, with one volume created for each task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TaskManagedEBSVolumeConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<TaskManagedEBSVolumeConfiguration.Builder, TaskManagedEBSVolumeConfiguration> {
    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("encrypted").getter(getter(TaskManagedEBSVolumeConfiguration::encrypted))
            .setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encrypted").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyId").getter(getter(TaskManagedEBSVolumeConfiguration::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyId").build()).build();

    private static final SdkField<String> VOLUME_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("volumeType").getter(getter(TaskManagedEBSVolumeConfiguration::volumeType))
            .setter(setter(Builder::volumeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeType").build()).build();

    private static final SdkField<Integer> SIZE_IN_GIB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("sizeInGiB").getter(getter(TaskManagedEBSVolumeConfiguration::sizeInGiB))
            .setter(setter(Builder::sizeInGiB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeInGiB").build()).build();

    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("snapshotId").getter(getter(TaskManagedEBSVolumeConfiguration::snapshotId))
            .setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("snapshotId").build()).build();

    private static final SdkField<Integer> VOLUME_INITIALIZATION_RATE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("volumeInitializationRate").getter(getter(TaskManagedEBSVolumeConfiguration::volumeInitializationRate))
            .setter(setter(Builder::volumeInitializationRate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeInitializationRate").build())
            .build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("iops")
            .getter(getter(TaskManagedEBSVolumeConfiguration::iops)).setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("iops").build()).build();

    private static final SdkField<Integer> THROUGHPUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("throughput").getter(getter(TaskManagedEBSVolumeConfiguration::throughput))
            .setter(setter(Builder::throughput))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("throughput").build()).build();

    private static final SdkField<List<EBSTagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<EBSTagSpecification>> builder(MarshallingType.LIST)
            .memberName("tagSpecifications")
            .getter(getter(TaskManagedEBSVolumeConfiguration::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tagSpecifications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EBSTagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(EBSTagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(TaskManagedEBSVolumeConfiguration::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<TaskManagedEBSVolumeTerminationPolicy> TERMINATION_POLICY_FIELD = SdkField
            .<TaskManagedEBSVolumeTerminationPolicy> builder(MarshallingType.SDK_POJO).memberName("terminationPolicy")
            .getter(getter(TaskManagedEBSVolumeConfiguration::terminationPolicy)).setter(setter(Builder::terminationPolicy))
            .constructor(TaskManagedEBSVolumeTerminationPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("terminationPolicy").build()).build();

    private static final SdkField<String> FILESYSTEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("filesystemType").getter(getter(TaskManagedEBSVolumeConfiguration::filesystemTypeAsString))
            .setter(setter(Builder::filesystemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filesystemType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENCRYPTED_FIELD,
            KMS_KEY_ID_FIELD, VOLUME_TYPE_FIELD, SIZE_IN_GIB_FIELD, SNAPSHOT_ID_FIELD, VOLUME_INITIALIZATION_RATE_FIELD,
            IOPS_FIELD, THROUGHPUT_FIELD, TAG_SPECIFICATIONS_FIELD, ROLE_ARN_FIELD, TERMINATION_POLICY_FIELD,
            FILESYSTEM_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean encrypted;

    private final String kmsKeyId;

    private final String volumeType;

    private final Integer sizeInGiB;

    private final String snapshotId;

    private final Integer volumeInitializationRate;

    private final Integer iops;

    private final Integer throughput;

    private final List<EBSTagSpecification> tagSpecifications;

    private final String roleArn;

    private final TaskManagedEBSVolumeTerminationPolicy terminationPolicy;

    private final String filesystemType;

    private TaskManagedEBSVolumeConfiguration(BuilderImpl builder) {
        this.encrypted = builder.encrypted;
        this.kmsKeyId = builder.kmsKeyId;
        this.volumeType = builder.volumeType;
        this.sizeInGiB = builder.sizeInGiB;
        this.snapshotId = builder.snapshotId;
        this.volumeInitializationRate = builder.volumeInitializationRate;
        this.iops = builder.iops;
        this.throughput = builder.throughput;
        this.tagSpecifications = builder.tagSpecifications;
        this.roleArn = builder.roleArn;
        this.terminationPolicy = builder.terminationPolicy;
        this.filesystemType = builder.filesystemType;
    }

    /**
     * <p>
     * Indicates whether the volume should be encrypted. If you turn on Region-level Amazon EBS encryption by default
     * but set this value as <code>false</code>, the setting is overridden and the volume is encrypted with the KMS key
     * specified for Amazon EBS encryption by default. This parameter maps 1:1 with the <code>Encrypted</code> parameter
     * of the <a href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
     * API</a> in the <i>Amazon EC2 API Reference</i>.
     * </p>
     * 
     * @return Indicates whether the volume should be encrypted. If you turn on Region-level Amazon EBS encryption by
     *         default but set this value as <code>false</code>, the setting is overridden and the volume is encrypted
     *         with the KMS key specified for Amazon EBS encryption by default. This parameter maps 1:1 with the
     *         <code>Encrypted</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a>
     *         in the <i>Amazon EC2 API Reference</i>.
     */
    public final Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) identifier of the Amazon Web Services Key Management Service key to use for Amazon
     * EBS encryption. When a key is specified using this parameter, it overrides Amazon EBS default encryption or any
     * KMS key that you specified for cluster-level managed storage encryption. This parameter maps 1:1 with the
     * <code>KmsKeyId</code> parameter of the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in the
     * <i>Amazon EC2 API Reference</i>. For more information about encrypting Amazon EBS volumes attached to a task, see
     * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-kms-encryption.html">Encrypt data stored
     * in Amazon EBS volumes attached to Amazon ECS tasks</a>.
     * </p>
     * <important>
     * <p>
     * Amazon Web Services authenticates the Amazon Web Services Key Management Service key asynchronously. Therefore,
     * if you specify an ID, alias, or ARN that is invalid, the action can appear to complete, but eventually fails.
     * </p>
     * </important>
     * 
     * @return The Amazon Resource Name (ARN) identifier of the Amazon Web Services Key Management Service key to use
     *         for Amazon EBS encryption. When a key is specified using this parameter, it overrides Amazon EBS default
     *         encryption or any KMS key that you specified for cluster-level managed storage encryption. This parameter
     *         maps 1:1 with the <code>KmsKeyId</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a>
     *         in the <i>Amazon EC2 API Reference</i>. For more information about encrypting Amazon EBS volumes attached
     *         to a task, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-kms-encryption.html">Encrypt data
     *         stored in Amazon EBS volumes attached to Amazon ECS tasks</a>.</p> <important>
     *         <p>
     *         Amazon Web Services authenticates the Amazon Web Services Key Management Service key asynchronously.
     *         Therefore, if you specify an ID, alias, or ARN that is invalid, the action can appear to complete, but
     *         eventually fails.
     *         </p>
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The volume type. This parameter maps 1:1 with the <code>VolumeType</code> parameter of the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in the
     * <i>Amazon EC2 API Reference</i>. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html">Amazon EBS volume types</a> in
     * the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * The following are the supported volume types.
     * </p>
     * <ul>
     * <li>
     * <p>
     * General Purpose SSD: <code>gp2</code>|<code>gp3</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned IOPS SSD: <code>io1</code>|<code>io2</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Throughput Optimized HDD: <code>st1</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Cold HDD: <code>sc1</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Magnetic: <code>standard</code>
     * </p>
     * <note>
     * <p>
     * The magnetic volume type is not supported on Fargate.
     * </p>
     * </note></li>
     * </ul>
     * 
     * @return The volume type. This parameter maps 1:1 with the <code>VolumeType</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a>
     *         in the <i>Amazon EC2 API Reference</i>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html">Amazon EBS volume
     *         types</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         The following are the supported volume types.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         General Purpose SSD: <code>gp2</code>|<code>gp3</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Provisioned IOPS SSD: <code>io1</code>|<code>io2</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Throughput Optimized HDD: <code>st1</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cold HDD: <code>sc1</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Magnetic: <code>standard</code>
     *         </p>
     *         <note>
     *         <p>
     *         The magnetic volume type is not supported on Fargate.
     *         </p>
     *         </note></li>
     */
    public final String volumeType() {
        return volumeType;
    }

    /**
     * <p>
     * The size of the volume in GiB. You must specify either a volume size or a snapshot ID. If you specify a snapshot
     * ID, the snapshot size is used for the volume size by default. You can optionally specify a volume size greater
     * than or equal to the snapshot size. This parameter maps 1:1 with the <code>Size</code> parameter of the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in the
     * <i>Amazon EC2 API Reference</i>.
     * </p>
     * <p>
     * The following are the supported volume size values for each volume type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>gp2</code> and <code>gp3</code>: 1-16,384
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>io1</code> and <code>io2</code>: 4-16,384
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>st1</code> and <code>sc1</code>: 125-16,384
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>standard</code>: 1-1,024
     * </p>
     * </li>
     * </ul>
     * 
     * @return The size of the volume in GiB. You must specify either a volume size or a snapshot ID. If you specify a
     *         snapshot ID, the snapshot size is used for the volume size by default. You can optionally specify a
     *         volume size greater than or equal to the snapshot size. This parameter maps 1:1 with the
     *         <code>Size</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a>
     *         in the <i>Amazon EC2 API Reference</i>.</p>
     *         <p>
     *         The following are the supported volume size values for each volume type.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>gp2</code> and <code>gp3</code>: 1-16,384
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>io1</code> and <code>io2</code>: 4-16,384
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>st1</code> and <code>sc1</code>: 125-16,384
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>standard</code>: 1-1,024
     *         </p>
     *         </li>
     */
    public final Integer sizeInGiB() {
        return sizeInGiB;
    }

    /**
     * <p>
     * The snapshot that Amazon ECS uses to create the volume. You must specify either a snapshot ID or a volume size.
     * This parameter maps 1:1 with the <code>SnapshotId</code> parameter of the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in the
     * <i>Amazon EC2 API Reference</i>.
     * </p>
     * 
     * @return The snapshot that Amazon ECS uses to create the volume. You must specify either a snapshot ID or a volume
     *         size. This parameter maps 1:1 with the <code>SnapshotId</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a>
     *         in the <i>Amazon EC2 API Reference</i>.
     */
    public final String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * The rate, in MiB/s, at which data is fetched from a snapshot of an existing Amazon EBS volume to create a new
     * volume for attachment to the task. This property can be specified only if you specify a <code>snapshotId</code>.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/ebs/latest/userguide/initalize-volume.html">Initialize Amazon EBS volumes</a>
     * in the <i>Amazon EBS User Guide</i>.
     * </p>
     * 
     * @return The rate, in MiB/s, at which data is fetched from a snapshot of an existing Amazon EBS volume to create a
     *         new volume for attachment to the task. This property can be specified only if you specify a
     *         <code>snapshotId</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/ebs/latest/userguide/initalize-volume.html">Initialize Amazon EBS
     *         volumes</a> in the <i>Amazon EBS User Guide</i>.
     */
    public final Integer volumeInitializationRate() {
        return volumeInitializationRate;
    }

    /**
     * <p>
     * The number of I/O operations per second (IOPS). For <code>gp3</code>, <code>io1</code>, and <code>io2</code>
     * volumes, this represents the number of IOPS that are provisioned for the volume. For <code>gp2</code> volumes,
     * this represents the baseline performance of the volume and the rate at which the volume accumulates I/O credits
     * for bursting.
     * </p>
     * <p>
     * The following are the supported values for each volume type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>gp3</code>: 3,000 - 16,000 IOPS
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>io1</code>: 100 - 64,000 IOPS
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>io2</code>: 100 - 256,000 IOPS
     * </p>
     * </li>
     * </ul>
     * <p>
     * This parameter is required for <code>io1</code> and <code>io2</code> volume types. The default for
     * <code>gp3</code> volumes is <code>3,000 IOPS</code>. This parameter is not supported for <code>st1</code>,
     * <code>sc1</code>, or <code>standard</code> volume types.
     * </p>
     * <p>
     * This parameter maps 1:1 with the <code>Iops</code> parameter of the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in the
     * <i>Amazon EC2 API Reference</i>.
     * </p>
     * 
     * @return The number of I/O operations per second (IOPS). For <code>gp3</code>, <code>io1</code>, and
     *         <code>io2</code> volumes, this represents the number of IOPS that are provisioned for the volume. For
     *         <code>gp2</code> volumes, this represents the baseline performance of the volume and the rate at which
     *         the volume accumulates I/O credits for bursting.</p>
     *         <p>
     *         The following are the supported values for each volume type.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>gp3</code>: 3,000 - 16,000 IOPS
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>io1</code>: 100 - 64,000 IOPS
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>io2</code>: 100 - 256,000 IOPS
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This parameter is required for <code>io1</code> and <code>io2</code> volume types. The default for
     *         <code>gp3</code> volumes is <code>3,000 IOPS</code>. This parameter is not supported for <code>st1</code>, <code>sc1</code>, or <code>standard</code> volume types.
     *         </p>
     *         <p>
     *         This parameter maps 1:1 with the <code>Iops</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a>
     *         in the <i>Amazon EC2 API Reference</i>.
     */
    public final Integer iops() {
        return iops;
    }

    /**
     * <p>
     * The throughput to provision for a volume, in MiB/s, with a maximum of 1,000 MiB/s. This parameter maps 1:1 with
     * the <code>Throughput</code> parameter of the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in the
     * <i>Amazon EC2 API Reference</i>.
     * </p>
     * <important>
     * <p>
     * This parameter is only supported for the <code>gp3</code> volume type.
     * </p>
     * </important>
     * 
     * @return The throughput to provision for a volume, in MiB/s, with a maximum of 1,000 MiB/s. This parameter maps
     *         1:1 with the <code>Throughput</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a>
     *         in the <i>Amazon EC2 API Reference</i>.</p> <important>
     *         <p>
     *         This parameter is only supported for the <code>gp3</code> volume type.
     *         </p>
     */
    public final Integer throughput() {
        return throughput;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to apply to the volume. Amazon ECS applies service-managed tags by default. This parameter maps 1:1 with
     * the <code>TagSpecifications.N</code> parameter of the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in the
     * <i>Amazon EC2 API Reference</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The tags to apply to the volume. Amazon ECS applies service-managed tags by default. This parameter maps
     *         1:1 with the <code>TagSpecifications.N</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a>
     *         in the <i>Amazon EC2 API Reference</i>.
     */
    public final List<EBSTagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * The ARN of the IAM role to associate with this volume. This is the Amazon ECS infrastructure IAM role that is
     * used to manage your Amazon Web Services infrastructure. We recommend using the Amazon ECS-managed
     * <code>AmazonECSInfrastructureRolePolicyForVolumes</code> IAM policy with this role. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/infrastructure_IAM_role.html">Amazon ECS
     * infrastructure IAM role</a> in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * 
     * @return The ARN of the IAM role to associate with this volume. This is the Amazon ECS infrastructure IAM role
     *         that is used to manage your Amazon Web Services infrastructure. We recommend using the Amazon ECS-managed
     *         <code>AmazonECSInfrastructureRolePolicyForVolumes</code> IAM policy with this role. For more information,
     *         see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/infrastructure_IAM_role.html">Amazon
     *         ECS infrastructure IAM role</a> in the <i>Amazon ECS Developer Guide</i>.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The termination policy for the volume when the task exits. This provides a way to control whether Amazon ECS
     * terminates the Amazon EBS volume when the task stops.
     * </p>
     * 
     * @return The termination policy for the volume when the task exits. This provides a way to control whether Amazon
     *         ECS terminates the Amazon EBS volume when the task stops.
     */
    public final TaskManagedEBSVolumeTerminationPolicy terminationPolicy() {
        return terminationPolicy;
    }

    /**
     * <p>
     * The Linux filesystem type for the volume. For volumes created from a snapshot, you must specify the same
     * filesystem type that the volume was using when the snapshot was created. If there is a filesystem type mismatch,
     * the task will fail to start.
     * </p>
     * <p>
     * The available filesystem types are&#x2028; <code>ext3</code>, <code>ext4</code>, and <code>xfs</code>. If no
     * value is specified, the <code>xfs</code> filesystem type is used by default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #filesystemType}
     * will return {@link TaskFilesystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #filesystemTypeAsString}.
     * </p>
     * 
     * @return The Linux filesystem type for the volume. For volumes created from a snapshot, you must specify the same
     *         filesystem type that the volume was using when the snapshot was created. If there is a filesystem type
     *         mismatch, the task will fail to start.</p>
     *         <p>
     *         The available filesystem types are&#x2028; <code>ext3</code>, <code>ext4</code>, and <code>xfs</code>. If
     *         no value is specified, the <code>xfs</code> filesystem type is used by default.
     * @see TaskFilesystemType
     */
    public final TaskFilesystemType filesystemType() {
        return TaskFilesystemType.fromValue(filesystemType);
    }

    /**
     * <p>
     * The Linux filesystem type for the volume. For volumes created from a snapshot, you must specify the same
     * filesystem type that the volume was using when the snapshot was created. If there is a filesystem type mismatch,
     * the task will fail to start.
     * </p>
     * <p>
     * The available filesystem types are&#x2028; <code>ext3</code>, <code>ext4</code>, and <code>xfs</code>. If no
     * value is specified, the <code>xfs</code> filesystem type is used by default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #filesystemType}
     * will return {@link TaskFilesystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #filesystemTypeAsString}.
     * </p>
     * 
     * @return The Linux filesystem type for the volume. For volumes created from a snapshot, you must specify the same
     *         filesystem type that the volume was using when the snapshot was created. If there is a filesystem type
     *         mismatch, the task will fail to start.</p>
     *         <p>
     *         The available filesystem types are&#x2028; <code>ext3</code>, <code>ext4</code>, and <code>xfs</code>. If
     *         no value is specified, the <code>xfs</code> filesystem type is used by default.
     * @see TaskFilesystemType
     */
    public final String filesystemTypeAsString() {
        return filesystemType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(volumeType());
        hashCode = 31 * hashCode + Objects.hashCode(sizeInGiB());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(volumeInitializationRate());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(throughput());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(terminationPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(filesystemTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TaskManagedEBSVolumeConfiguration)) {
            return false;
        }
        TaskManagedEBSVolumeConfiguration other = (TaskManagedEBSVolumeConfiguration) obj;
        return Objects.equals(encrypted(), other.encrypted()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(volumeType(), other.volumeType()) && Objects.equals(sizeInGiB(), other.sizeInGiB())
                && Objects.equals(snapshotId(), other.snapshotId())
                && Objects.equals(volumeInitializationRate(), other.volumeInitializationRate())
                && Objects.equals(iops(), other.iops()) && Objects.equals(throughput(), other.throughput())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(terminationPolicy(), other.terminationPolicy())
                && Objects.equals(filesystemTypeAsString(), other.filesystemTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TaskManagedEBSVolumeConfiguration").add("Encrypted", encrypted()).add("KmsKeyId", kmsKeyId())
                .add("VolumeType", volumeType()).add("SizeInGiB", sizeInGiB()).add("SnapshotId", snapshotId())
                .add("VolumeInitializationRate", volumeInitializationRate()).add("Iops", iops()).add("Throughput", throughput())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).add("RoleArn", roleArn())
                .add("TerminationPolicy", terminationPolicy()).add("FilesystemType", filesystemTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "kmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "volumeType":
            return Optional.ofNullable(clazz.cast(volumeType()));
        case "sizeInGiB":
            return Optional.ofNullable(clazz.cast(sizeInGiB()));
        case "snapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "volumeInitializationRate":
            return Optional.ofNullable(clazz.cast(volumeInitializationRate()));
        case "iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "throughput":
            return Optional.ofNullable(clazz.cast(throughput()));
        case "tagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "terminationPolicy":
            return Optional.ofNullable(clazz.cast(terminationPolicy()));
        case "filesystemType":
            return Optional.ofNullable(clazz.cast(filesystemTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("encrypted", ENCRYPTED_FIELD);
        map.put("kmsKeyId", KMS_KEY_ID_FIELD);
        map.put("volumeType", VOLUME_TYPE_FIELD);
        map.put("sizeInGiB", SIZE_IN_GIB_FIELD);
        map.put("snapshotId", SNAPSHOT_ID_FIELD);
        map.put("volumeInitializationRate", VOLUME_INITIALIZATION_RATE_FIELD);
        map.put("iops", IOPS_FIELD);
        map.put("throughput", THROUGHPUT_FIELD);
        map.put("tagSpecifications", TAG_SPECIFICATIONS_FIELD);
        map.put("roleArn", ROLE_ARN_FIELD);
        map.put("terminationPolicy", TERMINATION_POLICY_FIELD);
        map.put("filesystemType", FILESYSTEM_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TaskManagedEBSVolumeConfiguration, T> g) {
        return obj -> g.apply((TaskManagedEBSVolumeConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TaskManagedEBSVolumeConfiguration> {
        /**
         * <p>
         * Indicates whether the volume should be encrypted. If you turn on Region-level Amazon EBS encryption by
         * default but set this value as <code>false</code>, the setting is overridden and the volume is encrypted with
         * the KMS key specified for Amazon EBS encryption by default. This parameter maps 1:1 with the
         * <code>Encrypted</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>.
         * </p>
         * 
         * @param encrypted
         *        Indicates whether the volume should be encrypted. If you turn on Region-level Amazon EBS encryption by
         *        default but set this value as <code>false</code>, the setting is overridden and the volume is
         *        encrypted with the KMS key specified for Amazon EBS encryption by default. This parameter maps 1:1
         *        with the <code>Encrypted</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * The Amazon Resource Name (ARN) identifier of the Amazon Web Services Key Management Service key to use for
         * Amazon EBS encryption. When a key is specified using this parameter, it overrides Amazon EBS default
         * encryption or any KMS key that you specified for cluster-level managed storage encryption. This parameter
         * maps 1:1 with the <code>KmsKeyId</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>. For more information about encrypting Amazon EBS volumes attached to a
         * task, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-kms-encryption.html">Encrypt data
         * stored in Amazon EBS volumes attached to Amazon ECS tasks</a>.
         * </p>
         * <important>
         * <p>
         * Amazon Web Services authenticates the Amazon Web Services Key Management Service key asynchronously.
         * Therefore, if you specify an ID, alias, or ARN that is invalid, the action can appear to complete, but
         * eventually fails.
         * </p>
         * </important>
         * 
         * @param kmsKeyId
         *        The Amazon Resource Name (ARN) identifier of the Amazon Web Services Key Management Service key to use
         *        for Amazon EBS encryption. When a key is specified using this parameter, it overrides Amazon EBS
         *        default encryption or any KMS key that you specified for cluster-level managed storage encryption.
         *        This parameter maps 1:1 with the <code>KmsKeyId</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>. For more information about encrypting Amazon EBS
         *        volumes attached to a task, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-kms-encryption.html">Encrypt
         *        data stored in Amazon EBS volumes attached to Amazon ECS tasks</a>.</p> <important>
         *        <p>
         *        Amazon Web Services authenticates the Amazon Web Services Key Management Service key asynchronously.
         *        Therefore, if you specify an ID, alias, or ARN that is invalid, the action can appear to complete, but
         *        eventually fails.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The volume type. This parameter maps 1:1 with the <code>VolumeType</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html">Amazon EBS volume types</a>
         * in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * The following are the supported volume types.
         * </p>
         * <ul>
         * <li>
         * <p>
         * General Purpose SSD: <code>gp2</code>|<code>gp3</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Provisioned IOPS SSD: <code>io1</code>|<code>io2</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Throughput Optimized HDD: <code>st1</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Cold HDD: <code>sc1</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Magnetic: <code>standard</code>
         * </p>
         * <note>
         * <p>
         * The magnetic volume type is not supported on Fargate.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param volumeType
         *        The volume type. This parameter maps 1:1 with the <code>VolumeType</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html">Amazon EBS volume
         *        types</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        The following are the supported volume types.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        General Purpose SSD: <code>gp2</code>|<code>gp3</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Provisioned IOPS SSD: <code>io1</code>|<code>io2</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Throughput Optimized HDD: <code>st1</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cold HDD: <code>sc1</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Magnetic: <code>standard</code>
         *        </p>
         *        <note>
         *        <p>
         *        The magnetic volume type is not supported on Fargate.
         *        </p>
         *        </note></li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeType(String volumeType);

        /**
         * <p>
         * The size of the volume in GiB. You must specify either a volume size or a snapshot ID. If you specify a
         * snapshot ID, the snapshot size is used for the volume size by default. You can optionally specify a volume
         * size greater than or equal to the snapshot size. This parameter maps 1:1 with the <code>Size</code> parameter
         * of the <a href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         * API</a> in the <i>Amazon EC2 API Reference</i>.
         * </p>
         * <p>
         * The following are the supported volume size values for each volume type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>gp2</code> and <code>gp3</code>: 1-16,384
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>io1</code> and <code>io2</code>: 4-16,384
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>st1</code> and <code>sc1</code>: 125-16,384
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>standard</code>: 1-1,024
         * </p>
         * </li>
         * </ul>
         * 
         * @param sizeInGiB
         *        The size of the volume in GiB. You must specify either a volume size or a snapshot ID. If you specify
         *        a snapshot ID, the snapshot size is used for the volume size by default. You can optionally specify a
         *        volume size greater than or equal to the snapshot size. This parameter maps 1:1 with the
         *        <code>Size</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>.</p>
         *        <p>
         *        The following are the supported volume size values for each volume type.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>gp2</code> and <code>gp3</code>: 1-16,384
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>io1</code> and <code>io2</code>: 4-16,384
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>st1</code> and <code>sc1</code>: 125-16,384
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>standard</code>: 1-1,024
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInGiB(Integer sizeInGiB);

        /**
         * <p>
         * The snapshot that Amazon ECS uses to create the volume. You must specify either a snapshot ID or a volume
         * size. This parameter maps 1:1 with the <code>SnapshotId</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>.
         * </p>
         * 
         * @param snapshotId
         *        The snapshot that Amazon ECS uses to create the volume. You must specify either a snapshot ID or a
         *        volume size. This parameter maps 1:1 with the <code>SnapshotId</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * The rate, in MiB/s, at which data is fetched from a snapshot of an existing Amazon EBS volume to create a new
         * volume for attachment to the task. This property can be specified only if you specify a
         * <code>snapshotId</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/initalize-volume.html">Initialize Amazon EBS
         * volumes</a> in the <i>Amazon EBS User Guide</i>.
         * </p>
         * 
         * @param volumeInitializationRate
         *        The rate, in MiB/s, at which data is fetched from a snapshot of an existing Amazon EBS volume to
         *        create a new volume for attachment to the task. This property can be specified only if you specify a
         *        <code>snapshotId</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/ebs/latest/userguide/initalize-volume.html">Initialize Amazon EBS
         *        volumes</a> in the <i>Amazon EBS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeInitializationRate(Integer volumeInitializationRate);

        /**
         * <p>
         * The number of I/O operations per second (IOPS). For <code>gp3</code>, <code>io1</code>, and <code>io2</code>
         * volumes, this represents the number of IOPS that are provisioned for the volume. For <code>gp2</code>
         * volumes, this represents the baseline performance of the volume and the rate at which the volume accumulates
         * I/O credits for bursting.
         * </p>
         * <p>
         * The following are the supported values for each volume type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>gp3</code>: 3,000 - 16,000 IOPS
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>io1</code>: 100 - 64,000 IOPS
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>io2</code>: 100 - 256,000 IOPS
         * </p>
         * </li>
         * </ul>
         * <p>
         * This parameter is required for <code>io1</code> and <code>io2</code> volume types. The default for
         * <code>gp3</code> volumes is <code>3,000 IOPS</code>. This parameter is not supported for <code>st1</code>,
         * <code>sc1</code>, or <code>standard</code> volume types.
         * </p>
         * <p>
         * This parameter maps 1:1 with the <code>Iops</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>.
         * </p>
         * 
         * @param iops
         *        The number of I/O operations per second (IOPS). For <code>gp3</code>, <code>io1</code>, and
         *        <code>io2</code> volumes, this represents the number of IOPS that are provisioned for the volume. For
         *        <code>gp2</code> volumes, this represents the baseline performance of the volume and the rate at which
         *        the volume accumulates I/O credits for bursting.</p>
         *        <p>
         *        The following are the supported values for each volume type.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>gp3</code>: 3,000 - 16,000 IOPS
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>io1</code>: 100 - 64,000 IOPS
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>io2</code>: 100 - 256,000 IOPS
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This parameter is required for <code>io1</code> and <code>io2</code> volume types. The default for
         *        <code>gp3</code> volumes is <code>3,000 IOPS</code>. This parameter is not supported for
         *        <code>st1</code>, <code>sc1</code>, or <code>standard</code> volume types.
         *        </p>
         *        <p>
         *        This parameter maps 1:1 with the <code>Iops</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * The throughput to provision for a volume, in MiB/s, with a maximum of 1,000 MiB/s. This parameter maps 1:1
         * with the <code>Throughput</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>.
         * </p>
         * <important>
         * <p>
         * This parameter is only supported for the <code>gp3</code> volume type.
         * </p>
         * </important>
         * 
         * @param throughput
         *        The throughput to provision for a volume, in MiB/s, with a maximum of 1,000 MiB/s. This parameter maps
         *        1:1 with the <code>Throughput</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>.</p> <important>
         *        <p>
         *        This parameter is only supported for the <code>gp3</code> volume type.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throughput(Integer throughput);

        /**
         * <p>
         * The tags to apply to the volume. Amazon ECS applies service-managed tags by default. This parameter maps 1:1
         * with the <code>TagSpecifications.N</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the volume. Amazon ECS applies service-managed tags by default. This parameter
         *        maps 1:1 with the <code>TagSpecifications.N</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<EBSTagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to apply to the volume. Amazon ECS applies service-managed tags by default. This parameter maps 1:1
         * with the <code>TagSpecifications.N</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the volume. Amazon ECS applies service-managed tags by default. This parameter
         *        maps 1:1 with the <code>TagSpecifications.N</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume
         *        API</a> in the <i>Amazon EC2 API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(EBSTagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to apply to the volume. Amazon ECS applies service-managed tags by default. This parameter maps 1:1
         * with the <code>TagSpecifications.N</code> parameter of the <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html">CreateVolume API</a> in
         * the <i>Amazon EC2 API Reference</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.EBSTagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.EBSTagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.EBSTagSpecification.Builder#build()} is called immediately
         * and its result is passed to {@link #tagSpecifications(List<EBSTagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.EBSTagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<EBSTagSpecification>)
         */
        Builder tagSpecifications(Consumer<EBSTagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * The ARN of the IAM role to associate with this volume. This is the Amazon ECS infrastructure IAM role that is
         * used to manage your Amazon Web Services infrastructure. We recommend using the Amazon ECS-managed
         * <code>AmazonECSInfrastructureRolePolicyForVolumes</code> IAM policy with this role. For more information, see
         * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/infrastructure_IAM_role.html">Amazon ECS
         * infrastructure IAM role</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the IAM role to associate with this volume. This is the Amazon ECS infrastructure IAM role
         *        that is used to manage your Amazon Web Services infrastructure. We recommend using the Amazon
         *        ECS-managed <code>AmazonECSInfrastructureRolePolicyForVolumes</code> IAM policy with this role. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/infrastructure_IAM_role.html">Amazon
         *        ECS infrastructure IAM role</a> in the <i>Amazon ECS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The termination policy for the volume when the task exits. This provides a way to control whether Amazon ECS
         * terminates the Amazon EBS volume when the task stops.
         * </p>
         * 
         * @param terminationPolicy
         *        The termination policy for the volume when the task exits. This provides a way to control whether
         *        Amazon ECS terminates the Amazon EBS volume when the task stops.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationPolicy(TaskManagedEBSVolumeTerminationPolicy terminationPolicy);

        /**
         * <p>
         * The termination policy for the volume when the task exits. This provides a way to control whether Amazon ECS
         * terminates the Amazon EBS volume when the task stops.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link TaskManagedEBSVolumeTerminationPolicy.Builder} avoiding the need to create one manually via
         * {@link TaskManagedEBSVolumeTerminationPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TaskManagedEBSVolumeTerminationPolicy.Builder#build()} is called
         * immediately and its result is passed to {@link #terminationPolicy(TaskManagedEBSVolumeTerminationPolicy)}.
         * 
         * @param terminationPolicy
         *        a consumer that will call methods on {@link TaskManagedEBSVolumeTerminationPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #terminationPolicy(TaskManagedEBSVolumeTerminationPolicy)
         */
        default Builder terminationPolicy(Consumer<TaskManagedEBSVolumeTerminationPolicy.Builder> terminationPolicy) {
            return terminationPolicy(TaskManagedEBSVolumeTerminationPolicy.builder().applyMutation(terminationPolicy).build());
        }

        /**
         * <p>
         * The Linux filesystem type for the volume. For volumes created from a snapshot, you must specify the same
         * filesystem type that the volume was using when the snapshot was created. If there is a filesystem type
         * mismatch, the task will fail to start.
         * </p>
         * <p>
         * The available filesystem types are&#x2028; <code>ext3</code>, <code>ext4</code>, and <code>xfs</code>. If no
         * value is specified, the <code>xfs</code> filesystem type is used by default.
         * </p>
         * 
         * @param filesystemType
         *        The Linux filesystem type for the volume. For volumes created from a snapshot, you must specify the
         *        same filesystem type that the volume was using when the snapshot was created. If there is a filesystem
         *        type mismatch, the task will fail to start.</p>
         *        <p>
         *        The available filesystem types are&#x2028; <code>ext3</code>, <code>ext4</code>, and <code>xfs</code>.
         *        If no value is specified, the <code>xfs</code> filesystem type is used by default.
         * @see TaskFilesystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskFilesystemType
         */
        Builder filesystemType(String filesystemType);

        /**
         * <p>
         * The Linux filesystem type for the volume. For volumes created from a snapshot, you must specify the same
         * filesystem type that the volume was using when the snapshot was created. If there is a filesystem type
         * mismatch, the task will fail to start.
         * </p>
         * <p>
         * The available filesystem types are&#x2028; <code>ext3</code>, <code>ext4</code>, and <code>xfs</code>. If no
         * value is specified, the <code>xfs</code> filesystem type is used by default.
         * </p>
         * 
         * @param filesystemType
         *        The Linux filesystem type for the volume. For volumes created from a snapshot, you must specify the
         *        same filesystem type that the volume was using when the snapshot was created. If there is a filesystem
         *        type mismatch, the task will fail to start.</p>
         *        <p>
         *        The available filesystem types are&#x2028; <code>ext3</code>, <code>ext4</code>, and <code>xfs</code>.
         *        If no value is specified, the <code>xfs</code> filesystem type is used by default.
         * @see TaskFilesystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskFilesystemType
         */
        Builder filesystemType(TaskFilesystemType filesystemType);
    }

    static final class BuilderImpl implements Builder {
        private Boolean encrypted;

        private String kmsKeyId;

        private String volumeType;

        private Integer sizeInGiB;

        private String snapshotId;

        private Integer volumeInitializationRate;

        private Integer iops;

        private Integer throughput;

        private List<EBSTagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private String roleArn;

        private TaskManagedEBSVolumeTerminationPolicy terminationPolicy;

        private String filesystemType;

        private BuilderImpl() {
        }

        private BuilderImpl(TaskManagedEBSVolumeConfiguration model) {
            encrypted(model.encrypted);
            kmsKeyId(model.kmsKeyId);
            volumeType(model.volumeType);
            sizeInGiB(model.sizeInGiB);
            snapshotId(model.snapshotId);
            volumeInitializationRate(model.volumeInitializationRate);
            iops(model.iops);
            throughput(model.throughput);
            tagSpecifications(model.tagSpecifications);
            roleArn(model.roleArn);
            terminationPolicy(model.terminationPolicy);
            filesystemType(model.filesystemType);
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getVolumeType() {
            return volumeType;
        }

        public final void setVolumeType(String volumeType) {
            this.volumeType = volumeType;
        }

        @Override
        public final Builder volumeType(String volumeType) {
            this.volumeType = volumeType;
            return this;
        }

        public final Integer getSizeInGiB() {
            return sizeInGiB;
        }

        public final void setSizeInGiB(Integer sizeInGiB) {
            this.sizeInGiB = sizeInGiB;
        }

        @Override
        public final Builder sizeInGiB(Integer sizeInGiB) {
            this.sizeInGiB = sizeInGiB;
            return this;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final Integer getVolumeInitializationRate() {
            return volumeInitializationRate;
        }

        public final void setVolumeInitializationRate(Integer volumeInitializationRate) {
            this.volumeInitializationRate = volumeInitializationRate;
        }

        @Override
        public final Builder volumeInitializationRate(Integer volumeInitializationRate) {
            this.volumeInitializationRate = volumeInitializationRate;
            return this;
        }

        public final Integer getIops() {
            return iops;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final Integer getThroughput() {
            return throughput;
        }

        public final void setThroughput(Integer throughput) {
            this.throughput = throughput;
        }

        @Override
        public final Builder throughput(Integer throughput) {
            this.throughput = throughput;
            return this;
        }

        public final List<EBSTagSpecification.Builder> getTagSpecifications() {
            List<EBSTagSpecification.Builder> result = EBSTagSpecificationsCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<EBSTagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = EBSTagSpecificationsCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<EBSTagSpecification> tagSpecifications) {
            this.tagSpecifications = EBSTagSpecificationsCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(EBSTagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<EBSTagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> EBSTagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final TaskManagedEBSVolumeTerminationPolicy.Builder getTerminationPolicy() {
            return terminationPolicy != null ? terminationPolicy.toBuilder() : null;
        }

        public final void setTerminationPolicy(TaskManagedEBSVolumeTerminationPolicy.BuilderImpl terminationPolicy) {
            this.terminationPolicy = terminationPolicy != null ? terminationPolicy.build() : null;
        }

        @Override
        public final Builder terminationPolicy(TaskManagedEBSVolumeTerminationPolicy terminationPolicy) {
            this.terminationPolicy = terminationPolicy;
            return this;
        }

        public final String getFilesystemType() {
            return filesystemType;
        }

        public final void setFilesystemType(String filesystemType) {
            this.filesystemType = filesystemType;
        }

        @Override
        public final Builder filesystemType(String filesystemType) {
            this.filesystemType = filesystemType;
            return this;
        }

        @Override
        public final Builder filesystemType(TaskFilesystemType filesystemType) {
            this.filesystemType(filesystemType == null ? null : filesystemType.toString());
            return this;
        }

        @Override
        public TaskManagedEBSVolumeConfiguration build() {
            return new TaskManagedEBSVolumeConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
