/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Service Connect configuration of your Amazon ECS service. The configuration for this service to discover and
 * connect to services, and be discovered by, and connected from, other services within a namespace.
 * </p>
 * <p>
 * Tasks that run in a namespace can use short names to connect to services in the namespace. Tasks can connect to
 * services across all of the clusters in the namespace. Tasks connect through a managed proxy container that collects
 * logs and metrics for increased visibility. Only the tasks that Amazon ECS services create are supported with Service
 * Connect. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service Connect</a> in the
 * <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceConnectConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ServiceConnectConfiguration.Builder, ServiceConnectConfiguration> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enabled").getter(getter(ServiceConnectConfiguration::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enabled").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespace").getter(getter(ServiceConnectConfiguration::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build()).build();

    private static final SdkField<List<ServiceConnectService>> SERVICES_FIELD = SdkField
            .<List<ServiceConnectService>> builder(MarshallingType.LIST)
            .memberName("services")
            .getter(getter(ServiceConnectConfiguration::services))
            .setter(setter(Builder::services))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("services").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ServiceConnectService> builder(MarshallingType.SDK_POJO)
                                            .constructor(ServiceConnectService::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LogConfiguration> LOG_CONFIGURATION_FIELD = SdkField
            .<LogConfiguration> builder(MarshallingType.SDK_POJO).memberName("logConfiguration")
            .getter(getter(ServiceConnectConfiguration::logConfiguration)).setter(setter(Builder::logConfiguration))
            .constructor(LogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD,
            NAMESPACE_FIELD, SERVICES_FIELD, LOG_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final String namespace;

    private final List<ServiceConnectService> services;

    private final LogConfiguration logConfiguration;

    private ServiceConnectConfiguration(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.namespace = builder.namespace;
        this.services = builder.services;
        this.logConfiguration = builder.logConfiguration;
    }

    /**
     * <p>
     * Specifies whether to use Service Connect with this service.
     * </p>
     * 
     * @return Specifies whether to use Service Connect with this service.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The namespace name or full Amazon Resource Name (ARN) of the Cloud Map namespace for use with Service Connect.
     * The namespace must be in the same Amazon Web Services Region as the Amazon ECS service and cluster. The type of
     * namespace doesn't affect Service Connect. For more information about Cloud Map, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/dg/working-with-services.html">Working with Services</a> in
     * the <i>Cloud Map Developer Guide</i>.
     * </p>
     * 
     * @return The namespace name or full Amazon Resource Name (ARN) of the Cloud Map namespace for use with Service
     *         Connect. The namespace must be in the same Amazon Web Services Region as the Amazon ECS service and
     *         cluster. The type of namespace doesn't affect Service Connect. For more information about Cloud Map, see
     *         <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/working-with-services.html">Working with
     *         Services</a> in the <i>Cloud Map Developer Guide</i>.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * For responses, this returns true if the service returned a value for the Services property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasServices() {
        return services != null && !(services instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of Service Connect service objects. These are names and aliases (also known as endpoints) that are used
     * by other Amazon ECS services to connect to this service.
     * </p>
     * <p>
     * This field is not required for a "client" Amazon ECS service that's a member of a namespace only to connect to
     * other services within the namespace. An example of this would be a frontend application that accepts incoming
     * requests from either a load balancer that's attached to the service or by other means.
     * </p>
     * <p>
     * An object selects a port from the task definition, assigns a name for the Cloud Map service, and a list of
     * aliases (endpoints) and ports for client applications to refer to this service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasServices} method.
     * </p>
     * 
     * @return The list of Service Connect service objects. These are names and aliases (also known as endpoints) that
     *         are used by other Amazon ECS services to connect to this service. </p>
     *         <p>
     *         This field is not required for a "client" Amazon ECS service that's a member of a namespace only to
     *         connect to other services within the namespace. An example of this would be a frontend application that
     *         accepts incoming requests from either a load balancer that's attached to the service or by other means.
     *         </p>
     *         <p>
     *         An object selects a port from the task definition, assigns a name for the Cloud Map service, and a list
     *         of aliases (endpoints) and ports for client applications to refer to this service.
     */
    public final List<ServiceConnectService> services() {
        return services;
    }

    /**
     * Returns the value of the LogConfiguration property for this object.
     * 
     * @return The value of the LogConfiguration property for this object.
     */
    public final LogConfiguration logConfiguration() {
        return logConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(hasServices() ? services() : null);
        hashCode = 31 * hashCode + Objects.hashCode(logConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceConnectConfiguration)) {
            return false;
        }
        ServiceConnectConfiguration other = (ServiceConnectConfiguration) obj;
        return Objects.equals(enabled(), other.enabled()) && Objects.equals(namespace(), other.namespace())
                && hasServices() == other.hasServices() && Objects.equals(services(), other.services())
                && Objects.equals(logConfiguration(), other.logConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServiceConnectConfiguration").add("Enabled", enabled()).add("Namespace", namespace())
                .add("Services", hasServices() ? services() : null).add("LogConfiguration", logConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "services":
            return Optional.ofNullable(clazz.cast(services()));
        case "logConfiguration":
            return Optional.ofNullable(clazz.cast(logConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ServiceConnectConfiguration, T> g) {
        return obj -> g.apply((ServiceConnectConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceConnectConfiguration> {
        /**
         * <p>
         * Specifies whether to use Service Connect with this service.
         * </p>
         * 
         * @param enabled
         *        Specifies whether to use Service Connect with this service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The namespace name or full Amazon Resource Name (ARN) of the Cloud Map namespace for use with Service
         * Connect. The namespace must be in the same Amazon Web Services Region as the Amazon ECS service and cluster.
         * The type of namespace doesn't affect Service Connect. For more information about Cloud Map, see <a
         * href="https://docs.aws.amazon.com/cloud-map/latest/dg/working-with-services.html">Working with Services</a>
         * in the <i>Cloud Map Developer Guide</i>.
         * </p>
         * 
         * @param namespace
         *        The namespace name or full Amazon Resource Name (ARN) of the Cloud Map namespace for use with Service
         *        Connect. The namespace must be in the same Amazon Web Services Region as the Amazon ECS service and
         *        cluster. The type of namespace doesn't affect Service Connect. For more information about Cloud Map,
         *        see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/working-with-services.html">Working with
         *        Services</a> in the <i>Cloud Map Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The list of Service Connect service objects. These are names and aliases (also known as endpoints) that are
         * used by other Amazon ECS services to connect to this service.
         * </p>
         * <p>
         * This field is not required for a "client" Amazon ECS service that's a member of a namespace only to connect
         * to other services within the namespace. An example of this would be a frontend application that accepts
         * incoming requests from either a load balancer that's attached to the service or by other means.
         * </p>
         * <p>
         * An object selects a port from the task definition, assigns a name for the Cloud Map service, and a list of
         * aliases (endpoints) and ports for client applications to refer to this service.
         * </p>
         * 
         * @param services
         *        The list of Service Connect service objects. These are names and aliases (also known as endpoints)
         *        that are used by other Amazon ECS services to connect to this service. </p>
         *        <p>
         *        This field is not required for a "client" Amazon ECS service that's a member of a namespace only to
         *        connect to other services within the namespace. An example of this would be a frontend application
         *        that accepts incoming requests from either a load balancer that's attached to the service or by other
         *        means.
         *        </p>
         *        <p>
         *        An object selects a port from the task definition, assigns a name for the Cloud Map service, and a
         *        list of aliases (endpoints) and ports for client applications to refer to this service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder services(Collection<ServiceConnectService> services);

        /**
         * <p>
         * The list of Service Connect service objects. These are names and aliases (also known as endpoints) that are
         * used by other Amazon ECS services to connect to this service.
         * </p>
         * <p>
         * This field is not required for a "client" Amazon ECS service that's a member of a namespace only to connect
         * to other services within the namespace. An example of this would be a frontend application that accepts
         * incoming requests from either a load balancer that's attached to the service or by other means.
         * </p>
         * <p>
         * An object selects a port from the task definition, assigns a name for the Cloud Map service, and a list of
         * aliases (endpoints) and ports for client applications to refer to this service.
         * </p>
         * 
         * @param services
         *        The list of Service Connect service objects. These are names and aliases (also known as endpoints)
         *        that are used by other Amazon ECS services to connect to this service. </p>
         *        <p>
         *        This field is not required for a "client" Amazon ECS service that's a member of a namespace only to
         *        connect to other services within the namespace. An example of this would be a frontend application
         *        that accepts incoming requests from either a load balancer that's attached to the service or by other
         *        means.
         *        </p>
         *        <p>
         *        An object selects a port from the task definition, assigns a name for the Cloud Map service, and a
         *        list of aliases (endpoints) and ports for client applications to refer to this service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder services(ServiceConnectService... services);

        /**
         * <p>
         * The list of Service Connect service objects. These are names and aliases (also known as endpoints) that are
         * used by other Amazon ECS services to connect to this service.
         * </p>
         * <p>
         * This field is not required for a "client" Amazon ECS service that's a member of a namespace only to connect
         * to other services within the namespace. An example of this would be a frontend application that accepts
         * incoming requests from either a load balancer that's attached to the service or by other means.
         * </p>
         * <p>
         * An object selects a port from the task definition, assigns a name for the Cloud Map service, and a list of
         * aliases (endpoints) and ports for client applications to refer to this service.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ServiceConnectService.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ecs.model.ServiceConnectService#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ServiceConnectService.Builder#build()} is called immediately
         * and its result is passed to {@link #services(List<ServiceConnectService>)}.
         * 
         * @param services
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ServiceConnectService.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #services(java.util.Collection<ServiceConnectService>)
         */
        Builder services(Consumer<ServiceConnectService.Builder>... services);

        /**
         * Sets the value of the LogConfiguration property for this object.
         *
         * @param logConfiguration
         *        The new value for the LogConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfiguration(LogConfiguration logConfiguration);

        /**
         * Sets the value of the LogConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link LogConfiguration.Builder} avoiding the
         * need to create one manually via {@link LogConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LogConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #logConfiguration(LogConfiguration)}.
         * 
         * @param logConfiguration
         *        a consumer that will call methods on {@link LogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfiguration(LogConfiguration)
         */
        default Builder logConfiguration(Consumer<LogConfiguration.Builder> logConfiguration) {
            return logConfiguration(LogConfiguration.builder().applyMutation(logConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private String namespace;

        private List<ServiceConnectService> services = DefaultSdkAutoConstructList.getInstance();

        private LogConfiguration logConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceConnectConfiguration model) {
            enabled(model.enabled);
            namespace(model.namespace);
            services(model.services);
            logConfiguration(model.logConfiguration);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final List<ServiceConnectService.Builder> getServices() {
            List<ServiceConnectService.Builder> result = ServiceConnectServiceListCopier.copyToBuilder(this.services);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setServices(Collection<ServiceConnectService.BuilderImpl> services) {
            this.services = ServiceConnectServiceListCopier.copyFromBuilder(services);
        }

        @Override
        public final Builder services(Collection<ServiceConnectService> services) {
            this.services = ServiceConnectServiceListCopier.copy(services);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder services(ServiceConnectService... services) {
            services(Arrays.asList(services));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder services(Consumer<ServiceConnectService.Builder>... services) {
            services(Stream.of(services).map(c -> ServiceConnectService.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final LogConfiguration.Builder getLogConfiguration() {
            return logConfiguration != null ? logConfiguration.toBuilder() : null;
        }

        public final void setLogConfiguration(LogConfiguration.BuilderImpl logConfiguration) {
            this.logConfiguration = logConfiguration != null ? logConfiguration.build() : null;
        }

        @Override
        public final Builder logConfiguration(LogConfiguration logConfiguration) {
            this.logConfiguration = logConfiguration;
            return this;
        }

        @Override
        public ServiceConnectConfiguration build() {
            return new ServiceConnectConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
