/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecr.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that describes an image returned by a <a>DescribeImages</a> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageDetail implements SdkPojo, Serializable, ToCopyableBuilder<ImageDetail.Builder, ImageDetail> {
    private static final SdkField<String> REGISTRY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("registryId").getter(getter(ImageDetail::registryId)).setter(setter(Builder::registryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registryId").build()).build();

    private static final SdkField<String> REPOSITORY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("repositoryName").getter(getter(ImageDetail::repositoryName)).setter(setter(Builder::repositoryName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryName").build()).build();

    private static final SdkField<String> IMAGE_DIGEST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageDigest").getter(getter(ImageDetail::imageDigest)).setter(setter(Builder::imageDigest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageDigest").build()).build();

    private static final SdkField<List<String>> IMAGE_TAGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("imageTags")
            .getter(getter(ImageDetail::imageTags))
            .setter(setter(Builder::imageTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> IMAGE_SIZE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("imageSizeInBytes").getter(getter(ImageDetail::imageSizeInBytes))
            .setter(setter(Builder::imageSizeInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageSizeInBytes").build()).build();

    private static final SdkField<Instant> IMAGE_PUSHED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("imagePushedAt").getter(getter(ImageDetail::imagePushedAt)).setter(setter(Builder::imagePushedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imagePushedAt").build()).build();

    private static final SdkField<ImageScanStatus> IMAGE_SCAN_STATUS_FIELD = SdkField
            .<ImageScanStatus> builder(MarshallingType.SDK_POJO).memberName("imageScanStatus")
            .getter(getter(ImageDetail::imageScanStatus)).setter(setter(Builder::imageScanStatus))
            .constructor(ImageScanStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageScanStatus").build()).build();

    private static final SdkField<ImageScanFindingsSummary> IMAGE_SCAN_FINDINGS_SUMMARY_FIELD = SdkField
            .<ImageScanFindingsSummary> builder(MarshallingType.SDK_POJO).memberName("imageScanFindingsSummary")
            .getter(getter(ImageDetail::imageScanFindingsSummary)).setter(setter(Builder::imageScanFindingsSummary))
            .constructor(ImageScanFindingsSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageScanFindingsSummary").build())
            .build();

    private static final SdkField<String> IMAGE_MANIFEST_MEDIA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageManifestMediaType").getter(getter(ImageDetail::imageManifestMediaType))
            .setter(setter(Builder::imageManifestMediaType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageManifestMediaType").build())
            .build();

    private static final SdkField<String> ARTIFACT_MEDIA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("artifactMediaType").getter(getter(ImageDetail::artifactMediaType))
            .setter(setter(Builder::artifactMediaType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("artifactMediaType").build()).build();

    private static final SdkField<Instant> LAST_RECORDED_PULL_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRecordedPullTime").getter(getter(ImageDetail::lastRecordedPullTime))
            .setter(setter(Builder::lastRecordedPullTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRecordedPullTime").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REGISTRY_ID_FIELD,
            REPOSITORY_NAME_FIELD, IMAGE_DIGEST_FIELD, IMAGE_TAGS_FIELD, IMAGE_SIZE_IN_BYTES_FIELD, IMAGE_PUSHED_AT_FIELD,
            IMAGE_SCAN_STATUS_FIELD, IMAGE_SCAN_FINDINGS_SUMMARY_FIELD, IMAGE_MANIFEST_MEDIA_TYPE_FIELD,
            ARTIFACT_MEDIA_TYPE_FIELD, LAST_RECORDED_PULL_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String registryId;

    private final String repositoryName;

    private final String imageDigest;

    private final List<String> imageTags;

    private final Long imageSizeInBytes;

    private final Instant imagePushedAt;

    private final ImageScanStatus imageScanStatus;

    private final ImageScanFindingsSummary imageScanFindingsSummary;

    private final String imageManifestMediaType;

    private final String artifactMediaType;

    private final Instant lastRecordedPullTime;

    private ImageDetail(BuilderImpl builder) {
        this.registryId = builder.registryId;
        this.repositoryName = builder.repositoryName;
        this.imageDigest = builder.imageDigest;
        this.imageTags = builder.imageTags;
        this.imageSizeInBytes = builder.imageSizeInBytes;
        this.imagePushedAt = builder.imagePushedAt;
        this.imageScanStatus = builder.imageScanStatus;
        this.imageScanFindingsSummary = builder.imageScanFindingsSummary;
        this.imageManifestMediaType = builder.imageManifestMediaType;
        this.artifactMediaType = builder.artifactMediaType;
        this.lastRecordedPullTime = builder.lastRecordedPullTime;
    }

    /**
     * <p>
     * The Amazon Web Services account ID associated with the registry to which this image belongs.
     * </p>
     * 
     * @return The Amazon Web Services account ID associated with the registry to which this image belongs.
     */
    public final String registryId() {
        return registryId;
    }

    /**
     * <p>
     * The name of the repository to which this image belongs.
     * </p>
     * 
     * @return The name of the repository to which this image belongs.
     */
    public final String repositoryName() {
        return repositoryName;
    }

    /**
     * <p>
     * The <code>sha256</code> digest of the image manifest.
     * </p>
     * 
     * @return The <code>sha256</code> digest of the image manifest.
     */
    public final String imageDigest() {
        return imageDigest;
    }

    /**
     * For responses, this returns true if the service returned a value for the ImageTags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasImageTags() {
        return imageTags != null && !(imageTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of tags associated with this image.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImageTags} method.
     * </p>
     * 
     * @return The list of tags associated with this image.
     */
    public final List<String> imageTags() {
        return imageTags;
    }

    /**
     * <p>
     * The size, in bytes, of the image in the repository.
     * </p>
     * <p>
     * If the image is a manifest list, this will be the max size of all manifests in the list.
     * </p>
     * <note>
     * <p>
     * Beginning with Docker version 1.9, the Docker client compresses image layers before pushing them to a V2 Docker
     * registry. The output of the <code>docker images</code> command shows the uncompressed image size, so it may
     * return a larger image size than the image sizes returned by <a>DescribeImages</a>.
     * </p>
     * </note>
     * 
     * @return The size, in bytes, of the image in the repository.</p>
     *         <p>
     *         If the image is a manifest list, this will be the max size of all manifests in the list.
     *         </p>
     *         <note>
     *         <p>
     *         Beginning with Docker version 1.9, the Docker client compresses image layers before pushing them to a V2
     *         Docker registry. The output of the <code>docker images</code> command shows the uncompressed image size,
     *         so it may return a larger image size than the image sizes returned by <a>DescribeImages</a>.
     *         </p>
     */
    public final Long imageSizeInBytes() {
        return imageSizeInBytes;
    }

    /**
     * <p>
     * The date and time, expressed in standard JavaScript date format, at which the current image was pushed to the
     * repository.
     * </p>
     * 
     * @return The date and time, expressed in standard JavaScript date format, at which the current image was pushed to
     *         the repository.
     */
    public final Instant imagePushedAt() {
        return imagePushedAt;
    }

    /**
     * <p>
     * The current state of the scan.
     * </p>
     * 
     * @return The current state of the scan.
     */
    public final ImageScanStatus imageScanStatus() {
        return imageScanStatus;
    }

    /**
     * <p>
     * A summary of the last completed image scan.
     * </p>
     * 
     * @return A summary of the last completed image scan.
     */
    public final ImageScanFindingsSummary imageScanFindingsSummary() {
        return imageScanFindingsSummary;
    }

    /**
     * <p>
     * The media type of the image manifest.
     * </p>
     * 
     * @return The media type of the image manifest.
     */
    public final String imageManifestMediaType() {
        return imageManifestMediaType;
    }

    /**
     * <p>
     * The artifact media type of the image.
     * </p>
     * 
     * @return The artifact media type of the image.
     */
    public final String artifactMediaType() {
        return artifactMediaType;
    }

    /**
     * <p>
     * The date and time, expressed in standard JavaScript date format, when Amazon ECR recorded the last image pull.
     * </p>
     * <note>
     * <p>
     * Amazon ECR refreshes the last image pull timestamp at least once every 24 hours. For example, if you pull an
     * image once a day then the <code>lastRecordedPullTime</code> timestamp will indicate the exact time that the image
     * was last pulled. However, if you pull an image once an hour, because Amazon ECR refreshes the
     * <code>lastRecordedPullTime</code> timestamp at least once every 24 hours, the result may not be the exact time
     * that the image was last pulled.
     * </p>
     * </note>
     * 
     * @return The date and time, expressed in standard JavaScript date format, when Amazon ECR recorded the last image
     *         pull.</p> <note>
     *         <p>
     *         Amazon ECR refreshes the last image pull timestamp at least once every 24 hours. For example, if you pull
     *         an image once a day then the <code>lastRecordedPullTime</code> timestamp will indicate the exact time
     *         that the image was last pulled. However, if you pull an image once an hour, because Amazon ECR refreshes
     *         the <code>lastRecordedPullTime</code> timestamp at least once every 24 hours, the result may not be the
     *         exact time that the image was last pulled.
     *         </p>
     */
    public final Instant lastRecordedPullTime() {
        return lastRecordedPullTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(registryId());
        hashCode = 31 * hashCode + Objects.hashCode(repositoryName());
        hashCode = 31 * hashCode + Objects.hashCode(imageDigest());
        hashCode = 31 * hashCode + Objects.hashCode(hasImageTags() ? imageTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(imageSizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(imagePushedAt());
        hashCode = 31 * hashCode + Objects.hashCode(imageScanStatus());
        hashCode = 31 * hashCode + Objects.hashCode(imageScanFindingsSummary());
        hashCode = 31 * hashCode + Objects.hashCode(imageManifestMediaType());
        hashCode = 31 * hashCode + Objects.hashCode(artifactMediaType());
        hashCode = 31 * hashCode + Objects.hashCode(lastRecordedPullTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageDetail)) {
            return false;
        }
        ImageDetail other = (ImageDetail) obj;
        return Objects.equals(registryId(), other.registryId()) && Objects.equals(repositoryName(), other.repositoryName())
                && Objects.equals(imageDigest(), other.imageDigest()) && hasImageTags() == other.hasImageTags()
                && Objects.equals(imageTags(), other.imageTags()) && Objects.equals(imageSizeInBytes(), other.imageSizeInBytes())
                && Objects.equals(imagePushedAt(), other.imagePushedAt())
                && Objects.equals(imageScanStatus(), other.imageScanStatus())
                && Objects.equals(imageScanFindingsSummary(), other.imageScanFindingsSummary())
                && Objects.equals(imageManifestMediaType(), other.imageManifestMediaType())
                && Objects.equals(artifactMediaType(), other.artifactMediaType())
                && Objects.equals(lastRecordedPullTime(), other.lastRecordedPullTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImageDetail").add("RegistryId", registryId()).add("RepositoryName", repositoryName())
                .add("ImageDigest", imageDigest()).add("ImageTags", hasImageTags() ? imageTags() : null)
                .add("ImageSizeInBytes", imageSizeInBytes()).add("ImagePushedAt", imagePushedAt())
                .add("ImageScanStatus", imageScanStatus()).add("ImageScanFindingsSummary", imageScanFindingsSummary())
                .add("ImageManifestMediaType", imageManifestMediaType()).add("ArtifactMediaType", artifactMediaType())
                .add("LastRecordedPullTime", lastRecordedPullTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "registryId":
            return Optional.ofNullable(clazz.cast(registryId()));
        case "repositoryName":
            return Optional.ofNullable(clazz.cast(repositoryName()));
        case "imageDigest":
            return Optional.ofNullable(clazz.cast(imageDigest()));
        case "imageTags":
            return Optional.ofNullable(clazz.cast(imageTags()));
        case "imageSizeInBytes":
            return Optional.ofNullable(clazz.cast(imageSizeInBytes()));
        case "imagePushedAt":
            return Optional.ofNullable(clazz.cast(imagePushedAt()));
        case "imageScanStatus":
            return Optional.ofNullable(clazz.cast(imageScanStatus()));
        case "imageScanFindingsSummary":
            return Optional.ofNullable(clazz.cast(imageScanFindingsSummary()));
        case "imageManifestMediaType":
            return Optional.ofNullable(clazz.cast(imageManifestMediaType()));
        case "artifactMediaType":
            return Optional.ofNullable(clazz.cast(artifactMediaType()));
        case "lastRecordedPullTime":
            return Optional.ofNullable(clazz.cast(lastRecordedPullTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImageDetail, T> g) {
        return obj -> g.apply((ImageDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageDetail> {
        /**
         * <p>
         * The Amazon Web Services account ID associated with the registry to which this image belongs.
         * </p>
         * 
         * @param registryId
         *        The Amazon Web Services account ID associated with the registry to which this image belongs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registryId(String registryId);

        /**
         * <p>
         * The name of the repository to which this image belongs.
         * </p>
         * 
         * @param repositoryName
         *        The name of the repository to which this image belongs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryName(String repositoryName);

        /**
         * <p>
         * The <code>sha256</code> digest of the image manifest.
         * </p>
         * 
         * @param imageDigest
         *        The <code>sha256</code> digest of the image manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageDigest(String imageDigest);

        /**
         * <p>
         * The list of tags associated with this image.
         * </p>
         * 
         * @param imageTags
         *        The list of tags associated with this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageTags(Collection<String> imageTags);

        /**
         * <p>
         * The list of tags associated with this image.
         * </p>
         * 
         * @param imageTags
         *        The list of tags associated with this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageTags(String... imageTags);

        /**
         * <p>
         * The size, in bytes, of the image in the repository.
         * </p>
         * <p>
         * If the image is a manifest list, this will be the max size of all manifests in the list.
         * </p>
         * <note>
         * <p>
         * Beginning with Docker version 1.9, the Docker client compresses image layers before pushing them to a V2
         * Docker registry. The output of the <code>docker images</code> command shows the uncompressed image size, so
         * it may return a larger image size than the image sizes returned by <a>DescribeImages</a>.
         * </p>
         * </note>
         * 
         * @param imageSizeInBytes
         *        The size, in bytes, of the image in the repository.</p>
         *        <p>
         *        If the image is a manifest list, this will be the max size of all manifests in the list.
         *        </p>
         *        <note>
         *        <p>
         *        Beginning with Docker version 1.9, the Docker client compresses image layers before pushing them to a
         *        V2 Docker registry. The output of the <code>docker images</code> command shows the uncompressed image
         *        size, so it may return a larger image size than the image sizes returned by <a>DescribeImages</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageSizeInBytes(Long imageSizeInBytes);

        /**
         * <p>
         * The date and time, expressed in standard JavaScript date format, at which the current image was pushed to the
         * repository.
         * </p>
         * 
         * @param imagePushedAt
         *        The date and time, expressed in standard JavaScript date format, at which the current image was pushed
         *        to the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imagePushedAt(Instant imagePushedAt);

        /**
         * <p>
         * The current state of the scan.
         * </p>
         * 
         * @param imageScanStatus
         *        The current state of the scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageScanStatus(ImageScanStatus imageScanStatus);

        /**
         * <p>
         * The current state of the scan.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageScanStatus.Builder} avoiding the
         * need to create one manually via {@link ImageScanStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageScanStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #imageScanStatus(ImageScanStatus)}.
         * 
         * @param imageScanStatus
         *        a consumer that will call methods on {@link ImageScanStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageScanStatus(ImageScanStatus)
         */
        default Builder imageScanStatus(Consumer<ImageScanStatus.Builder> imageScanStatus) {
            return imageScanStatus(ImageScanStatus.builder().applyMutation(imageScanStatus).build());
        }

        /**
         * <p>
         * A summary of the last completed image scan.
         * </p>
         * 
         * @param imageScanFindingsSummary
         *        A summary of the last completed image scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageScanFindingsSummary(ImageScanFindingsSummary imageScanFindingsSummary);

        /**
         * <p>
         * A summary of the last completed image scan.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageScanFindingsSummary.Builder}
         * avoiding the need to create one manually via {@link ImageScanFindingsSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageScanFindingsSummary.Builder#build()} is called immediately
         * and its result is passed to {@link #imageScanFindingsSummary(ImageScanFindingsSummary)}.
         * 
         * @param imageScanFindingsSummary
         *        a consumer that will call methods on {@link ImageScanFindingsSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageScanFindingsSummary(ImageScanFindingsSummary)
         */
        default Builder imageScanFindingsSummary(Consumer<ImageScanFindingsSummary.Builder> imageScanFindingsSummary) {
            return imageScanFindingsSummary(ImageScanFindingsSummary.builder().applyMutation(imageScanFindingsSummary).build());
        }

        /**
         * <p>
         * The media type of the image manifest.
         * </p>
         * 
         * @param imageManifestMediaType
         *        The media type of the image manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageManifestMediaType(String imageManifestMediaType);

        /**
         * <p>
         * The artifact media type of the image.
         * </p>
         * 
         * @param artifactMediaType
         *        The artifact media type of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifactMediaType(String artifactMediaType);

        /**
         * <p>
         * The date and time, expressed in standard JavaScript date format, when Amazon ECR recorded the last image
         * pull.
         * </p>
         * <note>
         * <p>
         * Amazon ECR refreshes the last image pull timestamp at least once every 24 hours. For example, if you pull an
         * image once a day then the <code>lastRecordedPullTime</code> timestamp will indicate the exact time that the
         * image was last pulled. However, if you pull an image once an hour, because Amazon ECR refreshes the
         * <code>lastRecordedPullTime</code> timestamp at least once every 24 hours, the result may not be the exact
         * time that the image was last pulled.
         * </p>
         * </note>
         * 
         * @param lastRecordedPullTime
         *        The date and time, expressed in standard JavaScript date format, when Amazon ECR recorded the last
         *        image pull.</p> <note>
         *        <p>
         *        Amazon ECR refreshes the last image pull timestamp at least once every 24 hours. For example, if you
         *        pull an image once a day then the <code>lastRecordedPullTime</code> timestamp will indicate the exact
         *        time that the image was last pulled. However, if you pull an image once an hour, because Amazon ECR
         *        refreshes the <code>lastRecordedPullTime</code> timestamp at least once every 24 hours, the result may
         *        not be the exact time that the image was last pulled.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRecordedPullTime(Instant lastRecordedPullTime);
    }

    static final class BuilderImpl implements Builder {
        private String registryId;

        private String repositoryName;

        private String imageDigest;

        private List<String> imageTags = DefaultSdkAutoConstructList.getInstance();

        private Long imageSizeInBytes;

        private Instant imagePushedAt;

        private ImageScanStatus imageScanStatus;

        private ImageScanFindingsSummary imageScanFindingsSummary;

        private String imageManifestMediaType;

        private String artifactMediaType;

        private Instant lastRecordedPullTime;

        private BuilderImpl() {
        }

        private BuilderImpl(ImageDetail model) {
            registryId(model.registryId);
            repositoryName(model.repositoryName);
            imageDigest(model.imageDigest);
            imageTags(model.imageTags);
            imageSizeInBytes(model.imageSizeInBytes);
            imagePushedAt(model.imagePushedAt);
            imageScanStatus(model.imageScanStatus);
            imageScanFindingsSummary(model.imageScanFindingsSummary);
            imageManifestMediaType(model.imageManifestMediaType);
            artifactMediaType(model.artifactMediaType);
            lastRecordedPullTime(model.lastRecordedPullTime);
        }

        public final String getRegistryId() {
            return registryId;
        }

        public final void setRegistryId(String registryId) {
            this.registryId = registryId;
        }

        @Override
        public final Builder registryId(String registryId) {
            this.registryId = registryId;
            return this;
        }

        public final String getRepositoryName() {
            return repositoryName;
        }

        public final void setRepositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
        }

        @Override
        public final Builder repositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
            return this;
        }

        public final String getImageDigest() {
            return imageDigest;
        }

        public final void setImageDigest(String imageDigest) {
            this.imageDigest = imageDigest;
        }

        @Override
        public final Builder imageDigest(String imageDigest) {
            this.imageDigest = imageDigest;
            return this;
        }

        public final Collection<String> getImageTags() {
            if (imageTags instanceof SdkAutoConstructList) {
                return null;
            }
            return imageTags;
        }

        public final void setImageTags(Collection<String> imageTags) {
            this.imageTags = ImageTagListCopier.copy(imageTags);
        }

        @Override
        public final Builder imageTags(Collection<String> imageTags) {
            this.imageTags = ImageTagListCopier.copy(imageTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageTags(String... imageTags) {
            imageTags(Arrays.asList(imageTags));
            return this;
        }

        public final Long getImageSizeInBytes() {
            return imageSizeInBytes;
        }

        public final void setImageSizeInBytes(Long imageSizeInBytes) {
            this.imageSizeInBytes = imageSizeInBytes;
        }

        @Override
        public final Builder imageSizeInBytes(Long imageSizeInBytes) {
            this.imageSizeInBytes = imageSizeInBytes;
            return this;
        }

        public final Instant getImagePushedAt() {
            return imagePushedAt;
        }

        public final void setImagePushedAt(Instant imagePushedAt) {
            this.imagePushedAt = imagePushedAt;
        }

        @Override
        public final Builder imagePushedAt(Instant imagePushedAt) {
            this.imagePushedAt = imagePushedAt;
            return this;
        }

        public final ImageScanStatus.Builder getImageScanStatus() {
            return imageScanStatus != null ? imageScanStatus.toBuilder() : null;
        }

        public final void setImageScanStatus(ImageScanStatus.BuilderImpl imageScanStatus) {
            this.imageScanStatus = imageScanStatus != null ? imageScanStatus.build() : null;
        }

        @Override
        public final Builder imageScanStatus(ImageScanStatus imageScanStatus) {
            this.imageScanStatus = imageScanStatus;
            return this;
        }

        public final ImageScanFindingsSummary.Builder getImageScanFindingsSummary() {
            return imageScanFindingsSummary != null ? imageScanFindingsSummary.toBuilder() : null;
        }

        public final void setImageScanFindingsSummary(ImageScanFindingsSummary.BuilderImpl imageScanFindingsSummary) {
            this.imageScanFindingsSummary = imageScanFindingsSummary != null ? imageScanFindingsSummary.build() : null;
        }

        @Override
        public final Builder imageScanFindingsSummary(ImageScanFindingsSummary imageScanFindingsSummary) {
            this.imageScanFindingsSummary = imageScanFindingsSummary;
            return this;
        }

        public final String getImageManifestMediaType() {
            return imageManifestMediaType;
        }

        public final void setImageManifestMediaType(String imageManifestMediaType) {
            this.imageManifestMediaType = imageManifestMediaType;
        }

        @Override
        public final Builder imageManifestMediaType(String imageManifestMediaType) {
            this.imageManifestMediaType = imageManifestMediaType;
            return this;
        }

        public final String getArtifactMediaType() {
            return artifactMediaType;
        }

        public final void setArtifactMediaType(String artifactMediaType) {
            this.artifactMediaType = artifactMediaType;
        }

        @Override
        public final Builder artifactMediaType(String artifactMediaType) {
            this.artifactMediaType = artifactMediaType;
            return this;
        }

        public final Instant getLastRecordedPullTime() {
            return lastRecordedPullTime;
        }

        public final void setLastRecordedPullTime(Instant lastRecordedPullTime) {
            this.lastRecordedPullTime = lastRecordedPullTime;
        }

        @Override
        public final Builder lastRecordedPullTime(Instant lastRecordedPullTime) {
            this.lastRecordedPullTime = lastRecordedPullTime;
            return this;
        }

        @Override
        public ImageDetail build() {
            return new ImageDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
