/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon EC2 instance that is enabled for SQL Server High Availability standby detection monitoring.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RegisteredInstance implements SdkPojo, Serializable,
        ToCopyableBuilder<RegisteredInstance.Builder, RegisteredInstance> {
    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceId")
            .getter(getter(RegisteredInstance::instanceId))
            .setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId")
                    .unmarshallLocationName("instanceId").build()).build();

    private static final SdkField<String> SQL_SERVER_LICENSE_USAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SqlServerLicenseUsage")
            .getter(getter(RegisteredInstance::sqlServerLicenseUsageAsString))
            .setter(setter(Builder::sqlServerLicenseUsage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SqlServerLicenseUsage")
                    .unmarshallLocationName("sqlServerLicenseUsage").build()).build();

    private static final SdkField<String> HA_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("HaStatus")
            .getter(getter(RegisteredInstance::haStatusAsString))
            .setter(setter(Builder::haStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HaStatus")
                    .unmarshallLocationName("haStatus").build()).build();

    private static final SdkField<String> PROCESSING_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ProcessingStatus")
            .getter(getter(RegisteredInstance::processingStatus))
            .setter(setter(Builder::processingStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessingStatus")
                    .unmarshallLocationName("processingStatus").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime")
            .getter(getter(RegisteredInstance::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime")
                    .unmarshallLocationName("lastUpdatedTime").build()).build();

    private static final SdkField<String> SQL_SERVER_CREDENTIALS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SqlServerCredentials")
            .getter(getter(RegisteredInstance::sqlServerCredentials))
            .setter(setter(Builder::sqlServerCredentials))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SqlServerCredentials")
                    .unmarshallLocationName("sqlServerCredentials").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(RegisteredInstance::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ID_FIELD,
            SQL_SERVER_LICENSE_USAGE_FIELD, HA_STATUS_FIELD, PROCESSING_STATUS_FIELD, LAST_UPDATED_TIME_FIELD,
            SQL_SERVER_CREDENTIALS_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceId;

    private final String sqlServerLicenseUsage;

    private final String haStatus;

    private final String processingStatus;

    private final Instant lastUpdatedTime;

    private final String sqlServerCredentials;

    private final List<Tag> tags;

    private RegisteredInstance(BuilderImpl builder) {
        this.instanceId = builder.instanceId;
        this.sqlServerLicenseUsage = builder.sqlServerLicenseUsage;
        this.haStatus = builder.haStatus;
        this.processingStatus = builder.processingStatus;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.sqlServerCredentials = builder.sqlServerCredentials;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the SQL Server High Availability instance.
     * </p>
     * 
     * @return The ID of the SQL Server High Availability instance.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The license type for the SQL Server license. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>full</code> - The SQL Server High Availability instance is using a full SQL Server license.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>waived</code> - The SQL Server High Availability instance is waived from the SQL Server license.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #sqlServerLicenseUsage} will return {@link SqlServerLicenseUsage#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #sqlServerLicenseUsageAsString}.
     * </p>
     * 
     * @return The license type for the SQL Server license. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>full</code> - The SQL Server High Availability instance is using a full SQL Server license.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>waived</code> - The SQL Server High Availability instance is waived from the SQL Server license.
     *         </p>
     *         </li>
     * @see SqlServerLicenseUsage
     */
    public final SqlServerLicenseUsage sqlServerLicenseUsage() {
        return SqlServerLicenseUsage.fromValue(sqlServerLicenseUsage);
    }

    /**
     * <p>
     * The license type for the SQL Server license. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>full</code> - The SQL Server High Availability instance is using a full SQL Server license.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>waived</code> - The SQL Server High Availability instance is waived from the SQL Server license.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #sqlServerLicenseUsage} will return {@link SqlServerLicenseUsage#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #sqlServerLicenseUsageAsString}.
     * </p>
     * 
     * @return The license type for the SQL Server license. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>full</code> - The SQL Server High Availability instance is using a full SQL Server license.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>waived</code> - The SQL Server High Availability instance is waived from the SQL Server license.
     *         </p>
     *         </li>
     * @see SqlServerLicenseUsage
     */
    public final String sqlServerLicenseUsageAsString() {
        return sqlServerLicenseUsage;
    }

    /**
     * <p>
     * The SQL Server High Availability status of the instance. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>processing</code> - The SQL Server High Availability status for the SQL Server High Availability instance
     * is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>active</code> - The SQL Server High Availability instance is an active node in an SQL Server High
     * Availability cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>standby</code> - The SQL Server High Availability instance is a standby failover node in an SQL Server High
     * Availability cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>invalid</code> - An error occurred due to misconfigured permissions, or unable to dertemine SQL Server High
     * Availability status for the SQL Server High Availability instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #haStatus} will
     * return {@link HaStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #haStatusAsString}.
     * </p>
     * 
     * @return The SQL Server High Availability status of the instance. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>processing</code> - The SQL Server High Availability status for the SQL Server High Availability
     *         instance is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>active</code> - The SQL Server High Availability instance is an active node in an SQL Server High
     *         Availability cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>standby</code> - The SQL Server High Availability instance is a standby failover node in an SQL
     *         Server High Availability cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>invalid</code> - An error occurred due to misconfigured permissions, or unable to dertemine SQL
     *         Server High Availability status for the SQL Server High Availability instance.
     *         </p>
     *         </li>
     * @see HaStatus
     */
    public final HaStatus haStatus() {
        return HaStatus.fromValue(haStatus);
    }

    /**
     * <p>
     * The SQL Server High Availability status of the instance. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>processing</code> - The SQL Server High Availability status for the SQL Server High Availability instance
     * is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>active</code> - The SQL Server High Availability instance is an active node in an SQL Server High
     * Availability cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>standby</code> - The SQL Server High Availability instance is a standby failover node in an SQL Server High
     * Availability cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>invalid</code> - An error occurred due to misconfigured permissions, or unable to dertemine SQL Server High
     * Availability status for the SQL Server High Availability instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #haStatus} will
     * return {@link HaStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #haStatusAsString}.
     * </p>
     * 
     * @return The SQL Server High Availability status of the instance. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>processing</code> - The SQL Server High Availability status for the SQL Server High Availability
     *         instance is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>active</code> - The SQL Server High Availability instance is an active node in an SQL Server High
     *         Availability cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>standby</code> - The SQL Server High Availability instance is a standby failover node in an SQL
     *         Server High Availability cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>invalid</code> - An error occurred due to misconfigured permissions, or unable to dertemine SQL
     *         Server High Availability status for the SQL Server High Availability instance.
     *         </p>
     *         </li>
     * @see HaStatus
     */
    public final String haStatusAsString() {
        return haStatus;
    }

    /**
     * <p>
     * A brief description of the SQL Server High Availability status. If the instance is in the <code>invalid</code>
     * High Availability status, this parameter includes the error message.
     * </p>
     * 
     * @return A brief description of the SQL Server High Availability status. If the instance is in the
     *         <code>invalid</code> High Availability status, this parameter includes the error message.
     */
    public final String processingStatus() {
        return processingStatus;
    }

    /**
     * <p>
     * The date and time when the instance's SQL Server High Availability status was last updated, in the ISO 8601
     * format in the UTC time zone (<code>YYYY-MM-DDThh:mm:ss.sssZ</code>).
     * </p>
     * 
     * @return The date and time when the instance's SQL Server High Availability status was last updated, in the ISO
     *         8601 format in the UTC time zone (<code>YYYY-MM-DDThh:mm:ss.sssZ</code>).
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The ARN of the Secrets Manager secret containing the SQL Server access credentials for the SQL Server High
     * Availability instance. If not specified, deafult local user credentials will be used by the Amazon Web Services
     * Systems Manager agent.
     * </p>
     * 
     * @return The ARN of the Secrets Manager secret containing the SQL Server access credentials for the SQL Server
     *         High Availability instance. If not specified, deafult local user credentials will be used by the Amazon
     *         Web Services Systems Manager agent.
     */
    public final String sqlServerCredentials() {
        return sqlServerCredentials;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags assigned to the SQL Server High Availability instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the SQL Server High Availability instance.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(sqlServerLicenseUsageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(haStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(processingStatus());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(sqlServerCredentials());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RegisteredInstance)) {
            return false;
        }
        RegisteredInstance other = (RegisteredInstance) obj;
        return Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(sqlServerLicenseUsageAsString(), other.sqlServerLicenseUsageAsString())
                && Objects.equals(haStatusAsString(), other.haStatusAsString())
                && Objects.equals(processingStatus(), other.processingStatus())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(sqlServerCredentials(), other.sqlServerCredentials()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RegisteredInstance").add("InstanceId", instanceId())
                .add("SqlServerLicenseUsage", sqlServerLicenseUsageAsString()).add("HaStatus", haStatusAsString())
                .add("ProcessingStatus", processingStatus()).add("LastUpdatedTime", lastUpdatedTime())
                .add("SqlServerCredentials", sqlServerCredentials()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "SqlServerLicenseUsage":
            return Optional.ofNullable(clazz.cast(sqlServerLicenseUsageAsString()));
        case "HaStatus":
            return Optional.ofNullable(clazz.cast(haStatusAsString()));
        case "ProcessingStatus":
            return Optional.ofNullable(clazz.cast(processingStatus()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "SqlServerCredentials":
            return Optional.ofNullable(clazz.cast(sqlServerCredentials()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceId", INSTANCE_ID_FIELD);
        map.put("SqlServerLicenseUsage", SQL_SERVER_LICENSE_USAGE_FIELD);
        map.put("HaStatus", HA_STATUS_FIELD);
        map.put("ProcessingStatus", PROCESSING_STATUS_FIELD);
        map.put("LastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        map.put("SqlServerCredentials", SQL_SERVER_CREDENTIALS_FIELD);
        map.put("TagSet", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RegisteredInstance, T> g) {
        return obj -> g.apply((RegisteredInstance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RegisteredInstance> {
        /**
         * <p>
         * The ID of the SQL Server High Availability instance.
         * </p>
         * 
         * @param instanceId
         *        The ID of the SQL Server High Availability instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The license type for the SQL Server license. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>full</code> - The SQL Server High Availability instance is using a full SQL Server license.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>waived</code> - The SQL Server High Availability instance is waived from the SQL Server license.
         * </p>
         * </li>
         * </ul>
         * 
         * @param sqlServerLicenseUsage
         *        The license type for the SQL Server license. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>full</code> - The SQL Server High Availability instance is using a full SQL Server license.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>waived</code> - The SQL Server High Availability instance is waived from the SQL Server license.
         *        </p>
         *        </li>
         * @see SqlServerLicenseUsage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SqlServerLicenseUsage
         */
        Builder sqlServerLicenseUsage(String sqlServerLicenseUsage);

        /**
         * <p>
         * The license type for the SQL Server license. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>full</code> - The SQL Server High Availability instance is using a full SQL Server license.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>waived</code> - The SQL Server High Availability instance is waived from the SQL Server license.
         * </p>
         * </li>
         * </ul>
         * 
         * @param sqlServerLicenseUsage
         *        The license type for the SQL Server license. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>full</code> - The SQL Server High Availability instance is using a full SQL Server license.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>waived</code> - The SQL Server High Availability instance is waived from the SQL Server license.
         *        </p>
         *        </li>
         * @see SqlServerLicenseUsage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SqlServerLicenseUsage
         */
        Builder sqlServerLicenseUsage(SqlServerLicenseUsage sqlServerLicenseUsage);

        /**
         * <p>
         * The SQL Server High Availability status of the instance. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>processing</code> - The SQL Server High Availability status for the SQL Server High Availability
         * instance is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>active</code> - The SQL Server High Availability instance is an active node in an SQL Server High
         * Availability cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>standby</code> - The SQL Server High Availability instance is a standby failover node in an SQL Server
         * High Availability cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>invalid</code> - An error occurred due to misconfigured permissions, or unable to dertemine SQL Server
         * High Availability status for the SQL Server High Availability instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param haStatus
         *        The SQL Server High Availability status of the instance. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>processing</code> - The SQL Server High Availability status for the SQL Server High Availability
         *        instance is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>active</code> - The SQL Server High Availability instance is an active node in an SQL Server
         *        High Availability cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>standby</code> - The SQL Server High Availability instance is a standby failover node in an SQL
         *        Server High Availability cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>invalid</code> - An error occurred due to misconfigured permissions, or unable to dertemine SQL
         *        Server High Availability status for the SQL Server High Availability instance.
         *        </p>
         *        </li>
         * @see HaStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HaStatus
         */
        Builder haStatus(String haStatus);

        /**
         * <p>
         * The SQL Server High Availability status of the instance. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>processing</code> - The SQL Server High Availability status for the SQL Server High Availability
         * instance is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>active</code> - The SQL Server High Availability instance is an active node in an SQL Server High
         * Availability cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>standby</code> - The SQL Server High Availability instance is a standby failover node in an SQL Server
         * High Availability cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>invalid</code> - An error occurred due to misconfigured permissions, or unable to dertemine SQL Server
         * High Availability status for the SQL Server High Availability instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param haStatus
         *        The SQL Server High Availability status of the instance. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>processing</code> - The SQL Server High Availability status for the SQL Server High Availability
         *        instance is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>active</code> - The SQL Server High Availability instance is an active node in an SQL Server
         *        High Availability cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>standby</code> - The SQL Server High Availability instance is a standby failover node in an SQL
         *        Server High Availability cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>invalid</code> - An error occurred due to misconfigured permissions, or unable to dertemine SQL
         *        Server High Availability status for the SQL Server High Availability instance.
         *        </p>
         *        </li>
         * @see HaStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HaStatus
         */
        Builder haStatus(HaStatus haStatus);

        /**
         * <p>
         * A brief description of the SQL Server High Availability status. If the instance is in the
         * <code>invalid</code> High Availability status, this parameter includes the error message.
         * </p>
         * 
         * @param processingStatus
         *        A brief description of the SQL Server High Availability status. If the instance is in the
         *        <code>invalid</code> High Availability status, this parameter includes the error message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingStatus(String processingStatus);

        /**
         * <p>
         * The date and time when the instance's SQL Server High Availability status was last updated, in the ISO 8601
         * format in the UTC time zone (<code>YYYY-MM-DDThh:mm:ss.sssZ</code>).
         * </p>
         * 
         * @param lastUpdatedTime
         *        The date and time when the instance's SQL Server High Availability status was last updated, in the ISO
         *        8601 format in the UTC time zone (<code>YYYY-MM-DDThh:mm:ss.sssZ</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The ARN of the Secrets Manager secret containing the SQL Server access credentials for the SQL Server High
         * Availability instance. If not specified, deafult local user credentials will be used by the Amazon Web
         * Services Systems Manager agent.
         * </p>
         * 
         * @param sqlServerCredentials
         *        The ARN of the Secrets Manager secret containing the SQL Server access credentials for the SQL Server
         *        High Availability instance. If not specified, deafult local user credentials will be used by the
         *        Amazon Web Services Systems Manager agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sqlServerCredentials(String sqlServerCredentials);

        /**
         * <p>
         * The tags assigned to the SQL Server High Availability instance.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the SQL Server High Availability instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags assigned to the SQL Server High Availability instance.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the SQL Server High Availability instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags assigned to the SQL Server High Availability instance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String instanceId;

        private String sqlServerLicenseUsage;

        private String haStatus;

        private String processingStatus;

        private Instant lastUpdatedTime;

        private String sqlServerCredentials;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RegisteredInstance model) {
            instanceId(model.instanceId);
            sqlServerLicenseUsage(model.sqlServerLicenseUsage);
            haStatus(model.haStatus);
            processingStatus(model.processingStatus);
            lastUpdatedTime(model.lastUpdatedTime);
            sqlServerCredentials(model.sqlServerCredentials);
            tags(model.tags);
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getSqlServerLicenseUsage() {
            return sqlServerLicenseUsage;
        }

        public final void setSqlServerLicenseUsage(String sqlServerLicenseUsage) {
            this.sqlServerLicenseUsage = sqlServerLicenseUsage;
        }

        @Override
        public final Builder sqlServerLicenseUsage(String sqlServerLicenseUsage) {
            this.sqlServerLicenseUsage = sqlServerLicenseUsage;
            return this;
        }

        @Override
        public final Builder sqlServerLicenseUsage(SqlServerLicenseUsage sqlServerLicenseUsage) {
            this.sqlServerLicenseUsage(sqlServerLicenseUsage == null ? null : sqlServerLicenseUsage.toString());
            return this;
        }

        public final String getHaStatus() {
            return haStatus;
        }

        public final void setHaStatus(String haStatus) {
            this.haStatus = haStatus;
        }

        @Override
        public final Builder haStatus(String haStatus) {
            this.haStatus = haStatus;
            return this;
        }

        @Override
        public final Builder haStatus(HaStatus haStatus) {
            this.haStatus(haStatus == null ? null : haStatus.toString());
            return this;
        }

        public final String getProcessingStatus() {
            return processingStatus;
        }

        public final void setProcessingStatus(String processingStatus) {
            this.processingStatus = processingStatus;
        }

        @Override
        public final Builder processingStatus(String processingStatus) {
            this.processingStatus = processingStatus;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final String getSqlServerCredentials() {
            return sqlServerCredentials;
        }

        public final void setSqlServerCredentials(String sqlServerCredentials) {
            this.sqlServerCredentials = sqlServerCredentials;
        }

        @Override
        public final Builder sqlServerCredentials(String sqlServerCredentials) {
            this.sqlServerCredentials = sqlServerCredentials;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public RegisteredInstance build() {
            return new RegisteredInstance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
