/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the GPU accelerators for the instance type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GpuDeviceInfo implements SdkPojo, Serializable, ToCopyableBuilder<GpuDeviceInfo.Builder, GpuDeviceInfo> {
    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(GpuDeviceInfo::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("name").build()).build();

    private static final SdkField<String> MANUFACTURER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Manufacturer")
            .getter(getter(GpuDeviceInfo::manufacturer))
            .setter(setter(Builder::manufacturer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Manufacturer")
                    .unmarshallLocationName("manufacturer").build()).build();

    private static final SdkField<Integer> COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Count")
            .getter(getter(GpuDeviceInfo::count))
            .setter(setter(Builder::count))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Count")
                    .unmarshallLocationName("count").build()).build();

    private static final SdkField<Integer> LOGICAL_GPU_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("LogicalGpuCount")
            .getter(getter(GpuDeviceInfo::logicalGpuCount))
            .setter(setter(Builder::logicalGpuCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalGpuCount")
                    .unmarshallLocationName("logicalGpuCount").build()).build();

    private static final SdkField<Double> GPU_PARTITION_SIZE_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("GpuPartitionSize")
            .getter(getter(GpuDeviceInfo::gpuPartitionSize))
            .setter(setter(Builder::gpuPartitionSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GpuPartitionSize")
                    .unmarshallLocationName("gpuPartitionSize").build()).build();

    private static final SdkField<List<String>> WORKLOADS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Workloads")
            .getter(getter(GpuDeviceInfo::workloads))
            .setter(setter(Builder::workloads))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkloadSet")
                    .unmarshallLocationName("workloadSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<GpuDeviceMemoryInfo> MEMORY_INFO_FIELD = SdkField
            .<GpuDeviceMemoryInfo> builder(MarshallingType.SDK_POJO)
            .memberName("MemoryInfo")
            .getter(getter(GpuDeviceInfo::memoryInfo))
            .setter(setter(Builder::memoryInfo))
            .constructor(GpuDeviceMemoryInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MemoryInfo")
                    .unmarshallLocationName("memoryInfo").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            MANUFACTURER_FIELD, COUNT_FIELD, LOGICAL_GPU_COUNT_FIELD, GPU_PARTITION_SIZE_FIELD, WORKLOADS_FIELD,
            MEMORY_INFO_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String manufacturer;

    private final Integer count;

    private final Integer logicalGpuCount;

    private final Double gpuPartitionSize;

    private final List<String> workloads;

    private final GpuDeviceMemoryInfo memoryInfo;

    private GpuDeviceInfo(BuilderImpl builder) {
        this.name = builder.name;
        this.manufacturer = builder.manufacturer;
        this.count = builder.count;
        this.logicalGpuCount = builder.logicalGpuCount;
        this.gpuPartitionSize = builder.gpuPartitionSize;
        this.workloads = builder.workloads;
        this.memoryInfo = builder.memoryInfo;
    }

    /**
     * <p>
     * The name of the GPU accelerator.
     * </p>
     * 
     * @return The name of the GPU accelerator.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The manufacturer of the GPU accelerator.
     * </p>
     * 
     * @return The manufacturer of the GPU accelerator.
     */
    public final String manufacturer() {
        return manufacturer;
    }

    /**
     * <p>
     * The number of GPUs for the instance type.
     * </p>
     * 
     * @return The number of GPUs for the instance type.
     */
    public final Integer count() {
        return count;
    }

    /**
     * <p>
     * Total number of GPU devices of this type.
     * </p>
     * 
     * @return Total number of GPU devices of this type.
     */
    public final Integer logicalGpuCount() {
        return logicalGpuCount;
    }

    /**
     * <p>
     * The size of each GPU as a fraction of a full GPU, between 0 (excluded) and 1 (included).
     * </p>
     * 
     * @return The size of each GPU as a fraction of a full GPU, between 0 (excluded) and 1 (included).
     */
    public final Double gpuPartitionSize() {
        return gpuPartitionSize;
    }

    /**
     * For responses, this returns true if the service returned a value for the Workloads property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasWorkloads() {
        return workloads != null && !(workloads instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of workload types this GPU supports.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWorkloads} method.
     * </p>
     * 
     * @return A list of workload types this GPU supports.
     */
    public final List<String> workloads() {
        return workloads;
    }

    /**
     * <p>
     * Describes the memory available to the GPU accelerator.
     * </p>
     * 
     * @return Describes the memory available to the GPU accelerator.
     */
    public final GpuDeviceMemoryInfo memoryInfo() {
        return memoryInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(manufacturer());
        hashCode = 31 * hashCode + Objects.hashCode(count());
        hashCode = 31 * hashCode + Objects.hashCode(logicalGpuCount());
        hashCode = 31 * hashCode + Objects.hashCode(gpuPartitionSize());
        hashCode = 31 * hashCode + Objects.hashCode(hasWorkloads() ? workloads() : null);
        hashCode = 31 * hashCode + Objects.hashCode(memoryInfo());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GpuDeviceInfo)) {
            return false;
        }
        GpuDeviceInfo other = (GpuDeviceInfo) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(manufacturer(), other.manufacturer())
                && Objects.equals(count(), other.count()) && Objects.equals(logicalGpuCount(), other.logicalGpuCount())
                && Objects.equals(gpuPartitionSize(), other.gpuPartitionSize()) && hasWorkloads() == other.hasWorkloads()
                && Objects.equals(workloads(), other.workloads()) && Objects.equals(memoryInfo(), other.memoryInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GpuDeviceInfo").add("Name", name()).add("Manufacturer", manufacturer()).add("Count", count())
                .add("LogicalGpuCount", logicalGpuCount()).add("GpuPartitionSize", gpuPartitionSize())
                .add("Workloads", hasWorkloads() ? workloads() : null).add("MemoryInfo", memoryInfo()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Manufacturer":
            return Optional.ofNullable(clazz.cast(manufacturer()));
        case "Count":
            return Optional.ofNullable(clazz.cast(count()));
        case "LogicalGpuCount":
            return Optional.ofNullable(clazz.cast(logicalGpuCount()));
        case "GpuPartitionSize":
            return Optional.ofNullable(clazz.cast(gpuPartitionSize()));
        case "Workloads":
            return Optional.ofNullable(clazz.cast(workloads()));
        case "MemoryInfo":
            return Optional.ofNullable(clazz.cast(memoryInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Manufacturer", MANUFACTURER_FIELD);
        map.put("Count", COUNT_FIELD);
        map.put("LogicalGpuCount", LOGICAL_GPU_COUNT_FIELD);
        map.put("GpuPartitionSize", GPU_PARTITION_SIZE_FIELD);
        map.put("WorkloadSet", WORKLOADS_FIELD);
        map.put("MemoryInfo", MEMORY_INFO_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GpuDeviceInfo, T> g) {
        return obj -> g.apply((GpuDeviceInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GpuDeviceInfo> {
        /**
         * <p>
         * The name of the GPU accelerator.
         * </p>
         * 
         * @param name
         *        The name of the GPU accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The manufacturer of the GPU accelerator.
         * </p>
         * 
         * @param manufacturer
         *        The manufacturer of the GPU accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manufacturer(String manufacturer);

        /**
         * <p>
         * The number of GPUs for the instance type.
         * </p>
         * 
         * @param count
         *        The number of GPUs for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder count(Integer count);

        /**
         * <p>
         * Total number of GPU devices of this type.
         * </p>
         * 
         * @param logicalGpuCount
         *        Total number of GPU devices of this type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalGpuCount(Integer logicalGpuCount);

        /**
         * <p>
         * The size of each GPU as a fraction of a full GPU, between 0 (excluded) and 1 (included).
         * </p>
         * 
         * @param gpuPartitionSize
         *        The size of each GPU as a fraction of a full GPU, between 0 (excluded) and 1 (included).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gpuPartitionSize(Double gpuPartitionSize);

        /**
         * <p>
         * A list of workload types this GPU supports.
         * </p>
         * 
         * @param workloads
         *        A list of workload types this GPU supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workloads(Collection<String> workloads);

        /**
         * <p>
         * A list of workload types this GPU supports.
         * </p>
         * 
         * @param workloads
         *        A list of workload types this GPU supports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workloads(String... workloads);

        /**
         * <p>
         * Describes the memory available to the GPU accelerator.
         * </p>
         * 
         * @param memoryInfo
         *        Describes the memory available to the GPU accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryInfo(GpuDeviceMemoryInfo memoryInfo);

        /**
         * <p>
         * Describes the memory available to the GPU accelerator.
         * </p>
         * This is a convenience method that creates an instance of the {@link GpuDeviceMemoryInfo.Builder} avoiding the
         * need to create one manually via {@link GpuDeviceMemoryInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GpuDeviceMemoryInfo.Builder#build()} is called immediately and
         * its result is passed to {@link #memoryInfo(GpuDeviceMemoryInfo)}.
         * 
         * @param memoryInfo
         *        a consumer that will call methods on {@link GpuDeviceMemoryInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryInfo(GpuDeviceMemoryInfo)
         */
        default Builder memoryInfo(Consumer<GpuDeviceMemoryInfo.Builder> memoryInfo) {
            return memoryInfo(GpuDeviceMemoryInfo.builder().applyMutation(memoryInfo).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String manufacturer;

        private Integer count;

        private Integer logicalGpuCount;

        private Double gpuPartitionSize;

        private List<String> workloads = DefaultSdkAutoConstructList.getInstance();

        private GpuDeviceMemoryInfo memoryInfo;

        private BuilderImpl() {
        }

        private BuilderImpl(GpuDeviceInfo model) {
            name(model.name);
            manufacturer(model.manufacturer);
            count(model.count);
            logicalGpuCount(model.logicalGpuCount);
            gpuPartitionSize(model.gpuPartitionSize);
            workloads(model.workloads);
            memoryInfo(model.memoryInfo);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getManufacturer() {
            return manufacturer;
        }

        public final void setManufacturer(String manufacturer) {
            this.manufacturer = manufacturer;
        }

        @Override
        public final Builder manufacturer(String manufacturer) {
            this.manufacturer = manufacturer;
            return this;
        }

        public final Integer getCount() {
            return count;
        }

        public final void setCount(Integer count) {
            this.count = count;
        }

        @Override
        public final Builder count(Integer count) {
            this.count = count;
            return this;
        }

        public final Integer getLogicalGpuCount() {
            return logicalGpuCount;
        }

        public final void setLogicalGpuCount(Integer logicalGpuCount) {
            this.logicalGpuCount = logicalGpuCount;
        }

        @Override
        public final Builder logicalGpuCount(Integer logicalGpuCount) {
            this.logicalGpuCount = logicalGpuCount;
            return this;
        }

        public final Double getGpuPartitionSize() {
            return gpuPartitionSize;
        }

        public final void setGpuPartitionSize(Double gpuPartitionSize) {
            this.gpuPartitionSize = gpuPartitionSize;
        }

        @Override
        public final Builder gpuPartitionSize(Double gpuPartitionSize) {
            this.gpuPartitionSize = gpuPartitionSize;
            return this;
        }

        public final Collection<String> getWorkloads() {
            if (workloads instanceof SdkAutoConstructList) {
                return null;
            }
            return workloads;
        }

        public final void setWorkloads(Collection<String> workloads) {
            this.workloads = WorkloadsListCopier.copy(workloads);
        }

        @Override
        public final Builder workloads(Collection<String> workloads) {
            this.workloads = WorkloadsListCopier.copy(workloads);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder workloads(String... workloads) {
            workloads(Arrays.asList(workloads));
            return this;
        }

        public final GpuDeviceMemoryInfo.Builder getMemoryInfo() {
            return memoryInfo != null ? memoryInfo.toBuilder() : null;
        }

        public final void setMemoryInfo(GpuDeviceMemoryInfo.BuilderImpl memoryInfo) {
            this.memoryInfo = memoryInfo != null ? memoryInfo.build() : null;
        }

        @Override
        public final Builder memoryInfo(GpuDeviceMemoryInfo memoryInfo) {
            this.memoryInfo = memoryInfo;
            return this;
        }

        @Override
        public GpuDeviceInfo build() {
            return new GpuDeviceInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
