/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the Amazon EBS features supported by the instance type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EbsInfo implements SdkPojo, Serializable, ToCopyableBuilder<EbsInfo.Builder, EbsInfo> {
    private static final SdkField<String> EBS_OPTIMIZED_SUPPORT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EbsOptimizedSupport")
            .getter(getter(EbsInfo::ebsOptimizedSupportAsString))
            .setter(setter(Builder::ebsOptimizedSupport))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsOptimizedSupport")
                    .unmarshallLocationName("ebsOptimizedSupport").build()).build();

    private static final SdkField<String> ENCRYPTION_SUPPORT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EncryptionSupport")
            .getter(getter(EbsInfo::encryptionSupportAsString))
            .setter(setter(Builder::encryptionSupport))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionSupport")
                    .unmarshallLocationName("encryptionSupport").build()).build();

    private static final SdkField<EbsOptimizedInfo> EBS_OPTIMIZED_INFO_FIELD = SdkField
            .<EbsOptimizedInfo> builder(MarshallingType.SDK_POJO)
            .memberName("EbsOptimizedInfo")
            .getter(getter(EbsInfo::ebsOptimizedInfo))
            .setter(setter(Builder::ebsOptimizedInfo))
            .constructor(EbsOptimizedInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsOptimizedInfo")
                    .unmarshallLocationName("ebsOptimizedInfo").build()).build();

    private static final SdkField<String> NVME_SUPPORT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NvmeSupport")
            .getter(getter(EbsInfo::nvmeSupportAsString))
            .setter(setter(Builder::nvmeSupport))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NvmeSupport")
                    .unmarshallLocationName("nvmeSupport").build()).build();

    private static final SdkField<Integer> MAXIMUM_EBS_ATTACHMENTS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaximumEbsAttachments")
            .getter(getter(EbsInfo::maximumEbsAttachments))
            .setter(setter(Builder::maximumEbsAttachments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumEbsAttachments")
                    .unmarshallLocationName("maximumEbsAttachments").build()).build();

    private static final SdkField<String> ATTACHMENT_LIMIT_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AttachmentLimitType")
            .getter(getter(EbsInfo::attachmentLimitTypeAsString))
            .setter(setter(Builder::attachmentLimitType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentLimitType")
                    .unmarshallLocationName("attachmentLimitType").build()).build();

    private static final SdkField<Integer> MAXIMUM_EBS_CARDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaximumEbsCards")
            .getter(getter(EbsInfo::maximumEbsCards))
            .setter(setter(Builder::maximumEbsCards))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumEbsCards")
                    .unmarshallLocationName("maximumEbsCards").build()).build();

    private static final SdkField<List<EbsCardInfo>> EBS_CARDS_FIELD = SdkField
            .<List<EbsCardInfo>> builder(MarshallingType.LIST)
            .memberName("EbsCards")
            .getter(getter(EbsInfo::ebsCards))
            .setter(setter(Builder::ebsCards))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsCardSet")
                    .unmarshallLocationName("ebsCardSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<EbsCardInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(EbsCardInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EBS_OPTIMIZED_SUPPORT_FIELD,
            ENCRYPTION_SUPPORT_FIELD, EBS_OPTIMIZED_INFO_FIELD, NVME_SUPPORT_FIELD, MAXIMUM_EBS_ATTACHMENTS_FIELD,
            ATTACHMENT_LIMIT_TYPE_FIELD, MAXIMUM_EBS_CARDS_FIELD, EBS_CARDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ebsOptimizedSupport;

    private final String encryptionSupport;

    private final EbsOptimizedInfo ebsOptimizedInfo;

    private final String nvmeSupport;

    private final Integer maximumEbsAttachments;

    private final String attachmentLimitType;

    private final Integer maximumEbsCards;

    private final List<EbsCardInfo> ebsCards;

    private EbsInfo(BuilderImpl builder) {
        this.ebsOptimizedSupport = builder.ebsOptimizedSupport;
        this.encryptionSupport = builder.encryptionSupport;
        this.ebsOptimizedInfo = builder.ebsOptimizedInfo;
        this.nvmeSupport = builder.nvmeSupport;
        this.maximumEbsAttachments = builder.maximumEbsAttachments;
        this.attachmentLimitType = builder.attachmentLimitType;
        this.maximumEbsCards = builder.maximumEbsCards;
        this.ebsCards = builder.ebsCards;
    }

    /**
     * <p>
     * Indicates whether the instance type is Amazon EBS-optimized. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html">Amazon EBS-optimized instances</a>
     * in <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #ebsOptimizedSupport} will return {@link EbsOptimizedSupport#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #ebsOptimizedSupportAsString}.
     * </p>
     * 
     * @return Indicates whether the instance type is Amazon EBS-optimized. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html">Amazon EBS-optimized
     *         instances</a> in <i>Amazon EC2 User Guide</i>.
     * @see EbsOptimizedSupport
     */
    public final EbsOptimizedSupport ebsOptimizedSupport() {
        return EbsOptimizedSupport.fromValue(ebsOptimizedSupport);
    }

    /**
     * <p>
     * Indicates whether the instance type is Amazon EBS-optimized. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html">Amazon EBS-optimized instances</a>
     * in <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #ebsOptimizedSupport} will return {@link EbsOptimizedSupport#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #ebsOptimizedSupportAsString}.
     * </p>
     * 
     * @return Indicates whether the instance type is Amazon EBS-optimized. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html">Amazon EBS-optimized
     *         instances</a> in <i>Amazon EC2 User Guide</i>.
     * @see EbsOptimizedSupport
     */
    public final String ebsOptimizedSupportAsString() {
        return ebsOptimizedSupport;
    }

    /**
     * <p>
     * Indicates whether Amazon EBS encryption is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionSupport}
     * will return {@link EbsEncryptionSupport#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #encryptionSupportAsString}.
     * </p>
     * 
     * @return Indicates whether Amazon EBS encryption is supported.
     * @see EbsEncryptionSupport
     */
    public final EbsEncryptionSupport encryptionSupport() {
        return EbsEncryptionSupport.fromValue(encryptionSupport);
    }

    /**
     * <p>
     * Indicates whether Amazon EBS encryption is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionSupport}
     * will return {@link EbsEncryptionSupport#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #encryptionSupportAsString}.
     * </p>
     * 
     * @return Indicates whether Amazon EBS encryption is supported.
     * @see EbsEncryptionSupport
     */
    public final String encryptionSupportAsString() {
        return encryptionSupport;
    }

    /**
     * <p>
     * Describes the optimized EBS performance for the instance type.
     * </p>
     * 
     * @return Describes the optimized EBS performance for the instance type.
     */
    public final EbsOptimizedInfo ebsOptimizedInfo() {
        return ebsOptimizedInfo;
    }

    /**
     * <p>
     * Indicates whether non-volatile memory express (NVMe) is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #nvmeSupport} will
     * return {@link EbsNvmeSupport#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nvmeSupportAsString}.
     * </p>
     * 
     * @return Indicates whether non-volatile memory express (NVMe) is supported.
     * @see EbsNvmeSupport
     */
    public final EbsNvmeSupport nvmeSupport() {
        return EbsNvmeSupport.fromValue(nvmeSupport);
    }

    /**
     * <p>
     * Indicates whether non-volatile memory express (NVMe) is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #nvmeSupport} will
     * return {@link EbsNvmeSupport#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nvmeSupportAsString}.
     * </p>
     * 
     * @return Indicates whether non-volatile memory express (NVMe) is supported.
     * @see EbsNvmeSupport
     */
    public final String nvmeSupportAsString() {
        return nvmeSupport;
    }

    /**
     * <p>
     * Indicates the maximum number of Amazon EBS volumes that can be attached to the instance type. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS
     * volume limits for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * 
     * @return Indicates the maximum number of Amazon EBS volumes that can be attached to the instance type. For more
     *         information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon
     *         EBS volume limits for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final Integer maximumEbsAttachments() {
        return maximumEbsAttachments;
    }

    /**
     * <p>
     * Indicates whether the instance type features a shared or dedicated Amazon EBS volume attachment limit. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS
     * volume limits for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #attachmentLimitType} will return {@link AttachmentLimitType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #attachmentLimitTypeAsString}.
     * </p>
     * 
     * @return Indicates whether the instance type features a shared or dedicated Amazon EBS volume attachment limit.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS volume limits
     *         for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
     * @see AttachmentLimitType
     */
    public final AttachmentLimitType attachmentLimitType() {
        return AttachmentLimitType.fromValue(attachmentLimitType);
    }

    /**
     * <p>
     * Indicates whether the instance type features a shared or dedicated Amazon EBS volume attachment limit. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS
     * volume limits for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #attachmentLimitType} will return {@link AttachmentLimitType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #attachmentLimitTypeAsString}.
     * </p>
     * 
     * @return Indicates whether the instance type features a shared or dedicated Amazon EBS volume attachment limit.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS volume limits
     *         for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
     * @see AttachmentLimitType
     */
    public final String attachmentLimitTypeAsString() {
        return attachmentLimitType;
    }

    /**
     * <p>
     * Indicates the number of EBS cards supported by the instance type.
     * </p>
     * 
     * @return Indicates the number of EBS cards supported by the instance type.
     */
    public final Integer maximumEbsCards() {
        return maximumEbsCards;
    }

    /**
     * For responses, this returns true if the service returned a value for the EbsCards property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEbsCards() {
        return ebsCards != null && !(ebsCards instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the EBS cards available for the instance type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEbsCards} method.
     * </p>
     * 
     * @return Describes the EBS cards available for the instance type.
     */
    public final List<EbsCardInfo> ebsCards() {
        return ebsCards;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ebsOptimizedSupportAsString());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionSupportAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ebsOptimizedInfo());
        hashCode = 31 * hashCode + Objects.hashCode(nvmeSupportAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maximumEbsAttachments());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentLimitTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maximumEbsCards());
        hashCode = 31 * hashCode + Objects.hashCode(hasEbsCards() ? ebsCards() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EbsInfo)) {
            return false;
        }
        EbsInfo other = (EbsInfo) obj;
        return Objects.equals(ebsOptimizedSupportAsString(), other.ebsOptimizedSupportAsString())
                && Objects.equals(encryptionSupportAsString(), other.encryptionSupportAsString())
                && Objects.equals(ebsOptimizedInfo(), other.ebsOptimizedInfo())
                && Objects.equals(nvmeSupportAsString(), other.nvmeSupportAsString())
                && Objects.equals(maximumEbsAttachments(), other.maximumEbsAttachments())
                && Objects.equals(attachmentLimitTypeAsString(), other.attachmentLimitTypeAsString())
                && Objects.equals(maximumEbsCards(), other.maximumEbsCards()) && hasEbsCards() == other.hasEbsCards()
                && Objects.equals(ebsCards(), other.ebsCards());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EbsInfo").add("EbsOptimizedSupport", ebsOptimizedSupportAsString())
                .add("EncryptionSupport", encryptionSupportAsString()).add("EbsOptimizedInfo", ebsOptimizedInfo())
                .add("NvmeSupport", nvmeSupportAsString()).add("MaximumEbsAttachments", maximumEbsAttachments())
                .add("AttachmentLimitType", attachmentLimitTypeAsString()).add("MaximumEbsCards", maximumEbsCards())
                .add("EbsCards", hasEbsCards() ? ebsCards() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EbsOptimizedSupport":
            return Optional.ofNullable(clazz.cast(ebsOptimizedSupportAsString()));
        case "EncryptionSupport":
            return Optional.ofNullable(clazz.cast(encryptionSupportAsString()));
        case "EbsOptimizedInfo":
            return Optional.ofNullable(clazz.cast(ebsOptimizedInfo()));
        case "NvmeSupport":
            return Optional.ofNullable(clazz.cast(nvmeSupportAsString()));
        case "MaximumEbsAttachments":
            return Optional.ofNullable(clazz.cast(maximumEbsAttachments()));
        case "AttachmentLimitType":
            return Optional.ofNullable(clazz.cast(attachmentLimitTypeAsString()));
        case "MaximumEbsCards":
            return Optional.ofNullable(clazz.cast(maximumEbsCards()));
        case "EbsCards":
            return Optional.ofNullable(clazz.cast(ebsCards()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EbsOptimizedSupport", EBS_OPTIMIZED_SUPPORT_FIELD);
        map.put("EncryptionSupport", ENCRYPTION_SUPPORT_FIELD);
        map.put("EbsOptimizedInfo", EBS_OPTIMIZED_INFO_FIELD);
        map.put("NvmeSupport", NVME_SUPPORT_FIELD);
        map.put("MaximumEbsAttachments", MAXIMUM_EBS_ATTACHMENTS_FIELD);
        map.put("AttachmentLimitType", ATTACHMENT_LIMIT_TYPE_FIELD);
        map.put("MaximumEbsCards", MAXIMUM_EBS_CARDS_FIELD);
        map.put("EbsCardSet", EBS_CARDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EbsInfo, T> g) {
        return obj -> g.apply((EbsInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EbsInfo> {
        /**
         * <p>
         * Indicates whether the instance type is Amazon EBS-optimized. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html">Amazon EBS-optimized
         * instances</a> in <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param ebsOptimizedSupport
         *        Indicates whether the instance type is Amazon EBS-optimized. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html">Amazon EBS-optimized
         *        instances</a> in <i>Amazon EC2 User Guide</i>.
         * @see EbsOptimizedSupport
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EbsOptimizedSupport
         */
        Builder ebsOptimizedSupport(String ebsOptimizedSupport);

        /**
         * <p>
         * Indicates whether the instance type is Amazon EBS-optimized. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html">Amazon EBS-optimized
         * instances</a> in <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param ebsOptimizedSupport
         *        Indicates whether the instance type is Amazon EBS-optimized. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSOptimized.html">Amazon EBS-optimized
         *        instances</a> in <i>Amazon EC2 User Guide</i>.
         * @see EbsOptimizedSupport
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EbsOptimizedSupport
         */
        Builder ebsOptimizedSupport(EbsOptimizedSupport ebsOptimizedSupport);

        /**
         * <p>
         * Indicates whether Amazon EBS encryption is supported.
         * </p>
         * 
         * @param encryptionSupport
         *        Indicates whether Amazon EBS encryption is supported.
         * @see EbsEncryptionSupport
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EbsEncryptionSupport
         */
        Builder encryptionSupport(String encryptionSupport);

        /**
         * <p>
         * Indicates whether Amazon EBS encryption is supported.
         * </p>
         * 
         * @param encryptionSupport
         *        Indicates whether Amazon EBS encryption is supported.
         * @see EbsEncryptionSupport
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EbsEncryptionSupport
         */
        Builder encryptionSupport(EbsEncryptionSupport encryptionSupport);

        /**
         * <p>
         * Describes the optimized EBS performance for the instance type.
         * </p>
         * 
         * @param ebsOptimizedInfo
         *        Describes the optimized EBS performance for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsOptimizedInfo(EbsOptimizedInfo ebsOptimizedInfo);

        /**
         * <p>
         * Describes the optimized EBS performance for the instance type.
         * </p>
         * This is a convenience method that creates an instance of the {@link EbsOptimizedInfo.Builder} avoiding the
         * need to create one manually via {@link EbsOptimizedInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EbsOptimizedInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #ebsOptimizedInfo(EbsOptimizedInfo)}.
         * 
         * @param ebsOptimizedInfo
         *        a consumer that will call methods on {@link EbsOptimizedInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsOptimizedInfo(EbsOptimizedInfo)
         */
        default Builder ebsOptimizedInfo(Consumer<EbsOptimizedInfo.Builder> ebsOptimizedInfo) {
            return ebsOptimizedInfo(EbsOptimizedInfo.builder().applyMutation(ebsOptimizedInfo).build());
        }

        /**
         * <p>
         * Indicates whether non-volatile memory express (NVMe) is supported.
         * </p>
         * 
         * @param nvmeSupport
         *        Indicates whether non-volatile memory express (NVMe) is supported.
         * @see EbsNvmeSupport
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EbsNvmeSupport
         */
        Builder nvmeSupport(String nvmeSupport);

        /**
         * <p>
         * Indicates whether non-volatile memory express (NVMe) is supported.
         * </p>
         * 
         * @param nvmeSupport
         *        Indicates whether non-volatile memory express (NVMe) is supported.
         * @see EbsNvmeSupport
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EbsNvmeSupport
         */
        Builder nvmeSupport(EbsNvmeSupport nvmeSupport);

        /**
         * <p>
         * Indicates the maximum number of Amazon EBS volumes that can be attached to the instance type. For more
         * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS
         * volume limits for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param maximumEbsAttachments
         *        Indicates the maximum number of Amazon EBS volumes that can be attached to the instance type. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS volume limits
         *        for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumEbsAttachments(Integer maximumEbsAttachments);

        /**
         * <p>
         * Indicates whether the instance type features a shared or dedicated Amazon EBS volume attachment limit. For
         * more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon
         * EBS volume limits for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param attachmentLimitType
         *        Indicates whether the instance type features a shared or dedicated Amazon EBS volume attachment limit.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS volume limits
         *        for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * @see AttachmentLimitType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AttachmentLimitType
         */
        Builder attachmentLimitType(String attachmentLimitType);

        /**
         * <p>
         * Indicates whether the instance type features a shared or dedicated Amazon EBS volume attachment limit. For
         * more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon
         * EBS volume limits for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param attachmentLimitType
         *        Indicates whether the instance type features a shared or dedicated Amazon EBS volume attachment limit.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/volume_limits.html">Amazon EBS volume limits
         *        for Amazon EC2 instances</a> in the <i>Amazon EC2 User Guide</i>.
         * @see AttachmentLimitType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AttachmentLimitType
         */
        Builder attachmentLimitType(AttachmentLimitType attachmentLimitType);

        /**
         * <p>
         * Indicates the number of EBS cards supported by the instance type.
         * </p>
         * 
         * @param maximumEbsCards
         *        Indicates the number of EBS cards supported by the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumEbsCards(Integer maximumEbsCards);

        /**
         * <p>
         * Describes the EBS cards available for the instance type.
         * </p>
         * 
         * @param ebsCards
         *        Describes the EBS cards available for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsCards(Collection<EbsCardInfo> ebsCards);

        /**
         * <p>
         * Describes the EBS cards available for the instance type.
         * </p>
         * 
         * @param ebsCards
         *        Describes the EBS cards available for the instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsCards(EbsCardInfo... ebsCards);

        /**
         * <p>
         * Describes the EBS cards available for the instance type.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.EbsCardInfo.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.EbsCardInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.EbsCardInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #ebsCards(List<EbsCardInfo>)}.
         * 
         * @param ebsCards
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.EbsCardInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsCards(java.util.Collection<EbsCardInfo>)
         */
        Builder ebsCards(Consumer<EbsCardInfo.Builder>... ebsCards);
    }

    static final class BuilderImpl implements Builder {
        private String ebsOptimizedSupport;

        private String encryptionSupport;

        private EbsOptimizedInfo ebsOptimizedInfo;

        private String nvmeSupport;

        private Integer maximumEbsAttachments;

        private String attachmentLimitType;

        private Integer maximumEbsCards;

        private List<EbsCardInfo> ebsCards = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EbsInfo model) {
            ebsOptimizedSupport(model.ebsOptimizedSupport);
            encryptionSupport(model.encryptionSupport);
            ebsOptimizedInfo(model.ebsOptimizedInfo);
            nvmeSupport(model.nvmeSupport);
            maximumEbsAttachments(model.maximumEbsAttachments);
            attachmentLimitType(model.attachmentLimitType);
            maximumEbsCards(model.maximumEbsCards);
            ebsCards(model.ebsCards);
        }

        public final String getEbsOptimizedSupport() {
            return ebsOptimizedSupport;
        }

        public final void setEbsOptimizedSupport(String ebsOptimizedSupport) {
            this.ebsOptimizedSupport = ebsOptimizedSupport;
        }

        @Override
        public final Builder ebsOptimizedSupport(String ebsOptimizedSupport) {
            this.ebsOptimizedSupport = ebsOptimizedSupport;
            return this;
        }

        @Override
        public final Builder ebsOptimizedSupport(EbsOptimizedSupport ebsOptimizedSupport) {
            this.ebsOptimizedSupport(ebsOptimizedSupport == null ? null : ebsOptimizedSupport.toString());
            return this;
        }

        public final String getEncryptionSupport() {
            return encryptionSupport;
        }

        public final void setEncryptionSupport(String encryptionSupport) {
            this.encryptionSupport = encryptionSupport;
        }

        @Override
        public final Builder encryptionSupport(String encryptionSupport) {
            this.encryptionSupport = encryptionSupport;
            return this;
        }

        @Override
        public final Builder encryptionSupport(EbsEncryptionSupport encryptionSupport) {
            this.encryptionSupport(encryptionSupport == null ? null : encryptionSupport.toString());
            return this;
        }

        public final EbsOptimizedInfo.Builder getEbsOptimizedInfo() {
            return ebsOptimizedInfo != null ? ebsOptimizedInfo.toBuilder() : null;
        }

        public final void setEbsOptimizedInfo(EbsOptimizedInfo.BuilderImpl ebsOptimizedInfo) {
            this.ebsOptimizedInfo = ebsOptimizedInfo != null ? ebsOptimizedInfo.build() : null;
        }

        @Override
        public final Builder ebsOptimizedInfo(EbsOptimizedInfo ebsOptimizedInfo) {
            this.ebsOptimizedInfo = ebsOptimizedInfo;
            return this;
        }

        public final String getNvmeSupport() {
            return nvmeSupport;
        }

        public final void setNvmeSupport(String nvmeSupport) {
            this.nvmeSupport = nvmeSupport;
        }

        @Override
        public final Builder nvmeSupport(String nvmeSupport) {
            this.nvmeSupport = nvmeSupport;
            return this;
        }

        @Override
        public final Builder nvmeSupport(EbsNvmeSupport nvmeSupport) {
            this.nvmeSupport(nvmeSupport == null ? null : nvmeSupport.toString());
            return this;
        }

        public final Integer getMaximumEbsAttachments() {
            return maximumEbsAttachments;
        }

        public final void setMaximumEbsAttachments(Integer maximumEbsAttachments) {
            this.maximumEbsAttachments = maximumEbsAttachments;
        }

        @Override
        public final Builder maximumEbsAttachments(Integer maximumEbsAttachments) {
            this.maximumEbsAttachments = maximumEbsAttachments;
            return this;
        }

        public final String getAttachmentLimitType() {
            return attachmentLimitType;
        }

        public final void setAttachmentLimitType(String attachmentLimitType) {
            this.attachmentLimitType = attachmentLimitType;
        }

        @Override
        public final Builder attachmentLimitType(String attachmentLimitType) {
            this.attachmentLimitType = attachmentLimitType;
            return this;
        }

        @Override
        public final Builder attachmentLimitType(AttachmentLimitType attachmentLimitType) {
            this.attachmentLimitType(attachmentLimitType == null ? null : attachmentLimitType.toString());
            return this;
        }

        public final Integer getMaximumEbsCards() {
            return maximumEbsCards;
        }

        public final void setMaximumEbsCards(Integer maximumEbsCards) {
            this.maximumEbsCards = maximumEbsCards;
        }

        @Override
        public final Builder maximumEbsCards(Integer maximumEbsCards) {
            this.maximumEbsCards = maximumEbsCards;
            return this;
        }

        public final List<EbsCardInfo.Builder> getEbsCards() {
            List<EbsCardInfo.Builder> result = EbsCardInfoListCopier.copyToBuilder(this.ebsCards);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEbsCards(Collection<EbsCardInfo.BuilderImpl> ebsCards) {
            this.ebsCards = EbsCardInfoListCopier.copyFromBuilder(ebsCards);
        }

        @Override
        public final Builder ebsCards(Collection<EbsCardInfo> ebsCards) {
            this.ebsCards = EbsCardInfoListCopier.copy(ebsCards);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsCards(EbsCardInfo... ebsCards) {
            ebsCards(Arrays.asList(ebsCards));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ebsCards(Consumer<EbsCardInfo.Builder>... ebsCards) {
            ebsCards(Stream.of(ebsCards).map(c -> EbsCardInfo.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public EbsInfo build() {
            return new EbsInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
