/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a version of an IPAM prefix list resolver.
 * </p>
 * <p>
 * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number increments
 * every time the CIDR list changes due to infrastructure changes.
 * </p>
 * <p>
 * <b>Version example:</b>
 * </p>
 * <p>
 * <b>Initial State (Version 1)</b>
 * </p>
 * <p>
 * Production environment:
 * </p>
 * <ul>
 * <li>
 * <p>
 * vpc-prod-web (10.1.0.0/16) - tagged env=prod
 * </p>
 * </li>
 * <li>
 * <p>
 * vpc-prod-db (10.2.0.0/16) - tagged env=prod
 * </p>
 * </li>
 * </ul>
 * <p>
 * Resolver rule: Include all VPCs tagged env=prod
 * </p>
 * <p>
 * <b>Version 1 CIDRs:</b> 10.1.0.0/16, 10.2.0.0/16
 * </p>
 * <p>
 * <b>Infrastructure Change (Version 2)</b>
 * </p>
 * <p>
 * New VPC added:
 * </p>
 * <ul>
 * <li>
 * <p>
 * vpc-prod-api (10.3.0.0/16) - tagged env=prod
 * </p>
 * </li>
 * </ul>
 * <p>
 * IPAM automatically detects the change and creates a new version.
 * </p>
 * <p>
 * <b>Version 2 CIDRs:</b> 10.1.0.0/16, 10.2.0.0/16, 10.3.0.0/16
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamPrefixListResolverVersion implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamPrefixListResolverVersion.Builder, IpamPrefixListResolverVersion> {
    private static final SdkField<Long> VERSION_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("Version")
            .getter(getter(IpamPrefixListResolverVersion::version))
            .setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version")
                    .unmarshallLocationName("version").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long version;

    private IpamPrefixListResolverVersion(BuilderImpl builder) {
        this.version = builder.version;
    }

    /**
     * <p>
     * The version number of the IPAM prefix list resolver.
     * </p>
     * <p>
     * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number increments
     * every time the CIDR list changes due to infrastructure changes.
     * </p>
     * 
     * @return The version number of the IPAM prefix list resolver.</p>
     *         <p>
     *         Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
     *         increments every time the CIDR list changes due to infrastructure changes.
     */
    public final Long version() {
        return version;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamPrefixListResolverVersion)) {
            return false;
        }
        IpamPrefixListResolverVersion other = (IpamPrefixListResolverVersion) obj;
        return Objects.equals(version(), other.version());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamPrefixListResolverVersion").add("Version", version()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Version", VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IpamPrefixListResolverVersion, T> g) {
        return obj -> g.apply((IpamPrefixListResolverVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamPrefixListResolverVersion> {
        /**
         * <p>
         * The version number of the IPAM prefix list resolver.
         * </p>
         * <p>
         * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
         * increments every time the CIDR list changes due to infrastructure changes.
         * </p>
         * 
         * @param version
         *        The version number of the IPAM prefix list resolver.</p>
         *        <p>
         *        Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
         *        increments every time the CIDR list changes due to infrastructure changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Long version);
    }

    static final class BuilderImpl implements Builder {
        private Long version;

        private BuilderImpl() {
        }

        private BuilderImpl(IpamPrefixListResolverVersion model) {
            version(model.version);
        }

        public final Long getVersion() {
            return version;
        }

        public final void setVersion(Long version) {
            this.version = version;
        }

        @Override
        public final Builder version(Long version) {
            this.version = version;
            return this;
        }

        @Override
        public IpamPrefixListResolverVersion build() {
            return new IpamPrefixListResolverVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
