/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * For regional NAT gateways only: The configuration specifying which Elastic IP address (EIP) to use for handling
 * outbound NAT traffic from a specific Availability Zone.
 * </p>
 * <p>
 * A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in your VPC,
 * providing redundancy, scalability and availability across all the AZs in a Region.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/vpc/latest/userguide/nat-gateways-regional.html">Regional NAT gateways for
 * automatic multi-AZ expansion</a> in the <i>Amazon VPC User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AvailabilityZoneAddress implements SdkPojo, Serializable,
        ToCopyableBuilder<AvailabilityZoneAddress.Builder, AvailabilityZoneAddress> {
    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(AvailabilityZoneAddress::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("AvailabilityZone").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneId")
            .getter(getter(AvailabilityZoneAddress::availabilityZoneId))
            .setter(setter(Builder::availabilityZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneId")
                    .unmarshallLocationName("AvailabilityZoneId").build()).build();

    private static final SdkField<List<String>> ALLOCATION_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AllocationIds")
            .getter(getter(AvailabilityZoneAddress::allocationIds))
            .setter(setter(Builder::allocationIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationId")
                    .unmarshallLocationName("AllocationId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AllocationId")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AllocationId").unmarshallLocationName("AllocationId").build())
                                            .build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AVAILABILITY_ZONE_FIELD,
            AVAILABILITY_ZONE_ID_FIELD, ALLOCATION_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String availabilityZone;

    private final String availabilityZoneId;

    private final List<String> allocationIds;

    private AvailabilityZoneAddress(BuilderImpl builder) {
        this.availabilityZone = builder.availabilityZone;
        this.availabilityZoneId = builder.availabilityZoneId;
        this.allocationIds = builder.allocationIds;
    }

    /**
     * <p>
     * For regional NAT gateways only: The Availability Zone where this specific NAT gateway configuration will be
     * active. Each AZ in a regional NAT gateway has its own configuration to handle outbound NAT traffic from that AZ.
     * </p>
     * <p>
     * A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in your VPC,
     * providing redundancy, scalability and availability across all the AZs in a Region.
     * </p>
     * 
     * @return For regional NAT gateways only: The Availability Zone where this specific NAT gateway configuration will
     *         be active. Each AZ in a regional NAT gateway has its own configuration to handle outbound NAT traffic
     *         from that AZ. </p>
     *         <p>
     *         A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in
     *         your VPC, providing redundancy, scalability and availability across all the AZs in a Region.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * For regional NAT gateways only: The ID of the Availability Zone where this specific NAT gateway configuration
     * will be active. Each AZ in a regional NAT gateway has its own configuration to handle outbound NAT traffic from
     * that AZ. Use this instead of AvailabilityZone for consistent identification of AZs across Amazon Web Services
     * Regions.
     * </p>
     * <p>
     * A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in your VPC,
     * providing redundancy, scalability and availability across all the AZs in a Region.
     * </p>
     * 
     * @return For regional NAT gateways only: The ID of the Availability Zone where this specific NAT gateway
     *         configuration will be active. Each AZ in a regional NAT gateway has its own configuration to handle
     *         outbound NAT traffic from that AZ. Use this instead of AvailabilityZone for consistent identification of
     *         AZs across Amazon Web Services Regions. </p>
     *         <p>
     *         A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in
     *         your VPC, providing redundancy, scalability and availability across all the AZs in a Region.
     */
    public final String availabilityZoneId() {
        return availabilityZoneId;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllocationIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllocationIds() {
        return allocationIds != null && !(allocationIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The allocation IDs of the Elastic IP addresses (EIPs) to be used for handling outbound NAT traffic in this
     * specific Availability Zone.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllocationIds} method.
     * </p>
     * 
     * @return The allocation IDs of the Elastic IP addresses (EIPs) to be used for handling outbound NAT traffic in
     *         this specific Availability Zone.
     */
    public final List<String> allocationIds() {
        return allocationIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAllocationIds() ? allocationIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AvailabilityZoneAddress)) {
            return false;
        }
        AvailabilityZoneAddress other = (AvailabilityZoneAddress) obj;
        return Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(availabilityZoneId(), other.availabilityZoneId())
                && hasAllocationIds() == other.hasAllocationIds() && Objects.equals(allocationIds(), other.allocationIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AvailabilityZoneAddress").add("AvailabilityZone", availabilityZone())
                .add("AvailabilityZoneId", availabilityZoneId())
                .add("AllocationIds", hasAllocationIds() ? allocationIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "AvailabilityZoneId":
            return Optional.ofNullable(clazz.cast(availabilityZoneId()));
        case "AllocationIds":
            return Optional.ofNullable(clazz.cast(allocationIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AvailabilityZone", AVAILABILITY_ZONE_FIELD);
        map.put("AvailabilityZoneId", AVAILABILITY_ZONE_ID_FIELD);
        map.put("AllocationId", ALLOCATION_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AvailabilityZoneAddress, T> g) {
        return obj -> g.apply((AvailabilityZoneAddress) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AvailabilityZoneAddress> {
        /**
         * <p>
         * For regional NAT gateways only: The Availability Zone where this specific NAT gateway configuration will be
         * active. Each AZ in a regional NAT gateway has its own configuration to handle outbound NAT traffic from that
         * AZ.
         * </p>
         * <p>
         * A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in your
         * VPC, providing redundancy, scalability and availability across all the AZs in a Region.
         * </p>
         * 
         * @param availabilityZone
         *        For regional NAT gateways only: The Availability Zone where this specific NAT gateway configuration
         *        will be active. Each AZ in a regional NAT gateway has its own configuration to handle outbound NAT
         *        traffic from that AZ. </p>
         *        <p>
         *        A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in
         *        your VPC, providing redundancy, scalability and availability across all the AZs in a Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * For regional NAT gateways only: The ID of the Availability Zone where this specific NAT gateway configuration
         * will be active. Each AZ in a regional NAT gateway has its own configuration to handle outbound NAT traffic
         * from that AZ. Use this instead of AvailabilityZone for consistent identification of AZs across Amazon Web
         * Services Regions.
         * </p>
         * <p>
         * A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in your
         * VPC, providing redundancy, scalability and availability across all the AZs in a Region.
         * </p>
         * 
         * @param availabilityZoneId
         *        For regional NAT gateways only: The ID of the Availability Zone where this specific NAT gateway
         *        configuration will be active. Each AZ in a regional NAT gateway has its own configuration to handle
         *        outbound NAT traffic from that AZ. Use this instead of AvailabilityZone for consistent identification
         *        of AZs across Amazon Web Services Regions. </p>
         *        <p>
         *        A regional NAT gateway is a single NAT Gateway that works across multiple availability zones (AZs) in
         *        your VPC, providing redundancy, scalability and availability across all the AZs in a Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneId(String availabilityZoneId);

        /**
         * <p>
         * The allocation IDs of the Elastic IP addresses (EIPs) to be used for handling outbound NAT traffic in this
         * specific Availability Zone.
         * </p>
         * 
         * @param allocationIds
         *        The allocation IDs of the Elastic IP addresses (EIPs) to be used for handling outbound NAT traffic in
         *        this specific Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationIds(Collection<String> allocationIds);

        /**
         * <p>
         * The allocation IDs of the Elastic IP addresses (EIPs) to be used for handling outbound NAT traffic in this
         * specific Availability Zone.
         * </p>
         * 
         * @param allocationIds
         *        The allocation IDs of the Elastic IP addresses (EIPs) to be used for handling outbound NAT traffic in
         *        this specific Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationIds(String... allocationIds);
    }

    static final class BuilderImpl implements Builder {
        private String availabilityZone;

        private String availabilityZoneId;

        private List<String> allocationIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AvailabilityZoneAddress model) {
            availabilityZone(model.availabilityZone);
            availabilityZoneId(model.availabilityZoneId);
            allocationIds(model.allocationIds);
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final String getAvailabilityZoneId() {
            return availabilityZoneId;
        }

        public final void setAvailabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
        }

        @Override
        public final Builder availabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
            return this;
        }

        public final Collection<String> getAllocationIds() {
            if (allocationIds instanceof SdkAutoConstructList) {
                return null;
            }
            return allocationIds;
        }

        public final void setAllocationIds(Collection<String> allocationIds) {
            this.allocationIds = AllocationIdListCopier.copy(allocationIds);
        }

        @Override
        public final Builder allocationIds(Collection<String> allocationIds) {
            this.allocationIds = AllocationIdListCopier.copy(allocationIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allocationIds(String... allocationIds) {
            allocationIds(Arrays.asList(allocationIds));
            return this;
        }

        @Override
        public AvailabilityZoneAddress build() {
            return new AvailabilityZoneAddress(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
