/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains a single data point from a capacity metrics query, including the dimension values, timestamp, and metric
 * values for that specific combination.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricDataResult implements SdkPojo, Serializable,
        ToCopyableBuilder<MetricDataResult.Builder, MetricDataResult> {
    private static final SdkField<CapacityManagerDimension> DIMENSION_FIELD = SdkField
            .<CapacityManagerDimension> builder(MarshallingType.SDK_POJO)
            .memberName("Dimension")
            .getter(getter(MetricDataResult::dimension))
            .setter(setter(Builder::dimension))
            .constructor(CapacityManagerDimension::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimension")
                    .unmarshallLocationName("dimension").build()).build();

    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp")
            .getter(getter(MetricDataResult::timestamp))
            .setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp")
                    .unmarshallLocationName("timestamp").build()).build();

    private static final SdkField<List<MetricValue>> METRIC_VALUES_FIELD = SdkField
            .<List<MetricValue>> builder(MarshallingType.LIST)
            .memberName("MetricValues")
            .getter(getter(MetricDataResult::metricValues))
            .setter(setter(Builder::metricValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricValueSet")
                    .unmarshallLocationName("metricValueSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<MetricValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIMENSION_FIELD,
            TIMESTAMP_FIELD, METRIC_VALUES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final CapacityManagerDimension dimension;

    private final Instant timestamp;

    private final List<MetricValue> metricValues;

    private MetricDataResult(BuilderImpl builder) {
        this.dimension = builder.dimension;
        this.timestamp = builder.timestamp;
        this.metricValues = builder.metricValues;
    }

    /**
     * <p>
     * The dimension values that identify this specific data point, such as account ID, region, and instance family.
     * </p>
     * 
     * @return The dimension values that identify this specific data point, such as account ID, region, and instance
     *         family.
     */
    public final CapacityManagerDimension dimension() {
        return dimension;
    }

    /**
     * <p>
     * The timestamp for this data point, indicating when the capacity usage occurred.
     * </p>
     * 
     * @return The timestamp for this data point, indicating when the capacity usage occurred.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricValues() {
        return metricValues != null && !(metricValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metric values and statistics for this data point, containing the actual capacity usage numbers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricValues} method.
     * </p>
     * 
     * @return The metric values and statistics for this data point, containing the actual capacity usage numbers.
     */
    public final List<MetricValue> metricValues() {
        return metricValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dimension());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricValues() ? metricValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricDataResult)) {
            return false;
        }
        MetricDataResult other = (MetricDataResult) obj;
        return Objects.equals(dimension(), other.dimension()) && Objects.equals(timestamp(), other.timestamp())
                && hasMetricValues() == other.hasMetricValues() && Objects.equals(metricValues(), other.metricValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricDataResult").add("Dimension", dimension()).add("Timestamp", timestamp())
                .add("MetricValues", hasMetricValues() ? metricValues() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Dimension":
            return Optional.ofNullable(clazz.cast(dimension()));
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "MetricValues":
            return Optional.ofNullable(clazz.cast(metricValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Dimension", DIMENSION_FIELD);
        map.put("Timestamp", TIMESTAMP_FIELD);
        map.put("MetricValueSet", METRIC_VALUES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MetricDataResult, T> g) {
        return obj -> g.apply((MetricDataResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricDataResult> {
        /**
         * <p>
         * The dimension values that identify this specific data point, such as account ID, region, and instance family.
         * </p>
         * 
         * @param dimension
         *        The dimension values that identify this specific data point, such as account ID, region, and instance
         *        family.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimension(CapacityManagerDimension dimension);

        /**
         * <p>
         * The dimension values that identify this specific data point, such as account ID, region, and instance family.
         * </p>
         * This is a convenience method that creates an instance of the {@link CapacityManagerDimension.Builder}
         * avoiding the need to create one manually via {@link CapacityManagerDimension#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CapacityManagerDimension.Builder#build()} is called immediately
         * and its result is passed to {@link #dimension(CapacityManagerDimension)}.
         * 
         * @param dimension
         *        a consumer that will call methods on {@link CapacityManagerDimension.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimension(CapacityManagerDimension)
         */
        default Builder dimension(Consumer<CapacityManagerDimension.Builder> dimension) {
            return dimension(CapacityManagerDimension.builder().applyMutation(dimension).build());
        }

        /**
         * <p>
         * The timestamp for this data point, indicating when the capacity usage occurred.
         * </p>
         * 
         * @param timestamp
         *        The timestamp for this data point, indicating when the capacity usage occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The metric values and statistics for this data point, containing the actual capacity usage numbers.
         * </p>
         * 
         * @param metricValues
         *        The metric values and statistics for this data point, containing the actual capacity usage numbers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricValues(Collection<MetricValue> metricValues);

        /**
         * <p>
         * The metric values and statistics for this data point, containing the actual capacity usage numbers.
         * </p>
         * 
         * @param metricValues
         *        The metric values and statistics for this data point, containing the actual capacity usage numbers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricValues(MetricValue... metricValues);

        /**
         * <p>
         * The metric values and statistics for this data point, containing the actual capacity usage numbers.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.MetricValue.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.MetricValue#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.MetricValue.Builder#build()} is called immediately and its
         * result is passed to {@link #metricValues(List<MetricValue>)}.
         * 
         * @param metricValues
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.MetricValue.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricValues(java.util.Collection<MetricValue>)
         */
        Builder metricValues(Consumer<MetricValue.Builder>... metricValues);
    }

    static final class BuilderImpl implements Builder {
        private CapacityManagerDimension dimension;

        private Instant timestamp;

        private List<MetricValue> metricValues = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MetricDataResult model) {
            dimension(model.dimension);
            timestamp(model.timestamp);
            metricValues(model.metricValues);
        }

        public final CapacityManagerDimension.Builder getDimension() {
            return dimension != null ? dimension.toBuilder() : null;
        }

        public final void setDimension(CapacityManagerDimension.BuilderImpl dimension) {
            this.dimension = dimension != null ? dimension.build() : null;
        }

        @Override
        public final Builder dimension(CapacityManagerDimension dimension) {
            this.dimension = dimension;
            return this;
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final List<MetricValue.Builder> getMetricValues() {
            List<MetricValue.Builder> result = MetricValueSetCopier.copyToBuilder(this.metricValues);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricValues(Collection<MetricValue.BuilderImpl> metricValues) {
            this.metricValues = MetricValueSetCopier.copyFromBuilder(metricValues);
        }

        @Override
        public final Builder metricValues(Collection<MetricValue> metricValues) {
            this.metricValues = MetricValueSetCopier.copy(metricValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricValues(MetricValue... metricValues) {
            metricValues(Arrays.asList(metricValues));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricValues(Consumer<MetricValue.Builder>... metricValues) {
            metricValues(Stream.of(metricValues).map(c -> MetricValue.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public MetricDataResult build() {
            return new MetricDataResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
