/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the volume initialization. For more information, see <a
 * href="https://docs.aws.amazon.com/ebs/latest/userguide/initalize-volume.html">Initialize Amazon EBS volumes</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InitializationStatusDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<InitializationStatusDetails.Builder, InitializationStatusDetails> {
    private static final SdkField<String> INITIALIZATION_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InitializationType")
            .getter(getter(InitializationStatusDetails::initializationTypeAsString))
            .setter(setter(Builder::initializationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitializationType")
                    .unmarshallLocationName("initializationType").build()).build();

    private static final SdkField<Long> PROGRESS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("Progress")
            .getter(getter(InitializationStatusDetails::progress))
            .setter(setter(Builder::progress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Progress")
                    .unmarshallLocationName("progress").build()).build();

    private static final SdkField<Long> ESTIMATED_TIME_TO_COMPLETE_IN_SECONDS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("EstimatedTimeToCompleteInSeconds")
            .getter(getter(InitializationStatusDetails::estimatedTimeToCompleteInSeconds))
            .setter(setter(Builder::estimatedTimeToCompleteInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EstimatedTimeToCompleteInSeconds")
                    .unmarshallLocationName("estimatedTimeToCompleteInSeconds").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INITIALIZATION_TYPE_FIELD,
            PROGRESS_FIELD, ESTIMATED_TIME_TO_COMPLETE_IN_SECONDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String initializationType;

    private final Long progress;

    private final Long estimatedTimeToCompleteInSeconds;

    private InitializationStatusDetails(BuilderImpl builder) {
        this.initializationType = builder.initializationType;
        this.progress = builder.progress;
        this.estimatedTimeToCompleteInSeconds = builder.estimatedTimeToCompleteInSeconds;
    }

    /**
     * <p>
     * The method used for volume initialization. Possible values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>default</code> - Volume initialized using the default volume initialization rate or fast snapshot restore.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>provisioned-rate</code> - Volume initialized using an Amazon EBS Provisioned Rate for Volume
     * Initialization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #initializationType} will return {@link InitializationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #initializationTypeAsString}.
     * </p>
     * 
     * @return The method used for volume initialization. Possible values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>default</code> - Volume initialized using the default volume initialization rate or fast snapshot
     *         restore.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>provisioned-rate</code> - Volume initialized using an Amazon EBS Provisioned Rate for Volume
     *         Initialization.
     *         </p>
     *         </li>
     * @see InitializationType
     */
    public final InitializationType initializationType() {
        return InitializationType.fromValue(initializationType);
    }

    /**
     * <p>
     * The method used for volume initialization. Possible values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>default</code> - Volume initialized using the default volume initialization rate or fast snapshot restore.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>provisioned-rate</code> - Volume initialized using an Amazon EBS Provisioned Rate for Volume
     * Initialization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #initializationType} will return {@link InitializationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #initializationTypeAsString}.
     * </p>
     * 
     * @return The method used for volume initialization. Possible values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>default</code> - Volume initialized using the default volume initialization rate or fast snapshot
     *         restore.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>provisioned-rate</code> - Volume initialized using an Amazon EBS Provisioned Rate for Volume
     *         Initialization.
     *         </p>
     *         </li>
     * @see InitializationType
     */
    public final String initializationTypeAsString() {
        return initializationType;
    }

    /**
     * <p>
     * The current volume initialization progress as a percentage (0-100). Returns <code>100</code> when volume
     * initialization has completed.
     * </p>
     * 
     * @return The current volume initialization progress as a percentage (0-100). Returns <code>100</code> when volume
     *         initialization has completed.
     */
    public final Long progress() {
        return progress;
    }

    /**
     * <p>
     * The estimated remaining time, in seconds, for volume initialization to complete. Returns <code>0</code> when
     * volume initialization has completed.
     * </p>
     * <p>
     * Only available for volumes created with Amazon EBS Provisioned Rate for Volume Initialization.
     * </p>
     * 
     * @return The estimated remaining time, in seconds, for volume initialization to complete. Returns <code>0</code>
     *         when volume initialization has completed.</p>
     *         <p>
     *         Only available for volumes created with Amazon EBS Provisioned Rate for Volume Initialization.
     */
    public final Long estimatedTimeToCompleteInSeconds() {
        return estimatedTimeToCompleteInSeconds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(initializationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(progress());
        hashCode = 31 * hashCode + Objects.hashCode(estimatedTimeToCompleteInSeconds());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InitializationStatusDetails)) {
            return false;
        }
        InitializationStatusDetails other = (InitializationStatusDetails) obj;
        return Objects.equals(initializationTypeAsString(), other.initializationTypeAsString())
                && Objects.equals(progress(), other.progress())
                && Objects.equals(estimatedTimeToCompleteInSeconds(), other.estimatedTimeToCompleteInSeconds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InitializationStatusDetails").add("InitializationType", initializationTypeAsString())
                .add("Progress", progress()).add("EstimatedTimeToCompleteInSeconds", estimatedTimeToCompleteInSeconds()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InitializationType":
            return Optional.ofNullable(clazz.cast(initializationTypeAsString()));
        case "Progress":
            return Optional.ofNullable(clazz.cast(progress()));
        case "EstimatedTimeToCompleteInSeconds":
            return Optional.ofNullable(clazz.cast(estimatedTimeToCompleteInSeconds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InitializationType", INITIALIZATION_TYPE_FIELD);
        map.put("Progress", PROGRESS_FIELD);
        map.put("EstimatedTimeToCompleteInSeconds", ESTIMATED_TIME_TO_COMPLETE_IN_SECONDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InitializationStatusDetails, T> g) {
        return obj -> g.apply((InitializationStatusDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InitializationStatusDetails> {
        /**
         * <p>
         * The method used for volume initialization. Possible values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>default</code> - Volume initialized using the default volume initialization rate or fast snapshot
         * restore.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>provisioned-rate</code> - Volume initialized using an Amazon EBS Provisioned Rate for Volume
         * Initialization.
         * </p>
         * </li>
         * </ul>
         * 
         * @param initializationType
         *        The method used for volume initialization. Possible values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>default</code> - Volume initialized using the default volume initialization rate or fast
         *        snapshot restore.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>provisioned-rate</code> - Volume initialized using an Amazon EBS Provisioned Rate for Volume
         *        Initialization.
         *        </p>
         *        </li>
         * @see InitializationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InitializationType
         */
        Builder initializationType(String initializationType);

        /**
         * <p>
         * The method used for volume initialization. Possible values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>default</code> - Volume initialized using the default volume initialization rate or fast snapshot
         * restore.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>provisioned-rate</code> - Volume initialized using an Amazon EBS Provisioned Rate for Volume
         * Initialization.
         * </p>
         * </li>
         * </ul>
         * 
         * @param initializationType
         *        The method used for volume initialization. Possible values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>default</code> - Volume initialized using the default volume initialization rate or fast
         *        snapshot restore.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>provisioned-rate</code> - Volume initialized using an Amazon EBS Provisioned Rate for Volume
         *        Initialization.
         *        </p>
         *        </li>
         * @see InitializationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InitializationType
         */
        Builder initializationType(InitializationType initializationType);

        /**
         * <p>
         * The current volume initialization progress as a percentage (0-100). Returns <code>100</code> when volume
         * initialization has completed.
         * </p>
         * 
         * @param progress
         *        The current volume initialization progress as a percentage (0-100). Returns <code>100</code> when
         *        volume initialization has completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder progress(Long progress);

        /**
         * <p>
         * The estimated remaining time, in seconds, for volume initialization to complete. Returns <code>0</code> when
         * volume initialization has completed.
         * </p>
         * <p>
         * Only available for volumes created with Amazon EBS Provisioned Rate for Volume Initialization.
         * </p>
         * 
         * @param estimatedTimeToCompleteInSeconds
         *        The estimated remaining time, in seconds, for volume initialization to complete. Returns
         *        <code>0</code> when volume initialization has completed.</p>
         *        <p>
         *        Only available for volumes created with Amazon EBS Provisioned Rate for Volume Initialization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder estimatedTimeToCompleteInSeconds(Long estimatedTimeToCompleteInSeconds);
    }

    static final class BuilderImpl implements Builder {
        private String initializationType;

        private Long progress;

        private Long estimatedTimeToCompleteInSeconds;

        private BuilderImpl() {
        }

        private BuilderImpl(InitializationStatusDetails model) {
            initializationType(model.initializationType);
            progress(model.progress);
            estimatedTimeToCompleteInSeconds(model.estimatedTimeToCompleteInSeconds);
        }

        public final String getInitializationType() {
            return initializationType;
        }

        public final void setInitializationType(String initializationType) {
            this.initializationType = initializationType;
        }

        @Override
        public final Builder initializationType(String initializationType) {
            this.initializationType = initializationType;
            return this;
        }

        @Override
        public final Builder initializationType(InitializationType initializationType) {
            this.initializationType(initializationType == null ? null : initializationType.toString());
            return this;
        }

        public final Long getProgress() {
            return progress;
        }

        public final void setProgress(Long progress) {
            this.progress = progress;
        }

        @Override
        public final Builder progress(Long progress) {
            this.progress = progress;
            return this;
        }

        public final Long getEstimatedTimeToCompleteInSeconds() {
            return estimatedTimeToCompleteInSeconds;
        }

        public final void setEstimatedTimeToCompleteInSeconds(Long estimatedTimeToCompleteInSeconds) {
            this.estimatedTimeToCompleteInSeconds = estimatedTimeToCompleteInSeconds;
        }

        @Override
        public final Builder estimatedTimeToCompleteInSeconds(Long estimatedTimeToCompleteInSeconds) {
            this.estimatedTimeToCompleteInSeconds = estimatedTimeToCompleteInSeconds;
            return this;
        }

        @Override
        public InitializationStatusDetails build() {
            return new InitializationStatusDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
