/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A VPC BPA exclusion is a mode that can be applied to a single VPC or subnet that exempts it from the account’s BPA
 * mode and will allow bidirectional or egress-only access. You can create BPA exclusions for VPCs and subnets even when
 * BPA is not enabled on the account to ensure that there is no traffic disruption to the exclusions when VPC BPA is
 * turned on. To learn more about VPC BPA, see <a
 * href="https://docs.aws.amazon.com/vpc/latest/userguide/security-vpc-bpa.html">Block public access to VPCs and
 * subnets</a> in the <i>Amazon VPC User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VpcBlockPublicAccessExclusion implements SdkPojo, Serializable,
        ToCopyableBuilder<VpcBlockPublicAccessExclusion.Builder, VpcBlockPublicAccessExclusion> {
    private static final SdkField<String> EXCLUSION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ExclusionId")
            .getter(getter(VpcBlockPublicAccessExclusion::exclusionId))
            .setter(setter(Builder::exclusionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionId")
                    .unmarshallLocationName("exclusionId").build()).build();

    private static final SdkField<String> INTERNET_GATEWAY_EXCLUSION_MODE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InternetGatewayExclusionMode")
            .getter(getter(VpcBlockPublicAccessExclusion::internetGatewayExclusionModeAsString))
            .setter(setter(Builder::internetGatewayExclusionMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InternetGatewayExclusionMode")
                    .unmarshallLocationName("internetGatewayExclusionMode").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn")
            .getter(getter(VpcBlockPublicAccessExclusion::resourceArn))
            .setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn")
                    .unmarshallLocationName("resourceArn").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(VpcBlockPublicAccessExclusion::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> REASON_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Reason")
            .getter(getter(VpcBlockPublicAccessExclusion::reason))
            .setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Reason")
                    .unmarshallLocationName("reason").build()).build();

    private static final SdkField<Instant> CREATION_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTimestamp")
            .getter(getter(VpcBlockPublicAccessExclusion::creationTimestamp))
            .setter(setter(Builder::creationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTimestamp")
                    .unmarshallLocationName("creationTimestamp").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdateTimestamp")
            .getter(getter(VpcBlockPublicAccessExclusion::lastUpdateTimestamp))
            .setter(setter(Builder::lastUpdateTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateTimestamp")
                    .unmarshallLocationName("lastUpdateTimestamp").build()).build();

    private static final SdkField<Instant> DELETION_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeletionTimestamp")
            .getter(getter(VpcBlockPublicAccessExclusion::deletionTimestamp))
            .setter(setter(Builder::deletionTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionTimestamp")
                    .unmarshallLocationName("deletionTimestamp").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(VpcBlockPublicAccessExclusion::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXCLUSION_ID_FIELD,
            INTERNET_GATEWAY_EXCLUSION_MODE_FIELD, RESOURCE_ARN_FIELD, STATE_FIELD, REASON_FIELD, CREATION_TIMESTAMP_FIELD,
            LAST_UPDATE_TIMESTAMP_FIELD, DELETION_TIMESTAMP_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("ExclusionId", EXCLUSION_ID_FIELD);
                    put("InternetGatewayExclusionMode", INTERNET_GATEWAY_EXCLUSION_MODE_FIELD);
                    put("ResourceArn", RESOURCE_ARN_FIELD);
                    put("State", STATE_FIELD);
                    put("Reason", REASON_FIELD);
                    put("CreationTimestamp", CREATION_TIMESTAMP_FIELD);
                    put("LastUpdateTimestamp", LAST_UPDATE_TIMESTAMP_FIELD);
                    put("DeletionTimestamp", DELETION_TIMESTAMP_FIELD);
                    put("TagSet", TAGS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String exclusionId;

    private final String internetGatewayExclusionMode;

    private final String resourceArn;

    private final String state;

    private final String reason;

    private final Instant creationTimestamp;

    private final Instant lastUpdateTimestamp;

    private final Instant deletionTimestamp;

    private final List<Tag> tags;

    private VpcBlockPublicAccessExclusion(BuilderImpl builder) {
        this.exclusionId = builder.exclusionId;
        this.internetGatewayExclusionMode = builder.internetGatewayExclusionMode;
        this.resourceArn = builder.resourceArn;
        this.state = builder.state;
        this.reason = builder.reason;
        this.creationTimestamp = builder.creationTimestamp;
        this.lastUpdateTimestamp = builder.lastUpdateTimestamp;
        this.deletionTimestamp = builder.deletionTimestamp;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the exclusion.
     * </p>
     * 
     * @return The ID of the exclusion.
     */
    public final String exclusionId() {
        return exclusionId;
    }

    /**
     * <p>
     * The exclusion mode for internet gateway traffic.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>bidirectional-access-allowed</code>: Allow all internet traffic to and from the excluded VPCs and subnets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>egress-access-allowed</code>: Allow outbound internet traffic from the excluded VPCs and subnets. Block
     * inbound internet traffic to the excluded VPCs and subnets. Only applies when VPC Block Public Access is set to
     * Bidirectional.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #internetGatewayExclusionMode} will return {@link InternetGatewayExclusionMode#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #internetGatewayExclusionModeAsString}.
     * </p>
     * 
     * @return The exclusion mode for internet gateway traffic.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>bidirectional-access-allowed</code>: Allow all internet traffic to and from the excluded VPCs and
     *         subnets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>egress-access-allowed</code>: Allow outbound internet traffic from the excluded VPCs and subnets.
     *         Block inbound internet traffic to the excluded VPCs and subnets. Only applies when VPC Block Public
     *         Access is set to Bidirectional.
     *         </p>
     *         </li>
     * @see InternetGatewayExclusionMode
     */
    public final InternetGatewayExclusionMode internetGatewayExclusionMode() {
        return InternetGatewayExclusionMode.fromValue(internetGatewayExclusionMode);
    }

    /**
     * <p>
     * The exclusion mode for internet gateway traffic.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>bidirectional-access-allowed</code>: Allow all internet traffic to and from the excluded VPCs and subnets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>egress-access-allowed</code>: Allow outbound internet traffic from the excluded VPCs and subnets. Block
     * inbound internet traffic to the excluded VPCs and subnets. Only applies when VPC Block Public Access is set to
     * Bidirectional.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #internetGatewayExclusionMode} will return {@link InternetGatewayExclusionMode#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #internetGatewayExclusionModeAsString}.
     * </p>
     * 
     * @return The exclusion mode for internet gateway traffic.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>bidirectional-access-allowed</code>: Allow all internet traffic to and from the excluded VPCs and
     *         subnets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>egress-access-allowed</code>: Allow outbound internet traffic from the excluded VPCs and subnets.
     *         Block inbound internet traffic to the excluded VPCs and subnets. Only applies when VPC Block Public
     *         Access is set to Bidirectional.
     *         </p>
     *         </li>
     * @see InternetGatewayExclusionMode
     */
    public final String internetGatewayExclusionModeAsString() {
        return internetGatewayExclusionMode;
    }

    /**
     * <p>
     * The ARN of the exclusion.
     * </p>
     * 
     * @return The ARN of the exclusion.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * The state of the exclusion.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VpcBlockPublicAccessExclusionState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the exclusion.
     * @see VpcBlockPublicAccessExclusionState
     */
    public final VpcBlockPublicAccessExclusionState state() {
        return VpcBlockPublicAccessExclusionState.fromValue(state);
    }

    /**
     * <p>
     * The state of the exclusion.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VpcBlockPublicAccessExclusionState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the exclusion.
     * @see VpcBlockPublicAccessExclusionState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The reason for the current exclusion state.
     * </p>
     * 
     * @return The reason for the current exclusion state.
     */
    public final String reason() {
        return reason;
    }

    /**
     * <p>
     * When the exclusion was created.
     * </p>
     * 
     * @return When the exclusion was created.
     */
    public final Instant creationTimestamp() {
        return creationTimestamp;
    }

    /**
     * <p>
     * When the exclusion was last updated.
     * </p>
     * 
     * @return When the exclusion was last updated.
     */
    public final Instant lastUpdateTimestamp() {
        return lastUpdateTimestamp;
    }

    /**
     * <p>
     * When the exclusion was deleted.
     * </p>
     * 
     * @return When the exclusion was deleted.
     */
    public final Instant deletionTimestamp() {
        return deletionTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <code>tag</code> - The key/value combination of a tag assigned to the resource. Use the tag key in the filter
     * name and the tag value as the filter value. For example, to find all resources that have a tag with the key
     * <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and
     * <code>TeamA</code> for the filter value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return <code>tag</code> - The key/value combination of a tag assigned to the resource. Use the tag key in the
     *         filter name and the tag value as the filter value. For example, to find all resources that have a tag
     *         with the key <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the
     *         filter name and <code>TeamA</code> for the filter value.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(exclusionId());
        hashCode = 31 * hashCode + Objects.hashCode(internetGatewayExclusionModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(deletionTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VpcBlockPublicAccessExclusion)) {
            return false;
        }
        VpcBlockPublicAccessExclusion other = (VpcBlockPublicAccessExclusion) obj;
        return Objects.equals(exclusionId(), other.exclusionId())
                && Objects.equals(internetGatewayExclusionModeAsString(), other.internetGatewayExclusionModeAsString())
                && Objects.equals(resourceArn(), other.resourceArn()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(reason(), other.reason()) && Objects.equals(creationTimestamp(), other.creationTimestamp())
                && Objects.equals(lastUpdateTimestamp(), other.lastUpdateTimestamp())
                && Objects.equals(deletionTimestamp(), other.deletionTimestamp()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VpcBlockPublicAccessExclusion").add("ExclusionId", exclusionId())
                .add("InternetGatewayExclusionMode", internetGatewayExclusionModeAsString()).add("ResourceArn", resourceArn())
                .add("State", stateAsString()).add("Reason", reason()).add("CreationTimestamp", creationTimestamp())
                .add("LastUpdateTimestamp", lastUpdateTimestamp()).add("DeletionTimestamp", deletionTimestamp())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExclusionId":
            return Optional.ofNullable(clazz.cast(exclusionId()));
        case "InternetGatewayExclusionMode":
            return Optional.ofNullable(clazz.cast(internetGatewayExclusionModeAsString()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "CreationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        case "LastUpdateTimestamp":
            return Optional.ofNullable(clazz.cast(lastUpdateTimestamp()));
        case "DeletionTimestamp":
            return Optional.ofNullable(clazz.cast(deletionTimestamp()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<VpcBlockPublicAccessExclusion, T> g) {
        return obj -> g.apply((VpcBlockPublicAccessExclusion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VpcBlockPublicAccessExclusion> {
        /**
         * <p>
         * The ID of the exclusion.
         * </p>
         * 
         * @param exclusionId
         *        The ID of the exclusion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionId(String exclusionId);

        /**
         * <p>
         * The exclusion mode for internet gateway traffic.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>bidirectional-access-allowed</code>: Allow all internet traffic to and from the excluded VPCs and
         * subnets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>egress-access-allowed</code>: Allow outbound internet traffic from the excluded VPCs and subnets. Block
         * inbound internet traffic to the excluded VPCs and subnets. Only applies when VPC Block Public Access is set
         * to Bidirectional.
         * </p>
         * </li>
         * </ul>
         * 
         * @param internetGatewayExclusionMode
         *        The exclusion mode for internet gateway traffic.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>bidirectional-access-allowed</code>: Allow all internet traffic to and from the excluded VPCs
         *        and subnets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>egress-access-allowed</code>: Allow outbound internet traffic from the excluded VPCs and
         *        subnets. Block inbound internet traffic to the excluded VPCs and subnets. Only applies when VPC Block
         *        Public Access is set to Bidirectional.
         *        </p>
         *        </li>
         * @see InternetGatewayExclusionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InternetGatewayExclusionMode
         */
        Builder internetGatewayExclusionMode(String internetGatewayExclusionMode);

        /**
         * <p>
         * The exclusion mode for internet gateway traffic.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>bidirectional-access-allowed</code>: Allow all internet traffic to and from the excluded VPCs and
         * subnets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>egress-access-allowed</code>: Allow outbound internet traffic from the excluded VPCs and subnets. Block
         * inbound internet traffic to the excluded VPCs and subnets. Only applies when VPC Block Public Access is set
         * to Bidirectional.
         * </p>
         * </li>
         * </ul>
         * 
         * @param internetGatewayExclusionMode
         *        The exclusion mode for internet gateway traffic.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>bidirectional-access-allowed</code>: Allow all internet traffic to and from the excluded VPCs
         *        and subnets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>egress-access-allowed</code>: Allow outbound internet traffic from the excluded VPCs and
         *        subnets. Block inbound internet traffic to the excluded VPCs and subnets. Only applies when VPC Block
         *        Public Access is set to Bidirectional.
         *        </p>
         *        </li>
         * @see InternetGatewayExclusionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InternetGatewayExclusionMode
         */
        Builder internetGatewayExclusionMode(InternetGatewayExclusionMode internetGatewayExclusionMode);

        /**
         * <p>
         * The ARN of the exclusion.
         * </p>
         * 
         * @param resourceArn
         *        The ARN of the exclusion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The state of the exclusion.
         * </p>
         * 
         * @param state
         *        The state of the exclusion.
         * @see VpcBlockPublicAccessExclusionState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcBlockPublicAccessExclusionState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the exclusion.
         * </p>
         * 
         * @param state
         *        The state of the exclusion.
         * @see VpcBlockPublicAccessExclusionState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcBlockPublicAccessExclusionState
         */
        Builder state(VpcBlockPublicAccessExclusionState state);

        /**
         * <p>
         * The reason for the current exclusion state.
         * </p>
         * 
         * @param reason
         *        The reason for the current exclusion state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * When the exclusion was created.
         * </p>
         * 
         * @param creationTimestamp
         *        When the exclusion was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Instant creationTimestamp);

        /**
         * <p>
         * When the exclusion was last updated.
         * </p>
         * 
         * @param lastUpdateTimestamp
         *        When the exclusion was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTimestamp(Instant lastUpdateTimestamp);

        /**
         * <p>
         * When the exclusion was deleted.
         * </p>
         * 
         * @param deletionTimestamp
         *        When the exclusion was deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionTimestamp(Instant deletionTimestamp);

        /**
         * <p>
         * <code>tag</code> - The key/value combination of a tag assigned to the resource. Use the tag key in the filter
         * name and the tag value as the filter value. For example, to find all resources that have a tag with the key
         * <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and
         * <code>TeamA</code> for the filter value.
         * </p>
         * 
         * @param tags
         *        <code>tag</code> - The key/value combination of a tag assigned to the resource. Use the tag key in the
         *        filter name and the tag value as the filter value. For example, to find all resources that have a tag
         *        with the key <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for
         *        the filter name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * <code>tag</code> - The key/value combination of a tag assigned to the resource. Use the tag key in the filter
         * name and the tag value as the filter value. For example, to find all resources that have a tag with the key
         * <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and
         * <code>TeamA</code> for the filter value.
         * </p>
         * 
         * @param tags
         *        <code>tag</code> - The key/value combination of a tag assigned to the resource. Use the tag key in the
         *        filter name and the tag value as the filter value. For example, to find all resources that have a tag
         *        with the key <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for
         *        the filter name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * <code>tag</code> - The key/value combination of a tag assigned to the resource. Use the tag key in the filter
         * name and the tag value as the filter value. For example, to find all resources that have a tag with the key
         * <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and
         * <code>TeamA</code> for the filter value.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String exclusionId;

        private String internetGatewayExclusionMode;

        private String resourceArn;

        private String state;

        private String reason;

        private Instant creationTimestamp;

        private Instant lastUpdateTimestamp;

        private Instant deletionTimestamp;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(VpcBlockPublicAccessExclusion model) {
            exclusionId(model.exclusionId);
            internetGatewayExclusionMode(model.internetGatewayExclusionMode);
            resourceArn(model.resourceArn);
            state(model.state);
            reason(model.reason);
            creationTimestamp(model.creationTimestamp);
            lastUpdateTimestamp(model.lastUpdateTimestamp);
            deletionTimestamp(model.deletionTimestamp);
            tags(model.tags);
        }

        public final String getExclusionId() {
            return exclusionId;
        }

        public final void setExclusionId(String exclusionId) {
            this.exclusionId = exclusionId;
        }

        @Override
        public final Builder exclusionId(String exclusionId) {
            this.exclusionId = exclusionId;
            return this;
        }

        public final String getInternetGatewayExclusionMode() {
            return internetGatewayExclusionMode;
        }

        public final void setInternetGatewayExclusionMode(String internetGatewayExclusionMode) {
            this.internetGatewayExclusionMode = internetGatewayExclusionMode;
        }

        @Override
        public final Builder internetGatewayExclusionMode(String internetGatewayExclusionMode) {
            this.internetGatewayExclusionMode = internetGatewayExclusionMode;
            return this;
        }

        @Override
        public final Builder internetGatewayExclusionMode(InternetGatewayExclusionMode internetGatewayExclusionMode) {
            this.internetGatewayExclusionMode(internetGatewayExclusionMode == null ? null : internetGatewayExclusionMode
                    .toString());
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(VpcBlockPublicAccessExclusionState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getReason() {
            return reason;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final Instant getCreationTimestamp() {
            return creationTimestamp;
        }

        public final void setCreationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        public final Instant getLastUpdateTimestamp() {
            return lastUpdateTimestamp;
        }

        public final void setLastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
        }

        @Override
        public final Builder lastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
            return this;
        }

        public final Instant getDeletionTimestamp() {
            return deletionTimestamp;
        }

        public final void setDeletionTimestamp(Instant deletionTimestamp) {
            this.deletionTimestamp = deletionTimestamp;
        }

        @Override
        public final Builder deletionTimestamp(Instant deletionTimestamp) {
            this.deletionTimestamp = deletionTimestamp;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public VpcBlockPublicAccessExclusion build() {
            return new VpcBlockPublicAccessExclusion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
