/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateVpcRequest extends Ec2Request implements ToCopyableBuilder<CreateVpcRequest.Builder, CreateVpcRequest> {
    private static final SdkField<String> CIDR_BLOCK_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CidrBlock")
            .getter(getter(CreateVpcRequest::cidrBlock))
            .setter(setter(Builder::cidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CidrBlock")
                    .unmarshallLocationName("CidrBlock").build()).build();

    private static final SdkField<String> IPV6_POOL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Ipv6Pool")
            .getter(getter(CreateVpcRequest::ipv6Pool))
            .setter(setter(Builder::ipv6Pool))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6Pool")
                    .unmarshallLocationName("Ipv6Pool").build()).build();

    private static final SdkField<String> IPV6_CIDR_BLOCK_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Ipv6CidrBlock")
            .getter(getter(CreateVpcRequest::ipv6CidrBlock))
            .setter(setter(Builder::ipv6CidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6CidrBlock")
                    .unmarshallLocationName("Ipv6CidrBlock").build()).build();

    private static final SdkField<String> IPV4_IPAM_POOL_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Ipv4IpamPoolId")
            .getter(getter(CreateVpcRequest::ipv4IpamPoolId))
            .setter(setter(Builder::ipv4IpamPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv4IpamPoolId")
                    .unmarshallLocationName("Ipv4IpamPoolId").build()).build();

    private static final SdkField<Integer> IPV4_NETMASK_LENGTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Ipv4NetmaskLength")
            .getter(getter(CreateVpcRequest::ipv4NetmaskLength))
            .setter(setter(Builder::ipv4NetmaskLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv4NetmaskLength")
                    .unmarshallLocationName("Ipv4NetmaskLength").build()).build();

    private static final SdkField<String> IPV6_IPAM_POOL_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Ipv6IpamPoolId")
            .getter(getter(CreateVpcRequest::ipv6IpamPoolId))
            .setter(setter(Builder::ipv6IpamPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6IpamPoolId")
                    .unmarshallLocationName("Ipv6IpamPoolId").build()).build();

    private static final SdkField<Integer> IPV6_NETMASK_LENGTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Ipv6NetmaskLength")
            .getter(getter(CreateVpcRequest::ipv6NetmaskLength))
            .setter(setter(Builder::ipv6NetmaskLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6NetmaskLength")
                    .unmarshallLocationName("Ipv6NetmaskLength").build()).build();

    private static final SdkField<String> IPV6_CIDR_BLOCK_NETWORK_BORDER_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Ipv6CidrBlockNetworkBorderGroup")
            .getter(getter(CreateVpcRequest::ipv6CidrBlockNetworkBorderGroup))
            .setter(setter(Builder::ipv6CidrBlockNetworkBorderGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6CidrBlockNetworkBorderGroup")
                    .unmarshallLocationName("Ipv6CidrBlockNetworkBorderGroup").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CreateVpcRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(CreateVpcRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("dryRun").build()).build();

    private static final SdkField<String> INSTANCE_TENANCY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceTenancy")
            .getter(getter(CreateVpcRequest::instanceTenancyAsString))
            .setter(setter(Builder::instanceTenancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceTenancy")
                    .unmarshallLocationName("instanceTenancy").build()).build();

    private static final SdkField<Boolean> AMAZON_PROVIDED_IPV6_CIDR_BLOCK_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AmazonProvidedIpv6CidrBlock")
            .getter(getter(CreateVpcRequest::amazonProvidedIpv6CidrBlock))
            .setter(setter(Builder::amazonProvidedIpv6CidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AmazonProvidedIpv6CidrBlock")
                    .unmarshallLocationName("amazonProvidedIpv6CidrBlock").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CIDR_BLOCK_FIELD,
            IPV6_POOL_FIELD, IPV6_CIDR_BLOCK_FIELD, IPV4_IPAM_POOL_ID_FIELD, IPV4_NETMASK_LENGTH_FIELD, IPV6_IPAM_POOL_ID_FIELD,
            IPV6_NETMASK_LENGTH_FIELD, IPV6_CIDR_BLOCK_NETWORK_BORDER_GROUP_FIELD, TAG_SPECIFICATIONS_FIELD, DRY_RUN_FIELD,
            INSTANCE_TENANCY_FIELD, AMAZON_PROVIDED_IPV6_CIDR_BLOCK_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("CidrBlock", CIDR_BLOCK_FIELD);
                    put("Ipv6Pool", IPV6_POOL_FIELD);
                    put("Ipv6CidrBlock", IPV6_CIDR_BLOCK_FIELD);
                    put("Ipv4IpamPoolId", IPV4_IPAM_POOL_ID_FIELD);
                    put("Ipv4NetmaskLength", IPV4_NETMASK_LENGTH_FIELD);
                    put("Ipv6IpamPoolId", IPV6_IPAM_POOL_ID_FIELD);
                    put("Ipv6NetmaskLength", IPV6_NETMASK_LENGTH_FIELD);
                    put("Ipv6CidrBlockNetworkBorderGroup", IPV6_CIDR_BLOCK_NETWORK_BORDER_GROUP_FIELD);
                    put("TagSpecification", TAG_SPECIFICATIONS_FIELD);
                    put("DryRun", DRY_RUN_FIELD);
                    put("InstanceTenancy", INSTANCE_TENANCY_FIELD);
                    put("AmazonProvidedIpv6CidrBlock", AMAZON_PROVIDED_IPV6_CIDR_BLOCK_FIELD);
                }
            });

    private final String cidrBlock;

    private final String ipv6Pool;

    private final String ipv6CidrBlock;

    private final String ipv4IpamPoolId;

    private final Integer ipv4NetmaskLength;

    private final String ipv6IpamPoolId;

    private final Integer ipv6NetmaskLength;

    private final String ipv6CidrBlockNetworkBorderGroup;

    private final List<TagSpecification> tagSpecifications;

    private final Boolean dryRun;

    private final String instanceTenancy;

    private final Boolean amazonProvidedIpv6CidrBlock;

    private CreateVpcRequest(BuilderImpl builder) {
        super(builder);
        this.cidrBlock = builder.cidrBlock;
        this.ipv6Pool = builder.ipv6Pool;
        this.ipv6CidrBlock = builder.ipv6CidrBlock;
        this.ipv4IpamPoolId = builder.ipv4IpamPoolId;
        this.ipv4NetmaskLength = builder.ipv4NetmaskLength;
        this.ipv6IpamPoolId = builder.ipv6IpamPoolId;
        this.ipv6NetmaskLength = builder.ipv6NetmaskLength;
        this.ipv6CidrBlockNetworkBorderGroup = builder.ipv6CidrBlockNetworkBorderGroup;
        this.tagSpecifications = builder.tagSpecifications;
        this.dryRun = builder.dryRun;
        this.instanceTenancy = builder.instanceTenancy;
        this.amazonProvidedIpv6CidrBlock = builder.amazonProvidedIpv6CidrBlock;
    }

    /**
     * <p>
     * The IPv4 network range for the VPC, in CIDR notation. For example, <code>10.0.0.0/16</code>. We modify the
     * specified CIDR block to its canonical form; for example, if you specify <code>100.68.0.18/18</code>, we modify it
     * to <code>100.68.0.0/18</code>.
     * </p>
     * 
     * @return The IPv4 network range for the VPC, in CIDR notation. For example, <code>10.0.0.0/16</code>. We modify
     *         the specified CIDR block to its canonical form; for example, if you specify <code>100.68.0.18/18</code>,
     *         we modify it to <code>100.68.0.0/18</code>.
     */
    public final String cidrBlock() {
        return cidrBlock;
    }

    /**
     * <p>
     * The ID of an IPv6 address pool from which to allocate the IPv6 CIDR block.
     * </p>
     * 
     * @return The ID of an IPv6 address pool from which to allocate the IPv6 CIDR block.
     */
    public final String ipv6Pool() {
        return ipv6Pool;
    }

    /**
     * <p>
     * The IPv6 CIDR block from the IPv6 address pool. You must also specify <code>Ipv6Pool</code> in the request.
     * </p>
     * <p>
     * To let Amazon choose the IPv6 CIDR block for you, omit this parameter.
     * </p>
     * 
     * @return The IPv6 CIDR block from the IPv6 address pool. You must also specify <code>Ipv6Pool</code> in the
     *         request.</p>
     *         <p>
     *         To let Amazon choose the IPv6 CIDR block for you, omit this parameter.
     */
    public final String ipv6CidrBlock() {
        return ipv6CidrBlock;
    }

    /**
     * <p>
     * The ID of an IPv4 IPAM pool you want to use for allocating this VPC's CIDR. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon VPC
     * IPAM User Guide</i>.
     * </p>
     * 
     * @return The ID of an IPv4 IPAM pool you want to use for allocating this VPC's CIDR. For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the
     *         <i>Amazon VPC IPAM User Guide</i>.
     */
    public final String ipv4IpamPoolId() {
        return ipv4IpamPoolId;
    }

    /**
     * <p>
     * The netmask length of the IPv4 CIDR you want to allocate to this VPC from an Amazon VPC IP Address Manager (IPAM)
     * pool. For more information about IPAM, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon VPC
     * IPAM User Guide</i>.
     * </p>
     * 
     * @return The netmask length of the IPv4 CIDR you want to allocate to this VPC from an Amazon VPC IP Address
     *         Manager (IPAM) pool. For more information about IPAM, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the
     *         <i>Amazon VPC IPAM User Guide</i>.
     */
    public final Integer ipv4NetmaskLength() {
        return ipv4NetmaskLength;
    }

    /**
     * <p>
     * The ID of an IPv6 IPAM pool which will be used to allocate this VPC an IPv6 CIDR. IPAM is a VPC feature that you
     * can use to automate your IP address management workflows including assigning, tracking, troubleshooting, and
     * auditing IP addresses across Amazon Web Services Regions and accounts throughout your Amazon Web Services
     * Organization. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon VPC
     * IPAM User Guide</i>.
     * </p>
     * 
     * @return The ID of an IPv6 IPAM pool which will be used to allocate this VPC an IPv6 CIDR. IPAM is a VPC feature
     *         that you can use to automate your IP address management workflows including assigning, tracking,
     *         troubleshooting, and auditing IP addresses across Amazon Web Services Regions and accounts throughout
     *         your Amazon Web Services Organization. For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the
     *         <i>Amazon VPC IPAM User Guide</i>.
     */
    public final String ipv6IpamPoolId() {
        return ipv6IpamPoolId;
    }

    /**
     * <p>
     * The netmask length of the IPv6 CIDR you want to allocate to this VPC from an Amazon VPC IP Address Manager (IPAM)
     * pool. For more information about IPAM, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon VPC
     * IPAM User Guide</i>.
     * </p>
     * 
     * @return The netmask length of the IPv6 CIDR you want to allocate to this VPC from an Amazon VPC IP Address
     *         Manager (IPAM) pool. For more information about IPAM, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the
     *         <i>Amazon VPC IPAM User Guide</i>.
     */
    public final Integer ipv6NetmaskLength() {
        return ipv6NetmaskLength;
    }

    /**
     * <p>
     * The name of the location from which we advertise the IPV6 CIDR block. Use this parameter to limit the address to
     * this location.
     * </p>
     * <p>
     * You must set <code>AmazonProvidedIpv6CidrBlock</code> to <code>true</code> to use this parameter.
     * </p>
     * 
     * @return The name of the location from which we advertise the IPV6 CIDR block. Use this parameter to limit the
     *         address to this location.</p>
     *         <p>
     *         You must set <code>AmazonProvidedIpv6CidrBlock</code> to <code>true</code> to use this parameter.
     */
    public final String ipv6CidrBlockNetworkBorderGroup() {
        return ipv6CidrBlockNetworkBorderGroup;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to assign to the VPC.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The tags to assign to the VPC.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    /**
     * <p>
     * The tenancy options for instances launched into the VPC. For <code>default</code>, instances are launched with
     * shared tenancy by default. You can launch instances with any tenancy into a shared tenancy VPC. For
     * <code>dedicated</code>, instances are launched as dedicated tenancy instances by default. You can only launch
     * instances with a tenancy of <code>dedicated</code> or <code>host</code> into a dedicated tenancy VPC.
     * </p>
     * <p>
     * <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the <code>default</code> or
     * <code>dedicated</code> values only.
     * </p>
     * <p>
     * Default: <code>default</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceTenancy}
     * will return {@link Tenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTenancyAsString}.
     * </p>
     * 
     * @return The tenancy options for instances launched into the VPC. For <code>default</code>, instances are launched
     *         with shared tenancy by default. You can launch instances with any tenancy into a shared tenancy VPC. For
     *         <code>dedicated</code>, instances are launched as dedicated tenancy instances by default. You can only
     *         launch instances with a tenancy of <code>dedicated</code> or <code>host</code> into a dedicated tenancy
     *         VPC. </p>
     *         <p>
     *         <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
     *         <code>default</code> or <code>dedicated</code> values only.
     *         </p>
     *         <p>
     *         Default: <code>default</code>
     * @see Tenancy
     */
    public final Tenancy instanceTenancy() {
        return Tenancy.fromValue(instanceTenancy);
    }

    /**
     * <p>
     * The tenancy options for instances launched into the VPC. For <code>default</code>, instances are launched with
     * shared tenancy by default. You can launch instances with any tenancy into a shared tenancy VPC. For
     * <code>dedicated</code>, instances are launched as dedicated tenancy instances by default. You can only launch
     * instances with a tenancy of <code>dedicated</code> or <code>host</code> into a dedicated tenancy VPC.
     * </p>
     * <p>
     * <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the <code>default</code> or
     * <code>dedicated</code> values only.
     * </p>
     * <p>
     * Default: <code>default</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceTenancy}
     * will return {@link Tenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTenancyAsString}.
     * </p>
     * 
     * @return The tenancy options for instances launched into the VPC. For <code>default</code>, instances are launched
     *         with shared tenancy by default. You can launch instances with any tenancy into a shared tenancy VPC. For
     *         <code>dedicated</code>, instances are launched as dedicated tenancy instances by default. You can only
     *         launch instances with a tenancy of <code>dedicated</code> or <code>host</code> into a dedicated tenancy
     *         VPC. </p>
     *         <p>
     *         <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
     *         <code>default</code> or <code>dedicated</code> values only.
     *         </p>
     *         <p>
     *         Default: <code>default</code>
     * @see Tenancy
     */
    public final String instanceTenancyAsString() {
        return instanceTenancy;
    }

    /**
     * <p>
     * Requests an Amazon-provided IPv6 CIDR block with a /56 prefix length for the VPC. You cannot specify the range of
     * IP addresses, or the size of the CIDR block.
     * </p>
     * 
     * @return Requests an Amazon-provided IPv6 CIDR block with a /56 prefix length for the VPC. You cannot specify the
     *         range of IP addresses, or the size of the CIDR block.
     */
    public final Boolean amazonProvidedIpv6CidrBlock() {
        return amazonProvidedIpv6CidrBlock;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(cidrBlock());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6Pool());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6CidrBlock());
        hashCode = 31 * hashCode + Objects.hashCode(ipv4IpamPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(ipv4NetmaskLength());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6IpamPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6NetmaskLength());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6CidrBlockNetworkBorderGroup());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTenancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(amazonProvidedIpv6CidrBlock());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateVpcRequest)) {
            return false;
        }
        CreateVpcRequest other = (CreateVpcRequest) obj;
        return Objects.equals(cidrBlock(), other.cidrBlock()) && Objects.equals(ipv6Pool(), other.ipv6Pool())
                && Objects.equals(ipv6CidrBlock(), other.ipv6CidrBlock())
                && Objects.equals(ipv4IpamPoolId(), other.ipv4IpamPoolId())
                && Objects.equals(ipv4NetmaskLength(), other.ipv4NetmaskLength())
                && Objects.equals(ipv6IpamPoolId(), other.ipv6IpamPoolId())
                && Objects.equals(ipv6NetmaskLength(), other.ipv6NetmaskLength())
                && Objects.equals(ipv6CidrBlockNetworkBorderGroup(), other.ipv6CidrBlockNetworkBorderGroup())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications()) && Objects.equals(dryRun(), other.dryRun())
                && Objects.equals(instanceTenancyAsString(), other.instanceTenancyAsString())
                && Objects.equals(amazonProvidedIpv6CidrBlock(), other.amazonProvidedIpv6CidrBlock());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateVpcRequest").add("CidrBlock", cidrBlock()).add("Ipv6Pool", ipv6Pool())
                .add("Ipv6CidrBlock", ipv6CidrBlock()).add("Ipv4IpamPoolId", ipv4IpamPoolId())
                .add("Ipv4NetmaskLength", ipv4NetmaskLength()).add("Ipv6IpamPoolId", ipv6IpamPoolId())
                .add("Ipv6NetmaskLength", ipv6NetmaskLength())
                .add("Ipv6CidrBlockNetworkBorderGroup", ipv6CidrBlockNetworkBorderGroup())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).add("DryRun", dryRun())
                .add("InstanceTenancy", instanceTenancyAsString())
                .add("AmazonProvidedIpv6CidrBlock", amazonProvidedIpv6CidrBlock()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CidrBlock":
            return Optional.ofNullable(clazz.cast(cidrBlock()));
        case "Ipv6Pool":
            return Optional.ofNullable(clazz.cast(ipv6Pool()));
        case "Ipv6CidrBlock":
            return Optional.ofNullable(clazz.cast(ipv6CidrBlock()));
        case "Ipv4IpamPoolId":
            return Optional.ofNullable(clazz.cast(ipv4IpamPoolId()));
        case "Ipv4NetmaskLength":
            return Optional.ofNullable(clazz.cast(ipv4NetmaskLength()));
        case "Ipv6IpamPoolId":
            return Optional.ofNullable(clazz.cast(ipv6IpamPoolId()));
        case "Ipv6NetmaskLength":
            return Optional.ofNullable(clazz.cast(ipv6NetmaskLength()));
        case "Ipv6CidrBlockNetworkBorderGroup":
            return Optional.ofNullable(clazz.cast(ipv6CidrBlockNetworkBorderGroup()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        case "InstanceTenancy":
            return Optional.ofNullable(clazz.cast(instanceTenancyAsString()));
        case "AmazonProvidedIpv6CidrBlock":
            return Optional.ofNullable(clazz.cast(amazonProvidedIpv6CidrBlock()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CreateVpcRequest, T> g) {
        return obj -> g.apply((CreateVpcRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateVpcRequest> {
        /**
         * <p>
         * The IPv4 network range for the VPC, in CIDR notation. For example, <code>10.0.0.0/16</code>. We modify the
         * specified CIDR block to its canonical form; for example, if you specify <code>100.68.0.18/18</code>, we
         * modify it to <code>100.68.0.0/18</code>.
         * </p>
         * 
         * @param cidrBlock
         *        The IPv4 network range for the VPC, in CIDR notation. For example, <code>10.0.0.0/16</code>. We modify
         *        the specified CIDR block to its canonical form; for example, if you specify
         *        <code>100.68.0.18/18</code>, we modify it to <code>100.68.0.0/18</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrBlock(String cidrBlock);

        /**
         * <p>
         * The ID of an IPv6 address pool from which to allocate the IPv6 CIDR block.
         * </p>
         * 
         * @param ipv6Pool
         *        The ID of an IPv6 address pool from which to allocate the IPv6 CIDR block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Pool(String ipv6Pool);

        /**
         * <p>
         * The IPv6 CIDR block from the IPv6 address pool. You must also specify <code>Ipv6Pool</code> in the request.
         * </p>
         * <p>
         * To let Amazon choose the IPv6 CIDR block for you, omit this parameter.
         * </p>
         * 
         * @param ipv6CidrBlock
         *        The IPv6 CIDR block from the IPv6 address pool. You must also specify <code>Ipv6Pool</code> in the
         *        request.</p>
         *        <p>
         *        To let Amazon choose the IPv6 CIDR block for you, omit this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6CidrBlock(String ipv6CidrBlock);

        /**
         * <p>
         * The ID of an IPv4 IPAM pool you want to use for allocating this VPC's CIDR. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon
         * VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param ipv4IpamPoolId
         *        The ID of an IPv4 IPAM pool you want to use for allocating this VPC's CIDR. For more information, see
         *        <a href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv4IpamPoolId(String ipv4IpamPoolId);

        /**
         * <p>
         * The netmask length of the IPv4 CIDR you want to allocate to this VPC from an Amazon VPC IP Address Manager
         * (IPAM) pool. For more information about IPAM, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon
         * VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param ipv4NetmaskLength
         *        The netmask length of the IPv4 CIDR you want to allocate to this VPC from an Amazon VPC IP Address
         *        Manager (IPAM) pool. For more information about IPAM, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv4NetmaskLength(Integer ipv4NetmaskLength);

        /**
         * <p>
         * The ID of an IPv6 IPAM pool which will be used to allocate this VPC an IPv6 CIDR. IPAM is a VPC feature that
         * you can use to automate your IP address management workflows including assigning, tracking, troubleshooting,
         * and auditing IP addresses across Amazon Web Services Regions and accounts throughout your Amazon Web Services
         * Organization. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon
         * VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param ipv6IpamPoolId
         *        The ID of an IPv6 IPAM pool which will be used to allocate this VPC an IPv6 CIDR. IPAM is a VPC
         *        feature that you can use to automate your IP address management workflows including assigning,
         *        tracking, troubleshooting, and auditing IP addresses across Amazon Web Services Regions and accounts
         *        throughout your Amazon Web Services Organization. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6IpamPoolId(String ipv6IpamPoolId);

        /**
         * <p>
         * The netmask length of the IPv6 CIDR you want to allocate to this VPC from an Amazon VPC IP Address Manager
         * (IPAM) pool. For more information about IPAM, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon
         * VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param ipv6NetmaskLength
         *        The netmask length of the IPv6 CIDR you want to allocate to this VPC from an Amazon VPC IP Address
         *        Manager (IPAM) pool. For more information about IPAM, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6NetmaskLength(Integer ipv6NetmaskLength);

        /**
         * <p>
         * The name of the location from which we advertise the IPV6 CIDR block. Use this parameter to limit the address
         * to this location.
         * </p>
         * <p>
         * You must set <code>AmazonProvidedIpv6CidrBlock</code> to <code>true</code> to use this parameter.
         * </p>
         * 
         * @param ipv6CidrBlockNetworkBorderGroup
         *        The name of the location from which we advertise the IPV6 CIDR block. Use this parameter to limit the
         *        address to this location.</p>
         *        <p>
         *        You must set <code>AmazonProvidedIpv6CidrBlock</code> to <code>true</code> to use this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6CidrBlockNetworkBorderGroup(String ipv6CidrBlockNetworkBorderGroup);

        /**
         * <p>
         * The tags to assign to the VPC.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to assign to the VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to assign to the VPC.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to assign to the VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to assign to the VPC.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        /**
         * <p>
         * The tenancy options for instances launched into the VPC. For <code>default</code>, instances are launched
         * with shared tenancy by default. You can launch instances with any tenancy into a shared tenancy VPC. For
         * <code>dedicated</code>, instances are launched as dedicated tenancy instances by default. You can only launch
         * instances with a tenancy of <code>dedicated</code> or <code>host</code> into a dedicated tenancy VPC.
         * </p>
         * <p>
         * <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
         * <code>default</code> or <code>dedicated</code> values only.
         * </p>
         * <p>
         * Default: <code>default</code>
         * </p>
         * 
         * @param instanceTenancy
         *        The tenancy options for instances launched into the VPC. For <code>default</code>, instances are
         *        launched with shared tenancy by default. You can launch instances with any tenancy into a shared
         *        tenancy VPC. For <code>dedicated</code>, instances are launched as dedicated tenancy instances by
         *        default. You can only launch instances with a tenancy of <code>dedicated</code> or <code>host</code>
         *        into a dedicated tenancy VPC. </p>
         *        <p>
         *        <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
         *        <code>default</code> or <code>dedicated</code> values only.
         *        </p>
         *        <p>
         *        Default: <code>default</code>
         * @see Tenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tenancy
         */
        Builder instanceTenancy(String instanceTenancy);

        /**
         * <p>
         * The tenancy options for instances launched into the VPC. For <code>default</code>, instances are launched
         * with shared tenancy by default. You can launch instances with any tenancy into a shared tenancy VPC. For
         * <code>dedicated</code>, instances are launched as dedicated tenancy instances by default. You can only launch
         * instances with a tenancy of <code>dedicated</code> or <code>host</code> into a dedicated tenancy VPC.
         * </p>
         * <p>
         * <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
         * <code>default</code> or <code>dedicated</code> values only.
         * </p>
         * <p>
         * Default: <code>default</code>
         * </p>
         * 
         * @param instanceTenancy
         *        The tenancy options for instances launched into the VPC. For <code>default</code>, instances are
         *        launched with shared tenancy by default. You can launch instances with any tenancy into a shared
         *        tenancy VPC. For <code>dedicated</code>, instances are launched as dedicated tenancy instances by
         *        default. You can only launch instances with a tenancy of <code>dedicated</code> or <code>host</code>
         *        into a dedicated tenancy VPC. </p>
         *        <p>
         *        <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
         *        <code>default</code> or <code>dedicated</code> values only.
         *        </p>
         *        <p>
         *        Default: <code>default</code>
         * @see Tenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tenancy
         */
        Builder instanceTenancy(Tenancy instanceTenancy);

        /**
         * <p>
         * Requests an Amazon-provided IPv6 CIDR block with a /56 prefix length for the VPC. You cannot specify the
         * range of IP addresses, or the size of the CIDR block.
         * </p>
         * 
         * @param amazonProvidedIpv6CidrBlock
         *        Requests an Amazon-provided IPv6 CIDR block with a /56 prefix length for the VPC. You cannot specify
         *        the range of IP addresses, or the size of the CIDR block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder amazonProvidedIpv6CidrBlock(Boolean amazonProvidedIpv6CidrBlock);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String cidrBlock;

        private String ipv6Pool;

        private String ipv6CidrBlock;

        private String ipv4IpamPoolId;

        private Integer ipv4NetmaskLength;

        private String ipv6IpamPoolId;

        private Integer ipv6NetmaskLength;

        private String ipv6CidrBlockNetworkBorderGroup;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private Boolean dryRun;

        private String instanceTenancy;

        private Boolean amazonProvidedIpv6CidrBlock;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateVpcRequest model) {
            super(model);
            cidrBlock(model.cidrBlock);
            ipv6Pool(model.ipv6Pool);
            ipv6CidrBlock(model.ipv6CidrBlock);
            ipv4IpamPoolId(model.ipv4IpamPoolId);
            ipv4NetmaskLength(model.ipv4NetmaskLength);
            ipv6IpamPoolId(model.ipv6IpamPoolId);
            ipv6NetmaskLength(model.ipv6NetmaskLength);
            ipv6CidrBlockNetworkBorderGroup(model.ipv6CidrBlockNetworkBorderGroup);
            tagSpecifications(model.tagSpecifications);
            dryRun(model.dryRun);
            instanceTenancy(model.instanceTenancy);
            amazonProvidedIpv6CidrBlock(model.amazonProvidedIpv6CidrBlock);
        }

        public final String getCidrBlock() {
            return cidrBlock;
        }

        public final void setCidrBlock(String cidrBlock) {
            this.cidrBlock = cidrBlock;
        }

        @Override
        public final Builder cidrBlock(String cidrBlock) {
            this.cidrBlock = cidrBlock;
            return this;
        }

        public final String getIpv6Pool() {
            return ipv6Pool;
        }

        public final void setIpv6Pool(String ipv6Pool) {
            this.ipv6Pool = ipv6Pool;
        }

        @Override
        public final Builder ipv6Pool(String ipv6Pool) {
            this.ipv6Pool = ipv6Pool;
            return this;
        }

        public final String getIpv6CidrBlock() {
            return ipv6CidrBlock;
        }

        public final void setIpv6CidrBlock(String ipv6CidrBlock) {
            this.ipv6CidrBlock = ipv6CidrBlock;
        }

        @Override
        public final Builder ipv6CidrBlock(String ipv6CidrBlock) {
            this.ipv6CidrBlock = ipv6CidrBlock;
            return this;
        }

        public final String getIpv4IpamPoolId() {
            return ipv4IpamPoolId;
        }

        public final void setIpv4IpamPoolId(String ipv4IpamPoolId) {
            this.ipv4IpamPoolId = ipv4IpamPoolId;
        }

        @Override
        public final Builder ipv4IpamPoolId(String ipv4IpamPoolId) {
            this.ipv4IpamPoolId = ipv4IpamPoolId;
            return this;
        }

        public final Integer getIpv4NetmaskLength() {
            return ipv4NetmaskLength;
        }

        public final void setIpv4NetmaskLength(Integer ipv4NetmaskLength) {
            this.ipv4NetmaskLength = ipv4NetmaskLength;
        }

        @Override
        public final Builder ipv4NetmaskLength(Integer ipv4NetmaskLength) {
            this.ipv4NetmaskLength = ipv4NetmaskLength;
            return this;
        }

        public final String getIpv6IpamPoolId() {
            return ipv6IpamPoolId;
        }

        public final void setIpv6IpamPoolId(String ipv6IpamPoolId) {
            this.ipv6IpamPoolId = ipv6IpamPoolId;
        }

        @Override
        public final Builder ipv6IpamPoolId(String ipv6IpamPoolId) {
            this.ipv6IpamPoolId = ipv6IpamPoolId;
            return this;
        }

        public final Integer getIpv6NetmaskLength() {
            return ipv6NetmaskLength;
        }

        public final void setIpv6NetmaskLength(Integer ipv6NetmaskLength) {
            this.ipv6NetmaskLength = ipv6NetmaskLength;
        }

        @Override
        public final Builder ipv6NetmaskLength(Integer ipv6NetmaskLength) {
            this.ipv6NetmaskLength = ipv6NetmaskLength;
            return this;
        }

        public final String getIpv6CidrBlockNetworkBorderGroup() {
            return ipv6CidrBlockNetworkBorderGroup;
        }

        public final void setIpv6CidrBlockNetworkBorderGroup(String ipv6CidrBlockNetworkBorderGroup) {
            this.ipv6CidrBlockNetworkBorderGroup = ipv6CidrBlockNetworkBorderGroup;
        }

        @Override
        public final Builder ipv6CidrBlockNetworkBorderGroup(String ipv6CidrBlockNetworkBorderGroup) {
            this.ipv6CidrBlockNetworkBorderGroup = ipv6CidrBlockNetworkBorderGroup;
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        public final String getInstanceTenancy() {
            return instanceTenancy;
        }

        public final void setInstanceTenancy(String instanceTenancy) {
            this.instanceTenancy = instanceTenancy;
        }

        @Override
        public final Builder instanceTenancy(String instanceTenancy) {
            this.instanceTenancy = instanceTenancy;
            return this;
        }

        @Override
        public final Builder instanceTenancy(Tenancy instanceTenancy) {
            this.instanceTenancy(instanceTenancy == null ? null : instanceTenancy.toString());
            return this;
        }

        public final Boolean getAmazonProvidedIpv6CidrBlock() {
            return amazonProvidedIpv6CidrBlock;
        }

        public final void setAmazonProvidedIpv6CidrBlock(Boolean amazonProvidedIpv6CidrBlock) {
            this.amazonProvidedIpv6CidrBlock = amazonProvidedIpv6CidrBlock;
        }

        @Override
        public final Builder amazonProvidedIpv6CidrBlock(Boolean amazonProvidedIpv6CidrBlock) {
            this.amazonProvidedIpv6CidrBlock = amazonProvidedIpv6CidrBlock;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateVpcRequest build() {
            return new CreateVpcRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
