/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * IPAM is a VPC feature that you can use to automate your IP address management workflows including assigning,
 * tracking, troubleshooting, and auditing IP addresses across Amazon Web Services Regions and accounts throughout your
 * Amazon Web Services Organization. For more information, see <a
 * href="https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html">What is IPAM?</a> in the <i>Amazon VPC IPAM
 * User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Ipam implements SdkPojo, Serializable, ToCopyableBuilder<Ipam.Builder, Ipam> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(Ipam::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> IPAM_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamId")
            .getter(getter(Ipam::ipamId))
            .setter(setter(Builder::ipamId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamId")
                    .unmarshallLocationName("ipamId").build()).build();

    private static final SdkField<String> IPAM_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamArn")
            .getter(getter(Ipam::ipamArn))
            .setter(setter(Builder::ipamArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamArn")
                    .unmarshallLocationName("ipamArn").build()).build();

    private static final SdkField<String> IPAM_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamRegion")
            .getter(getter(Ipam::ipamRegion))
            .setter(setter(Builder::ipamRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamRegion")
                    .unmarshallLocationName("ipamRegion").build()).build();

    private static final SdkField<String> PUBLIC_DEFAULT_SCOPE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PublicDefaultScopeId")
            .getter(getter(Ipam::publicDefaultScopeId))
            .setter(setter(Builder::publicDefaultScopeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicDefaultScopeId")
                    .unmarshallLocationName("publicDefaultScopeId").build()).build();

    private static final SdkField<String> PRIVATE_DEFAULT_SCOPE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PrivateDefaultScopeId")
            .getter(getter(Ipam::privateDefaultScopeId))
            .setter(setter(Builder::privateDefaultScopeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateDefaultScopeId")
                    .unmarshallLocationName("privateDefaultScopeId").build()).build();

    private static final SdkField<Integer> SCOPE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ScopeCount")
            .getter(getter(Ipam::scopeCount))
            .setter(setter(Builder::scopeCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScopeCount")
                    .unmarshallLocationName("scopeCount").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(Ipam::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<List<IpamOperatingRegion>> OPERATING_REGIONS_FIELD = SdkField
            .<List<IpamOperatingRegion>> builder(MarshallingType.LIST)
            .memberName("OperatingRegions")
            .getter(getter(Ipam::operatingRegions))
            .setter(setter(Builder::operatingRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingRegionSet")
                    .unmarshallLocationName("operatingRegionSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<IpamOperatingRegion> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpamOperatingRegion::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(Ipam::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Ipam::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> DEFAULT_RESOURCE_DISCOVERY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DefaultResourceDiscoveryId")
            .getter(getter(Ipam::defaultResourceDiscoveryId))
            .setter(setter(Builder::defaultResourceDiscoveryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultResourceDiscoveryId")
                    .unmarshallLocationName("defaultResourceDiscoveryId").build()).build();

    private static final SdkField<String> DEFAULT_RESOURCE_DISCOVERY_ASSOCIATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DefaultResourceDiscoveryAssociationId")
            .getter(getter(Ipam::defaultResourceDiscoveryAssociationId))
            .setter(setter(Builder::defaultResourceDiscoveryAssociationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("DefaultResourceDiscoveryAssociationId")
                    .unmarshallLocationName("defaultResourceDiscoveryAssociationId").build()).build();

    private static final SdkField<Integer> RESOURCE_DISCOVERY_ASSOCIATION_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ResourceDiscoveryAssociationCount")
            .getter(getter(Ipam::resourceDiscoveryAssociationCount))
            .setter(setter(Builder::resourceDiscoveryAssociationCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceDiscoveryAssociationCount")
                    .unmarshallLocationName("resourceDiscoveryAssociationCount").build()).build();

    private static final SdkField<String> STATE_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StateMessage")
            .getter(getter(Ipam::stateMessage))
            .setter(setter(Builder::stateMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateMessage")
                    .unmarshallLocationName("stateMessage").build()).build();

    private static final SdkField<String> TIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Tier")
            .getter(getter(Ipam::tierAsString))
            .setter(setter(Builder::tier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tier")
                    .unmarshallLocationName("tier").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD, IPAM_ID_FIELD,
            IPAM_ARN_FIELD, IPAM_REGION_FIELD, PUBLIC_DEFAULT_SCOPE_ID_FIELD, PRIVATE_DEFAULT_SCOPE_ID_FIELD, SCOPE_COUNT_FIELD,
            DESCRIPTION_FIELD, OPERATING_REGIONS_FIELD, STATE_FIELD, TAGS_FIELD, DEFAULT_RESOURCE_DISCOVERY_ID_FIELD,
            DEFAULT_RESOURCE_DISCOVERY_ASSOCIATION_ID_FIELD, RESOURCE_DISCOVERY_ASSOCIATION_COUNT_FIELD, STATE_MESSAGE_FIELD,
            TIER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final String ipamId;

    private final String ipamArn;

    private final String ipamRegion;

    private final String publicDefaultScopeId;

    private final String privateDefaultScopeId;

    private final Integer scopeCount;

    private final String description;

    private final List<IpamOperatingRegion> operatingRegions;

    private final String state;

    private final List<Tag> tags;

    private final String defaultResourceDiscoveryId;

    private final String defaultResourceDiscoveryAssociationId;

    private final Integer resourceDiscoveryAssociationCount;

    private final String stateMessage;

    private final String tier;

    private Ipam(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.ipamId = builder.ipamId;
        this.ipamArn = builder.ipamArn;
        this.ipamRegion = builder.ipamRegion;
        this.publicDefaultScopeId = builder.publicDefaultScopeId;
        this.privateDefaultScopeId = builder.privateDefaultScopeId;
        this.scopeCount = builder.scopeCount;
        this.description = builder.description;
        this.operatingRegions = builder.operatingRegions;
        this.state = builder.state;
        this.tags = builder.tags;
        this.defaultResourceDiscoveryId = builder.defaultResourceDiscoveryId;
        this.defaultResourceDiscoveryAssociationId = builder.defaultResourceDiscoveryAssociationId;
        this.resourceDiscoveryAssociationCount = builder.resourceDiscoveryAssociationCount;
        this.stateMessage = builder.stateMessage;
        this.tier = builder.tier;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the owner of the IPAM.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the owner of the IPAM.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The ID of the IPAM.
     * </p>
     * 
     * @return The ID of the IPAM.
     */
    public final String ipamId() {
        return ipamId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IPAM.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IPAM.
     */
    public final String ipamArn() {
        return ipamArn;
    }

    /**
     * <p>
     * The Amazon Web Services Region of the IPAM.
     * </p>
     * 
     * @return The Amazon Web Services Region of the IPAM.
     */
    public final String ipamRegion() {
        return ipamRegion;
    }

    /**
     * <p>
     * The ID of the IPAM's default public scope.
     * </p>
     * 
     * @return The ID of the IPAM's default public scope.
     */
    public final String publicDefaultScopeId() {
        return publicDefaultScopeId;
    }

    /**
     * <p>
     * The ID of the IPAM's default private scope.
     * </p>
     * 
     * @return The ID of the IPAM's default private scope.
     */
    public final String privateDefaultScopeId() {
        return privateDefaultScopeId;
    }

    /**
     * <p>
     * The number of scopes in the IPAM. The scope quota is 5. For more information on quotas, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas in IPAM</a> in the <i>Amazon VPC IPAM
     * User Guide</i>.
     * </p>
     * 
     * @return The number of scopes in the IPAM. The scope quota is 5. For more information on quotas, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas in IPAM</a> in the <i>Amazon
     *         VPC IPAM User Guide</i>.
     */
    public final Integer scopeCount() {
        return scopeCount;
    }

    /**
     * <p>
     * The description for the IPAM.
     * </p>
     * 
     * @return The description for the IPAM.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the OperatingRegions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOperatingRegions() {
        return operatingRegions != null && !(operatingRegions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The operating Regions for an IPAM. Operating Regions are Amazon Web Services Regions where the IPAM is allowed to
     * manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web Services Regions you select
     * as operating Regions.
     * </p>
     * <p>
     * For more information about operating Regions, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/create-ipam.html">Create an IPAM</a> in the <i>Amazon VPC IPAM
     * User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOperatingRegions} method.
     * </p>
     * 
     * @return The operating Regions for an IPAM. Operating Regions are Amazon Web Services Regions where the IPAM is
     *         allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web Services
     *         Regions you select as operating Regions.</p>
     *         <p>
     *         For more information about operating Regions, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/create-ipam.html">Create an IPAM</a> in the <i>Amazon
     *         VPC IPAM User Guide</i>.
     */
    public final List<IpamOperatingRegion> operatingRegions() {
        return operatingRegions;
    }

    /**
     * <p>
     * The state of the IPAM.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the IPAM.
     * @see IpamState
     */
    public final IpamState state() {
        return IpamState.fromValue(state);
    }

    /**
     * <p>
     * The state of the IPAM.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the IPAM.
     * @see IpamState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value
     * as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code> and the
     * value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code> for the
     * filter value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the
     *         tag value as the filter value. For example, to find all resources that have a tag with the key
     *         <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name
     *         and <code>TeamA</code> for the filter value.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The IPAM's default resource discovery ID.
     * </p>
     * 
     * @return The IPAM's default resource discovery ID.
     */
    public final String defaultResourceDiscoveryId() {
        return defaultResourceDiscoveryId;
    }

    /**
     * <p>
     * The IPAM's default resource discovery association ID.
     * </p>
     * 
     * @return The IPAM's default resource discovery association ID.
     */
    public final String defaultResourceDiscoveryAssociationId() {
        return defaultResourceDiscoveryAssociationId;
    }

    /**
     * <p>
     * The IPAM's resource discovery association count.
     * </p>
     * 
     * @return The IPAM's resource discovery association count.
     */
    public final Integer resourceDiscoveryAssociationCount() {
        return resourceDiscoveryAssociationCount;
    }

    /**
     * <p>
     * The state message.
     * </p>
     * 
     * @return The state message.
     */
    public final String stateMessage() {
        return stateMessage;
    }

    /**
     * <p>
     * IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available in each
     * tier and the costs associated with the tiers, see <a href="http://aws.amazon.com/vpc/pricing/">Amazon VPC pricing
     * &gt; IPAM tab</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link IpamTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available in
     *         each tier and the costs associated with the tiers, see <a
     *         href="http://aws.amazon.com/vpc/pricing/">Amazon VPC pricing &gt; IPAM tab</a>.
     * @see IpamTier
     */
    public final IpamTier tier() {
        return IpamTier.fromValue(tier);
    }

    /**
     * <p>
     * IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available in each
     * tier and the costs associated with the tiers, see <a href="http://aws.amazon.com/vpc/pricing/">Amazon VPC pricing
     * &gt; IPAM tab</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tier} will return
     * {@link IpamTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tierAsString}.
     * </p>
     * 
     * @return IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available in
     *         each tier and the costs associated with the tiers, see <a
     *         href="http://aws.amazon.com/vpc/pricing/">Amazon VPC pricing &gt; IPAM tab</a>.
     * @see IpamTier
     */
    public final String tierAsString() {
        return tier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamRegion());
        hashCode = 31 * hashCode + Objects.hashCode(publicDefaultScopeId());
        hashCode = 31 * hashCode + Objects.hashCode(privateDefaultScopeId());
        hashCode = 31 * hashCode + Objects.hashCode(scopeCount());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasOperatingRegions() ? operatingRegions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(defaultResourceDiscoveryId());
        hashCode = 31 * hashCode + Objects.hashCode(defaultResourceDiscoveryAssociationId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceDiscoveryAssociationCount());
        hashCode = 31 * hashCode + Objects.hashCode(stateMessage());
        hashCode = 31 * hashCode + Objects.hashCode(tierAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Ipam)) {
            return false;
        }
        Ipam other = (Ipam) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(ipamId(), other.ipamId())
                && Objects.equals(ipamArn(), other.ipamArn()) && Objects.equals(ipamRegion(), other.ipamRegion())
                && Objects.equals(publicDefaultScopeId(), other.publicDefaultScopeId())
                && Objects.equals(privateDefaultScopeId(), other.privateDefaultScopeId())
                && Objects.equals(scopeCount(), other.scopeCount()) && Objects.equals(description(), other.description())
                && hasOperatingRegions() == other.hasOperatingRegions()
                && Objects.equals(operatingRegions(), other.operatingRegions())
                && Objects.equals(stateAsString(), other.stateAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(defaultResourceDiscoveryId(), other.defaultResourceDiscoveryId())
                && Objects.equals(defaultResourceDiscoveryAssociationId(), other.defaultResourceDiscoveryAssociationId())
                && Objects.equals(resourceDiscoveryAssociationCount(), other.resourceDiscoveryAssociationCount())
                && Objects.equals(stateMessage(), other.stateMessage()) && Objects.equals(tierAsString(), other.tierAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Ipam").add("OwnerId", ownerId()).add("IpamId", ipamId()).add("IpamArn", ipamArn())
                .add("IpamRegion", ipamRegion()).add("PublicDefaultScopeId", publicDefaultScopeId())
                .add("PrivateDefaultScopeId", privateDefaultScopeId()).add("ScopeCount", scopeCount())
                .add("Description", description()).add("OperatingRegions", hasOperatingRegions() ? operatingRegions() : null)
                .add("State", stateAsString()).add("Tags", hasTags() ? tags() : null)
                .add("DefaultResourceDiscoveryId", defaultResourceDiscoveryId())
                .add("DefaultResourceDiscoveryAssociationId", defaultResourceDiscoveryAssociationId())
                .add("ResourceDiscoveryAssociationCount", resourceDiscoveryAssociationCount())
                .add("StateMessage", stateMessage()).add("Tier", tierAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "IpamId":
            return Optional.ofNullable(clazz.cast(ipamId()));
        case "IpamArn":
            return Optional.ofNullable(clazz.cast(ipamArn()));
        case "IpamRegion":
            return Optional.ofNullable(clazz.cast(ipamRegion()));
        case "PublicDefaultScopeId":
            return Optional.ofNullable(clazz.cast(publicDefaultScopeId()));
        case "PrivateDefaultScopeId":
            return Optional.ofNullable(clazz.cast(privateDefaultScopeId()));
        case "ScopeCount":
            return Optional.ofNullable(clazz.cast(scopeCount()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OperatingRegions":
            return Optional.ofNullable(clazz.cast(operatingRegions()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "DefaultResourceDiscoveryId":
            return Optional.ofNullable(clazz.cast(defaultResourceDiscoveryId()));
        case "DefaultResourceDiscoveryAssociationId":
            return Optional.ofNullable(clazz.cast(defaultResourceDiscoveryAssociationId()));
        case "ResourceDiscoveryAssociationCount":
            return Optional.ofNullable(clazz.cast(resourceDiscoveryAssociationCount()));
        case "StateMessage":
            return Optional.ofNullable(clazz.cast(stateMessage()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tierAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Ipam, T> g) {
        return obj -> g.apply((Ipam) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Ipam> {
        /**
         * <p>
         * The Amazon Web Services account ID of the owner of the IPAM.
         * </p>
         * 
         * @param ownerId
         *        The Amazon Web Services account ID of the owner of the IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The ID of the IPAM.
         * </p>
         * 
         * @param ipamId
         *        The ID of the IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamId(String ipamId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IPAM.
         * </p>
         * 
         * @param ipamArn
         *        The Amazon Resource Name (ARN) of the IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamArn(String ipamArn);

        /**
         * <p>
         * The Amazon Web Services Region of the IPAM.
         * </p>
         * 
         * @param ipamRegion
         *        The Amazon Web Services Region of the IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamRegion(String ipamRegion);

        /**
         * <p>
         * The ID of the IPAM's default public scope.
         * </p>
         * 
         * @param publicDefaultScopeId
         *        The ID of the IPAM's default public scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicDefaultScopeId(String publicDefaultScopeId);

        /**
         * <p>
         * The ID of the IPAM's default private scope.
         * </p>
         * 
         * @param privateDefaultScopeId
         *        The ID of the IPAM's default private scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateDefaultScopeId(String privateDefaultScopeId);

        /**
         * <p>
         * The number of scopes in the IPAM. The scope quota is 5. For more information on quotas, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas in IPAM</a> in the <i>Amazon VPC
         * IPAM User Guide</i>.
         * </p>
         * 
         * @param scopeCount
         *        The number of scopes in the IPAM. The scope quota is 5. For more information on quotas, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas in IPAM</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopeCount(Integer scopeCount);

        /**
         * <p>
         * The description for the IPAM.
         * </p>
         * 
         * @param description
         *        The description for the IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The operating Regions for an IPAM. Operating Regions are Amazon Web Services Regions where the IPAM is
         * allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web Services
         * Regions you select as operating Regions.
         * </p>
         * <p>
         * For more information about operating Regions, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/create-ipam.html">Create an IPAM</a> in the <i>Amazon VPC
         * IPAM User Guide</i>.
         * </p>
         * 
         * @param operatingRegions
         *        The operating Regions for an IPAM. Operating Regions are Amazon Web Services Regions where the IPAM is
         *        allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web
         *        Services Regions you select as operating Regions.</p>
         *        <p>
         *        For more information about operating Regions, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/create-ipam.html">Create an IPAM</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operatingRegions(Collection<IpamOperatingRegion> operatingRegions);

        /**
         * <p>
         * The operating Regions for an IPAM. Operating Regions are Amazon Web Services Regions where the IPAM is
         * allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web Services
         * Regions you select as operating Regions.
         * </p>
         * <p>
         * For more information about operating Regions, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/create-ipam.html">Create an IPAM</a> in the <i>Amazon VPC
         * IPAM User Guide</i>.
         * </p>
         * 
         * @param operatingRegions
         *        The operating Regions for an IPAM. Operating Regions are Amazon Web Services Regions where the IPAM is
         *        allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web
         *        Services Regions you select as operating Regions.</p>
         *        <p>
         *        For more information about operating Regions, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/create-ipam.html">Create an IPAM</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operatingRegions(IpamOperatingRegion... operatingRegions);

        /**
         * <p>
         * The operating Regions for an IPAM. Operating Regions are Amazon Web Services Regions where the IPAM is
         * allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web Services
         * Regions you select as operating Regions.
         * </p>
         * <p>
         * For more information about operating Regions, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/create-ipam.html">Create an IPAM</a> in the <i>Amazon VPC
         * IPAM User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.IpamOperatingRegion.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.IpamOperatingRegion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.IpamOperatingRegion.Builder#build()} is called immediately
         * and its result is passed to {@link #operatingRegions(List<IpamOperatingRegion>)}.
         * 
         * @param operatingRegions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.IpamOperatingRegion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operatingRegions(java.util.Collection<IpamOperatingRegion>)
         */
        Builder operatingRegions(Consumer<IpamOperatingRegion.Builder>... operatingRegions);

        /**
         * <p>
         * The state of the IPAM.
         * </p>
         * 
         * @param state
         *        The state of the IPAM.
         * @see IpamState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the IPAM.
         * </p>
         * 
         * @param state
         *        The state of the IPAM.
         * @see IpamState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamState
         */
        Builder state(IpamState state);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * 
         * @param tags
         *        The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and
         *        the tag value as the filter value. For example, to find all resources that have a tag with the key
         *        <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter
         *        name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * 
         * @param tags
         *        The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and
         *        the tag value as the filter value. For example, to find all resources that have a tag with the key
         *        <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter
         *        name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The IPAM's default resource discovery ID.
         * </p>
         * 
         * @param defaultResourceDiscoveryId
         *        The IPAM's default resource discovery ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultResourceDiscoveryId(String defaultResourceDiscoveryId);

        /**
         * <p>
         * The IPAM's default resource discovery association ID.
         * </p>
         * 
         * @param defaultResourceDiscoveryAssociationId
         *        The IPAM's default resource discovery association ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultResourceDiscoveryAssociationId(String defaultResourceDiscoveryAssociationId);

        /**
         * <p>
         * The IPAM's resource discovery association count.
         * </p>
         * 
         * @param resourceDiscoveryAssociationCount
         *        The IPAM's resource discovery association count.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceDiscoveryAssociationCount(Integer resourceDiscoveryAssociationCount);

        /**
         * <p>
         * The state message.
         * </p>
         * 
         * @param stateMessage
         *        The state message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateMessage(String stateMessage);

        /**
         * <p>
         * IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available in
         * each tier and the costs associated with the tiers, see <a href="http://aws.amazon.com/vpc/pricing/">Amazon
         * VPC pricing &gt; IPAM tab</a>.
         * </p>
         * 
         * @param tier
         *        IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available
         *        in each tier and the costs associated with the tiers, see <a
         *        href="http://aws.amazon.com/vpc/pricing/">Amazon VPC pricing &gt; IPAM tab</a>.
         * @see IpamTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamTier
         */
        Builder tier(String tier);

        /**
         * <p>
         * IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available in
         * each tier and the costs associated with the tiers, see <a href="http://aws.amazon.com/vpc/pricing/">Amazon
         * VPC pricing &gt; IPAM tab</a>.
         * </p>
         * 
         * @param tier
         *        IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available
         *        in each tier and the costs associated with the tiers, see <a
         *        href="http://aws.amazon.com/vpc/pricing/">Amazon VPC pricing &gt; IPAM tab</a>.
         * @see IpamTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamTier
         */
        Builder tier(IpamTier tier);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String ipamId;

        private String ipamArn;

        private String ipamRegion;

        private String publicDefaultScopeId;

        private String privateDefaultScopeId;

        private Integer scopeCount;

        private String description;

        private List<IpamOperatingRegion> operatingRegions = DefaultSdkAutoConstructList.getInstance();

        private String state;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String defaultResourceDiscoveryId;

        private String defaultResourceDiscoveryAssociationId;

        private Integer resourceDiscoveryAssociationCount;

        private String stateMessage;

        private String tier;

        private BuilderImpl() {
        }

        private BuilderImpl(Ipam model) {
            ownerId(model.ownerId);
            ipamId(model.ipamId);
            ipamArn(model.ipamArn);
            ipamRegion(model.ipamRegion);
            publicDefaultScopeId(model.publicDefaultScopeId);
            privateDefaultScopeId(model.privateDefaultScopeId);
            scopeCount(model.scopeCount);
            description(model.description);
            operatingRegions(model.operatingRegions);
            state(model.state);
            tags(model.tags);
            defaultResourceDiscoveryId(model.defaultResourceDiscoveryId);
            defaultResourceDiscoveryAssociationId(model.defaultResourceDiscoveryAssociationId);
            resourceDiscoveryAssociationCount(model.resourceDiscoveryAssociationCount);
            stateMessage(model.stateMessage);
            tier(model.tier);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getIpamId() {
            return ipamId;
        }

        public final void setIpamId(String ipamId) {
            this.ipamId = ipamId;
        }

        @Override
        public final Builder ipamId(String ipamId) {
            this.ipamId = ipamId;
            return this;
        }

        public final String getIpamArn() {
            return ipamArn;
        }

        public final void setIpamArn(String ipamArn) {
            this.ipamArn = ipamArn;
        }

        @Override
        public final Builder ipamArn(String ipamArn) {
            this.ipamArn = ipamArn;
            return this;
        }

        public final String getIpamRegion() {
            return ipamRegion;
        }

        public final void setIpamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
        }

        @Override
        public final Builder ipamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
            return this;
        }

        public final String getPublicDefaultScopeId() {
            return publicDefaultScopeId;
        }

        public final void setPublicDefaultScopeId(String publicDefaultScopeId) {
            this.publicDefaultScopeId = publicDefaultScopeId;
        }

        @Override
        public final Builder publicDefaultScopeId(String publicDefaultScopeId) {
            this.publicDefaultScopeId = publicDefaultScopeId;
            return this;
        }

        public final String getPrivateDefaultScopeId() {
            return privateDefaultScopeId;
        }

        public final void setPrivateDefaultScopeId(String privateDefaultScopeId) {
            this.privateDefaultScopeId = privateDefaultScopeId;
        }

        @Override
        public final Builder privateDefaultScopeId(String privateDefaultScopeId) {
            this.privateDefaultScopeId = privateDefaultScopeId;
            return this;
        }

        public final Integer getScopeCount() {
            return scopeCount;
        }

        public final void setScopeCount(Integer scopeCount) {
            this.scopeCount = scopeCount;
        }

        @Override
        public final Builder scopeCount(Integer scopeCount) {
            this.scopeCount = scopeCount;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<IpamOperatingRegion.Builder> getOperatingRegions() {
            List<IpamOperatingRegion.Builder> result = IpamOperatingRegionSetCopier.copyToBuilder(this.operatingRegions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOperatingRegions(Collection<IpamOperatingRegion.BuilderImpl> operatingRegions) {
            this.operatingRegions = IpamOperatingRegionSetCopier.copyFromBuilder(operatingRegions);
        }

        @Override
        public final Builder operatingRegions(Collection<IpamOperatingRegion> operatingRegions) {
            this.operatingRegions = IpamOperatingRegionSetCopier.copy(operatingRegions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operatingRegions(IpamOperatingRegion... operatingRegions) {
            operatingRegions(Arrays.asList(operatingRegions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operatingRegions(Consumer<IpamOperatingRegion.Builder>... operatingRegions) {
            operatingRegions(Stream.of(operatingRegions).map(c -> IpamOperatingRegion.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IpamState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getDefaultResourceDiscoveryId() {
            return defaultResourceDiscoveryId;
        }

        public final void setDefaultResourceDiscoveryId(String defaultResourceDiscoveryId) {
            this.defaultResourceDiscoveryId = defaultResourceDiscoveryId;
        }

        @Override
        public final Builder defaultResourceDiscoveryId(String defaultResourceDiscoveryId) {
            this.defaultResourceDiscoveryId = defaultResourceDiscoveryId;
            return this;
        }

        public final String getDefaultResourceDiscoveryAssociationId() {
            return defaultResourceDiscoveryAssociationId;
        }

        public final void setDefaultResourceDiscoveryAssociationId(String defaultResourceDiscoveryAssociationId) {
            this.defaultResourceDiscoveryAssociationId = defaultResourceDiscoveryAssociationId;
        }

        @Override
        public final Builder defaultResourceDiscoveryAssociationId(String defaultResourceDiscoveryAssociationId) {
            this.defaultResourceDiscoveryAssociationId = defaultResourceDiscoveryAssociationId;
            return this;
        }

        public final Integer getResourceDiscoveryAssociationCount() {
            return resourceDiscoveryAssociationCount;
        }

        public final void setResourceDiscoveryAssociationCount(Integer resourceDiscoveryAssociationCount) {
            this.resourceDiscoveryAssociationCount = resourceDiscoveryAssociationCount;
        }

        @Override
        public final Builder resourceDiscoveryAssociationCount(Integer resourceDiscoveryAssociationCount) {
            this.resourceDiscoveryAssociationCount = resourceDiscoveryAssociationCount;
            return this;
        }

        public final String getStateMessage() {
            return stateMessage;
        }

        public final void setStateMessage(String stateMessage) {
            this.stateMessage = stateMessage;
        }

        @Override
        public final Builder stateMessage(String stateMessage) {
            this.stateMessage = stateMessage;
            return this;
        }

        public final String getTier() {
            return tier;
        }

        public final void setTier(String tier) {
            this.tier = tier;
        }

        @Override
        public final Builder tier(String tier) {
            this.tier = tier;
            return this;
        }

        @Override
        public final Builder tier(IpamTier tier) {
            this.tier(tier == null ? null : tier.toString());
            return this;
        }

        @Override
        public Ipam build() {
            return new Ipam(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
