/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a disk image.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DiskImageDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<DiskImageDescription.Builder, DiskImageDescription> {
    private static final SdkField<String> CHECKSUM_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Checksum")
            .getter(getter(DiskImageDescription::checksum))
            .setter(setter(Builder::checksum))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Checksum")
                    .unmarshallLocationName("checksum").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Format")
            .getter(getter(DiskImageDescription::formatAsString))
            .setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format")
                    .unmarshallLocationName("format").build()).build();

    private static final SdkField<String> IMPORT_MANIFEST_URL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ImportManifestUrl")
            .getter(getter(DiskImageDescription::importManifestUrl))
            .setter(setter(Builder::importManifestUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportManifestUrl")
                    .unmarshallLocationName("importManifestUrl").build()).build();

    private static final SdkField<Long> SIZE_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("Size")
            .getter(getter(DiskImageDescription::size))
            .setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Size")
                    .unmarshallLocationName("size").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CHECKSUM_FIELD, FORMAT_FIELD,
            IMPORT_MANIFEST_URL_FIELD, SIZE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String checksum;

    private final String format;

    private final String importManifestUrl;

    private final Long size;

    private DiskImageDescription(BuilderImpl builder) {
        this.checksum = builder.checksum;
        this.format = builder.format;
        this.importManifestUrl = builder.importManifestUrl;
        this.size = builder.size;
    }

    /**
     * <p>
     * The checksum computed for the disk image.
     * </p>
     * 
     * @return The checksum computed for the disk image.
     */
    public final String checksum() {
        return checksum;
    }

    /**
     * <p>
     * The disk image format.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link DiskImageFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The disk image format.
     * @see DiskImageFormat
     */
    public final DiskImageFormat format() {
        return DiskImageFormat.fromValue(format);
    }

    /**
     * <p>
     * The disk image format.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link DiskImageFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The disk image format.
     * @see DiskImageFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * A presigned URL for the import manifest stored in Amazon S3. For information about creating a presigned URL for
     * an Amazon S3 object, read the "Query String Request Authentication Alternative" section of the <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/RESTAuthentication.html">Authenticating REST Requests</a>
     * topic in the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     * <p>
     * For information about the import manifest referenced by this API action, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/manifest.html">VM Import Manifest</a>.
     * </p>
     * 
     * @return A presigned URL for the import manifest stored in Amazon S3. For information about creating a presigned
     *         URL for an Amazon S3 object, read the "Query String Request Authentication Alternative" section of the <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/RESTAuthentication.html">Authenticating REST
     *         Requests</a> topic in the <i>Amazon Simple Storage Service Developer Guide</i>.</p>
     *         <p>
     *         For information about the import manifest referenced by this API action, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/manifest.html">VM Import Manifest</a>.
     */
    public final String importManifestUrl() {
        return importManifestUrl;
    }

    /**
     * <p>
     * The size of the disk image, in GiB.
     * </p>
     * 
     * @return The size of the disk image, in GiB.
     */
    public final Long size() {
        return size;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(checksum());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(importManifestUrl());
        hashCode = 31 * hashCode + Objects.hashCode(size());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DiskImageDescription)) {
            return false;
        }
        DiskImageDescription other = (DiskImageDescription) obj;
        return Objects.equals(checksum(), other.checksum()) && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(importManifestUrl(), other.importManifestUrl()) && Objects.equals(size(), other.size());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DiskImageDescription").add("Checksum", checksum()).add("Format", formatAsString())
                .add("ImportManifestUrl", importManifestUrl() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Size", size()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Checksum":
            return Optional.ofNullable(clazz.cast(checksum()));
        case "Format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "ImportManifestUrl":
            return Optional.ofNullable(clazz.cast(importManifestUrl()));
        case "Size":
            return Optional.ofNullable(clazz.cast(size()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DiskImageDescription, T> g) {
        return obj -> g.apply((DiskImageDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DiskImageDescription> {
        /**
         * <p>
         * The checksum computed for the disk image.
         * </p>
         * 
         * @param checksum
         *        The checksum computed for the disk image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checksum(String checksum);

        /**
         * <p>
         * The disk image format.
         * </p>
         * 
         * @param format
         *        The disk image format.
         * @see DiskImageFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiskImageFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The disk image format.
         * </p>
         * 
         * @param format
         *        The disk image format.
         * @see DiskImageFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiskImageFormat
         */
        Builder format(DiskImageFormat format);

        /**
         * <p>
         * A presigned URL for the import manifest stored in Amazon S3. For information about creating a presigned URL
         * for an Amazon S3 object, read the "Query String Request Authentication Alternative" section of the <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/RESTAuthentication.html">Authenticating REST
         * Requests</a> topic in the <i>Amazon Simple Storage Service Developer Guide</i>.
         * </p>
         * <p>
         * For information about the import manifest referenced by this API action, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/manifest.html">VM Import Manifest</a>.
         * </p>
         * 
         * @param importManifestUrl
         *        A presigned URL for the import manifest stored in Amazon S3. For information about creating a
         *        presigned URL for an Amazon S3 object, read the "Query String Request Authentication Alternative"
         *        section of the <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/RESTAuthentication.html">Authenticating REST
         *        Requests</a> topic in the <i>Amazon Simple Storage Service Developer Guide</i>.</p>
         *        <p>
         *        For information about the import manifest referenced by this API action, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/manifest.html">VM Import Manifest</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importManifestUrl(String importManifestUrl);

        /**
         * <p>
         * The size of the disk image, in GiB.
         * </p>
         * 
         * @param size
         *        The size of the disk image, in GiB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder size(Long size);
    }

    static final class BuilderImpl implements Builder {
        private String checksum;

        private String format;

        private String importManifestUrl;

        private Long size;

        private BuilderImpl() {
        }

        private BuilderImpl(DiskImageDescription model) {
            checksum(model.checksum);
            format(model.format);
            importManifestUrl(model.importManifestUrl);
            size(model.size);
        }

        public final String getChecksum() {
            return checksum;
        }

        public final void setChecksum(String checksum) {
            this.checksum = checksum;
        }

        @Override
        public final Builder checksum(String checksum) {
            this.checksum = checksum;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(DiskImageFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final String getImportManifestUrl() {
            return importManifestUrl;
        }

        public final void setImportManifestUrl(String importManifestUrl) {
            this.importManifestUrl = importManifestUrl;
        }

        @Override
        public final Builder importManifestUrl(String importManifestUrl) {
            this.importManifestUrl = importManifestUrl;
            return this;
        }

        public final Long getSize() {
            return size;
        }

        public final void setSize(Long size) {
            this.size = size;
        }

        @Override
        public final Builder size(Long size) {
            this.size = size;
            return this;
        }

        @Override
        public DiskImageDescription build() {
            return new DiskImageDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
