/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.waiters;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.dynamodb.DynamoDbAsyncClient;
import software.amazon.awssdk.services.dynamodb.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeImportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeImportResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeKinesisStreamingDestinationResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DynamoDbRequest;
import software.amazon.awssdk.services.dynamodb.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultDynamoDbAsyncWaiter implements DynamoDbAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final DynamoDbAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeContributorInsightsResponse> contributorInsightsEnabledWaiter;

    private final AsyncWaiter<DescribeExportResponse> exportCompletedWaiter;

    private final AsyncWaiter<DescribeImportResponse> importCompletedWaiter;

    private final AsyncWaiter<DescribeKinesisStreamingDestinationResponse> kinesisStreamingDestinationActiveWaiter;

    private final AsyncWaiter<DescribeTableResponse> tableExistsWaiter;

    private final AsyncWaiter<DescribeTableResponse> tableNotExistsWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultDynamoDbAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = DynamoDbAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.contributorInsightsEnabledWaiter = AsyncWaiter.builder(DescribeContributorInsightsResponse.class)
                .acceptors(contributorInsightsEnabledWaiterAcceptors())
                .overrideConfiguration(contributorInsightsEnabledWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.exportCompletedWaiter = AsyncWaiter.builder(DescribeExportResponse.class)
                .acceptors(exportCompletedWaiterAcceptors())
                .overrideConfiguration(exportCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.importCompletedWaiter = AsyncWaiter.builder(DescribeImportResponse.class)
                .acceptors(importCompletedWaiterAcceptors())
                .overrideConfiguration(importCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.kinesisStreamingDestinationActiveWaiter = AsyncWaiter.builder(DescribeKinesisStreamingDestinationResponse.class)
                .acceptors(kinesisStreamingDestinationActiveWaiterAcceptors())
                .overrideConfiguration(kinesisStreamingDestinationActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.tableExistsWaiter = AsyncWaiter.builder(DescribeTableResponse.class).acceptors(tableExistsWaiterAcceptors())
                .overrideConfiguration(tableExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.tableNotExistsWaiter = AsyncWaiter.builder(DescribeTableResponse.class).acceptors(tableNotExistsWaiterAcceptors())
                .overrideConfiguration(tableNotExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeContributorInsightsResponse>> waitUntilContributorInsightsEnabled(
            DescribeContributorInsightsRequest describeContributorInsightsRequest) {
        return contributorInsightsEnabledWaiter.runAsync(() -> client
                .describeContributorInsights(applyWaitersUserAgent(describeContributorInsightsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeContributorInsightsResponse>> waitUntilContributorInsightsEnabled(
            DescribeContributorInsightsRequest describeContributorInsightsRequest, WaiterOverrideConfiguration overrideConfig) {
        return contributorInsightsEnabledWaiter.runAsync(
                () -> client.describeContributorInsights(applyWaitersUserAgent(describeContributorInsightsRequest)),
                contributorInsightsEnabledWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeExportResponse>> waitUntilExportCompleted(
            DescribeExportRequest describeExportRequest) {
        return exportCompletedWaiter.runAsync(() -> client.describeExport(applyWaitersUserAgent(describeExportRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeExportResponse>> waitUntilExportCompleted(
            DescribeExportRequest describeExportRequest, WaiterOverrideConfiguration overrideConfig) {
        return exportCompletedWaiter.runAsync(() -> client.describeExport(applyWaitersUserAgent(describeExportRequest)),
                exportCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImportResponse>> waitUntilImportCompleted(
            DescribeImportRequest describeImportRequest) {
        return importCompletedWaiter.runAsync(() -> client.describeImport(applyWaitersUserAgent(describeImportRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImportResponse>> waitUntilImportCompleted(
            DescribeImportRequest describeImportRequest, WaiterOverrideConfiguration overrideConfig) {
        return importCompletedWaiter.runAsync(() -> client.describeImport(applyWaitersUserAgent(describeImportRequest)),
                importCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeKinesisStreamingDestinationResponse>> waitUntilKinesisStreamingDestinationActive(
            DescribeKinesisStreamingDestinationRequest describeKinesisStreamingDestinationRequest) {
        return kinesisStreamingDestinationActiveWaiter.runAsync(() -> client
                .describeKinesisStreamingDestination(applyWaitersUserAgent(describeKinesisStreamingDestinationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeKinesisStreamingDestinationResponse>> waitUntilKinesisStreamingDestinationActive(
            DescribeKinesisStreamingDestinationRequest describeKinesisStreamingDestinationRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return kinesisStreamingDestinationActiveWaiter.runAsync(() -> client
                .describeKinesisStreamingDestination(applyWaitersUserAgent(describeKinesisStreamingDestinationRequest)),
                kinesisStreamingDestinationActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableExists(DescribeTableRequest describeTableRequest) {
        return tableExistsWaiter.runAsync(() -> client.describeTable(applyWaitersUserAgent(describeTableRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableExists(
            DescribeTableRequest describeTableRequest, WaiterOverrideConfiguration overrideConfig) {
        return tableExistsWaiter.runAsync(() -> client.describeTable(applyWaitersUserAgent(describeTableRequest)),
                tableExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableNotExists(
            DescribeTableRequest describeTableRequest) {
        return tableNotExistsWaiter.runAsync(() -> client.describeTable(applyWaitersUserAgent(describeTableRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableNotExists(
            DescribeTableRequest describeTableRequest, WaiterOverrideConfiguration overrideConfig) {
        return tableNotExistsWaiter.runAsync(() -> client.describeTable(applyWaitersUserAgent(describeTableRequest)),
                tableNotExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeContributorInsightsResponse>> contributorInsightsEnabledWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeContributorInsightsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ContributorInsightsStatus").value(), "ENABLED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ContributorInsightsStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ContributorInsightsStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeExportResponse>> exportCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeExportResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ExportDescription").field("ExportStatus").value(), "COMPLETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ExportDescription").field("ExportStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ExportDescription.ExportStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImportResponse>> importCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImportResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ImportTableDescription").field("ImportStatus").value(), "COMPLETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ImportTableDescription").field("ImportStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ImportTableDescription.ImportStatus=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ImportTableDescription").field("ImportStatus").value(), "CANCELLED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ImportTableDescription.ImportStatus=CANCELLED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeKinesisStreamingDestinationResponse>> kinesisStreamingDestinationActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeKinesisStreamingDestinationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("KinesisDataStreamDestinations").flatten().field("DestinationStatus")
                    .values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ACTIVE"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(
                            input.field("KinesisDataStreamDestinations")
                                    .length()
                                    .compare(">", input.constant(new BigDecimal("0")))
                                    .and(input
                                            .field("KinesisDataStreamDestinations")
                                            .filter(x0 -> x0
                                                    .field("DestinationStatus")
                                                    .compare("==", x0.constant("DISABLED"))
                                                    .or(x0.field("DestinationStatus").compare("==", x0.constant("ENABLE_FAILED"))))
                                            .length().compare("==", input.field("KinesisDataStreamDestinations").length()))
                                    .value(), true);
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (length(KinesisDataStreamDestinations) > `0`  && length(KinesisDataStreamDestinations[?DestinationStatus == 'DISABLED' || DestinationStatus == 'ENABLE_FAILED']) ==  length(KinesisDataStreamDestinations)=true) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTableResponse>> tableExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTableResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Table").field("TableStatus").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTableResponse>> tableNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTableResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration contributorInsightsEnabledWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration exportCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration importCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration kinesisStreamingDestinationActiveWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration tableExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(25);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration tableNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(25);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static DynamoDbAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends DynamoDbRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements DynamoDbAsyncWaiter.Builder {
        private DynamoDbAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public DynamoDbAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public DynamoDbAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public DynamoDbAsyncWaiter.Builder client(DynamoDbAsyncClient client) {
            this.client = client;
            return this;
        }

        public DynamoDbAsyncWaiter build() {
            return new DefaultDynamoDbAsyncWaiter(this);
        }
    }
}
