/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the specific reason why a DynamoDB request was throttled and the ARN of the impacted resource. This helps
 * identify exactly what resource is being throttled, what type of operation caused it, and why the throttling occurred.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ThrottlingReason implements SdkPojo, Serializable,
        ToCopyableBuilder<ThrottlingReason.Builder, ThrottlingReason> {
    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("reason")
            .getter(getter(ThrottlingReason::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reason").build()).build();

    private static final SdkField<String> RESOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resource").getter(getter(ThrottlingReason::resource)).setter(setter(Builder::resource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REASON_FIELD, RESOURCE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String reason;

    private final String resource;

    private ThrottlingReason(BuilderImpl builder) {
        this.reason = builder.reason;
        this.resource = builder.resource;
    }

    /**
     * <p>
     * The reason for throttling. The throttling reason follows a specific format:
     * <code>ResourceType+OperationType+LimitType</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Resource Type (What is being throttled): Table or Index
     * </p>
     * </li>
     * <li>
     * <p>
     * Operation Type (What kind of operation): Read or Write
     * </p>
     * </li>
     * <li>
     * <p>
     * Limit Type (Why the throttling occurred):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ProvisionedThroughputExceeded</code>: The request rate is exceeding the <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/provisioned-capacity-mode.html"
     * >provisioned throughput capacity</a> (read or write capacity units) configured for a table or a global secondary
     * index (GSI) in provisioned capacity mode.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AccountLimitExceeded</code>: The request rate has caused a table or global secondary index (GSI) in
     * on-demand mode to exceed the <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ServiceQuotas.html#default-limits-throughput"
     * >per-table account-level service quotas</a> for read/write throughput in the current Amazon Web Services Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyRangeThroughputExceeded</code>: The request rate directed at a specific partition key value has exceeded
     * the <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html">internal
     * partition-level throughput limits</a>, indicating uneven access patterns across the table's or GSI's key space.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MaxOnDemandThroughputExceeded</code>: The request rate has exceeded the <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/on-demand-capacity-mode-max-throughput.html"
     * >configured maximum throughput limits</a> set for a table or index in on-demand capacity mode.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * Examples of complete throttling reasons:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TableReadProvisionedThroughputExceeded
     * </p>
     * </li>
     * <li>
     * <p>
     * IndexWriteAccountLimitExceeded
     * </p>
     * </li>
     * </ul>
     * <p>
     * This helps identify exactly what resource is being throttled, what type of operation caused it, and why the
     * throttling occurred.
     * </p>
     * 
     * @return The reason for throttling. The throttling reason follows a specific format:
     *         <code>ResourceType+OperationType+LimitType</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Resource Type (What is being throttled): Table or Index
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Operation Type (What kind of operation): Read or Write
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Limit Type (Why the throttling occurred):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ProvisionedThroughputExceeded</code>: The request rate is exceeding the <a
     *         href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/provisioned-capacity-mode.html"
     *         >provisioned throughput capacity</a> (read or write capacity units) configured for a table or a global
     *         secondary index (GSI) in provisioned capacity mode.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AccountLimitExceeded</code>: The request rate has caused a table or global secondary index (GSI) in
     *         on-demand mode to exceed the <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ServiceQuotas.html#default-limits-throughput"
     *         >per-table account-level service quotas</a> for read/write throughput in the current Amazon Web Services
     *         Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KeyRangeThroughputExceeded</code>: The request rate directed at a specific partition key value has
     *         exceeded the <a
     *         href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html"
     *         >internal partition-level throughput limits</a>, indicating uneven access patterns across the table's or
     *         GSI's key space.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MaxOnDemandThroughputExceeded</code>: The request rate has exceeded the <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/on-demand-capacity-mode-max-throughput.html"
     *         >configured maximum throughput limits</a> set for a table or index in on-demand capacity mode.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         Examples of complete throttling reasons:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         TableReadProvisionedThroughputExceeded
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IndexWriteAccountLimitExceeded
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This helps identify exactly what resource is being throttled, what type of operation caused it, and why
     *         the throttling occurred.
     */
    public final String reason() {
        return reason;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the DynamoDB table or index that experienced the throttling event.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the DynamoDB table or index that experienced the throttling event.
     */
    public final String resource() {
        return resource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(resource());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ThrottlingReason)) {
            return false;
        }
        ThrottlingReason other = (ThrottlingReason) obj;
        return Objects.equals(reason(), other.reason()) && Objects.equals(resource(), other.resource());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ThrottlingReason").add("Reason", reason()).add("Resource", resource()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "resource":
            return Optional.ofNullable(clazz.cast(resource()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("reason", REASON_FIELD);
        map.put("resource", RESOURCE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ThrottlingReason, T> g) {
        return obj -> g.apply((ThrottlingReason) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ThrottlingReason> {
        /**
         * <p>
         * The reason for throttling. The throttling reason follows a specific format:
         * <code>ResourceType+OperationType+LimitType</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Resource Type (What is being throttled): Table or Index
         * </p>
         * </li>
         * <li>
         * <p>
         * Operation Type (What kind of operation): Read or Write
         * </p>
         * </li>
         * <li>
         * <p>
         * Limit Type (Why the throttling occurred):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProvisionedThroughputExceeded</code>: The request rate is exceeding the <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/provisioned-capacity-mode.html"
         * >provisioned throughput capacity</a> (read or write capacity units) configured for a table or a global
         * secondary index (GSI) in provisioned capacity mode.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AccountLimitExceeded</code>: The request rate has caused a table or global secondary index (GSI) in
         * on-demand mode to exceed the <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ServiceQuotas.html#default-limits-throughput"
         * >per-table account-level service quotas</a> for read/write throughput in the current Amazon Web Services
         * Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeyRangeThroughputExceeded</code>: The request rate directed at a specific partition key value has
         * exceeded the <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html">internal
         * partition-level throughput limits</a>, indicating uneven access patterns across the table's or GSI's key
         * space.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MaxOnDemandThroughputExceeded</code>: The request rate has exceeded the <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/on-demand-capacity-mode-max-throughput.html"
         * >configured maximum throughput limits</a> set for a table or index in on-demand capacity mode.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * Examples of complete throttling reasons:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TableReadProvisionedThroughputExceeded
         * </p>
         * </li>
         * <li>
         * <p>
         * IndexWriteAccountLimitExceeded
         * </p>
         * </li>
         * </ul>
         * <p>
         * This helps identify exactly what resource is being throttled, what type of operation caused it, and why the
         * throttling occurred.
         * </p>
         * 
         * @param reason
         *        The reason for throttling. The throttling reason follows a specific format:
         *        <code>ResourceType+OperationType+LimitType</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Resource Type (What is being throttled): Table or Index
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Operation Type (What kind of operation): Read or Write
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Limit Type (Why the throttling occurred):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProvisionedThroughputExceeded</code>: The request rate is exceeding the <a
         *        href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/provisioned-capacity-mode.html"
         *        >provisioned throughput capacity</a> (read or write capacity units) configured for a table or a global
         *        secondary index (GSI) in provisioned capacity mode.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AccountLimitExceeded</code>: The request rate has caused a table or global secondary index (GSI)
         *        in on-demand mode to exceed the <a href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ServiceQuotas.html#default-limits-throughput"
         *        >per-table account-level service quotas</a> for read/write throughput in the current Amazon Web
         *        Services Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeyRangeThroughputExceeded</code>: The request rate directed at a specific partition key value
         *        has exceeded the <a
         *        href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html"
         *        >internal partition-level throughput limits</a>, indicating uneven access patterns across the table's
         *        or GSI's key space.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MaxOnDemandThroughputExceeded</code>: The request rate has exceeded the <a href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/on-demand-capacity-mode-max-throughput.html"
         *        >configured maximum throughput limits</a> set for a table or index in on-demand capacity mode.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        Examples of complete throttling reasons:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        TableReadProvisionedThroughputExceeded
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IndexWriteAccountLimitExceeded
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This helps identify exactly what resource is being throttled, what type of operation caused it, and
         *        why the throttling occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the DynamoDB table or index that experienced the throttling event.
         * </p>
         * 
         * @param resource
         *        The Amazon Resource Name (ARN) of the DynamoDB table or index that experienced the throttling event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resource(String resource);
    }

    static final class BuilderImpl implements Builder {
        private String reason;

        private String resource;

        private BuilderImpl() {
        }

        private BuilderImpl(ThrottlingReason model) {
            reason(model.reason);
            resource(model.resource);
        }

        public final String getReason() {
            return reason;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final String getResource() {
            return resource;
        }

        public final void setResource(String resource) {
            this.resource = resource;
        }

        @Override
        public final Builder resource(String resource) {
            this.resource = resource;
            return this;
        }

        @Override
        public ThrottlingReason build() {
            return new ThrottlingReason(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
