/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.enhanced.dynamodb.internal.converter;

import java.time.DateTimeException;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.TemporalQuery;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.enhanced.dynamodb.internal.converter.ConverterUtils;
import software.amazon.awssdk.enhanced.dynamodb.internal.converter.TypeConvertingVisitor;
import software.amazon.awssdk.enhanced.dynamodb.internal.converter.attribute.EnhancedAttributeValue;
import software.amazon.awssdk.services.dynamodb.model.AttributeValue;
import software.amazon.awssdk.utils.Validate;

@SdkInternalApi
public final class TimeConversion {
    private static final InstantVisitor INSTANT_VISITOR = new InstantVisitor();
    private static final OffsetDateTimeVisitor OFFSET_DATE_TIME_VISITOR = new OffsetDateTimeVisitor();
    private static final ZonedDateTimeVisitor ZONED_DATE_TIME_VISITOR = new ZonedDateTimeVisitor();

    private TimeConversion() {
    }

    public static AttributeValue toIntegerAttributeValue(Instant instant) {
        String value;
        long instantSeconds = instant.getEpochSecond();
        int nanos = instant.getNano();
        if (nanos == 0) {
            value = Long.toString(instantSeconds);
        } else if (instantSeconds >= 0L) {
            value = instantSeconds + "." + ConverterUtils.padLeft(9, nanos);
        } else {
            nanos = 1000000000 - nanos;
            value = "-" + Math.abs(++instantSeconds) + "." + ConverterUtils.padLeft(9, nanos);
        }
        return (AttributeValue)AttributeValue.builder().n(ConverterUtils.trimNumber(value)).build();
    }

    public static AttributeValue toStringAttributeValue(Instant instant) {
        return (AttributeValue)AttributeValue.builder().s(DateTimeFormatter.ISO_INSTANT.format(instant)).build();
    }

    public static AttributeValue toStringAttributeValue(OffsetDateTime accessor) {
        return (AttributeValue)AttributeValue.builder().s(DateTimeFormatter.ISO_OFFSET_DATE_TIME.format(accessor)).build();
    }

    public static AttributeValue toStringAttributeValue(ZonedDateTime accessor) {
        return (AttributeValue)AttributeValue.builder().s(DateTimeFormatter.ISO_ZONED_DATE_TIME.format(accessor)).build();
    }

    public static Instant instantFromAttributeValue(EnhancedAttributeValue itemAttributeValue) {
        return TimeConversion.convert(itemAttributeValue, INSTANT_VISITOR);
    }

    public static OffsetDateTime offsetDateTimeFromAttributeValue(EnhancedAttributeValue itemAttributeValue) {
        return TimeConversion.convert(itemAttributeValue, OFFSET_DATE_TIME_VISITOR);
    }

    public static ZonedDateTime zonedDateTimeFromAttributeValue(EnhancedAttributeValue itemAttributeValue) {
        return TimeConversion.convert(itemAttributeValue, ZONED_DATE_TIME_VISITOR);
    }

    private static <T> T convert(EnhancedAttributeValue itemAttributeValue, TypeConvertingVisitor<T> visitor) {
        try {
            return itemAttributeValue.convert(visitor);
        }
        catch (DateTimeException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static abstract class BaseVisitor<T>
    extends TypeConvertingVisitor<T> {
        private final TemporalQuery<T> query;

        protected BaseVisitor(Class<T> targetType, TemporalQuery<T> query) {
            super(targetType);
            this.query = query;
        }

        @Override
        public T convertString(String value) {
            return DateTimeFormatter.ISO_ZONED_DATE_TIME.parse((CharSequence)value, this.query);
        }

        @Override
        public T convertNumber(String value) {
            String[] splitOnDecimal = ConverterUtils.splitNumberOnDecimal(value);
            Validate.isTrue((splitOnDecimal[1].length() <= 9 ? 1 : 0) != 0, (String)"Nanoseconds must be expressed in 9 or fewer digits.", (Object[])new Object[0]);
            long epochSecond = splitOnDecimal[0].length() == 0 ? 0L : Long.parseLong(splitOnDecimal[0]);
            int nanoAdjustment = Integer.parseInt(ConverterUtils.padRight(9, splitOnDecimal[1]));
            if (value.startsWith("-")) {
                nanoAdjustment = -nanoAdjustment;
            }
            return this.fromUtcInstant(Instant.ofEpochSecond(epochSecond, nanoAdjustment));
        }

        protected abstract T fromUtcInstant(Instant var1);
    }

    private static final class ZonedDateTimeVisitor
    extends BaseVisitor<ZonedDateTime> {
        protected ZonedDateTimeVisitor() {
            super(ZonedDateTime.class, ZonedDateTime::from);
        }

        @Override
        protected ZonedDateTime fromUtcInstant(Instant instant) {
            return instant.atZone(ZoneOffset.UTC);
        }
    }

    private static final class OffsetDateTimeVisitor
    extends BaseVisitor<OffsetDateTime> {
        protected OffsetDateTimeVisitor() {
            super(OffsetDateTime.class, OffsetDateTime::from);
        }

        @Override
        protected OffsetDateTime fromUtcInstant(Instant instant) {
            return instant.atOffset(ZoneOffset.UTC);
        }
    }

    private static final class InstantVisitor
    extends BaseVisitor<Instant> {
        protected InstantVisitor() {
            super(Instant.class, Instant::from);
        }

        @Override
        public Instant convertString(String value) {
            try {
                return (Instant)super.convertString(value);
            }
            catch (DateTimeParseException e) {
                try {
                    return DateTimeFormatter.ISO_INSTANT.parse((CharSequence)value, Instant::from);
                }
                catch (DateTimeParseException e2) {
                    throw new IllegalArgumentException("Record could not be parsed with either DateTimeFormatter.ISO_ZONED_DATE_TIME (" + e.getMessage() + ") or DateTimeFormatter.ISO_INSTANT (" + e2.getMessage() + ").");
                }
            }
        }

        @Override
        protected Instant fromUtcInstant(Instant instant) {
            return instant;
        }
    }
}

