/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devopsguru.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about a reactive anomaly. This object is returned by <code>DescribeAnomaly.</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReactiveAnomalySummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ReactiveAnomalySummary.Builder, ReactiveAnomalySummary> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(ReactiveAnomalySummary::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(ReactiveAnomalySummary::severityAsString)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ReactiveAnomalySummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<AnomalyTimeRange> ANOMALY_TIME_RANGE_FIELD = SdkField
            .<AnomalyTimeRange> builder(MarshallingType.SDK_POJO).memberName("AnomalyTimeRange")
            .getter(getter(ReactiveAnomalySummary::anomalyTimeRange)).setter(setter(Builder::anomalyTimeRange))
            .constructor(AnomalyTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyTimeRange").build()).build();

    private static final SdkField<AnomalyReportedTimeRange> ANOMALY_REPORTED_TIME_RANGE_FIELD = SdkField
            .<AnomalyReportedTimeRange> builder(MarshallingType.SDK_POJO).memberName("AnomalyReportedTimeRange")
            .getter(getter(ReactiveAnomalySummary::anomalyReportedTimeRange)).setter(setter(Builder::anomalyReportedTimeRange))
            .constructor(AnomalyReportedTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyReportedTimeRange").build())
            .build();

    private static final SdkField<AnomalySourceDetails> SOURCE_DETAILS_FIELD = SdkField
            .<AnomalySourceDetails> builder(MarshallingType.SDK_POJO).memberName("SourceDetails")
            .getter(getter(ReactiveAnomalySummary::sourceDetails)).setter(setter(Builder::sourceDetails))
            .constructor(AnomalySourceDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDetails").build()).build();

    private static final SdkField<String> ASSOCIATED_INSIGHT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociatedInsightId").getter(getter(ReactiveAnomalySummary::associatedInsightId))
            .setter(setter(Builder::associatedInsightId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatedInsightId").build())
            .build();

    private static final SdkField<ResourceCollection> RESOURCE_COLLECTION_FIELD = SdkField
            .<ResourceCollection> builder(MarshallingType.SDK_POJO).memberName("ResourceCollection")
            .getter(getter(ReactiveAnomalySummary::resourceCollection)).setter(setter(Builder::resourceCollection))
            .constructor(ResourceCollection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceCollection").build())
            .build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(ReactiveAnomalySummary::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ReactiveAnomalySummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ReactiveAnomalySummary::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> CAUSAL_ANOMALY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CausalAnomalyId").getter(getter(ReactiveAnomalySummary::causalAnomalyId))
            .setter(setter(Builder::causalAnomalyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CausalAnomalyId").build()).build();

    private static final SdkField<List<AnomalyResource>> ANOMALY_RESOURCES_FIELD = SdkField
            .<List<AnomalyResource>> builder(MarshallingType.LIST)
            .memberName("AnomalyResources")
            .getter(getter(ReactiveAnomalySummary::anomalyResources))
            .setter(setter(Builder::anomalyResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AnomalyResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(AnomalyResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, SEVERITY_FIELD,
            STATUS_FIELD, ANOMALY_TIME_RANGE_FIELD, ANOMALY_REPORTED_TIME_RANGE_FIELD, SOURCE_DETAILS_FIELD,
            ASSOCIATED_INSIGHT_ID_FIELD, RESOURCE_COLLECTION_FIELD, TYPE_FIELD, NAME_FIELD, DESCRIPTION_FIELD,
            CAUSAL_ANOMALY_ID_FIELD, ANOMALY_RESOURCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String severity;

    private final String status;

    private final AnomalyTimeRange anomalyTimeRange;

    private final AnomalyReportedTimeRange anomalyReportedTimeRange;

    private final AnomalySourceDetails sourceDetails;

    private final String associatedInsightId;

    private final ResourceCollection resourceCollection;

    private final String type;

    private final String name;

    private final String description;

    private final String causalAnomalyId;

    private final List<AnomalyResource> anomalyResources;

    private ReactiveAnomalySummary(BuilderImpl builder) {
        this.id = builder.id;
        this.severity = builder.severity;
        this.status = builder.status;
        this.anomalyTimeRange = builder.anomalyTimeRange;
        this.anomalyReportedTimeRange = builder.anomalyReportedTimeRange;
        this.sourceDetails = builder.sourceDetails;
        this.associatedInsightId = builder.associatedInsightId;
        this.resourceCollection = builder.resourceCollection;
        this.type = builder.type;
        this.name = builder.name;
        this.description = builder.description;
        this.causalAnomalyId = builder.causalAnomalyId;
        this.anomalyResources = builder.anomalyResources;
    }

    /**
     * <p>
     * The ID of the reactive anomaly.
     * </p>
     * 
     * @return The ID of the reactive anomaly.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
     * severity. For more information, see <a href=
     * "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
     * >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link AnomalySeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
     *         severity. For more information, see <a href=
     *         "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
     *         >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
     * @see AnomalySeverity
     */
    public final AnomalySeverity severity() {
        return AnomalySeverity.fromValue(severity);
    }

    /**
     * <p>
     * The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
     * severity. For more information, see <a href=
     * "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
     * >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link AnomalySeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
     *         severity. For more information, see <a href=
     *         "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
     *         >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
     * @see AnomalySeverity
     */
    public final String severityAsString() {
        return severity;
    }

    /**
     * <p>
     * The status of the reactive anomaly.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AnomalyStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the reactive anomaly.
     * @see AnomalyStatus
     */
    public final AnomalyStatus status() {
        return AnomalyStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the reactive anomaly.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AnomalyStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the reactive anomaly.
     * @see AnomalyStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * Returns the value of the AnomalyTimeRange property for this object.
     * 
     * @return The value of the AnomalyTimeRange property for this object.
     */
    public final AnomalyTimeRange anomalyTimeRange() {
        return anomalyTimeRange;
    }

    /**
     * <p>
     * An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly is opened
     * and the time when it is closed.
     * </p>
     * 
     * @return An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly is
     *         opened and the time when it is closed.
     */
    public final AnomalyReportedTimeRange anomalyReportedTimeRange() {
        return anomalyReportedTimeRange;
    }

    /**
     * <p>
     * Details about the source of the analyzed operational data that triggered the anomaly. The one supported source is
     * Amazon CloudWatch metrics.
     * </p>
     * 
     * @return Details about the source of the analyzed operational data that triggered the anomaly. The one supported
     *         source is Amazon CloudWatch metrics.
     */
    public final AnomalySourceDetails sourceDetails() {
        return sourceDetails;
    }

    /**
     * <p>
     * The ID of the insight that contains this anomaly. An insight is composed of related anomalies.
     * </p>
     * 
     * @return The ID of the insight that contains this anomaly. An insight is composed of related anomalies.
     */
    public final String associatedInsightId() {
        return associatedInsightId;
    }

    /**
     * Returns the value of the ResourceCollection property for this object.
     * 
     * @return The value of the ResourceCollection property for this object.
     */
    public final ResourceCollection resourceCollection() {
        return resourceCollection;
    }

    /**
     * <p>
     * The type of the reactive anomaly. It can be one of the following types.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CAUSAL</code> - the anomaly can cause a new insight.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONTEXTUAL</code> - the anomaly contains additional information about an insight or its causal anomaly.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link AnomalyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the reactive anomaly. It can be one of the following types.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CAUSAL</code> - the anomaly can cause a new insight.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONTEXTUAL</code> - the anomaly contains additional information about an insight or its causal
     *         anomaly.
     *         </p>
     *         </li>
     * @see AnomalyType
     */
    public final AnomalyType type() {
        return AnomalyType.fromValue(type);
    }

    /**
     * <p>
     * The type of the reactive anomaly. It can be one of the following types.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CAUSAL</code> - the anomaly can cause a new insight.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONTEXTUAL</code> - the anomaly contains additional information about an insight or its causal anomaly.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link AnomalyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the reactive anomaly. It can be one of the following types.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CAUSAL</code> - the anomaly can cause a new insight.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONTEXTUAL</code> - the anomaly contains additional information about an insight or its causal
     *         anomaly.
     *         </p>
     *         </li>
     * @see AnomalyType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The name of the reactive anomaly.
     * </p>
     * 
     * @return The name of the reactive anomaly.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A description of the reactive anomaly.
     * </p>
     * 
     * @return A description of the reactive anomaly.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The ID of the causal anomaly that is associated with this reactive anomaly. The ID of a `CAUSAL` anomaly is
     * always `NULL`.
     * </p>
     * 
     * @return The ID of the causal anomaly that is associated with this reactive anomaly. The ID of a `CAUSAL` anomaly
     *         is always `NULL`.
     */
    public final String causalAnomalyId() {
        return causalAnomalyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the AnomalyResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAnomalyResources() {
        return anomalyResources != null && !(anomalyResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services resources in which anomalous behavior was detected by DevOps Guru.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAnomalyResources} method.
     * </p>
     * 
     * @return The Amazon Web Services resources in which anomalous behavior was detected by DevOps Guru.
     */
    public final List<AnomalyResource> anomalyResources() {
        return anomalyResources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(severityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(anomalyTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(anomalyReportedTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(sourceDetails());
        hashCode = 31 * hashCode + Objects.hashCode(associatedInsightId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceCollection());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(causalAnomalyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAnomalyResources() ? anomalyResources() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReactiveAnomalySummary)) {
            return false;
        }
        ReactiveAnomalySummary other = (ReactiveAnomalySummary) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(severityAsString(), other.severityAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(anomalyTimeRange(), other.anomalyTimeRange())
                && Objects.equals(anomalyReportedTimeRange(), other.anomalyReportedTimeRange())
                && Objects.equals(sourceDetails(), other.sourceDetails())
                && Objects.equals(associatedInsightId(), other.associatedInsightId())
                && Objects.equals(resourceCollection(), other.resourceCollection())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description())
                && Objects.equals(causalAnomalyId(), other.causalAnomalyId())
                && hasAnomalyResources() == other.hasAnomalyResources()
                && Objects.equals(anomalyResources(), other.anomalyResources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReactiveAnomalySummary").add("Id", id()).add("Severity", severityAsString())
                .add("Status", statusAsString()).add("AnomalyTimeRange", anomalyTimeRange())
                .add("AnomalyReportedTimeRange", anomalyReportedTimeRange()).add("SourceDetails", sourceDetails())
                .add("AssociatedInsightId", associatedInsightId()).add("ResourceCollection", resourceCollection())
                .add("Type", typeAsString()).add("Name", name()).add("Description", description())
                .add("CausalAnomalyId", causalAnomalyId())
                .add("AnomalyResources", hasAnomalyResources() ? anomalyResources() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severityAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "AnomalyTimeRange":
            return Optional.ofNullable(clazz.cast(anomalyTimeRange()));
        case "AnomalyReportedTimeRange":
            return Optional.ofNullable(clazz.cast(anomalyReportedTimeRange()));
        case "SourceDetails":
            return Optional.ofNullable(clazz.cast(sourceDetails()));
        case "AssociatedInsightId":
            return Optional.ofNullable(clazz.cast(associatedInsightId()));
        case "ResourceCollection":
            return Optional.ofNullable(clazz.cast(resourceCollection()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "CausalAnomalyId":
            return Optional.ofNullable(clazz.cast(causalAnomalyId()));
        case "AnomalyResources":
            return Optional.ofNullable(clazz.cast(anomalyResources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("Severity", SEVERITY_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("AnomalyTimeRange", ANOMALY_TIME_RANGE_FIELD);
        map.put("AnomalyReportedTimeRange", ANOMALY_REPORTED_TIME_RANGE_FIELD);
        map.put("SourceDetails", SOURCE_DETAILS_FIELD);
        map.put("AssociatedInsightId", ASSOCIATED_INSIGHT_ID_FIELD);
        map.put("ResourceCollection", RESOURCE_COLLECTION_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("CausalAnomalyId", CAUSAL_ANOMALY_ID_FIELD);
        map.put("AnomalyResources", ANOMALY_RESOURCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ReactiveAnomalySummary, T> g) {
        return obj -> g.apply((ReactiveAnomalySummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReactiveAnomalySummary> {
        /**
         * <p>
         * The ID of the reactive anomaly.
         * </p>
         * 
         * @param id
         *        The ID of the reactive anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
         * severity. For more information, see <a href=
         * "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
         * >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
         * </p>
         * 
         * @param severity
         *        The severity of the anomaly. The severity of anomalies that generate an insight determine that
         *        insight's severity. For more information, see <a href=
         *        "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
         *        >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
         * @see AnomalySeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalySeverity
         */
        Builder severity(String severity);

        /**
         * <p>
         * The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
         * severity. For more information, see <a href=
         * "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
         * >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
         * </p>
         * 
         * @param severity
         *        The severity of the anomaly. The severity of anomalies that generate an insight determine that
         *        insight's severity. For more information, see <a href=
         *        "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
         *        >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
         * @see AnomalySeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalySeverity
         */
        Builder severity(AnomalySeverity severity);

        /**
         * <p>
         * The status of the reactive anomaly.
         * </p>
         * 
         * @param status
         *        The status of the reactive anomaly.
         * @see AnomalyStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the reactive anomaly.
         * </p>
         * 
         * @param status
         *        The status of the reactive anomaly.
         * @see AnomalyStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyStatus
         */
        Builder status(AnomalyStatus status);

        /**
         * Sets the value of the AnomalyTimeRange property for this object.
         *
         * @param anomalyTimeRange
         *        The new value for the AnomalyTimeRange property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyTimeRange(AnomalyTimeRange anomalyTimeRange);

        /**
         * Sets the value of the AnomalyTimeRange property for this object.
         *
         * This is a convenience method that creates an instance of the {@link AnomalyTimeRange.Builder} avoiding the
         * need to create one manually via {@link AnomalyTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnomalyTimeRange.Builder#build()} is called immediately and its
         * result is passed to {@link #anomalyTimeRange(AnomalyTimeRange)}.
         * 
         * @param anomalyTimeRange
         *        a consumer that will call methods on {@link AnomalyTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anomalyTimeRange(AnomalyTimeRange)
         */
        default Builder anomalyTimeRange(Consumer<AnomalyTimeRange.Builder> anomalyTimeRange) {
            return anomalyTimeRange(AnomalyTimeRange.builder().applyMutation(anomalyTimeRange).build());
        }

        /**
         * <p>
         * An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly is
         * opened and the time when it is closed.
         * </p>
         * 
         * @param anomalyReportedTimeRange
         *        An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly
         *        is opened and the time when it is closed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyReportedTimeRange(AnomalyReportedTimeRange anomalyReportedTimeRange);

        /**
         * <p>
         * An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly is
         * opened and the time when it is closed.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnomalyReportedTimeRange.Builder}
         * avoiding the need to create one manually via {@link AnomalyReportedTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnomalyReportedTimeRange.Builder#build()} is called immediately
         * and its result is passed to {@link #anomalyReportedTimeRange(AnomalyReportedTimeRange)}.
         * 
         * @param anomalyReportedTimeRange
         *        a consumer that will call methods on {@link AnomalyReportedTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anomalyReportedTimeRange(AnomalyReportedTimeRange)
         */
        default Builder anomalyReportedTimeRange(Consumer<AnomalyReportedTimeRange.Builder> anomalyReportedTimeRange) {
            return anomalyReportedTimeRange(AnomalyReportedTimeRange.builder().applyMutation(anomalyReportedTimeRange).build());
        }

        /**
         * <p>
         * Details about the source of the analyzed operational data that triggered the anomaly. The one supported
         * source is Amazon CloudWatch metrics.
         * </p>
         * 
         * @param sourceDetails
         *        Details about the source of the analyzed operational data that triggered the anomaly. The one
         *        supported source is Amazon CloudWatch metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDetails(AnomalySourceDetails sourceDetails);

        /**
         * <p>
         * Details about the source of the analyzed operational data that triggered the anomaly. The one supported
         * source is Amazon CloudWatch metrics.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnomalySourceDetails.Builder} avoiding
         * the need to create one manually via {@link AnomalySourceDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnomalySourceDetails.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceDetails(AnomalySourceDetails)}.
         * 
         * @param sourceDetails
         *        a consumer that will call methods on {@link AnomalySourceDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceDetails(AnomalySourceDetails)
         */
        default Builder sourceDetails(Consumer<AnomalySourceDetails.Builder> sourceDetails) {
            return sourceDetails(AnomalySourceDetails.builder().applyMutation(sourceDetails).build());
        }

        /**
         * <p>
         * The ID of the insight that contains this anomaly. An insight is composed of related anomalies.
         * </p>
         * 
         * @param associatedInsightId
         *        The ID of the insight that contains this anomaly. An insight is composed of related anomalies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedInsightId(String associatedInsightId);

        /**
         * Sets the value of the ResourceCollection property for this object.
         *
         * @param resourceCollection
         *        The new value for the ResourceCollection property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceCollection(ResourceCollection resourceCollection);

        /**
         * Sets the value of the ResourceCollection property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ResourceCollection.Builder} avoiding the
         * need to create one manually via {@link ResourceCollection#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceCollection.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceCollection(ResourceCollection)}.
         * 
         * @param resourceCollection
         *        a consumer that will call methods on {@link ResourceCollection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceCollection(ResourceCollection)
         */
        default Builder resourceCollection(Consumer<ResourceCollection.Builder> resourceCollection) {
            return resourceCollection(ResourceCollection.builder().applyMutation(resourceCollection).build());
        }

        /**
         * <p>
         * The type of the reactive anomaly. It can be one of the following types.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CAUSAL</code> - the anomaly can cause a new insight.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONTEXTUAL</code> - the anomaly contains additional information about an insight or its causal anomaly.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the reactive anomaly. It can be one of the following types.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CAUSAL</code> - the anomaly can cause a new insight.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONTEXTUAL</code> - the anomaly contains additional information about an insight or its causal
         *        anomaly.
         *        </p>
         *        </li>
         * @see AnomalyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the reactive anomaly. It can be one of the following types.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CAUSAL</code> - the anomaly can cause a new insight.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONTEXTUAL</code> - the anomaly contains additional information about an insight or its causal anomaly.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of the reactive anomaly. It can be one of the following types.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CAUSAL</code> - the anomaly can cause a new insight.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONTEXTUAL</code> - the anomaly contains additional information about an insight or its causal
         *        anomaly.
         *        </p>
         *        </li>
         * @see AnomalyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyType
         */
        Builder type(AnomalyType type);

        /**
         * <p>
         * The name of the reactive anomaly.
         * </p>
         * 
         * @param name
         *        The name of the reactive anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the reactive anomaly.
         * </p>
         * 
         * @param description
         *        A description of the reactive anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ID of the causal anomaly that is associated with this reactive anomaly. The ID of a `CAUSAL` anomaly is
         * always `NULL`.
         * </p>
         * 
         * @param causalAnomalyId
         *        The ID of the causal anomaly that is associated with this reactive anomaly. The ID of a `CAUSAL`
         *        anomaly is always `NULL`.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder causalAnomalyId(String causalAnomalyId);

        /**
         * <p>
         * The Amazon Web Services resources in which anomalous behavior was detected by DevOps Guru.
         * </p>
         * 
         * @param anomalyResources
         *        The Amazon Web Services resources in which anomalous behavior was detected by DevOps Guru.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyResources(Collection<AnomalyResource> anomalyResources);

        /**
         * <p>
         * The Amazon Web Services resources in which anomalous behavior was detected by DevOps Guru.
         * </p>
         * 
         * @param anomalyResources
         *        The Amazon Web Services resources in which anomalous behavior was detected by DevOps Guru.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyResources(AnomalyResource... anomalyResources);

        /**
         * <p>
         * The Amazon Web Services resources in which anomalous behavior was detected by DevOps Guru.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.devopsguru.model.AnomalyResource.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.devopsguru.model.AnomalyResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.devopsguru.model.AnomalyResource.Builder#build()} is called
         * immediately and its result is passed to {@link #anomalyResources(List<AnomalyResource>)}.
         * 
         * @param anomalyResources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.devopsguru.model.AnomalyResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anomalyResources(java.util.Collection<AnomalyResource>)
         */
        Builder anomalyResources(Consumer<AnomalyResource.Builder>... anomalyResources);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String severity;

        private String status;

        private AnomalyTimeRange anomalyTimeRange;

        private AnomalyReportedTimeRange anomalyReportedTimeRange;

        private AnomalySourceDetails sourceDetails;

        private String associatedInsightId;

        private ResourceCollection resourceCollection;

        private String type;

        private String name;

        private String description;

        private String causalAnomalyId;

        private List<AnomalyResource> anomalyResources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ReactiveAnomalySummary model) {
            id(model.id);
            severity(model.severity);
            status(model.status);
            anomalyTimeRange(model.anomalyTimeRange);
            anomalyReportedTimeRange(model.anomalyReportedTimeRange);
            sourceDetails(model.sourceDetails);
            associatedInsightId(model.associatedInsightId);
            resourceCollection(model.resourceCollection);
            type(model.type);
            name(model.name);
            description(model.description);
            causalAnomalyId(model.causalAnomalyId);
            anomalyResources(model.anomalyResources);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        @Override
        public final Builder severity(AnomalySeverity severity) {
            this.severity(severity == null ? null : severity.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AnomalyStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final AnomalyTimeRange.Builder getAnomalyTimeRange() {
            return anomalyTimeRange != null ? anomalyTimeRange.toBuilder() : null;
        }

        public final void setAnomalyTimeRange(AnomalyTimeRange.BuilderImpl anomalyTimeRange) {
            this.anomalyTimeRange = anomalyTimeRange != null ? anomalyTimeRange.build() : null;
        }

        @Override
        public final Builder anomalyTimeRange(AnomalyTimeRange anomalyTimeRange) {
            this.anomalyTimeRange = anomalyTimeRange;
            return this;
        }

        public final AnomalyReportedTimeRange.Builder getAnomalyReportedTimeRange() {
            return anomalyReportedTimeRange != null ? anomalyReportedTimeRange.toBuilder() : null;
        }

        public final void setAnomalyReportedTimeRange(AnomalyReportedTimeRange.BuilderImpl anomalyReportedTimeRange) {
            this.anomalyReportedTimeRange = anomalyReportedTimeRange != null ? anomalyReportedTimeRange.build() : null;
        }

        @Override
        public final Builder anomalyReportedTimeRange(AnomalyReportedTimeRange anomalyReportedTimeRange) {
            this.anomalyReportedTimeRange = anomalyReportedTimeRange;
            return this;
        }

        public final AnomalySourceDetails.Builder getSourceDetails() {
            return sourceDetails != null ? sourceDetails.toBuilder() : null;
        }

        public final void setSourceDetails(AnomalySourceDetails.BuilderImpl sourceDetails) {
            this.sourceDetails = sourceDetails != null ? sourceDetails.build() : null;
        }

        @Override
        public final Builder sourceDetails(AnomalySourceDetails sourceDetails) {
            this.sourceDetails = sourceDetails;
            return this;
        }

        public final String getAssociatedInsightId() {
            return associatedInsightId;
        }

        public final void setAssociatedInsightId(String associatedInsightId) {
            this.associatedInsightId = associatedInsightId;
        }

        @Override
        public final Builder associatedInsightId(String associatedInsightId) {
            this.associatedInsightId = associatedInsightId;
            return this;
        }

        public final ResourceCollection.Builder getResourceCollection() {
            return resourceCollection != null ? resourceCollection.toBuilder() : null;
        }

        public final void setResourceCollection(ResourceCollection.BuilderImpl resourceCollection) {
            this.resourceCollection = resourceCollection != null ? resourceCollection.build() : null;
        }

        @Override
        public final Builder resourceCollection(ResourceCollection resourceCollection) {
            this.resourceCollection = resourceCollection;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(AnomalyType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getCausalAnomalyId() {
            return causalAnomalyId;
        }

        public final void setCausalAnomalyId(String causalAnomalyId) {
            this.causalAnomalyId = causalAnomalyId;
        }

        @Override
        public final Builder causalAnomalyId(String causalAnomalyId) {
            this.causalAnomalyId = causalAnomalyId;
            return this;
        }

        public final List<AnomalyResource.Builder> getAnomalyResources() {
            List<AnomalyResource.Builder> result = AnomalyResourcesCopier.copyToBuilder(this.anomalyResources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAnomalyResources(Collection<AnomalyResource.BuilderImpl> anomalyResources) {
            this.anomalyResources = AnomalyResourcesCopier.copyFromBuilder(anomalyResources);
        }

        @Override
        public final Builder anomalyResources(Collection<AnomalyResource> anomalyResources) {
            this.anomalyResources = AnomalyResourcesCopier.copy(anomalyResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder anomalyResources(AnomalyResource... anomalyResources) {
            anomalyResources(Arrays.asList(anomalyResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder anomalyResources(Consumer<AnomalyResource.Builder>... anomalyResources) {
            anomalyResources(Stream.of(anomalyResources).map(c -> AnomalyResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ReactiveAnomalySummary build() {
            return new ReactiveAnomalySummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
