/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devopsguru.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an anomaly. This object is returned by <code>ListAnomalies</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProactiveAnomaly implements SdkPojo, Serializable,
        ToCopyableBuilder<ProactiveAnomaly.Builder, ProactiveAnomaly> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(ProactiveAnomaly::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(ProactiveAnomaly::severityAsString)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ProactiveAnomaly::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdateTime").getter(getter(ProactiveAnomaly::updateTime)).setter(setter(Builder::updateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateTime").build()).build();

    private static final SdkField<AnomalyTimeRange> ANOMALY_TIME_RANGE_FIELD = SdkField
            .<AnomalyTimeRange> builder(MarshallingType.SDK_POJO).memberName("AnomalyTimeRange")
            .getter(getter(ProactiveAnomaly::anomalyTimeRange)).setter(setter(Builder::anomalyTimeRange))
            .constructor(AnomalyTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyTimeRange").build()).build();

    private static final SdkField<AnomalyReportedTimeRange> ANOMALY_REPORTED_TIME_RANGE_FIELD = SdkField
            .<AnomalyReportedTimeRange> builder(MarshallingType.SDK_POJO).memberName("AnomalyReportedTimeRange")
            .getter(getter(ProactiveAnomaly::anomalyReportedTimeRange)).setter(setter(Builder::anomalyReportedTimeRange))
            .constructor(AnomalyReportedTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyReportedTimeRange").build())
            .build();

    private static final SdkField<PredictionTimeRange> PREDICTION_TIME_RANGE_FIELD = SdkField
            .<PredictionTimeRange> builder(MarshallingType.SDK_POJO).memberName("PredictionTimeRange")
            .getter(getter(ProactiveAnomaly::predictionTimeRange)).setter(setter(Builder::predictionTimeRange))
            .constructor(PredictionTimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PredictionTimeRange").build())
            .build();

    private static final SdkField<AnomalySourceDetails> SOURCE_DETAILS_FIELD = SdkField
            .<AnomalySourceDetails> builder(MarshallingType.SDK_POJO).memberName("SourceDetails")
            .getter(getter(ProactiveAnomaly::sourceDetails)).setter(setter(Builder::sourceDetails))
            .constructor(AnomalySourceDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDetails").build()).build();

    private static final SdkField<String> ASSOCIATED_INSIGHT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociatedInsightId").getter(getter(ProactiveAnomaly::associatedInsightId))
            .setter(setter(Builder::associatedInsightId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatedInsightId").build())
            .build();

    private static final SdkField<ResourceCollection> RESOURCE_COLLECTION_FIELD = SdkField
            .<ResourceCollection> builder(MarshallingType.SDK_POJO).memberName("ResourceCollection")
            .getter(getter(ProactiveAnomaly::resourceCollection)).setter(setter(Builder::resourceCollection))
            .constructor(ResourceCollection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceCollection").build())
            .build();

    private static final SdkField<Double> LIMIT_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Limit")
            .getter(getter(ProactiveAnomaly::limit)).setter(setter(Builder::limit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Limit").build()).build();

    private static final SdkField<AnomalySourceMetadata> SOURCE_METADATA_FIELD = SdkField
            .<AnomalySourceMetadata> builder(MarshallingType.SDK_POJO).memberName("SourceMetadata")
            .getter(getter(ProactiveAnomaly::sourceMetadata)).setter(setter(Builder::sourceMetadata))
            .constructor(AnomalySourceMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceMetadata").build()).build();

    private static final SdkField<List<AnomalyResource>> ANOMALY_RESOURCES_FIELD = SdkField
            .<List<AnomalyResource>> builder(MarshallingType.LIST)
            .memberName("AnomalyResources")
            .getter(getter(ProactiveAnomaly::anomalyResources))
            .setter(setter(Builder::anomalyResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AnomalyResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(AnomalyResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ProactiveAnomaly::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, SEVERITY_FIELD,
            STATUS_FIELD, UPDATE_TIME_FIELD, ANOMALY_TIME_RANGE_FIELD, ANOMALY_REPORTED_TIME_RANGE_FIELD,
            PREDICTION_TIME_RANGE_FIELD, SOURCE_DETAILS_FIELD, ASSOCIATED_INSIGHT_ID_FIELD, RESOURCE_COLLECTION_FIELD,
            LIMIT_FIELD, SOURCE_METADATA_FIELD, ANOMALY_RESOURCES_FIELD, DESCRIPTION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String severity;

    private final String status;

    private final Instant updateTime;

    private final AnomalyTimeRange anomalyTimeRange;

    private final AnomalyReportedTimeRange anomalyReportedTimeRange;

    private final PredictionTimeRange predictionTimeRange;

    private final AnomalySourceDetails sourceDetails;

    private final String associatedInsightId;

    private final ResourceCollection resourceCollection;

    private final Double limit;

    private final AnomalySourceMetadata sourceMetadata;

    private final List<AnomalyResource> anomalyResources;

    private final String description;

    private ProactiveAnomaly(BuilderImpl builder) {
        this.id = builder.id;
        this.severity = builder.severity;
        this.status = builder.status;
        this.updateTime = builder.updateTime;
        this.anomalyTimeRange = builder.anomalyTimeRange;
        this.anomalyReportedTimeRange = builder.anomalyReportedTimeRange;
        this.predictionTimeRange = builder.predictionTimeRange;
        this.sourceDetails = builder.sourceDetails;
        this.associatedInsightId = builder.associatedInsightId;
        this.resourceCollection = builder.resourceCollection;
        this.limit = builder.limit;
        this.sourceMetadata = builder.sourceMetadata;
        this.anomalyResources = builder.anomalyResources;
        this.description = builder.description;
    }

    /**
     * <p>
     * The ID of a proactive anomaly.
     * </p>
     * 
     * @return The ID of a proactive anomaly.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
     * severity. For more information, see <a href=
     * "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
     * >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link AnomalySeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
     *         severity. For more information, see <a href=
     *         "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
     *         >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
     * @see AnomalySeverity
     */
    public final AnomalySeverity severity() {
        return AnomalySeverity.fromValue(severity);
    }

    /**
     * <p>
     * The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
     * severity. For more information, see <a href=
     * "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
     * >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link AnomalySeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
     *         severity. For more information, see <a href=
     *         "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
     *         >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
     * @see AnomalySeverity
     */
    public final String severityAsString() {
        return severity;
    }

    /**
     * <p>
     * The status of a proactive anomaly.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AnomalyStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a proactive anomaly.
     * @see AnomalyStatus
     */
    public final AnomalyStatus status() {
        return AnomalyStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of a proactive anomaly.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AnomalyStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a proactive anomaly.
     * @see AnomalyStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time of the anomaly's most recent update.
     * </p>
     * 
     * @return The time of the anomaly's most recent update.
     */
    public final Instant updateTime() {
        return updateTime;
    }

    /**
     * Returns the value of the AnomalyTimeRange property for this object.
     * 
     * @return The value of the AnomalyTimeRange property for this object.
     */
    public final AnomalyTimeRange anomalyTimeRange() {
        return anomalyTimeRange;
    }

    /**
     * <p>
     * An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly is opened
     * and the time when it is closed.
     * </p>
     * 
     * @return An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly is
     *         opened and the time when it is closed.
     */
    public final AnomalyReportedTimeRange anomalyReportedTimeRange() {
        return anomalyReportedTimeRange;
    }

    /**
     * Returns the value of the PredictionTimeRange property for this object.
     * 
     * @return The value of the PredictionTimeRange property for this object.
     */
    public final PredictionTimeRange predictionTimeRange() {
        return predictionTimeRange;
    }

    /**
     * <p>
     * Details about the source of the analyzed operational data that triggered the anomaly. The one supported source is
     * Amazon CloudWatch metrics.
     * </p>
     * 
     * @return Details about the source of the analyzed operational data that triggered the anomaly. The one supported
     *         source is Amazon CloudWatch metrics.
     */
    public final AnomalySourceDetails sourceDetails() {
        return sourceDetails;
    }

    /**
     * <p>
     * The ID of the insight that contains this anomaly. An insight is composed of related anomalies.
     * </p>
     * 
     * @return The ID of the insight that contains this anomaly. An insight is composed of related anomalies.
     */
    public final String associatedInsightId() {
        return associatedInsightId;
    }

    /**
     * Returns the value of the ResourceCollection property for this object.
     * 
     * @return The value of the ResourceCollection property for this object.
     */
    public final ResourceCollection resourceCollection() {
        return resourceCollection;
    }

    /**
     * <p>
     * A threshold that was exceeded by behavior in analyzed resources. Exceeding this threshold is related to the
     * anomalous behavior that generated this anomaly.
     * </p>
     * 
     * @return A threshold that was exceeded by behavior in analyzed resources. Exceeding this threshold is related to
     *         the anomalous behavior that generated this anomaly.
     */
    public final Double limit() {
        return limit;
    }

    /**
     * <p>
     * The metadata for the anomaly.
     * </p>
     * 
     * @return The metadata for the anomaly.
     */
    public final AnomalySourceMetadata sourceMetadata() {
        return sourceMetadata;
    }

    /**
     * For responses, this returns true if the service returned a value for the AnomalyResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAnomalyResources() {
        return anomalyResources != null && !(anomalyResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about a resource in which DevOps Guru detected anomalous behavior.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAnomalyResources} method.
     * </p>
     * 
     * @return Information about a resource in which DevOps Guru detected anomalous behavior.
     */
    public final List<AnomalyResource> anomalyResources() {
        return anomalyResources;
    }

    /**
     * <p>
     * A description of the proactive anomaly.
     * </p>
     * 
     * @return A description of the proactive anomaly.
     */
    public final String description() {
        return description;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(severityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updateTime());
        hashCode = 31 * hashCode + Objects.hashCode(anomalyTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(anomalyReportedTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(predictionTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(sourceDetails());
        hashCode = 31 * hashCode + Objects.hashCode(associatedInsightId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceCollection());
        hashCode = 31 * hashCode + Objects.hashCode(limit());
        hashCode = 31 * hashCode + Objects.hashCode(sourceMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(hasAnomalyResources() ? anomalyResources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(description());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProactiveAnomaly)) {
            return false;
        }
        ProactiveAnomaly other = (ProactiveAnomaly) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(severityAsString(), other.severityAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(updateTime(), other.updateTime())
                && Objects.equals(anomalyTimeRange(), other.anomalyTimeRange())
                && Objects.equals(anomalyReportedTimeRange(), other.anomalyReportedTimeRange())
                && Objects.equals(predictionTimeRange(), other.predictionTimeRange())
                && Objects.equals(sourceDetails(), other.sourceDetails())
                && Objects.equals(associatedInsightId(), other.associatedInsightId())
                && Objects.equals(resourceCollection(), other.resourceCollection()) && Objects.equals(limit(), other.limit())
                && Objects.equals(sourceMetadata(), other.sourceMetadata())
                && hasAnomalyResources() == other.hasAnomalyResources()
                && Objects.equals(anomalyResources(), other.anomalyResources())
                && Objects.equals(description(), other.description());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProactiveAnomaly").add("Id", id()).add("Severity", severityAsString())
                .add("Status", statusAsString()).add("UpdateTime", updateTime()).add("AnomalyTimeRange", anomalyTimeRange())
                .add("AnomalyReportedTimeRange", anomalyReportedTimeRange()).add("PredictionTimeRange", predictionTimeRange())
                .add("SourceDetails", sourceDetails()).add("AssociatedInsightId", associatedInsightId())
                .add("ResourceCollection", resourceCollection()).add("Limit", limit()).add("SourceMetadata", sourceMetadata())
                .add("AnomalyResources", hasAnomalyResources() ? anomalyResources() : null).add("Description", description())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severityAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "UpdateTime":
            return Optional.ofNullable(clazz.cast(updateTime()));
        case "AnomalyTimeRange":
            return Optional.ofNullable(clazz.cast(anomalyTimeRange()));
        case "AnomalyReportedTimeRange":
            return Optional.ofNullable(clazz.cast(anomalyReportedTimeRange()));
        case "PredictionTimeRange":
            return Optional.ofNullable(clazz.cast(predictionTimeRange()));
        case "SourceDetails":
            return Optional.ofNullable(clazz.cast(sourceDetails()));
        case "AssociatedInsightId":
            return Optional.ofNullable(clazz.cast(associatedInsightId()));
        case "ResourceCollection":
            return Optional.ofNullable(clazz.cast(resourceCollection()));
        case "Limit":
            return Optional.ofNullable(clazz.cast(limit()));
        case "SourceMetadata":
            return Optional.ofNullable(clazz.cast(sourceMetadata()));
        case "AnomalyResources":
            return Optional.ofNullable(clazz.cast(anomalyResources()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("Severity", SEVERITY_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("UpdateTime", UPDATE_TIME_FIELD);
        map.put("AnomalyTimeRange", ANOMALY_TIME_RANGE_FIELD);
        map.put("AnomalyReportedTimeRange", ANOMALY_REPORTED_TIME_RANGE_FIELD);
        map.put("PredictionTimeRange", PREDICTION_TIME_RANGE_FIELD);
        map.put("SourceDetails", SOURCE_DETAILS_FIELD);
        map.put("AssociatedInsightId", ASSOCIATED_INSIGHT_ID_FIELD);
        map.put("ResourceCollection", RESOURCE_COLLECTION_FIELD);
        map.put("Limit", LIMIT_FIELD);
        map.put("SourceMetadata", SOURCE_METADATA_FIELD);
        map.put("AnomalyResources", ANOMALY_RESOURCES_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ProactiveAnomaly, T> g) {
        return obj -> g.apply((ProactiveAnomaly) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProactiveAnomaly> {
        /**
         * <p>
         * The ID of a proactive anomaly.
         * </p>
         * 
         * @param id
         *        The ID of a proactive anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
         * severity. For more information, see <a href=
         * "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
         * >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
         * </p>
         * 
         * @param severity
         *        The severity of the anomaly. The severity of anomalies that generate an insight determine that
         *        insight's severity. For more information, see <a href=
         *        "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
         *        >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
         * @see AnomalySeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalySeverity
         */
        Builder severity(String severity);

        /**
         * <p>
         * The severity of the anomaly. The severity of anomalies that generate an insight determine that insight's
         * severity. For more information, see <a href=
         * "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
         * >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
         * </p>
         * 
         * @param severity
         *        The severity of the anomaly. The severity of anomalies that generate an insight determine that
         *        insight's severity. For more information, see <a href=
         *        "https://docs.aws.amazon.com/devops-guru/latest/userguide/working-with-insights.html#understanding-insights-severities"
         *        >Understanding insight severities</a> in the <i>Amazon DevOps Guru User Guide</i>.
         * @see AnomalySeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalySeverity
         */
        Builder severity(AnomalySeverity severity);

        /**
         * <p>
         * The status of a proactive anomaly.
         * </p>
         * 
         * @param status
         *        The status of a proactive anomaly.
         * @see AnomalyStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a proactive anomaly.
         * </p>
         * 
         * @param status
         *        The status of a proactive anomaly.
         * @see AnomalyStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyStatus
         */
        Builder status(AnomalyStatus status);

        /**
         * <p>
         * The time of the anomaly's most recent update.
         * </p>
         * 
         * @param updateTime
         *        The time of the anomaly's most recent update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateTime(Instant updateTime);

        /**
         * Sets the value of the AnomalyTimeRange property for this object.
         *
         * @param anomalyTimeRange
         *        The new value for the AnomalyTimeRange property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyTimeRange(AnomalyTimeRange anomalyTimeRange);

        /**
         * Sets the value of the AnomalyTimeRange property for this object.
         *
         * This is a convenience method that creates an instance of the {@link AnomalyTimeRange.Builder} avoiding the
         * need to create one manually via {@link AnomalyTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnomalyTimeRange.Builder#build()} is called immediately and its
         * result is passed to {@link #anomalyTimeRange(AnomalyTimeRange)}.
         * 
         * @param anomalyTimeRange
         *        a consumer that will call methods on {@link AnomalyTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anomalyTimeRange(AnomalyTimeRange)
         */
        default Builder anomalyTimeRange(Consumer<AnomalyTimeRange.Builder> anomalyTimeRange) {
            return anomalyTimeRange(AnomalyTimeRange.builder().applyMutation(anomalyTimeRange).build());
        }

        /**
         * <p>
         * An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly is
         * opened and the time when it is closed.
         * </p>
         * 
         * @param anomalyReportedTimeRange
         *        An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly
         *        is opened and the time when it is closed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyReportedTimeRange(AnomalyReportedTimeRange anomalyReportedTimeRange);

        /**
         * <p>
         * An <code>AnomalyReportedTimeRange</code> object that specifies the time range between when the anomaly is
         * opened and the time when it is closed.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnomalyReportedTimeRange.Builder}
         * avoiding the need to create one manually via {@link AnomalyReportedTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnomalyReportedTimeRange.Builder#build()} is called immediately
         * and its result is passed to {@link #anomalyReportedTimeRange(AnomalyReportedTimeRange)}.
         * 
         * @param anomalyReportedTimeRange
         *        a consumer that will call methods on {@link AnomalyReportedTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anomalyReportedTimeRange(AnomalyReportedTimeRange)
         */
        default Builder anomalyReportedTimeRange(Consumer<AnomalyReportedTimeRange.Builder> anomalyReportedTimeRange) {
            return anomalyReportedTimeRange(AnomalyReportedTimeRange.builder().applyMutation(anomalyReportedTimeRange).build());
        }

        /**
         * Sets the value of the PredictionTimeRange property for this object.
         *
         * @param predictionTimeRange
         *        The new value for the PredictionTimeRange property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predictionTimeRange(PredictionTimeRange predictionTimeRange);

        /**
         * Sets the value of the PredictionTimeRange property for this object.
         *
         * This is a convenience method that creates an instance of the {@link PredictionTimeRange.Builder} avoiding the
         * need to create one manually via {@link PredictionTimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PredictionTimeRange.Builder#build()} is called immediately and
         * its result is passed to {@link #predictionTimeRange(PredictionTimeRange)}.
         * 
         * @param predictionTimeRange
         *        a consumer that will call methods on {@link PredictionTimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #predictionTimeRange(PredictionTimeRange)
         */
        default Builder predictionTimeRange(Consumer<PredictionTimeRange.Builder> predictionTimeRange) {
            return predictionTimeRange(PredictionTimeRange.builder().applyMutation(predictionTimeRange).build());
        }

        /**
         * <p>
         * Details about the source of the analyzed operational data that triggered the anomaly. The one supported
         * source is Amazon CloudWatch metrics.
         * </p>
         * 
         * @param sourceDetails
         *        Details about the source of the analyzed operational data that triggered the anomaly. The one
         *        supported source is Amazon CloudWatch metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDetails(AnomalySourceDetails sourceDetails);

        /**
         * <p>
         * Details about the source of the analyzed operational data that triggered the anomaly. The one supported
         * source is Amazon CloudWatch metrics.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnomalySourceDetails.Builder} avoiding
         * the need to create one manually via {@link AnomalySourceDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnomalySourceDetails.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceDetails(AnomalySourceDetails)}.
         * 
         * @param sourceDetails
         *        a consumer that will call methods on {@link AnomalySourceDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceDetails(AnomalySourceDetails)
         */
        default Builder sourceDetails(Consumer<AnomalySourceDetails.Builder> sourceDetails) {
            return sourceDetails(AnomalySourceDetails.builder().applyMutation(sourceDetails).build());
        }

        /**
         * <p>
         * The ID of the insight that contains this anomaly. An insight is composed of related anomalies.
         * </p>
         * 
         * @param associatedInsightId
         *        The ID of the insight that contains this anomaly. An insight is composed of related anomalies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedInsightId(String associatedInsightId);

        /**
         * Sets the value of the ResourceCollection property for this object.
         *
         * @param resourceCollection
         *        The new value for the ResourceCollection property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceCollection(ResourceCollection resourceCollection);

        /**
         * Sets the value of the ResourceCollection property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ResourceCollection.Builder} avoiding the
         * need to create one manually via {@link ResourceCollection#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceCollection.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceCollection(ResourceCollection)}.
         * 
         * @param resourceCollection
         *        a consumer that will call methods on {@link ResourceCollection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceCollection(ResourceCollection)
         */
        default Builder resourceCollection(Consumer<ResourceCollection.Builder> resourceCollection) {
            return resourceCollection(ResourceCollection.builder().applyMutation(resourceCollection).build());
        }

        /**
         * <p>
         * A threshold that was exceeded by behavior in analyzed resources. Exceeding this threshold is related to the
         * anomalous behavior that generated this anomaly.
         * </p>
         * 
         * @param limit
         *        A threshold that was exceeded by behavior in analyzed resources. Exceeding this threshold is related
         *        to the anomalous behavior that generated this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limit(Double limit);

        /**
         * <p>
         * The metadata for the anomaly.
         * </p>
         * 
         * @param sourceMetadata
         *        The metadata for the anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceMetadata(AnomalySourceMetadata sourceMetadata);

        /**
         * <p>
         * The metadata for the anomaly.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnomalySourceMetadata.Builder} avoiding
         * the need to create one manually via {@link AnomalySourceMetadata#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnomalySourceMetadata.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceMetadata(AnomalySourceMetadata)}.
         * 
         * @param sourceMetadata
         *        a consumer that will call methods on {@link AnomalySourceMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceMetadata(AnomalySourceMetadata)
         */
        default Builder sourceMetadata(Consumer<AnomalySourceMetadata.Builder> sourceMetadata) {
            return sourceMetadata(AnomalySourceMetadata.builder().applyMutation(sourceMetadata).build());
        }

        /**
         * <p>
         * Information about a resource in which DevOps Guru detected anomalous behavior.
         * </p>
         * 
         * @param anomalyResources
         *        Information about a resource in which DevOps Guru detected anomalous behavior.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyResources(Collection<AnomalyResource> anomalyResources);

        /**
         * <p>
         * Information about a resource in which DevOps Guru detected anomalous behavior.
         * </p>
         * 
         * @param anomalyResources
         *        Information about a resource in which DevOps Guru detected anomalous behavior.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyResources(AnomalyResource... anomalyResources);

        /**
         * <p>
         * Information about a resource in which DevOps Guru detected anomalous behavior.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.devopsguru.model.AnomalyResource.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.devopsguru.model.AnomalyResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.devopsguru.model.AnomalyResource.Builder#build()} is called
         * immediately and its result is passed to {@link #anomalyResources(List<AnomalyResource>)}.
         * 
         * @param anomalyResources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.devopsguru.model.AnomalyResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anomalyResources(java.util.Collection<AnomalyResource>)
         */
        Builder anomalyResources(Consumer<AnomalyResource.Builder>... anomalyResources);

        /**
         * <p>
         * A description of the proactive anomaly.
         * </p>
         * 
         * @param description
         *        A description of the proactive anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String severity;

        private String status;

        private Instant updateTime;

        private AnomalyTimeRange anomalyTimeRange;

        private AnomalyReportedTimeRange anomalyReportedTimeRange;

        private PredictionTimeRange predictionTimeRange;

        private AnomalySourceDetails sourceDetails;

        private String associatedInsightId;

        private ResourceCollection resourceCollection;

        private Double limit;

        private AnomalySourceMetadata sourceMetadata;

        private List<AnomalyResource> anomalyResources = DefaultSdkAutoConstructList.getInstance();

        private String description;

        private BuilderImpl() {
        }

        private BuilderImpl(ProactiveAnomaly model) {
            id(model.id);
            severity(model.severity);
            status(model.status);
            updateTime(model.updateTime);
            anomalyTimeRange(model.anomalyTimeRange);
            anomalyReportedTimeRange(model.anomalyReportedTimeRange);
            predictionTimeRange(model.predictionTimeRange);
            sourceDetails(model.sourceDetails);
            associatedInsightId(model.associatedInsightId);
            resourceCollection(model.resourceCollection);
            limit(model.limit);
            sourceMetadata(model.sourceMetadata);
            anomalyResources(model.anomalyResources);
            description(model.description);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        @Override
        public final Builder severity(AnomalySeverity severity) {
            this.severity(severity == null ? null : severity.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AnomalyStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getUpdateTime() {
            return updateTime;
        }

        public final void setUpdateTime(Instant updateTime) {
            this.updateTime = updateTime;
        }

        @Override
        public final Builder updateTime(Instant updateTime) {
            this.updateTime = updateTime;
            return this;
        }

        public final AnomalyTimeRange.Builder getAnomalyTimeRange() {
            return anomalyTimeRange != null ? anomalyTimeRange.toBuilder() : null;
        }

        public final void setAnomalyTimeRange(AnomalyTimeRange.BuilderImpl anomalyTimeRange) {
            this.anomalyTimeRange = anomalyTimeRange != null ? anomalyTimeRange.build() : null;
        }

        @Override
        public final Builder anomalyTimeRange(AnomalyTimeRange anomalyTimeRange) {
            this.anomalyTimeRange = anomalyTimeRange;
            return this;
        }

        public final AnomalyReportedTimeRange.Builder getAnomalyReportedTimeRange() {
            return anomalyReportedTimeRange != null ? anomalyReportedTimeRange.toBuilder() : null;
        }

        public final void setAnomalyReportedTimeRange(AnomalyReportedTimeRange.BuilderImpl anomalyReportedTimeRange) {
            this.anomalyReportedTimeRange = anomalyReportedTimeRange != null ? anomalyReportedTimeRange.build() : null;
        }

        @Override
        public final Builder anomalyReportedTimeRange(AnomalyReportedTimeRange anomalyReportedTimeRange) {
            this.anomalyReportedTimeRange = anomalyReportedTimeRange;
            return this;
        }

        public final PredictionTimeRange.Builder getPredictionTimeRange() {
            return predictionTimeRange != null ? predictionTimeRange.toBuilder() : null;
        }

        public final void setPredictionTimeRange(PredictionTimeRange.BuilderImpl predictionTimeRange) {
            this.predictionTimeRange = predictionTimeRange != null ? predictionTimeRange.build() : null;
        }

        @Override
        public final Builder predictionTimeRange(PredictionTimeRange predictionTimeRange) {
            this.predictionTimeRange = predictionTimeRange;
            return this;
        }

        public final AnomalySourceDetails.Builder getSourceDetails() {
            return sourceDetails != null ? sourceDetails.toBuilder() : null;
        }

        public final void setSourceDetails(AnomalySourceDetails.BuilderImpl sourceDetails) {
            this.sourceDetails = sourceDetails != null ? sourceDetails.build() : null;
        }

        @Override
        public final Builder sourceDetails(AnomalySourceDetails sourceDetails) {
            this.sourceDetails = sourceDetails;
            return this;
        }

        public final String getAssociatedInsightId() {
            return associatedInsightId;
        }

        public final void setAssociatedInsightId(String associatedInsightId) {
            this.associatedInsightId = associatedInsightId;
        }

        @Override
        public final Builder associatedInsightId(String associatedInsightId) {
            this.associatedInsightId = associatedInsightId;
            return this;
        }

        public final ResourceCollection.Builder getResourceCollection() {
            return resourceCollection != null ? resourceCollection.toBuilder() : null;
        }

        public final void setResourceCollection(ResourceCollection.BuilderImpl resourceCollection) {
            this.resourceCollection = resourceCollection != null ? resourceCollection.build() : null;
        }

        @Override
        public final Builder resourceCollection(ResourceCollection resourceCollection) {
            this.resourceCollection = resourceCollection;
            return this;
        }

        public final Double getLimit() {
            return limit;
        }

        public final void setLimit(Double limit) {
            this.limit = limit;
        }

        @Override
        public final Builder limit(Double limit) {
            this.limit = limit;
            return this;
        }

        public final AnomalySourceMetadata.Builder getSourceMetadata() {
            return sourceMetadata != null ? sourceMetadata.toBuilder() : null;
        }

        public final void setSourceMetadata(AnomalySourceMetadata.BuilderImpl sourceMetadata) {
            this.sourceMetadata = sourceMetadata != null ? sourceMetadata.build() : null;
        }

        @Override
        public final Builder sourceMetadata(AnomalySourceMetadata sourceMetadata) {
            this.sourceMetadata = sourceMetadata;
            return this;
        }

        public final List<AnomalyResource.Builder> getAnomalyResources() {
            List<AnomalyResource.Builder> result = AnomalyResourcesCopier.copyToBuilder(this.anomalyResources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAnomalyResources(Collection<AnomalyResource.BuilderImpl> anomalyResources) {
            this.anomalyResources = AnomalyResourcesCopier.copyFromBuilder(anomalyResources);
        }

        @Override
        public final Builder anomalyResources(Collection<AnomalyResource> anomalyResources) {
            this.anomalyResources = AnomalyResourcesCopier.copy(anomalyResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder anomalyResources(AnomalyResource... anomalyResources) {
            anomalyResources(Arrays.asList(anomalyResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder anomalyResources(Consumer<AnomalyResource.Builder>... anomalyResources) {
            anomalyResources(Stream.of(anomalyResources).map(c -> AnomalyResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        @Override
        public ProactiveAnomaly build() {
            return new ProactiveAnomaly(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
