/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devopsguru;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.devopsguru.internal.DevOpsGuruServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.devopsguru.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.devopsguru.model.AccessDeniedException;
import software.amazon.awssdk.services.devopsguru.model.AddNotificationChannelRequest;
import software.amazon.awssdk.services.devopsguru.model.AddNotificationChannelResponse;
import software.amazon.awssdk.services.devopsguru.model.ConflictException;
import software.amazon.awssdk.services.devopsguru.model.DeleteInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.DeleteInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountOverviewRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountOverviewResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAnomalyRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAnomalyResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeEventSourcesConfigRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeEventSourcesConfigResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeFeedbackRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeFeedbackResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationOverviewRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationOverviewResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationResourceCollectionHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationResourceCollectionHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeServiceIntegrationRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeServiceIntegrationResponse;
import software.amazon.awssdk.services.devopsguru.model.DevOpsGuruException;
import software.amazon.awssdk.services.devopsguru.model.GetCostEstimationRequest;
import software.amazon.awssdk.services.devopsguru.model.GetCostEstimationResponse;
import software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionRequest;
import software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionResponse;
import software.amazon.awssdk.services.devopsguru.model.InternalServerException;
import software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.ListAnomalousLogGroupsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListAnomalousLogGroupsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListEventsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListEventsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListMonitoredResourcesRequest;
import software.amazon.awssdk.services.devopsguru.model.ListMonitoredResourcesResponse;
import software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListOrganizationInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListOrganizationInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListRecommendationsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListRecommendationsResponse;
import software.amazon.awssdk.services.devopsguru.model.PutFeedbackRequest;
import software.amazon.awssdk.services.devopsguru.model.PutFeedbackResponse;
import software.amazon.awssdk.services.devopsguru.model.RemoveNotificationChannelRequest;
import software.amazon.awssdk.services.devopsguru.model.RemoveNotificationChannelResponse;
import software.amazon.awssdk.services.devopsguru.model.ResourceNotFoundException;
import software.amazon.awssdk.services.devopsguru.model.SearchInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.SearchInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.SearchOrganizationInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.SearchOrganizationInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.devopsguru.model.StartCostEstimationRequest;
import software.amazon.awssdk.services.devopsguru.model.StartCostEstimationResponse;
import software.amazon.awssdk.services.devopsguru.model.ThrottlingException;
import software.amazon.awssdk.services.devopsguru.model.UpdateEventSourcesConfigRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateEventSourcesConfigResponse;
import software.amazon.awssdk.services.devopsguru.model.UpdateResourceCollectionRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateResourceCollectionResponse;
import software.amazon.awssdk.services.devopsguru.model.UpdateServiceIntegrationRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateServiceIntegrationResponse;
import software.amazon.awssdk.services.devopsguru.model.ValidationException;
import software.amazon.awssdk.services.devopsguru.transform.AddNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DeleteInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAccountHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAccountOverviewRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAnomalyRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeEventSourcesConfigRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeFeedbackRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeOrganizationHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeOrganizationOverviewRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeOrganizationResourceCollectionHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeResourceCollectionHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeServiceIntegrationRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.GetCostEstimationRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.GetResourceCollectionRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListAnomaliesForInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListAnomalousLogGroupsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListEventsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListMonitoredResourcesRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListNotificationChannelsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListOrganizationInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.PutFeedbackRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.RemoveNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.SearchInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.SearchOrganizationInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.StartCostEstimationRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateEventSourcesConfigRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateResourceCollectionRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateServiceIntegrationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link DevOpsGuruClient}.
 *
 * @see DevOpsGuruClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDevOpsGuruClient implements DevOpsGuruClient {
    private static final Logger log = Logger.loggerFor(DefaultDevOpsGuruClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDevOpsGuruClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "DevOps_Guru" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds a notification channel to DevOps Guru. A notification channel is used to notify you about important DevOps
     * Guru events, such as when an insight is generated.
     * </p>
     * <p>
     * If you use an Amazon SNS topic in another account, you must attach a policy to it that grants DevOps Guru
     * permission to send it notifications. DevOps Guru adds the required policy on your behalf to send notifications
     * using Amazon SNS in your account. DevOps Guru only supports standard SNS topics. For more information, see <a
     * href="https://docs.aws.amazon.com/devops-guru/latest/userguide/sns-required-permissions.html">Permissions for
     * Amazon SNS topics</a>.
     * </p>
     * <p>
     * If you use an Amazon SNS topic that is encrypted by an Amazon Web Services Key Management Service
     * customer-managed key (CMK), then you must add permissions to the CMK. For more information, see <a
     * href="https://docs.aws.amazon.com/devops-guru/latest/userguide/sns-kms-permissions.html">Permissions for Amazon
     * Web Services KMS–encrypted Amazon SNS topics</a>.
     * </p>
     *
     * @param addNotificationChannelRequest
     * @return Result of the AddNotificationChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ServiceQuotaExceededException
     *         The request contains a value that exceeds a maximum quota.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.AddNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/AddNotificationChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddNotificationChannelResponse addNotificationChannel(AddNotificationChannelRequest addNotificationChannelRequest)
            throws AccessDeniedException, ConflictException, InternalServerException, ResourceNotFoundException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddNotificationChannelResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddNotificationChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<AddNotificationChannelRequest, AddNotificationChannelResponse>()
                            .withOperationName("AddNotificationChannel").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(addNotificationChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AddNotificationChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the insight along with the associated anomalies, events and recommendations.
     * </p>
     *
     * @param deleteInsightRequest
     * @return Result of the DeleteInsight operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DeleteInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DeleteInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteInsightResponse deleteInsight(DeleteInsightRequest deleteInsightRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ConflictException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteInsightResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInsightRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInsight");

            return clientHandler.execute(new ClientExecutionParams<DeleteInsightRequest, DeleteInsightResponse>()
                    .withOperationName("DeleteInsight").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteInsightRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteInsightRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of open reactive insights, the number of open proactive insights, and the number of metrics
     * analyzed in your Amazon Web Services account. Use these numbers to gauge the health of operations in your Amazon
     * Web Services account.
     * </p>
     *
     * @param describeAccountHealthRequest
     * @return Result of the DescribeAccountHealth operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeAccountHealth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAccountHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccountHealthResponse describeAccountHealth(DescribeAccountHealthRequest describeAccountHealthRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAccountHealthResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAccountHealthResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAccountHealthRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountHealthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountHealth");

            return clientHandler.execute(new ClientExecutionParams<DescribeAccountHealthRequest, DescribeAccountHealthResponse>()
                    .withOperationName("DescribeAccountHealth").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeAccountHealthRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAccountHealthRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For the time range passed in, returns the number of open reactive insight that were created, the number of open
     * proactive insights that were created, and the Mean Time to Recover (MTTR) for all closed reactive insights.
     * </p>
     *
     * @param describeAccountOverviewRequest
     * @return Result of the DescribeAccountOverview operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeAccountOverview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAccountOverview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccountOverviewResponse describeAccountOverview(DescribeAccountOverviewRequest describeAccountOverviewRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAccountOverviewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAccountOverviewResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAccountOverviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountOverviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountOverview");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountOverviewRequest, DescribeAccountOverviewResponse>()
                            .withOperationName("DescribeAccountOverview").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeAccountOverviewRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAccountOverviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about an anomaly that you specify using its ID.
     * </p>
     *
     * @param describeAnomalyRequest
     * @return Result of the DescribeAnomaly operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeAnomaly
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAnomaly" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeAnomalyResponse describeAnomaly(DescribeAnomalyRequest describeAnomalyRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAnomalyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAnomalyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAnomalyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAnomalyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnomaly");

            return clientHandler.execute(new ClientExecutionParams<DescribeAnomalyRequest, DescribeAnomalyResponse>()
                    .withOperationName("DescribeAnomaly").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeAnomalyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAnomalyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the integration status of services that are integrated with DevOps Guru as Consumer via EventBridge. The
     * one service that can be integrated with DevOps Guru is Amazon CodeGuru Profiler, which can produce proactive
     * recommendations which can be stored and viewed in DevOps Guru.
     * </p>
     *
     * @param describeEventSourcesConfigRequest
     * @return Result of the DescribeEventSourcesConfig operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeEventSourcesConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeEventSourcesConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEventSourcesConfigResponse describeEventSourcesConfig(
            DescribeEventSourcesConfigRequest describeEventSourcesConfigRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEventSourcesConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEventSourcesConfigResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEventSourcesConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventSourcesConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventSourcesConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeEventSourcesConfigRequest, DescribeEventSourcesConfigResponse>()
                            .withOperationName("DescribeEventSourcesConfig").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeEventSourcesConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeEventSourcesConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the most recent feedback submitted in the current Amazon Web Services account and Region.
     * </p>
     *
     * @param describeFeedbackRequest
     * @return Result of the DescribeFeedback operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeFeedback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeFeedback" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeFeedbackResponse describeFeedback(DescribeFeedbackRequest describeFeedbackRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeFeedbackResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFeedbackRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFeedback");

            return clientHandler.execute(new ClientExecutionParams<DescribeFeedbackRequest, DescribeFeedbackResponse>()
                    .withOperationName("DescribeFeedback").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeFeedbackRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFeedbackRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about an insight that you specify using its ID.
     * </p>
     *
     * @param describeInsightRequest
     * @return Result of the DescribeInsight operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeInsight" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeInsightResponse describeInsight(DescribeInsightRequest describeInsightRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeInsightResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeInsightRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInsight");

            return clientHandler.execute(new ClientExecutionParams<DescribeInsightRequest, DescribeInsightResponse>()
                    .withOperationName("DescribeInsight").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeInsightRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeInsightRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns active insights, predictive insights, and resource hours analyzed in last hour.
     * </p>
     *
     * @param describeOrganizationHealthRequest
     * @return Result of the DescribeOrganizationHealth operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeOrganizationHealth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeOrganizationHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeOrganizationHealthResponse describeOrganizationHealth(
            DescribeOrganizationHealthRequest describeOrganizationHealthRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeOrganizationHealthResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeOrganizationHealthResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeOrganizationHealthRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOrganizationHealthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationHealth");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationHealthRequest, DescribeOrganizationHealthResponse>()
                            .withOperationName("DescribeOrganizationHealth").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeOrganizationHealthRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeOrganizationHealthRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an overview of your organization's history based on the specified time range. The overview includes the
     * total reactive and proactive insights.
     * </p>
     *
     * @param describeOrganizationOverviewRequest
     * @return Result of the DescribeOrganizationOverview operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeOrganizationOverview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeOrganizationOverview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeOrganizationOverviewResponse describeOrganizationOverview(
            DescribeOrganizationOverviewRequest describeOrganizationOverviewRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeOrganizationOverviewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeOrganizationOverviewResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeOrganizationOverviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOrganizationOverviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationOverview");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationOverviewRequest, DescribeOrganizationOverviewResponse>()
                            .withOperationName("DescribeOrganizationOverview").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeOrganizationOverviewRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeOrganizationOverviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides an overview of your system's health. If additional member accounts are part of your organization, you
     * can filter those accounts using the <code>AccountIds</code> field.
     * </p>
     *
     * @param describeOrganizationResourceCollectionHealthRequest
     * @return Result of the DescribeOrganizationResourceCollectionHealth operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeOrganizationResourceCollectionHealth
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeOrganizationResourceCollectionHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeOrganizationResourceCollectionHealthResponse describeOrganizationResourceCollectionHealth(
            DescribeOrganizationResourceCollectionHealthRequest describeOrganizationResourceCollectionHealthRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeOrganizationResourceCollectionHealthResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeOrganizationResourceCollectionHealthResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                describeOrganizationResourceCollectionHealthRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeOrganizationResourceCollectionHealthRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationResourceCollectionHealth");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationResourceCollectionHealthRequest, DescribeOrganizationResourceCollectionHealthResponse>()
                            .withOperationName("DescribeOrganizationResourceCollectionHealth")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(describeOrganizationResourceCollectionHealthRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeOrganizationResourceCollectionHealthRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of open proactive insights, open reactive insights, and the Mean Time to Recover (MTTR) for
     * all closed insights in resource collections in your account. You specify the type of Amazon Web Services
     * resources collection. The two types of Amazon Web Services resource collections supported are Amazon Web Services
     * CloudFormation stacks and Amazon Web Services resources that contain the same Amazon Web Services tag. DevOps
     * Guru can be configured to analyze the Amazon Web Services resources that are defined in the stacks or that are
     * tagged using the same tag <i>key</i>. You can specify up to 500 Amazon Web Services CloudFormation stacks.
     * </p>
     *
     * @param describeResourceCollectionHealthRequest
     * @return Result of the DescribeResourceCollectionHealth operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeResourceCollectionHealth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeResourceCollectionHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeResourceCollectionHealthResponse describeResourceCollectionHealth(
            DescribeResourceCollectionHealthRequest describeResourceCollectionHealthRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeResourceCollectionHealthResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeResourceCollectionHealthResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeResourceCollectionHealthRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeResourceCollectionHealthRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResourceCollectionHealth");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeResourceCollectionHealthRequest, DescribeResourceCollectionHealthResponse>()
                            .withOperationName("DescribeResourceCollectionHealth").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeResourceCollectionHealthRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeResourceCollectionHealthRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the integration status of services that are integrated with DevOps Guru. The one service that can be
     * integrated with DevOps Guru is Amazon Web Services Systems Manager, which can be used to create an OpsItem for
     * each generated insight.
     * </p>
     *
     * @param describeServiceIntegrationRequest
     * @return Result of the DescribeServiceIntegration operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.DescribeServiceIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeServiceIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeServiceIntegrationResponse describeServiceIntegration(
            DescribeServiceIntegrationRequest describeServiceIntegrationRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServiceIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeServiceIntegrationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeServiceIntegrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServiceIntegration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceIntegrationRequest, DescribeServiceIntegrationResponse>()
                            .withOperationName("DescribeServiceIntegration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeServiceIntegrationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeServiceIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an estimate of the monthly cost for DevOps Guru to analyze your Amazon Web Services resources. For more
     * information, see <a href="https://docs.aws.amazon.com/devops-guru/latest/userguide/cost-estimate.html">Estimate
     * your Amazon DevOps Guru costs</a> and <a href="http://aws.amazon.com/devops-guru/pricing/">Amazon DevOps Guru
     * pricing</a>.
     * </p>
     *
     * @param getCostEstimationRequest
     * @return Result of the GetCostEstimation operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.GetCostEstimation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/GetCostEstimation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetCostEstimationResponse getCostEstimation(GetCostEstimationRequest getCostEstimationRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCostEstimationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCostEstimationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCostEstimationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCostEstimationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCostEstimation");

            return clientHandler.execute(new ClientExecutionParams<GetCostEstimationRequest, GetCostEstimationResponse>()
                    .withOperationName("GetCostEstimation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getCostEstimationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCostEstimationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns lists Amazon Web Services resources that are of the specified resource collection type. The two types of
     * Amazon Web Services resource collections supported are Amazon Web Services CloudFormation stacks and Amazon Web
     * Services resources that contain the same Amazon Web Services tag. DevOps Guru can be configured to analyze the
     * Amazon Web Services resources that are defined in the stacks or that are tagged using the same tag <i>key</i>.
     * You can specify up to 500 Amazon Web Services CloudFormation stacks.
     * </p>
     *
     * @param getResourceCollectionRequest
     * @return Result of the GetResourceCollection operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.GetResourceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/GetResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceCollectionResponse getResourceCollection(GetResourceCollectionRequest getResourceCollectionRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceCollectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceCollectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceCollection");

            return clientHandler.execute(new ClientExecutionParams<GetResourceCollectionRequest, GetResourceCollectionResponse>()
                    .withOperationName("GetResourceCollection").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourceCollectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the anomalies that belong to an insight that you specify using its ID.
     * </p>
     *
     * @param listAnomaliesForInsightRequest
     * @return Result of the ListAnomaliesForInsight operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListAnomaliesForInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListAnomaliesForInsight"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAnomaliesForInsightResponse listAnomaliesForInsight(ListAnomaliesForInsightRequest listAnomaliesForInsightRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAnomaliesForInsightResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAnomaliesForInsightResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAnomaliesForInsightRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomaliesForInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomaliesForInsight");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAnomaliesForInsightRequest, ListAnomaliesForInsightResponse>()
                            .withOperationName("ListAnomaliesForInsight").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listAnomaliesForInsightRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAnomaliesForInsightRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of log groups that contain log anomalies.
     * </p>
     *
     * @param listAnomalousLogGroupsRequest
     * @return Result of the ListAnomalousLogGroups operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListAnomalousLogGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListAnomalousLogGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAnomalousLogGroupsResponse listAnomalousLogGroups(ListAnomalousLogGroupsRequest listAnomalousLogGroupsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAnomalousLogGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAnomalousLogGroupsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAnomalousLogGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomalousLogGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomalousLogGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAnomalousLogGroupsRequest, ListAnomalousLogGroupsResponse>()
                            .withOperationName("ListAnomalousLogGroups").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listAnomalousLogGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAnomalousLogGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the events emitted by the resources that are evaluated by DevOps Guru. You can use filters to
     * specify which events are returned.
     * </p>
     *
     * @param listEventsRequest
     * @return Result of the ListEvents operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEventsResponse listEvents(ListEventsRequest listEventsRequest) throws AccessDeniedException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEventsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEventsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEvents");

            return clientHandler.execute(new ClientExecutionParams<ListEventsRequest, ListEventsResponse>()
                    .withOperationName("ListEvents").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listEventsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of insights in your Amazon Web Services account. You can specify which insights are returned by
     * their start time and status (<code>ONGOING</code>, <code>CLOSED</code>, or <code>ANY</code>).
     * </p>
     *
     * @param listInsightsRequest
     * @return Result of the ListInsights operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListInsightsResponse listInsights(ListInsightsRequest listInsightsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInsightsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInsightsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInsights");

            return clientHandler.execute(new ClientExecutionParams<ListInsightsRequest, ListInsightsResponse>()
                    .withOperationName("ListInsights").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInsightsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of all log groups that are being monitored and tagged by DevOps Guru.
     * </p>
     *
     * @param listMonitoredResourcesRequest
     * @return Result of the ListMonitoredResources operation returned by the service.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListMonitoredResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListMonitoredResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMonitoredResourcesResponse listMonitoredResources(ListMonitoredResourcesRequest listMonitoredResourcesRequest)
            throws ResourceNotFoundException, InternalServerException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMonitoredResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMonitoredResourcesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMonitoredResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMonitoredResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMonitoredResources");

            return clientHandler
                    .execute(new ClientExecutionParams<ListMonitoredResourcesRequest, ListMonitoredResourcesResponse>()
                            .withOperationName("ListMonitoredResources").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listMonitoredResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListMonitoredResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of notification channels configured for DevOps Guru. Each notification channel is used to notify
     * you when DevOps Guru generates an insight that contains information about how to improve your operations. The one
     * supported notification channel is Amazon Simple Notification Service (Amazon SNS).
     * </p>
     *
     * @param listNotificationChannelsRequest
     * @return Result of the ListNotificationChannels operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListNotificationChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListNotificationChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationChannelsResponse listNotificationChannels(
            ListNotificationChannelsRequest listNotificationChannelsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotificationChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNotificationChannelsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNotificationChannelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotificationChannels");

            return clientHandler
                    .execute(new ClientExecutionParams<ListNotificationChannelsRequest, ListNotificationChannelsResponse>()
                            .withOperationName("ListNotificationChannels").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listNotificationChannelsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListNotificationChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of insights associated with the account or OU Id.
     * </p>
     *
     * @param listOrganizationInsightsRequest
     * @return Result of the ListOrganizationInsights operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListOrganizationInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListOrganizationInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListOrganizationInsightsResponse listOrganizationInsights(
            ListOrganizationInsightsRequest listOrganizationInsightsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOrganizationInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListOrganizationInsightsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOrganizationInsightsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOrganizationInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrganizationInsights");

            return clientHandler
                    .execute(new ClientExecutionParams<ListOrganizationInsightsRequest, ListOrganizationInsightsResponse>()
                            .withOperationName("ListOrganizationInsights").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listOrganizationInsightsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListOrganizationInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of a specified insight's recommendations. Each recommendation includes a list of related metrics
     * and a list of related events.
     * </p>
     *
     * @param listRecommendationsRequest
     * @return Result of the ListRecommendations operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.ListRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecommendationsResponse listRecommendations(ListRecommendationsRequest listRecommendationsRequest)
            throws AccessDeniedException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecommendationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecommendations");

            return clientHandler.execute(new ClientExecutionParams<ListRecommendationsRequest, ListRecommendationsResponse>()
                    .withOperationName("ListRecommendations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRecommendationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Collects customer feedback about the specified insight.
     * </p>
     *
     * @param putFeedbackRequest
     * @return Result of the PutFeedback operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.PutFeedback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/PutFeedback" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutFeedbackResponse putFeedback(PutFeedbackRequest putFeedbackRequest) throws AccessDeniedException,
            ConflictException, InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutFeedbackResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putFeedbackRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutFeedback");

            return clientHandler.execute(new ClientExecutionParams<PutFeedbackRequest, PutFeedbackResponse>()
                    .withOperationName("PutFeedback").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putFeedbackRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutFeedbackRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a notification channel from DevOps Guru. A notification channel is used to notify you when DevOps Guru
     * generates an insight that contains information about how to improve your operations.
     * </p>
     *
     * @param removeNotificationChannelRequest
     * @return Result of the RemoveNotificationChannel operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.RemoveNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/RemoveNotificationChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveNotificationChannelResponse removeNotificationChannel(
            RemoveNotificationChannelRequest removeNotificationChannelRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveNotificationChannelResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveNotificationChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveNotificationChannelRequest, RemoveNotificationChannelResponse>()
                            .withOperationName("RemoveNotificationChannel").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(removeNotificationChannelRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveNotificationChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of insights in your Amazon Web Services account. You can specify which insights are returned by
     * their start time, one or more statuses (<code>ONGOING</code> or <code>CLOSED</code>), one or more severities (
     * <code>LOW</code>, <code>MEDIUM</code>, and <code>HIGH</code>), and type (<code>REACTIVE</code> or
     * <code>PROACTIVE</code>).
     * </p>
     * <p>
     * Use the <code>Filters</code> parameter to specify status and severity search parameters. Use the
     * <code>Type</code> parameter to specify <code>REACTIVE</code> or <code>PROACTIVE</code> in your search.
     * </p>
     *
     * @param searchInsightsRequest
     * @return Result of the SearchInsights operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.SearchInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/SearchInsights" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public SearchInsightsResponse searchInsights(SearchInsightsRequest searchInsightsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchInsightsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchInsightsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchInsights");

            return clientHandler.execute(new ClientExecutionParams<SearchInsightsRequest, SearchInsightsResponse>()
                    .withOperationName("SearchInsights").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(searchInsightsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of insights in your organization. You can specify which insights are returned by their start time,
     * one or more statuses (<code>ONGOING</code>, <code>CLOSED</code>, and <code>CLOSED</code>), one or more severities
     * (<code>LOW</code>, <code>MEDIUM</code>, and <code>HIGH</code>), and type (<code>REACTIVE</code> or
     * <code>PROACTIVE</code>).
     * </p>
     * <p>
     * Use the <code>Filters</code> parameter to specify status and severity search parameters. Use the
     * <code>Type</code> parameter to specify <code>REACTIVE</code> or <code>PROACTIVE</code> in your search.
     * </p>
     *
     * @param searchOrganizationInsightsRequest
     * @return Result of the SearchOrganizationInsights operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.SearchOrganizationInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/SearchOrganizationInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchOrganizationInsightsResponse searchOrganizationInsights(
            SearchOrganizationInsightsRequest searchOrganizationInsightsRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchOrganizationInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SearchOrganizationInsightsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchOrganizationInsightsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchOrganizationInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchOrganizationInsights");

            return clientHandler
                    .execute(new ClientExecutionParams<SearchOrganizationInsightsRequest, SearchOrganizationInsightsResponse>()
                            .withOperationName("SearchOrganizationInsights").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(searchOrganizationInsightsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SearchOrganizationInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the creation of an estimate of the monthly cost to analyze your Amazon Web Services resources.
     * </p>
     *
     * @param startCostEstimationRequest
     * @return Result of the StartCostEstimation operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ResourceNotFoundException
     *         A requested resource could not be found
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.StartCostEstimation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/StartCostEstimation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartCostEstimationResponse startCostEstimation(StartCostEstimationRequest startCostEstimationRequest)
            throws AccessDeniedException, ConflictException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartCostEstimationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartCostEstimationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCostEstimationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCostEstimationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCostEstimation");

            return clientHandler.execute(new ClientExecutionParams<StartCostEstimationRequest, StartCostEstimationResponse>()
                    .withOperationName("StartCostEstimation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startCostEstimationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartCostEstimationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables or disables integration with a service that can be integrated with DevOps Guru. The one service that can
     * be integrated with DevOps Guru is Amazon CodeGuru Profiler, which can produce proactive recommendations which can
     * be stored and viewed in DevOps Guru.
     * </p>
     *
     * @param updateEventSourcesConfigRequest
     * @return Result of the UpdateEventSourcesConfig operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.UpdateEventSourcesConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateEventSourcesConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEventSourcesConfigResponse updateEventSourcesConfig(
            UpdateEventSourcesConfigRequest updateEventSourcesConfigRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEventSourcesConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEventSourcesConfigResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEventSourcesConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEventSourcesConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEventSourcesConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateEventSourcesConfigRequest, UpdateEventSourcesConfigResponse>()
                            .withOperationName("UpdateEventSourcesConfig").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateEventSourcesConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateEventSourcesConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the collection of resources that DevOps Guru analyzes. The two types of Amazon Web Services resource
     * collections supported are Amazon Web Services CloudFormation stacks and Amazon Web Services resources that
     * contain the same Amazon Web Services tag. DevOps Guru can be configured to analyze the Amazon Web Services
     * resources that are defined in the stacks or that are tagged using the same tag <i>key</i>. You can specify up to
     * 500 Amazon Web Services CloudFormation stacks. This method also creates the IAM role required for you to use
     * DevOps Guru.
     * </p>
     *
     * @param updateResourceCollectionRequest
     * @return Result of the UpdateResourceCollection operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.UpdateResourceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResourceCollectionResponse updateResourceCollection(
            UpdateResourceCollectionRequest updateResourceCollectionRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResourceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResourceCollectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResourceCollectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceCollection");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceCollectionRequest, UpdateResourceCollectionResponse>()
                            .withOperationName("UpdateResourceCollection").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateResourceCollectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateResourceCollectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables or disables integration with a service that can be integrated with DevOps Guru. The one service that can
     * be integrated with DevOps Guru is Amazon Web Services Systems Manager, which can be used to create an OpsItem for
     * each generated insight.
     * </p>
     *
     * @param updateServiceIntegrationRequest
     * @return Result of the UpdateServiceIntegration operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation. The user or role that is making the
     *         request must have at least one IAM permissions policy attached that grants the required permissions. For
     *         more information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access
     *         Management</a> in the <i>IAM User Guide</i>.
     * @throws ConflictException
     *         An exception that is thrown when a conflict occurs.
     * @throws InternalServerException
     *         An internal failure in an Amazon service occurred.
     * @throws ThrottlingException
     *         The request was denied due to a request throttling.
     * @throws ValidationException
     *         Contains information about data passed in to a field during a request that is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DevOpsGuruException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DevOpsGuruClient.UpdateServiceIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateServiceIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServiceIntegrationResponse updateServiceIntegration(
            UpdateServiceIntegrationRequest updateServiceIntegrationRequest) throws AccessDeniedException, ConflictException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            DevOpsGuruException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateServiceIntegrationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceIntegrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceIntegration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceIntegrationRequest, UpdateServiceIntegrationResponse>()
                            .withOperationName("UpdateServiceIntegration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateServiceIntegrationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateServiceIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        DevOpsGuruServiceClientConfigurationBuilder serviceConfigBuilder = new DevOpsGuruServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(DevOpsGuruException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final DevOpsGuruServiceClientConfiguration serviceClientConfiguration() {
        return new DevOpsGuruServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
