/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devopsguru;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.devopsguru.internal.DevOpsGuruServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.devopsguru.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.devopsguru.model.AccessDeniedException;
import software.amazon.awssdk.services.devopsguru.model.AddNotificationChannelRequest;
import software.amazon.awssdk.services.devopsguru.model.AddNotificationChannelResponse;
import software.amazon.awssdk.services.devopsguru.model.ConflictException;
import software.amazon.awssdk.services.devopsguru.model.DeleteInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.DeleteInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountOverviewRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountOverviewResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAnomalyRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAnomalyResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeEventSourcesConfigRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeEventSourcesConfigResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeFeedbackRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeFeedbackResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationOverviewRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationOverviewResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationResourceCollectionHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeOrganizationResourceCollectionHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeServiceIntegrationRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeServiceIntegrationResponse;
import software.amazon.awssdk.services.devopsguru.model.DevOpsGuruException;
import software.amazon.awssdk.services.devopsguru.model.GetCostEstimationRequest;
import software.amazon.awssdk.services.devopsguru.model.GetCostEstimationResponse;
import software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionRequest;
import software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionResponse;
import software.amazon.awssdk.services.devopsguru.model.InternalServerException;
import software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.ListAnomalousLogGroupsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListAnomalousLogGroupsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListEventsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListEventsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListMonitoredResourcesRequest;
import software.amazon.awssdk.services.devopsguru.model.ListMonitoredResourcesResponse;
import software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListOrganizationInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListOrganizationInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListRecommendationsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListRecommendationsResponse;
import software.amazon.awssdk.services.devopsguru.model.PutFeedbackRequest;
import software.amazon.awssdk.services.devopsguru.model.PutFeedbackResponse;
import software.amazon.awssdk.services.devopsguru.model.RemoveNotificationChannelRequest;
import software.amazon.awssdk.services.devopsguru.model.RemoveNotificationChannelResponse;
import software.amazon.awssdk.services.devopsguru.model.ResourceNotFoundException;
import software.amazon.awssdk.services.devopsguru.model.SearchInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.SearchInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.SearchOrganizationInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.SearchOrganizationInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.devopsguru.model.StartCostEstimationRequest;
import software.amazon.awssdk.services.devopsguru.model.StartCostEstimationResponse;
import software.amazon.awssdk.services.devopsguru.model.ThrottlingException;
import software.amazon.awssdk.services.devopsguru.model.UpdateEventSourcesConfigRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateEventSourcesConfigResponse;
import software.amazon.awssdk.services.devopsguru.model.UpdateResourceCollectionRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateResourceCollectionResponse;
import software.amazon.awssdk.services.devopsguru.model.UpdateServiceIntegrationRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateServiceIntegrationResponse;
import software.amazon.awssdk.services.devopsguru.model.ValidationException;
import software.amazon.awssdk.services.devopsguru.transform.AddNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DeleteInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAccountHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAccountOverviewRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAnomalyRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeEventSourcesConfigRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeFeedbackRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeOrganizationHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeOrganizationOverviewRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeOrganizationResourceCollectionHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeResourceCollectionHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeServiceIntegrationRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.GetCostEstimationRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.GetResourceCollectionRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListAnomaliesForInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListAnomalousLogGroupsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListEventsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListMonitoredResourcesRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListNotificationChannelsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListOrganizationInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.PutFeedbackRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.RemoveNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.SearchInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.SearchOrganizationInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.StartCostEstimationRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateEventSourcesConfigRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateResourceCollectionRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateServiceIntegrationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DevOpsGuruAsyncClient}.
 *
 * @see DevOpsGuruAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDevOpsGuruAsyncClient implements DevOpsGuruAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDevOpsGuruAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDevOpsGuruAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "DevOps_Guru" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds a notification channel to DevOps Guru. A notification channel is used to notify you about important DevOps
     * Guru events, such as when an insight is generated.
     * </p>
     * <p>
     * If you use an Amazon SNS topic in another account, you must attach a policy to it that grants DevOps Guru
     * permission to send it notifications. DevOps Guru adds the required policy on your behalf to send notifications
     * using Amazon SNS in your account. DevOps Guru only supports standard SNS topics. For more information, see <a
     * href="https://docs.aws.amazon.com/devops-guru/latest/userguide/sns-required-permissions.html">Permissions for
     * Amazon SNS topics</a>.
     * </p>
     * <p>
     * If you use an Amazon SNS topic that is encrypted by an Amazon Web Services Key Management Service
     * customer-managed key (CMK), then you must add permissions to the CMK. For more information, see <a
     * href="https://docs.aws.amazon.com/devops-guru/latest/userguide/sns-kms-permissions.html">Permissions for Amazon
     * Web Services KMS–encrypted Amazon SNS topics</a>.
     * </p>
     *
     * @param addNotificationChannelRequest
     * @return A Java Future containing the result of the AddNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ServiceQuotaExceededException The request contains a value that exceeds a maximum quota.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.AddNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/AddNotificationChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddNotificationChannelResponse> addNotificationChannel(
            AddNotificationChannelRequest addNotificationChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddNotificationChannelResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AddNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddNotificationChannelRequest, AddNotificationChannelResponse>()
                            .withOperationName("AddNotificationChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AddNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(addNotificationChannelRequest));
            CompletableFuture<AddNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the insight along with the associated anomalies, events and recommendations.
     * </p>
     *
     * @param deleteInsightRequest
     * @return A Java Future containing the result of the DeleteInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DeleteInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DeleteInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInsightResponse> deleteInsight(DeleteInsightRequest deleteInsightRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInsightRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteInsightResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInsightRequest, DeleteInsightResponse>()
                            .withOperationName("DeleteInsight").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteInsightRequest));
            CompletableFuture<DeleteInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of open reactive insights, the number of open proactive insights, and the number of metrics
     * analyzed in your Amazon Web Services account. Use these numbers to gauge the health of operations in your Amazon
     * Web Services account.
     * </p>
     *
     * @param describeAccountHealthRequest
     * @return A Java Future containing the result of the DescribeAccountHealth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeAccountHealth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAccountHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAccountHealthResponse> describeAccountHealth(
            DescribeAccountHealthRequest describeAccountHealthRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAccountHealthRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountHealthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountHealth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAccountHealthResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAccountHealthResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeAccountHealthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountHealthRequest, DescribeAccountHealthResponse>()
                            .withOperationName("DescribeAccountHealth").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAccountHealthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAccountHealthRequest));
            CompletableFuture<DescribeAccountHealthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For the time range passed in, returns the number of open reactive insight that were created, the number of open
     * proactive insights that were created, and the Mean Time to Recover (MTTR) for all closed reactive insights.
     * </p>
     *
     * @param describeAccountOverviewRequest
     * @return A Java Future containing the result of the DescribeAccountOverview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeAccountOverview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAccountOverview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAccountOverviewResponse> describeAccountOverview(
            DescribeAccountOverviewRequest describeAccountOverviewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAccountOverviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountOverviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountOverview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAccountOverviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAccountOverviewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeAccountOverviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountOverviewRequest, DescribeAccountOverviewResponse>()
                            .withOperationName("DescribeAccountOverview").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAccountOverviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAccountOverviewRequest));
            CompletableFuture<DescribeAccountOverviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about an anomaly that you specify using its ID.
     * </p>
     *
     * @param describeAnomalyRequest
     * @return A Java Future containing the result of the DescribeAnomaly operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeAnomaly
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAnomaly" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAnomalyResponse> describeAnomaly(DescribeAnomalyRequest describeAnomalyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAnomalyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAnomalyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnomaly");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAnomalyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAnomalyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeAnomalyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAnomalyRequest, DescribeAnomalyResponse>()
                            .withOperationName("DescribeAnomaly").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAnomalyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAnomalyRequest));
            CompletableFuture<DescribeAnomalyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the integration status of services that are integrated with DevOps Guru as Consumer via EventBridge. The
     * one service that can be integrated with DevOps Guru is Amazon CodeGuru Profiler, which can produce proactive
     * recommendations which can be stored and viewed in DevOps Guru.
     * </p>
     *
     * @param describeEventSourcesConfigRequest
     * @return A Java Future containing the result of the DescribeEventSourcesConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeEventSourcesConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeEventSourcesConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventSourcesConfigResponse> describeEventSourcesConfig(
            DescribeEventSourcesConfigRequest describeEventSourcesConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEventSourcesConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventSourcesConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventSourcesConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventSourcesConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventSourcesConfigResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeEventSourcesConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventSourcesConfigRequest, DescribeEventSourcesConfigResponse>()
                            .withOperationName("DescribeEventSourcesConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEventSourcesConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEventSourcesConfigRequest));
            CompletableFuture<DescribeEventSourcesConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most recent feedback submitted in the current Amazon Web Services account and Region.
     * </p>
     *
     * @param describeFeedbackRequest
     * @return A Java Future containing the result of the DescribeFeedback operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeFeedback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeFeedback" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFeedbackResponse> describeFeedback(DescribeFeedbackRequest describeFeedbackRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFeedbackRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFeedback");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFeedbackResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeFeedbackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFeedbackRequest, DescribeFeedbackResponse>()
                            .withOperationName("DescribeFeedback").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeFeedbackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeFeedbackRequest));
            CompletableFuture<DescribeFeedbackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about an insight that you specify using its ID.
     * </p>
     *
     * @param describeInsightRequest
     * @return A Java Future containing the result of the DescribeInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeInsight" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInsightResponse> describeInsight(DescribeInsightRequest describeInsightRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeInsightRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInsightResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInsightResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInsightRequest, DescribeInsightResponse>()
                            .withOperationName("DescribeInsight").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeInsightRequest));
            CompletableFuture<DescribeInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns active insights, predictive insights, and resource hours analyzed in last hour.
     * </p>
     *
     * @param describeOrganizationHealthRequest
     * @return A Java Future containing the result of the DescribeOrganizationHealth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeOrganizationHealth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeOrganizationHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOrganizationHealthResponse> describeOrganizationHealth(
            DescribeOrganizationHealthRequest describeOrganizationHealthRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeOrganizationHealthRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOrganizationHealthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationHealth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOrganizationHealthResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeOrganizationHealthResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeOrganizationHealthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationHealthRequest, DescribeOrganizationHealthResponse>()
                            .withOperationName("DescribeOrganizationHealth").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeOrganizationHealthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeOrganizationHealthRequest));
            CompletableFuture<DescribeOrganizationHealthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an overview of your organization's history based on the specified time range. The overview includes the
     * total reactive and proactive insights.
     * </p>
     *
     * @param describeOrganizationOverviewRequest
     * @return A Java Future containing the result of the DescribeOrganizationOverview operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeOrganizationOverview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeOrganizationOverview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOrganizationOverviewResponse> describeOrganizationOverview(
            DescribeOrganizationOverviewRequest describeOrganizationOverviewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeOrganizationOverviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOrganizationOverviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationOverview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOrganizationOverviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeOrganizationOverviewResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeOrganizationOverviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationOverviewRequest, DescribeOrganizationOverviewResponse>()
                            .withOperationName("DescribeOrganizationOverview").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeOrganizationOverviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeOrganizationOverviewRequest));
            CompletableFuture<DescribeOrganizationOverviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides an overview of your system's health. If additional member accounts are part of your organization, you
     * can filter those accounts using the <code>AccountIds</code> field.
     * </p>
     *
     * @param describeOrganizationResourceCollectionHealthRequest
     * @return A Java Future containing the result of the DescribeOrganizationResourceCollectionHealth operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeOrganizationResourceCollectionHealth
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeOrganizationResourceCollectionHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOrganizationResourceCollectionHealthResponse> describeOrganizationResourceCollectionHealth(
            DescribeOrganizationResourceCollectionHealthRequest describeOrganizationResourceCollectionHealthRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                describeOrganizationResourceCollectionHealthRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeOrganizationResourceCollectionHealthRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationResourceCollectionHealth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOrganizationResourceCollectionHealthResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeOrganizationResourceCollectionHealthResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeOrganizationResourceCollectionHealthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationResourceCollectionHealthRequest, DescribeOrganizationResourceCollectionHealthResponse>()
                            .withOperationName("DescribeOrganizationResourceCollectionHealth")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeOrganizationResourceCollectionHealthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeOrganizationResourceCollectionHealthRequest));
            CompletableFuture<DescribeOrganizationResourceCollectionHealthResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of open proactive insights, open reactive insights, and the Mean Time to Recover (MTTR) for
     * all closed insights in resource collections in your account. You specify the type of Amazon Web Services
     * resources collection. The two types of Amazon Web Services resource collections supported are Amazon Web Services
     * CloudFormation stacks and Amazon Web Services resources that contain the same Amazon Web Services tag. DevOps
     * Guru can be configured to analyze the Amazon Web Services resources that are defined in the stacks or that are
     * tagged using the same tag <i>key</i>. You can specify up to 500 Amazon Web Services CloudFormation stacks.
     * </p>
     *
     * @param describeResourceCollectionHealthRequest
     * @return A Java Future containing the result of the DescribeResourceCollectionHealth operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeResourceCollectionHealth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeResourceCollectionHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeResourceCollectionHealthResponse> describeResourceCollectionHealth(
            DescribeResourceCollectionHealthRequest describeResourceCollectionHealthRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeResourceCollectionHealthRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeResourceCollectionHealthRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResourceCollectionHealth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeResourceCollectionHealthResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeResourceCollectionHealthResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeResourceCollectionHealthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeResourceCollectionHealthRequest, DescribeResourceCollectionHealthResponse>()
                            .withOperationName("DescribeResourceCollectionHealth").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeResourceCollectionHealthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeResourceCollectionHealthRequest));
            CompletableFuture<DescribeResourceCollectionHealthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the integration status of services that are integrated with DevOps Guru. The one service that can be
     * integrated with DevOps Guru is Amazon Web Services Systems Manager, which can be used to create an OpsItem for
     * each generated insight.
     * </p>
     *
     * @param describeServiceIntegrationRequest
     * @return A Java Future containing the result of the DescribeServiceIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeServiceIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeServiceIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeServiceIntegrationResponse> describeServiceIntegration(
            DescribeServiceIntegrationRequest describeServiceIntegrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeServiceIntegrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServiceIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeServiceIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeServiceIntegrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeServiceIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceIntegrationRequest, DescribeServiceIntegrationResponse>()
                            .withOperationName("DescribeServiceIntegration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeServiceIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeServiceIntegrationRequest));
            CompletableFuture<DescribeServiceIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an estimate of the monthly cost for DevOps Guru to analyze your Amazon Web Services resources. For more
     * information, see <a href="https://docs.aws.amazon.com/devops-guru/latest/userguide/cost-estimate.html">Estimate
     * your Amazon DevOps Guru costs</a> and <a href="http://aws.amazon.com/devops-guru/pricing/">Amazon DevOps Guru
     * pricing</a>.
     * </p>
     *
     * @param getCostEstimationRequest
     * @return A Java Future containing the result of the GetCostEstimation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.GetCostEstimation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/GetCostEstimation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCostEstimationResponse> getCostEstimation(GetCostEstimationRequest getCostEstimationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCostEstimationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCostEstimationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCostEstimation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCostEstimationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCostEstimationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetCostEstimationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCostEstimationRequest, GetCostEstimationResponse>()
                            .withOperationName("GetCostEstimation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCostEstimationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCostEstimationRequest));
            CompletableFuture<GetCostEstimationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns lists Amazon Web Services resources that are of the specified resource collection type. The two types of
     * Amazon Web Services resource collections supported are Amazon Web Services CloudFormation stacks and Amazon Web
     * Services resources that contain the same Amazon Web Services tag. DevOps Guru can be configured to analyze the
     * Amazon Web Services resources that are defined in the stacks or that are tagged using the same tag <i>key</i>.
     * You can specify up to 500 Amazon Web Services CloudFormation stacks.
     * </p>
     *
     * @param getResourceCollectionRequest
     * @return A Java Future containing the result of the GetResourceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.GetResourceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/GetResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceCollectionResponse> getResourceCollection(
            GetResourceCollectionRequest getResourceCollectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceCollectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceCollectionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetResourceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceCollectionRequest, GetResourceCollectionResponse>()
                            .withOperationName("GetResourceCollection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceCollectionRequest));
            CompletableFuture<GetResourceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the anomalies that belong to an insight that you specify using its ID.
     * </p>
     *
     * @param listAnomaliesForInsightRequest
     * @return A Java Future containing the result of the ListAnomaliesForInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListAnomaliesForInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListAnomaliesForInsight"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnomaliesForInsightResponse> listAnomaliesForInsight(
            ListAnomaliesForInsightRequest listAnomaliesForInsightRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAnomaliesForInsightRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomaliesForInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomaliesForInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnomaliesForInsightResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnomaliesForInsightResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListAnomaliesForInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnomaliesForInsightRequest, ListAnomaliesForInsightResponse>()
                            .withOperationName("ListAnomaliesForInsight").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAnomaliesForInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAnomaliesForInsightRequest));
            CompletableFuture<ListAnomaliesForInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of log groups that contain log anomalies.
     * </p>
     *
     * @param listAnomalousLogGroupsRequest
     * @return A Java Future containing the result of the ListAnomalousLogGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListAnomalousLogGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListAnomalousLogGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnomalousLogGroupsResponse> listAnomalousLogGroups(
            ListAnomalousLogGroupsRequest listAnomalousLogGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAnomalousLogGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomalousLogGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomalousLogGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnomalousLogGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnomalousLogGroupsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListAnomalousLogGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnomalousLogGroupsRequest, ListAnomalousLogGroupsResponse>()
                            .withOperationName("ListAnomalousLogGroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAnomalousLogGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAnomalousLogGroupsRequest));
            CompletableFuture<ListAnomalousLogGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the events emitted by the resources that are evaluated by DevOps Guru. You can use filters to
     * specify which events are returned.
     * </p>
     *
     * @param listEventsRequest
     * @return A Java Future containing the result of the ListEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventsResponse> listEvents(ListEventsRequest listEventsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEventsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListEventsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventsRequest, ListEventsResponse>().withOperationName("ListEvents")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEventsRequest));
            CompletableFuture<ListEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of insights in your Amazon Web Services account. You can specify which insights are returned by
     * their start time and status (<code>ONGOING</code>, <code>CLOSED</code>, or <code>ANY</code>).
     * </p>
     *
     * @param listInsightsRequest
     * @return A Java Future containing the result of the ListInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInsightsResponse> listInsights(ListInsightsRequest listInsightsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInsightsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInsightsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInsightsRequest, ListInsightsResponse>()
                            .withOperationName("ListInsights").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInsightsRequest));
            CompletableFuture<ListInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of all log groups that are being monitored and tagged by DevOps Guru.
     * </p>
     *
     * @param listMonitoredResourcesRequest
     * @return A Java Future containing the result of the ListMonitoredResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListMonitoredResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListMonitoredResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMonitoredResourcesResponse> listMonitoredResources(
            ListMonitoredResourcesRequest listMonitoredResourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMonitoredResourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMonitoredResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMonitoredResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMonitoredResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMonitoredResourcesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListMonitoredResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMonitoredResourcesRequest, ListMonitoredResourcesResponse>()
                            .withOperationName("ListMonitoredResources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMonitoredResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMonitoredResourcesRequest));
            CompletableFuture<ListMonitoredResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of notification channels configured for DevOps Guru. Each notification channel is used to notify
     * you when DevOps Guru generates an insight that contains information about how to improve your operations. The one
     * supported notification channel is Amazon Simple Notification Service (Amazon SNS).
     * </p>
     *
     * @param listNotificationChannelsRequest
     * @return A Java Future containing the result of the ListNotificationChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListNotificationChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListNotificationChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNotificationChannelsResponse> listNotificationChannels(
            ListNotificationChannelsRequest listNotificationChannelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNotificationChannelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotificationChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNotificationChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNotificationChannelsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListNotificationChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNotificationChannelsRequest, ListNotificationChannelsResponse>()
                            .withOperationName("ListNotificationChannels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListNotificationChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNotificationChannelsRequest));
            CompletableFuture<ListNotificationChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of insights associated with the account or OU Id.
     * </p>
     *
     * @param listOrganizationInsightsRequest
     * @return A Java Future containing the result of the ListOrganizationInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListOrganizationInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListOrganizationInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrganizationInsightsResponse> listOrganizationInsights(
            ListOrganizationInsightsRequest listOrganizationInsightsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOrganizationInsightsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOrganizationInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrganizationInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrganizationInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOrganizationInsightsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListOrganizationInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrganizationInsightsRequest, ListOrganizationInsightsResponse>()
                            .withOperationName("ListOrganizationInsights").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOrganizationInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOrganizationInsightsRequest));
            CompletableFuture<ListOrganizationInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of a specified insight's recommendations. Each recommendation includes a list of related metrics
     * and a list of related events.
     * </p>
     *
     * @param listRecommendationsRequest
     * @return A Java Future containing the result of the ListRecommendations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecommendationsResponse> listRecommendations(
            ListRecommendationsRequest listRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRecommendationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecommendationsRequest, ListRecommendationsResponse>()
                            .withOperationName("ListRecommendations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRecommendationsRequest));
            CompletableFuture<ListRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Collects customer feedback about the specified insight.
     * </p>
     *
     * @param putFeedbackRequest
     * @return A Java Future containing the result of the PutFeedback operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.PutFeedback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/PutFeedback" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutFeedbackResponse> putFeedback(PutFeedbackRequest putFeedbackRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putFeedbackRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutFeedback");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutFeedbackResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutFeedbackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutFeedbackRequest, PutFeedbackResponse>()
                            .withOperationName("PutFeedback").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutFeedbackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putFeedbackRequest));
            CompletableFuture<PutFeedbackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a notification channel from DevOps Guru. A notification channel is used to notify you when DevOps Guru
     * generates an insight that contains information about how to improve your operations.
     * </p>
     *
     * @param removeNotificationChannelRequest
     * @return A Java Future containing the result of the RemoveNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.RemoveNotificationChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/RemoveNotificationChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveNotificationChannelResponse> removeNotificationChannel(
            RemoveNotificationChannelRequest removeNotificationChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeNotificationChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveNotificationChannelResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RemoveNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveNotificationChannelRequest, RemoveNotificationChannelResponse>()
                            .withOperationName("RemoveNotificationChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RemoveNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(removeNotificationChannelRequest));
            CompletableFuture<RemoveNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of insights in your Amazon Web Services account. You can specify which insights are returned by
     * their start time, one or more statuses (<code>ONGOING</code> or <code>CLOSED</code>), one or more severities (
     * <code>LOW</code>, <code>MEDIUM</code>, and <code>HIGH</code>), and type (<code>REACTIVE</code> or
     * <code>PROACTIVE</code>).
     * </p>
     * <p>
     * Use the <code>Filters</code> parameter to specify status and severity search parameters. Use the
     * <code>Type</code> parameter to specify <code>REACTIVE</code> or <code>PROACTIVE</code> in your search.
     * </p>
     *
     * @param searchInsightsRequest
     * @return A Java Future containing the result of the SearchInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.SearchInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/SearchInsights" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchInsightsResponse> searchInsights(SearchInsightsRequest searchInsightsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchInsightsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchInsightsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<SearchInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchInsightsRequest, SearchInsightsResponse>()
                            .withOperationName("SearchInsights").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchInsightsRequest));
            CompletableFuture<SearchInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of insights in your organization. You can specify which insights are returned by their start time,
     * one or more statuses (<code>ONGOING</code>, <code>CLOSED</code>, and <code>CLOSED</code>), one or more severities
     * (<code>LOW</code>, <code>MEDIUM</code>, and <code>HIGH</code>), and type (<code>REACTIVE</code> or
     * <code>PROACTIVE</code>).
     * </p>
     * <p>
     * Use the <code>Filters</code> parameter to specify status and severity search parameters. Use the
     * <code>Type</code> parameter to specify <code>REACTIVE</code> or <code>PROACTIVE</code> in your search.
     * </p>
     *
     * @param searchOrganizationInsightsRequest
     * @return A Java Future containing the result of the SearchOrganizationInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.SearchOrganizationInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/SearchOrganizationInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchOrganizationInsightsResponse> searchOrganizationInsights(
            SearchOrganizationInsightsRequest searchOrganizationInsightsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchOrganizationInsightsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchOrganizationInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchOrganizationInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchOrganizationInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchOrganizationInsightsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<SearchOrganizationInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchOrganizationInsightsRequest, SearchOrganizationInsightsResponse>()
                            .withOperationName("SearchOrganizationInsights").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchOrganizationInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchOrganizationInsightsRequest));
            CompletableFuture<SearchOrganizationInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the creation of an estimate of the monthly cost to analyze your Amazon Web Services resources.
     * </p>
     *
     * @param startCostEstimationRequest
     * @return A Java Future containing the result of the StartCostEstimation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.StartCostEstimation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/StartCostEstimation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartCostEstimationResponse> startCostEstimation(
            StartCostEstimationRequest startCostEstimationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCostEstimationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCostEstimationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCostEstimation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCostEstimationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartCostEstimationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartCostEstimationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartCostEstimationRequest, StartCostEstimationResponse>()
                            .withOperationName("StartCostEstimation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartCostEstimationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startCostEstimationRequest));
            CompletableFuture<StartCostEstimationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables or disables integration with a service that can be integrated with DevOps Guru. The one service that can
     * be integrated with DevOps Guru is Amazon CodeGuru Profiler, which can produce proactive recommendations which can
     * be stored and viewed in DevOps Guru.
     * </p>
     *
     * @param updateEventSourcesConfigRequest
     * @return A Java Future containing the result of the UpdateEventSourcesConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.UpdateEventSourcesConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateEventSourcesConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEventSourcesConfigResponse> updateEventSourcesConfig(
            UpdateEventSourcesConfigRequest updateEventSourcesConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEventSourcesConfigRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEventSourcesConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEventSourcesConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEventSourcesConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEventSourcesConfigResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateEventSourcesConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEventSourcesConfigRequest, UpdateEventSourcesConfigResponse>()
                            .withOperationName("UpdateEventSourcesConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEventSourcesConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEventSourcesConfigRequest));
            CompletableFuture<UpdateEventSourcesConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the collection of resources that DevOps Guru analyzes. The two types of Amazon Web Services resource
     * collections supported are Amazon Web Services CloudFormation stacks and Amazon Web Services resources that
     * contain the same Amazon Web Services tag. DevOps Guru can be configured to analyze the Amazon Web Services
     * resources that are defined in the stacks or that are tagged using the same tag <i>key</i>. You can specify up to
     * 500 Amazon Web Services CloudFormation stacks. This method also creates the IAM role required for you to use
     * DevOps Guru.
     * </p>
     *
     * @param updateResourceCollectionRequest
     * @return A Java Future containing the result of the UpdateResourceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.UpdateResourceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResourceCollectionResponse> updateResourceCollection(
            UpdateResourceCollectionRequest updateResourceCollectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResourceCollectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResourceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResourceCollectionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateResourceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceCollectionRequest, UpdateResourceCollectionResponse>()
                            .withOperationName("UpdateResourceCollection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateResourceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateResourceCollectionRequest));
            CompletableFuture<UpdateResourceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables or disables integration with a service that can be integrated with DevOps Guru. The one service that can
     * be integrated with DevOps Guru is Amazon Web Services Systems Manager, which can be used to create an OpsItem for
     * each generated insight.
     * </p>
     *
     * @param updateServiceIntegrationRequest
     * @return A Java Future containing the result of the UpdateServiceIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.UpdateServiceIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateServiceIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceIntegrationResponse> updateServiceIntegration(
            UpdateServiceIntegrationRequest updateServiceIntegrationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceIntegrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceIntegrationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateServiceIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceIntegrationRequest, UpdateServiceIntegrationResponse>()
                            .withOperationName("UpdateServiceIntegration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateServiceIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateServiceIntegrationRequest));
            CompletableFuture<UpdateServiceIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final DevOpsGuruServiceClientConfiguration serviceClientConfiguration() {
        return new DevOpsGuruServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(DevOpsGuruException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        DevOpsGuruServiceClientConfigurationBuilder serviceConfigBuilder = new DevOpsGuruServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
