/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateLocationSmbRequest extends DataSyncRequest implements
        ToCopyableBuilder<UpdateLocationSmbRequest.Builder, UpdateLocationSmbRequest> {
    private static final SdkField<String> LOCATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LocationArn").getter(getter(UpdateLocationSmbRequest::locationArn)).setter(setter(Builder::locationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocationArn").build()).build();

    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(UpdateLocationSmbRequest::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<String> SERVER_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerHostname").getter(getter(UpdateLocationSmbRequest::serverHostname))
            .setter(setter(Builder::serverHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerHostname").build()).build();

    private static final SdkField<String> USER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("User")
            .getter(getter(UpdateLocationSmbRequest::user)).setter(setter(Builder::user))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("User").build()).build();

    private static final SdkField<String> DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Domain")
            .getter(getter(UpdateLocationSmbRequest::domain)).setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Domain").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Password").getter(getter(UpdateLocationSmbRequest::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(UpdateLocationSmbRequest::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SmbMountOptions> MOUNT_OPTIONS_FIELD = SdkField
            .<SmbMountOptions> builder(MarshallingType.SDK_POJO).memberName("MountOptions")
            .getter(getter(UpdateLocationSmbRequest::mountOptions)).setter(setter(Builder::mountOptions))
            .constructor(SmbMountOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MountOptions").build()).build();

    private static final SdkField<String> AUTHENTICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthenticationType").getter(getter(UpdateLocationSmbRequest::authenticationTypeAsString))
            .setter(setter(Builder::authenticationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationType").build())
            .build();

    private static final SdkField<List<String>> DNS_IP_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIpAddresses")
            .getter(getter(UpdateLocationSmbRequest::dnsIpAddresses))
            .setter(setter(Builder::dnsIpAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIpAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> KERBEROS_PRINCIPAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KerberosPrincipal").getter(getter(UpdateLocationSmbRequest::kerberosPrincipal))
            .setter(setter(Builder::kerberosPrincipal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KerberosPrincipal").build()).build();

    private static final SdkField<SdkBytes> KERBEROS_KEYTAB_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("KerberosKeytab").getter(getter(UpdateLocationSmbRequest::kerberosKeytab))
            .setter(setter(Builder::kerberosKeytab))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KerberosKeytab").build()).build();

    private static final SdkField<SdkBytes> KERBEROS_KRB5_CONF_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("KerberosKrb5Conf").getter(getter(UpdateLocationSmbRequest::kerberosKrb5Conf))
            .setter(setter(Builder::kerberosKrb5Conf))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KerberosKrb5Conf").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCATION_ARN_FIELD,
            SUBDIRECTORY_FIELD, SERVER_HOSTNAME_FIELD, USER_FIELD, DOMAIN_FIELD, PASSWORD_FIELD, AGENT_ARNS_FIELD,
            MOUNT_OPTIONS_FIELD, AUTHENTICATION_TYPE_FIELD, DNS_IP_ADDRESSES_FIELD, KERBEROS_PRINCIPAL_FIELD,
            KERBEROS_KEYTAB_FIELD, KERBEROS_KRB5_CONF_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String locationArn;

    private final String subdirectory;

    private final String serverHostname;

    private final String user;

    private final String domain;

    private final String password;

    private final List<String> agentArns;

    private final SmbMountOptions mountOptions;

    private final String authenticationType;

    private final List<String> dnsIpAddresses;

    private final String kerberosPrincipal;

    private final SdkBytes kerberosKeytab;

    private final SdkBytes kerberosKrb5Conf;

    private UpdateLocationSmbRequest(BuilderImpl builder) {
        super(builder);
        this.locationArn = builder.locationArn;
        this.subdirectory = builder.subdirectory;
        this.serverHostname = builder.serverHostname;
        this.user = builder.user;
        this.domain = builder.domain;
        this.password = builder.password;
        this.agentArns = builder.agentArns;
        this.mountOptions = builder.mountOptions;
        this.authenticationType = builder.authenticationType;
        this.dnsIpAddresses = builder.dnsIpAddresses;
        this.kerberosPrincipal = builder.kerberosPrincipal;
        this.kerberosKeytab = builder.kerberosKeytab;
        this.kerberosKrb5Conf = builder.kerberosKrb5Conf;
    }

    /**
     * <p>
     * Specifies the ARN of the SMB location that you want to update.
     * </p>
     * 
     * @return Specifies the ARN of the SMB location that you want to update.
     */
    public final String locationArn() {
        return locationArn;
    }

    /**
     * <p>
     * Specifies the name of the share exported by your SMB file server where DataSync will read or write data. You can
     * include a subdirectory in the share path (for example, <code>/path/to/subdirectory</code>). Make sure that other
     * SMB clients in your network can also mount this path.
     * </p>
     * <p>
     * To copy all data in the specified subdirectory, DataSync must be able to mount the SMB share and access all of
     * its data. For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     * >Providing DataSync access to SMB file servers</a>.
     * </p>
     * 
     * @return Specifies the name of the share exported by your SMB file server where DataSync will read or write data.
     *         You can include a subdirectory in the share path (for example, <code>/path/to/subdirectory</code>). Make
     *         sure that other SMB clients in your network can also mount this path.</p>
     *         <p>
     *         To copy all data in the specified subdirectory, DataSync must be able to mount the SMB share and access
     *         all of its data. For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     *         >Providing DataSync access to SMB file servers</a>.
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * <p>
     * Specifies the domain name or IP address (IPv4 or IPv6) of the SMB file server that your DataSync agent connects
     * to.
     * </p>
     * <note>
     * <p>
     * If you're using Kerberos authentication, you must specify a domain name.
     * </p>
     * </note>
     * 
     * @return Specifies the domain name or IP address (IPv4 or IPv6) of the SMB file server that your DataSync agent
     *         connects to.</p> <note>
     *         <p>
     *         If you're using Kerberos authentication, you must specify a domain name.
     *         </p>
     */
    public final String serverHostname() {
        return serverHostname;
    }

    /**
     * <p>
     * Specifies the user name that can mount your SMB file server and has permission to access the files and folders
     * involved in your transfer. This parameter applies only if <code>AuthenticationType</code> is set to
     * <code>NTLM</code>.
     * </p>
     * <p>
     * For information about choosing a user with the right level of access for your transfer, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     * >Providing DataSync access to SMB file servers</a>.
     * </p>
     * 
     * @return Specifies the user name that can mount your SMB file server and has permission to access the files and
     *         folders involved in your transfer. This parameter applies only if <code>AuthenticationType</code> is set
     *         to <code>NTLM</code>.</p>
     *         <p>
     *         For information about choosing a user with the right level of access for your transfer, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     *         >Providing DataSync access to SMB file servers</a>.
     */
    public final String user() {
        return user;
    }

    /**
     * <p>
     * Specifies the Windows domain name that your SMB file server belongs to. This parameter applies only if
     * <code>AuthenticationType</code> is set to <code>NTLM</code>.
     * </p>
     * <p>
     * If you have multiple domains in your environment, configuring this parameter makes sure that DataSync connects to
     * the right file server.
     * </p>
     * 
     * @return Specifies the Windows domain name that your SMB file server belongs to. This parameter applies only if
     *         <code>AuthenticationType</code> is set to <code>NTLM</code>.</p>
     *         <p>
     *         If you have multiple domains in your environment, configuring this parameter makes sure that DataSync
     *         connects to the right file server.
     */
    public final String domain() {
        return domain;
    }

    /**
     * <p>
     * Specifies the password of the user who can mount your SMB file server and has permission to access the files and
     * folders involved in your transfer. This parameter applies only if <code>AuthenticationType</code> is set to
     * <code>NTLM</code>.
     * </p>
     * 
     * @return Specifies the password of the user who can mount your SMB file server and has permission to access the
     *         files and folders involved in your transfer. This parameter applies only if
     *         <code>AuthenticationType</code> is set to <code>NTLM</code>.
     */
    public final String password() {
        return password;
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the DataSync agent (or agents) that can connect to your SMB file server. You specify an agent by using
     * its Amazon Resource Name (ARN).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentArns} method.
     * </p>
     * 
     * @return Specifies the DataSync agent (or agents) that can connect to your SMB file server. You specify an agent
     *         by using its Amazon Resource Name (ARN).
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    /**
     * Returns the value of the MountOptions property for this object.
     * 
     * @return The value of the MountOptions property for this object.
     */
    public final SmbMountOptions mountOptions() {
        return mountOptions;
    }

    /**
     * <p>
     * Specifies the authentication protocol that DataSync uses to connect to your SMB file server. DataSync supports
     * <code>NTLM</code> (default) and <code>KERBEROS</code> authentication.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     * >Providing DataSync access to SMB file servers</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link SmbAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return Specifies the authentication protocol that DataSync uses to connect to your SMB file server. DataSync
     *         supports <code>NTLM</code> (default) and <code>KERBEROS</code> authentication.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     *         >Providing DataSync access to SMB file servers</a>.
     * @see SmbAuthenticationType
     */
    public final SmbAuthenticationType authenticationType() {
        return SmbAuthenticationType.fromValue(authenticationType);
    }

    /**
     * <p>
     * Specifies the authentication protocol that DataSync uses to connect to your SMB file server. DataSync supports
     * <code>NTLM</code> (default) and <code>KERBEROS</code> authentication.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     * >Providing DataSync access to SMB file servers</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link SmbAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return Specifies the authentication protocol that DataSync uses to connect to your SMB file server. DataSync
     *         supports <code>NTLM</code> (default) and <code>KERBEROS</code> authentication.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
     *         >Providing DataSync access to SMB file servers</a>.
     * @see SmbAuthenticationType
     */
    public final String authenticationTypeAsString() {
        return authenticationType;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIpAddresses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDnsIpAddresses() {
        return dnsIpAddresses != null && !(dnsIpAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the IP addresses (IPv4 or IPv6) for the DNS servers that your SMB file server belongs to. This
     * parameter applies only if <code>AuthenticationType</code> is set to <code>KERBEROS</code>.
     * </p>
     * <p>
     * If you have multiple domains in your environment, configuring this parameter makes sure that DataSync connects to
     * the right SMB file server.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIpAddresses} method.
     * </p>
     * 
     * @return Specifies the IP addresses (IPv4 or IPv6) for the DNS servers that your SMB file server belongs to. This
     *         parameter applies only if <code>AuthenticationType</code> is set to <code>KERBEROS</code>.</p>
     *         <p>
     *         If you have multiple domains in your environment, configuring this parameter makes sure that DataSync
     *         connects to the right SMB file server.
     */
    public final List<String> dnsIpAddresses() {
        return dnsIpAddresses;
    }

    /**
     * <p>
     * Specifies a Kerberos prinicpal, which is an identity in your Kerberos realm that has permission to access the
     * files, folders, and file metadata in your SMB file server.
     * </p>
     * <p>
     * A Kerberos principal might look like <code>HOST/kerberosuser@MYDOMAIN.ORG</code>.
     * </p>
     * <p>
     * Principal names are case sensitive. Your DataSync task execution will fail if the principal that you specify for
     * this parameter doesn’t exactly match the principal that you use to create the keytab file.
     * </p>
     * 
     * @return Specifies a Kerberos prinicpal, which is an identity in your Kerberos realm that has permission to access
     *         the files, folders, and file metadata in your SMB file server.</p>
     *         <p>
     *         A Kerberos principal might look like <code>HOST/kerberosuser@MYDOMAIN.ORG</code>.
     *         </p>
     *         <p>
     *         Principal names are case sensitive. Your DataSync task execution will fail if the principal that you
     *         specify for this parameter doesn’t exactly match the principal that you use to create the keytab file.
     */
    public final String kerberosPrincipal() {
        return kerberosPrincipal;
    }

    /**
     * <p>
     * Specifies your Kerberos key table (keytab) file, which includes mappings between your Kerberos principal and
     * encryption keys.
     * </p>
     * <p>
     * To avoid task execution errors, make sure that the Kerberos principal that you use to create the keytab file
     * matches exactly what you specify for <code>KerberosPrincipal</code>.
     * </p>
     * 
     * @return Specifies your Kerberos key table (keytab) file, which includes mappings between your Kerberos principal
     *         and encryption keys.</p>
     *         <p>
     *         To avoid task execution errors, make sure that the Kerberos principal that you use to create the keytab
     *         file matches exactly what you specify for <code>KerberosPrincipal</code>.
     */
    public final SdkBytes kerberosKeytab() {
        return kerberosKeytab;
    }

    /**
     * <p>
     * Specifies a Kerberos configuration file (<code>krb5.conf</code>) that defines your Kerberos realm configuration.
     * </p>
     * <p>
     * The file must be base64 encoded. If you're using the CLI, the encoding is done for you.
     * </p>
     * 
     * @return Specifies a Kerberos configuration file (<code>krb5.conf</code>) that defines your Kerberos realm
     *         configuration.</p>
     *         <p>
     *         The file must be base64 encoded. If you're using the CLI, the encoding is done for you.
     */
    public final SdkBytes kerberosKrb5Conf() {
        return kerberosKrb5Conf;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(locationArn());
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(serverHostname());
        hashCode = 31 * hashCode + Objects.hashCode(user());
        hashCode = 31 * hashCode + Objects.hashCode(domain());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(mountOptions());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIpAddresses() ? dnsIpAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kerberosPrincipal());
        hashCode = 31 * hashCode + Objects.hashCode(kerberosKeytab());
        hashCode = 31 * hashCode + Objects.hashCode(kerberosKrb5Conf());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateLocationSmbRequest)) {
            return false;
        }
        UpdateLocationSmbRequest other = (UpdateLocationSmbRequest) obj;
        return Objects.equals(locationArn(), other.locationArn()) && Objects.equals(subdirectory(), other.subdirectory())
                && Objects.equals(serverHostname(), other.serverHostname()) && Objects.equals(user(), other.user())
                && Objects.equals(domain(), other.domain()) && Objects.equals(password(), other.password())
                && hasAgentArns() == other.hasAgentArns() && Objects.equals(agentArns(), other.agentArns())
                && Objects.equals(mountOptions(), other.mountOptions())
                && Objects.equals(authenticationTypeAsString(), other.authenticationTypeAsString())
                && hasDnsIpAddresses() == other.hasDnsIpAddresses() && Objects.equals(dnsIpAddresses(), other.dnsIpAddresses())
                && Objects.equals(kerberosPrincipal(), other.kerberosPrincipal())
                && Objects.equals(kerberosKeytab(), other.kerberosKeytab())
                && Objects.equals(kerberosKrb5Conf(), other.kerberosKrb5Conf());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateLocationSmbRequest").add("LocationArn", locationArn()).add("Subdirectory", subdirectory())
                .add("ServerHostname", serverHostname()).add("User", user()).add("Domain", domain())
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AgentArns", hasAgentArns() ? agentArns() : null).add("MountOptions", mountOptions())
                .add("AuthenticationType", authenticationTypeAsString())
                .add("DnsIpAddresses", hasDnsIpAddresses() ? dnsIpAddresses() : null)
                .add("KerberosPrincipal", kerberosPrincipal()).add("KerberosKeytab", kerberosKeytab())
                .add("KerberosKrb5Conf", kerberosKrb5Conf()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LocationArn":
            return Optional.ofNullable(clazz.cast(locationArn()));
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "ServerHostname":
            return Optional.ofNullable(clazz.cast(serverHostname()));
        case "User":
            return Optional.ofNullable(clazz.cast(user()));
        case "Domain":
            return Optional.ofNullable(clazz.cast(domain()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        case "MountOptions":
            return Optional.ofNullable(clazz.cast(mountOptions()));
        case "AuthenticationType":
            return Optional.ofNullable(clazz.cast(authenticationTypeAsString()));
        case "DnsIpAddresses":
            return Optional.ofNullable(clazz.cast(dnsIpAddresses()));
        case "KerberosPrincipal":
            return Optional.ofNullable(clazz.cast(kerberosPrincipal()));
        case "KerberosKeytab":
            return Optional.ofNullable(clazz.cast(kerberosKeytab()));
        case "KerberosKrb5Conf":
            return Optional.ofNullable(clazz.cast(kerberosKrb5Conf()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LocationArn", LOCATION_ARN_FIELD);
        map.put("Subdirectory", SUBDIRECTORY_FIELD);
        map.put("ServerHostname", SERVER_HOSTNAME_FIELD);
        map.put("User", USER_FIELD);
        map.put("Domain", DOMAIN_FIELD);
        map.put("Password", PASSWORD_FIELD);
        map.put("AgentArns", AGENT_ARNS_FIELD);
        map.put("MountOptions", MOUNT_OPTIONS_FIELD);
        map.put("AuthenticationType", AUTHENTICATION_TYPE_FIELD);
        map.put("DnsIpAddresses", DNS_IP_ADDRESSES_FIELD);
        map.put("KerberosPrincipal", KERBEROS_PRINCIPAL_FIELD);
        map.put("KerberosKeytab", KERBEROS_KEYTAB_FIELD);
        map.put("KerberosKrb5Conf", KERBEROS_KRB5_CONF_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UpdateLocationSmbRequest, T> g) {
        return obj -> g.apply((UpdateLocationSmbRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends DataSyncRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateLocationSmbRequest> {
        /**
         * <p>
         * Specifies the ARN of the SMB location that you want to update.
         * </p>
         * 
         * @param locationArn
         *        Specifies the ARN of the SMB location that you want to update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationArn(String locationArn);

        /**
         * <p>
         * Specifies the name of the share exported by your SMB file server where DataSync will read or write data. You
         * can include a subdirectory in the share path (for example, <code>/path/to/subdirectory</code>). Make sure
         * that other SMB clients in your network can also mount this path.
         * </p>
         * <p>
         * To copy all data in the specified subdirectory, DataSync must be able to mount the SMB share and access all
         * of its data. For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         * >Providing DataSync access to SMB file servers</a>.
         * </p>
         * 
         * @param subdirectory
         *        Specifies the name of the share exported by your SMB file server where DataSync will read or write
         *        data. You can include a subdirectory in the share path (for example,
         *        <code>/path/to/subdirectory</code>). Make sure that other SMB clients in your network can also mount
         *        this path.</p>
         *        <p>
         *        To copy all data in the specified subdirectory, DataSync must be able to mount the SMB share and
         *        access all of its data. For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         *        >Providing DataSync access to SMB file servers</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * Specifies the domain name or IP address (IPv4 or IPv6) of the SMB file server that your DataSync agent
         * connects to.
         * </p>
         * <note>
         * <p>
         * If you're using Kerberos authentication, you must specify a domain name.
         * </p>
         * </note>
         * 
         * @param serverHostname
         *        Specifies the domain name or IP address (IPv4 or IPv6) of the SMB file server that your DataSync agent
         *        connects to.</p> <note>
         *        <p>
         *        If you're using Kerberos authentication, you must specify a domain name.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverHostname(String serverHostname);

        /**
         * <p>
         * Specifies the user name that can mount your SMB file server and has permission to access the files and
         * folders involved in your transfer. This parameter applies only if <code>AuthenticationType</code> is set to
         * <code>NTLM</code>.
         * </p>
         * <p>
         * For information about choosing a user with the right level of access for your transfer, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         * >Providing DataSync access to SMB file servers</a>.
         * </p>
         * 
         * @param user
         *        Specifies the user name that can mount your SMB file server and has permission to access the files and
         *        folders involved in your transfer. This parameter applies only if <code>AuthenticationType</code> is
         *        set to <code>NTLM</code>.</p>
         *        <p>
         *        For information about choosing a user with the right level of access for your transfer, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         *        >Providing DataSync access to SMB file servers</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);

        /**
         * <p>
         * Specifies the Windows domain name that your SMB file server belongs to. This parameter applies only if
         * <code>AuthenticationType</code> is set to <code>NTLM</code>.
         * </p>
         * <p>
         * If you have multiple domains in your environment, configuring this parameter makes sure that DataSync
         * connects to the right file server.
         * </p>
         * 
         * @param domain
         *        Specifies the Windows domain name that your SMB file server belongs to. This parameter applies only if
         *        <code>AuthenticationType</code> is set to <code>NTLM</code>.</p>
         *        <p>
         *        If you have multiple domains in your environment, configuring this parameter makes sure that DataSync
         *        connects to the right file server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domain(String domain);

        /**
         * <p>
         * Specifies the password of the user who can mount your SMB file server and has permission to access the files
         * and folders involved in your transfer. This parameter applies only if <code>AuthenticationType</code> is set
         * to <code>NTLM</code>.
         * </p>
         * 
         * @param password
         *        Specifies the password of the user who can mount your SMB file server and has permission to access the
         *        files and folders involved in your transfer. This parameter applies only if
         *        <code>AuthenticationType</code> is set to <code>NTLM</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * Specifies the DataSync agent (or agents) that can connect to your SMB file server. You specify an agent by
         * using its Amazon Resource Name (ARN).
         * </p>
         * 
         * @param agentArns
         *        Specifies the DataSync agent (or agents) that can connect to your SMB file server. You specify an
         *        agent by using its Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * Specifies the DataSync agent (or agents) that can connect to your SMB file server. You specify an agent by
         * using its Amazon Resource Name (ARN).
         * </p>
         * 
         * @param agentArns
         *        Specifies the DataSync agent (or agents) that can connect to your SMB file server. You specify an
         *        agent by using its Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        /**
         * Sets the value of the MountOptions property for this object.
         *
         * @param mountOptions
         *        The new value for the MountOptions property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountOptions(SmbMountOptions mountOptions);

        /**
         * Sets the value of the MountOptions property for this object.
         *
         * This is a convenience method that creates an instance of the {@link SmbMountOptions.Builder} avoiding the
         * need to create one manually via {@link SmbMountOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SmbMountOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #mountOptions(SmbMountOptions)}.
         * 
         * @param mountOptions
         *        a consumer that will call methods on {@link SmbMountOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountOptions(SmbMountOptions)
         */
        default Builder mountOptions(Consumer<SmbMountOptions.Builder> mountOptions) {
            return mountOptions(SmbMountOptions.builder().applyMutation(mountOptions).build());
        }

        /**
         * <p>
         * Specifies the authentication protocol that DataSync uses to connect to your SMB file server. DataSync
         * supports <code>NTLM</code> (default) and <code>KERBEROS</code> authentication.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         * >Providing DataSync access to SMB file servers</a>.
         * </p>
         * 
         * @param authenticationType
         *        Specifies the authentication protocol that DataSync uses to connect to your SMB file server. DataSync
         *        supports <code>NTLM</code> (default) and <code>KERBEROS</code> authentication.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         *        >Providing DataSync access to SMB file servers</a>.
         * @see SmbAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SmbAuthenticationType
         */
        Builder authenticationType(String authenticationType);

        /**
         * <p>
         * Specifies the authentication protocol that DataSync uses to connect to your SMB file server. DataSync
         * supports <code>NTLM</code> (default) and <code>KERBEROS</code> authentication.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         * >Providing DataSync access to SMB file servers</a>.
         * </p>
         * 
         * @param authenticationType
         *        Specifies the authentication protocol that DataSync uses to connect to your SMB file server. DataSync
         *        supports <code>NTLM</code> (default) and <code>KERBEROS</code> authentication.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html#configuring-smb-permissions"
         *        >Providing DataSync access to SMB file servers</a>.
         * @see SmbAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SmbAuthenticationType
         */
        Builder authenticationType(SmbAuthenticationType authenticationType);

        /**
         * <p>
         * Specifies the IP addresses (IPv4 or IPv6) for the DNS servers that your SMB file server belongs to. This
         * parameter applies only if <code>AuthenticationType</code> is set to <code>KERBEROS</code>.
         * </p>
         * <p>
         * If you have multiple domains in your environment, configuring this parameter makes sure that DataSync
         * connects to the right SMB file server.
         * </p>
         * 
         * @param dnsIpAddresses
         *        Specifies the IP addresses (IPv4 or IPv6) for the DNS servers that your SMB file server belongs to.
         *        This parameter applies only if <code>AuthenticationType</code> is set to <code>KERBEROS</code>.</p>
         *        <p>
         *        If you have multiple domains in your environment, configuring this parameter makes sure that DataSync
         *        connects to the right SMB file server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddresses(Collection<String> dnsIpAddresses);

        /**
         * <p>
         * Specifies the IP addresses (IPv4 or IPv6) for the DNS servers that your SMB file server belongs to. This
         * parameter applies only if <code>AuthenticationType</code> is set to <code>KERBEROS</code>.
         * </p>
         * <p>
         * If you have multiple domains in your environment, configuring this parameter makes sure that DataSync
         * connects to the right SMB file server.
         * </p>
         * 
         * @param dnsIpAddresses
         *        Specifies the IP addresses (IPv4 or IPv6) for the DNS servers that your SMB file server belongs to.
         *        This parameter applies only if <code>AuthenticationType</code> is set to <code>KERBEROS</code>.</p>
         *        <p>
         *        If you have multiple domains in your environment, configuring this parameter makes sure that DataSync
         *        connects to the right SMB file server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddresses(String... dnsIpAddresses);

        /**
         * <p>
         * Specifies a Kerberos prinicpal, which is an identity in your Kerberos realm that has permission to access the
         * files, folders, and file metadata in your SMB file server.
         * </p>
         * <p>
         * A Kerberos principal might look like <code>HOST/kerberosuser@MYDOMAIN.ORG</code>.
         * </p>
         * <p>
         * Principal names are case sensitive. Your DataSync task execution will fail if the principal that you specify
         * for this parameter doesn’t exactly match the principal that you use to create the keytab file.
         * </p>
         * 
         * @param kerberosPrincipal
         *        Specifies a Kerberos prinicpal, which is an identity in your Kerberos realm that has permission to
         *        access the files, folders, and file metadata in your SMB file server.</p>
         *        <p>
         *        A Kerberos principal might look like <code>HOST/kerberosuser@MYDOMAIN.ORG</code>.
         *        </p>
         *        <p>
         *        Principal names are case sensitive. Your DataSync task execution will fail if the principal that you
         *        specify for this parameter doesn’t exactly match the principal that you use to create the keytab file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kerberosPrincipal(String kerberosPrincipal);

        /**
         * <p>
         * Specifies your Kerberos key table (keytab) file, which includes mappings between your Kerberos principal and
         * encryption keys.
         * </p>
         * <p>
         * To avoid task execution errors, make sure that the Kerberos principal that you use to create the keytab file
         * matches exactly what you specify for <code>KerberosPrincipal</code>.
         * </p>
         * 
         * @param kerberosKeytab
         *        Specifies your Kerberos key table (keytab) file, which includes mappings between your Kerberos
         *        principal and encryption keys.</p>
         *        <p>
         *        To avoid task execution errors, make sure that the Kerberos principal that you use to create the
         *        keytab file matches exactly what you specify for <code>KerberosPrincipal</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kerberosKeytab(SdkBytes kerberosKeytab);

        /**
         * <p>
         * Specifies a Kerberos configuration file (<code>krb5.conf</code>) that defines your Kerberos realm
         * configuration.
         * </p>
         * <p>
         * The file must be base64 encoded. If you're using the CLI, the encoding is done for you.
         * </p>
         * 
         * @param kerberosKrb5Conf
         *        Specifies a Kerberos configuration file (<code>krb5.conf</code>) that defines your Kerberos realm
         *        configuration.</p>
         *        <p>
         *        The file must be base64 encoded. If you're using the CLI, the encoding is done for you.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kerberosKrb5Conf(SdkBytes kerberosKrb5Conf);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String locationArn;

        private String subdirectory;

        private String serverHostname;

        private String user;

        private String domain;

        private String password;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private SmbMountOptions mountOptions;

        private String authenticationType;

        private List<String> dnsIpAddresses = DefaultSdkAutoConstructList.getInstance();

        private String kerberosPrincipal;

        private SdkBytes kerberosKeytab;

        private SdkBytes kerberosKrb5Conf;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateLocationSmbRequest model) {
            super(model);
            locationArn(model.locationArn);
            subdirectory(model.subdirectory);
            serverHostname(model.serverHostname);
            user(model.user);
            domain(model.domain);
            password(model.password);
            agentArns(model.agentArns);
            mountOptions(model.mountOptions);
            authenticationType(model.authenticationType);
            dnsIpAddresses(model.dnsIpAddresses);
            kerberosPrincipal(model.kerberosPrincipal);
            kerberosKeytab(model.kerberosKeytab);
            kerberosKrb5Conf(model.kerberosKrb5Conf);
        }

        public final String getLocationArn() {
            return locationArn;
        }

        public final void setLocationArn(String locationArn) {
            this.locationArn = locationArn;
        }

        @Override
        public final Builder locationArn(String locationArn) {
            this.locationArn = locationArn;
            return this;
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final String getServerHostname() {
            return serverHostname;
        }

        public final void setServerHostname(String serverHostname) {
            this.serverHostname = serverHostname;
        }

        @Override
        public final Builder serverHostname(String serverHostname) {
            this.serverHostname = serverHostname;
            return this;
        }

        public final String getUser() {
            return user;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        public final String getDomain() {
            return domain;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        @Override
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public final String getPassword() {
            return password;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        public final SmbMountOptions.Builder getMountOptions() {
            return mountOptions != null ? mountOptions.toBuilder() : null;
        }

        public final void setMountOptions(SmbMountOptions.BuilderImpl mountOptions) {
            this.mountOptions = mountOptions != null ? mountOptions.build() : null;
        }

        @Override
        public final Builder mountOptions(SmbMountOptions mountOptions) {
            this.mountOptions = mountOptions;
            return this;
        }

        public final String getAuthenticationType() {
            return authenticationType;
        }

        public final void setAuthenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
        }

        @Override
        public final Builder authenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        @Override
        public final Builder authenticationType(SmbAuthenticationType authenticationType) {
            this.authenticationType(authenticationType == null ? null : authenticationType.toString());
            return this;
        }

        public final Collection<String> getDnsIpAddresses() {
            if (dnsIpAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIpAddresses;
        }

        public final void setDnsIpAddresses(Collection<String> dnsIpAddresses) {
            this.dnsIpAddresses = DnsIpListCopier.copy(dnsIpAddresses);
        }

        @Override
        public final Builder dnsIpAddresses(Collection<String> dnsIpAddresses) {
            this.dnsIpAddresses = DnsIpListCopier.copy(dnsIpAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIpAddresses(String... dnsIpAddresses) {
            dnsIpAddresses(Arrays.asList(dnsIpAddresses));
            return this;
        }

        public final String getKerberosPrincipal() {
            return kerberosPrincipal;
        }

        public final void setKerberosPrincipal(String kerberosPrincipal) {
            this.kerberosPrincipal = kerberosPrincipal;
        }

        @Override
        public final Builder kerberosPrincipal(String kerberosPrincipal) {
            this.kerberosPrincipal = kerberosPrincipal;
            return this;
        }

        public final ByteBuffer getKerberosKeytab() {
            return kerberosKeytab == null ? null : kerberosKeytab.asByteBuffer();
        }

        public final void setKerberosKeytab(ByteBuffer kerberosKeytab) {
            kerberosKeytab(kerberosKeytab == null ? null : SdkBytes.fromByteBuffer(kerberosKeytab));
        }

        @Override
        public final Builder kerberosKeytab(SdkBytes kerberosKeytab) {
            this.kerberosKeytab = kerberosKeytab;
            return this;
        }

        public final ByteBuffer getKerberosKrb5Conf() {
            return kerberosKrb5Conf == null ? null : kerberosKrb5Conf.asByteBuffer();
        }

        public final void setKerberosKrb5Conf(ByteBuffer kerberosKrb5Conf) {
            kerberosKrb5Conf(kerberosKrb5Conf == null ? null : SdkBytes.fromByteBuffer(kerberosKrb5Conf));
        }

        @Override
        public final Builder kerberosKrb5Conf(SdkBytes kerberosKrb5Conf) {
            this.kerberosKrb5Conf = kerberosKrb5Conf;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateLocationSmbRequest build() {
            return new UpdateLocationSmbRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
