/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databrew;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.databrew.model.AccessDeniedException;
import software.amazon.awssdk.services.databrew.model.BatchDeleteRecipeVersionRequest;
import software.amazon.awssdk.services.databrew.model.BatchDeleteRecipeVersionResponse;
import software.amazon.awssdk.services.databrew.model.ConflictException;
import software.amazon.awssdk.services.databrew.model.CreateDatasetRequest;
import software.amazon.awssdk.services.databrew.model.CreateDatasetResponse;
import software.amazon.awssdk.services.databrew.model.CreateProfileJobRequest;
import software.amazon.awssdk.services.databrew.model.CreateProfileJobResponse;
import software.amazon.awssdk.services.databrew.model.CreateProjectRequest;
import software.amazon.awssdk.services.databrew.model.CreateProjectResponse;
import software.amazon.awssdk.services.databrew.model.CreateRecipeJobRequest;
import software.amazon.awssdk.services.databrew.model.CreateRecipeJobResponse;
import software.amazon.awssdk.services.databrew.model.CreateRecipeRequest;
import software.amazon.awssdk.services.databrew.model.CreateRecipeResponse;
import software.amazon.awssdk.services.databrew.model.CreateRulesetRequest;
import software.amazon.awssdk.services.databrew.model.CreateRulesetResponse;
import software.amazon.awssdk.services.databrew.model.CreateScheduleRequest;
import software.amazon.awssdk.services.databrew.model.CreateScheduleResponse;
import software.amazon.awssdk.services.databrew.model.DataBrewException;
import software.amazon.awssdk.services.databrew.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.databrew.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.databrew.model.DeleteJobRequest;
import software.amazon.awssdk.services.databrew.model.DeleteJobResponse;
import software.amazon.awssdk.services.databrew.model.DeleteProjectRequest;
import software.amazon.awssdk.services.databrew.model.DeleteProjectResponse;
import software.amazon.awssdk.services.databrew.model.DeleteRecipeVersionRequest;
import software.amazon.awssdk.services.databrew.model.DeleteRecipeVersionResponse;
import software.amazon.awssdk.services.databrew.model.DeleteRulesetRequest;
import software.amazon.awssdk.services.databrew.model.DeleteRulesetResponse;
import software.amazon.awssdk.services.databrew.model.DeleteScheduleRequest;
import software.amazon.awssdk.services.databrew.model.DeleteScheduleResponse;
import software.amazon.awssdk.services.databrew.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.databrew.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.databrew.model.DescribeJobRequest;
import software.amazon.awssdk.services.databrew.model.DescribeJobResponse;
import software.amazon.awssdk.services.databrew.model.DescribeJobRunRequest;
import software.amazon.awssdk.services.databrew.model.DescribeJobRunResponse;
import software.amazon.awssdk.services.databrew.model.DescribeProjectRequest;
import software.amazon.awssdk.services.databrew.model.DescribeProjectResponse;
import software.amazon.awssdk.services.databrew.model.DescribeRecipeRequest;
import software.amazon.awssdk.services.databrew.model.DescribeRecipeResponse;
import software.amazon.awssdk.services.databrew.model.DescribeRulesetRequest;
import software.amazon.awssdk.services.databrew.model.DescribeRulesetResponse;
import software.amazon.awssdk.services.databrew.model.DescribeScheduleRequest;
import software.amazon.awssdk.services.databrew.model.DescribeScheduleResponse;
import software.amazon.awssdk.services.databrew.model.InternalServerException;
import software.amazon.awssdk.services.databrew.model.ListDatasetsRequest;
import software.amazon.awssdk.services.databrew.model.ListDatasetsResponse;
import software.amazon.awssdk.services.databrew.model.ListJobRunsRequest;
import software.amazon.awssdk.services.databrew.model.ListJobRunsResponse;
import software.amazon.awssdk.services.databrew.model.ListJobsRequest;
import software.amazon.awssdk.services.databrew.model.ListJobsResponse;
import software.amazon.awssdk.services.databrew.model.ListProjectsRequest;
import software.amazon.awssdk.services.databrew.model.ListProjectsResponse;
import software.amazon.awssdk.services.databrew.model.ListRecipeVersionsRequest;
import software.amazon.awssdk.services.databrew.model.ListRecipeVersionsResponse;
import software.amazon.awssdk.services.databrew.model.ListRecipesRequest;
import software.amazon.awssdk.services.databrew.model.ListRecipesResponse;
import software.amazon.awssdk.services.databrew.model.ListRulesetsRequest;
import software.amazon.awssdk.services.databrew.model.ListRulesetsResponse;
import software.amazon.awssdk.services.databrew.model.ListSchedulesRequest;
import software.amazon.awssdk.services.databrew.model.ListSchedulesResponse;
import software.amazon.awssdk.services.databrew.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.databrew.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.databrew.model.PublishRecipeRequest;
import software.amazon.awssdk.services.databrew.model.PublishRecipeResponse;
import software.amazon.awssdk.services.databrew.model.ResourceNotFoundException;
import software.amazon.awssdk.services.databrew.model.SendProjectSessionActionRequest;
import software.amazon.awssdk.services.databrew.model.SendProjectSessionActionResponse;
import software.amazon.awssdk.services.databrew.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.databrew.model.StartJobRunRequest;
import software.amazon.awssdk.services.databrew.model.StartJobRunResponse;
import software.amazon.awssdk.services.databrew.model.StartProjectSessionRequest;
import software.amazon.awssdk.services.databrew.model.StartProjectSessionResponse;
import software.amazon.awssdk.services.databrew.model.StopJobRunRequest;
import software.amazon.awssdk.services.databrew.model.StopJobRunResponse;
import software.amazon.awssdk.services.databrew.model.TagResourceRequest;
import software.amazon.awssdk.services.databrew.model.TagResourceResponse;
import software.amazon.awssdk.services.databrew.model.UntagResourceRequest;
import software.amazon.awssdk.services.databrew.model.UntagResourceResponse;
import software.amazon.awssdk.services.databrew.model.UpdateDatasetRequest;
import software.amazon.awssdk.services.databrew.model.UpdateDatasetResponse;
import software.amazon.awssdk.services.databrew.model.UpdateProfileJobRequest;
import software.amazon.awssdk.services.databrew.model.UpdateProfileJobResponse;
import software.amazon.awssdk.services.databrew.model.UpdateProjectRequest;
import software.amazon.awssdk.services.databrew.model.UpdateProjectResponse;
import software.amazon.awssdk.services.databrew.model.UpdateRecipeJobRequest;
import software.amazon.awssdk.services.databrew.model.UpdateRecipeJobResponse;
import software.amazon.awssdk.services.databrew.model.UpdateRecipeRequest;
import software.amazon.awssdk.services.databrew.model.UpdateRecipeResponse;
import software.amazon.awssdk.services.databrew.model.UpdateRulesetRequest;
import software.amazon.awssdk.services.databrew.model.UpdateRulesetResponse;
import software.amazon.awssdk.services.databrew.model.UpdateScheduleRequest;
import software.amazon.awssdk.services.databrew.model.UpdateScheduleResponse;
import software.amazon.awssdk.services.databrew.model.ValidationException;
import software.amazon.awssdk.services.databrew.transform.BatchDeleteRecipeVersionRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.CreateProfileJobRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.CreateRecipeJobRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.CreateRecipeRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.CreateRulesetRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.CreateScheduleRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DeleteJobRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DeleteRecipeVersionRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DeleteRulesetRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DeleteScheduleRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DescribeJobRunRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DescribeProjectRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DescribeRecipeRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DescribeRulesetRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.DescribeScheduleRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListJobRunsRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListRecipeVersionsRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListRecipesRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListRulesetsRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListSchedulesRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.PublishRecipeRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.SendProjectSessionActionRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.StartJobRunRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.StartProjectSessionRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.StopJobRunRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.UpdateDatasetRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.UpdateProfileJobRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.UpdateRecipeJobRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.UpdateRecipeRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.UpdateRulesetRequestMarshaller;
import software.amazon.awssdk.services.databrew.transform.UpdateScheduleRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DataBrewAsyncClient}.
 *
 * @see DataBrewAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDataBrewAsyncClient implements DataBrewAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDataBrewAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final DataBrewServiceClientConfiguration serviceClientConfiguration;

    protected DefaultDataBrewAsyncClient(DataBrewServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes one or more versions of a recipe at a time.
     * </p>
     * <p>
     * The entire request will be rejected if:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The recipe does not exist.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is an invalid version identifier in the list of versions.
     * </p>
     * </li>
     * <li>
     * <p>
     * The version list is empty.
     * </p>
     * </li>
     * <li>
     * <p>
     * The version list size exceeds 50.
     * </p>
     * </li>
     * <li>
     * <p>
     * The version list contains duplicate entries.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The request will complete successfully, but with partial failures, if:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A version does not exist.
     * </p>
     * </li>
     * <li>
     * <p>
     * A version is being used by a job.
     * </p>
     * </li>
     * <li>
     * <p>
     * You specify <code>LATEST_WORKING</code>, but it's being used by a project.
     * </p>
     * </li>
     * <li>
     * <p>
     * The version fails to be deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The <code>LATEST_WORKING</code> version will only be deleted if the recipe has no other versions. If you try to
     * delete <code>LATEST_WORKING</code> while other versions exist (or if they can't be deleted), then
     * <code>LATEST_WORKING</code> will be listed as partial failure in the response.
     * </p>
     *
     * @param batchDeleteRecipeVersionRequest
     * @return A Java Future containing the result of the BatchDeleteRecipeVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.BatchDeleteRecipeVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/BatchDeleteRecipeVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteRecipeVersionResponse> batchDeleteRecipeVersion(
            BatchDeleteRecipeVersionRequest batchDeleteRecipeVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteRecipeVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteRecipeVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteRecipeVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteRecipeVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDeleteRecipeVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteRecipeVersionRequest, BatchDeleteRecipeVersionResponse>()
                            .withOperationName("BatchDeleteRecipeVersion")
                            .withMarshaller(new BatchDeleteRecipeVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchDeleteRecipeVersionRequest));
            CompletableFuture<BatchDeleteRecipeVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new DataBrew dataset.
     * </p>
     *
     * @param createDatasetRequest
     * @return A Java Future containing the result of the CreateDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access to the specified resource was denied.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.CreateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/CreateDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetResponse> createDataset(CreateDatasetRequest createDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                            .withOperationName("CreateDataset")
                            .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDatasetRequest));
            CompletableFuture<CreateDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new job to analyze a dataset and create its data profile.
     * </p>
     *
     * @param createProfileJobRequest
     * @return A Java Future containing the result of the CreateProfileJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access to the specified resource was denied.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.CreateProfileJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/CreateProfileJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProfileJobResponse> createProfileJob(CreateProfileJobRequest createProfileJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProfileJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProfileJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProfileJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateProfileJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProfileJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProfileJobRequest, CreateProfileJobResponse>()
                            .withOperationName("CreateProfileJob")
                            .withMarshaller(new CreateProfileJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProfileJobRequest));
            CompletableFuture<CreateProfileJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new DataBrew project.
     * </p>
     *
     * @param createProjectRequest
     * @return A Java Future containing the result of the CreateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>InternalServerException An internal service failure occurred.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProjectResponse> createProject(CreateProjectRequest createProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                            .withOperationName("CreateProject")
                            .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProjectRequest));
            CompletableFuture<CreateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new DataBrew recipe.
     * </p>
     *
     * @param createRecipeRequest
     * @return A Java Future containing the result of the CreateRecipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.CreateRecipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/CreateRecipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRecipeResponse> createRecipe(CreateRecipeRequest createRecipeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRecipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRecipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRecipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateRecipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRecipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRecipeRequest, CreateRecipeResponse>()
                            .withOperationName("CreateRecipe").withMarshaller(new CreateRecipeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRecipeRequest));
            CompletableFuture<CreateRecipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new job to transform input data, using steps defined in an existing Glue DataBrew recipe
     * </p>
     *
     * @param createRecipeJobRequest
     * @return A Java Future containing the result of the CreateRecipeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access to the specified resource was denied.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.CreateRecipeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/CreateRecipeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRecipeJobResponse> createRecipeJob(CreateRecipeJobRequest createRecipeJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRecipeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRecipeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRecipeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRecipeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRecipeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRecipeJobRequest, CreateRecipeJobResponse>()
                            .withOperationName("CreateRecipeJob")
                            .withMarshaller(new CreateRecipeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRecipeJobRequest));
            CompletableFuture<CreateRecipeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new ruleset that can be used in a profile job to validate the data quality of a dataset.
     * </p>
     *
     * @param createRulesetRequest
     * @return A Java Future containing the result of the CreateRuleset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.CreateRuleset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/CreateRuleset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRulesetResponse> createRuleset(CreateRulesetRequest createRulesetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRulesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRuleset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRulesetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateRulesetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRulesetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRulesetRequest, CreateRulesetResponse>()
                            .withOperationName("CreateRuleset")
                            .withMarshaller(new CreateRulesetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRulesetRequest));
            CompletableFuture<CreateRulesetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new schedule for one or more DataBrew jobs. Jobs can be run at a specific date and time, or at regular
     * intervals.
     * </p>
     *
     * @param createScheduleRequest
     * @return A Java Future containing the result of the CreateSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.CreateSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/CreateSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateScheduleResponse> createSchedule(CreateScheduleRequest createScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateScheduleRequest, CreateScheduleResponse>()
                            .withOperationName("CreateSchedule")
                            .withMarshaller(new CreateScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createScheduleRequest));
            CompletableFuture<CreateScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a dataset from DataBrew.
     * </p>
     *
     * @param deleteDatasetRequest
     * @return A Java Future containing the result of the DeleteDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DeleteDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DeleteDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetResponse> deleteDataset(DeleteDatasetRequest deleteDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                            .withOperationName("DeleteDataset")
                            .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDatasetRequest));
            CompletableFuture<DeleteDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified DataBrew job.
     * </p>
     *
     * @param deleteJobRequest
     * @return A Java Future containing the result of the DeleteJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DeleteJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DeleteJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobResponse> deleteJob(DeleteJobRequest deleteJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobRequest, DeleteJobResponse>().withOperationName("DeleteJob")
                            .withMarshaller(new DeleteJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteJobRequest));
            CompletableFuture<DeleteJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing DataBrew project.
     * </p>
     *
     * @param deleteProjectRequest
     * @return A Java Future containing the result of the DeleteProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProjectResponse> deleteProject(DeleteProjectRequest deleteProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                            .withOperationName("DeleteProject")
                            .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProjectRequest));
            CompletableFuture<DeleteProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a single version of a DataBrew recipe.
     * </p>
     *
     * @param deleteRecipeVersionRequest
     * @return A Java Future containing the result of the DeleteRecipeVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DeleteRecipeVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DeleteRecipeVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRecipeVersionResponse> deleteRecipeVersion(
            DeleteRecipeVersionRequest deleteRecipeVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecipeVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecipeVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRecipeVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRecipeVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRecipeVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRecipeVersionRequest, DeleteRecipeVersionResponse>()
                            .withOperationName("DeleteRecipeVersion")
                            .withMarshaller(new DeleteRecipeVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRecipeVersionRequest));
            CompletableFuture<DeleteRecipeVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a ruleset.
     * </p>
     *
     * @param deleteRulesetRequest
     * @return A Java Future containing the result of the DeleteRuleset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DeleteRuleset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DeleteRuleset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRulesetResponse> deleteRuleset(DeleteRulesetRequest deleteRulesetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRulesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRuleset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRulesetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteRulesetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRulesetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRulesetRequest, DeleteRulesetResponse>()
                            .withOperationName("DeleteRuleset")
                            .withMarshaller(new DeleteRulesetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRulesetRequest));
            CompletableFuture<DeleteRulesetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified DataBrew schedule.
     * </p>
     *
     * @param deleteScheduleRequest
     * @return A Java Future containing the result of the DeleteSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DeleteSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DeleteSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScheduleResponse> deleteSchedule(DeleteScheduleRequest deleteScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScheduleRequest, DeleteScheduleResponse>()
                            .withOperationName("DeleteSchedule")
                            .withMarshaller(new DeleteScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteScheduleRequest));
            CompletableFuture<DeleteScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the definition of a specific DataBrew dataset.
     * </p>
     *
     * @param describeDatasetRequest
     * @return A Java Future containing the result of the DescribeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DescribeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DescribeDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetResponse> describeDataset(DescribeDatasetRequest describeDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                            .withOperationName("DescribeDataset")
                            .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDatasetRequest));
            CompletableFuture<DescribeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the definition of a specific DataBrew job.
     * </p>
     *
     * @param describeJobRequest
     * @return A Java Future containing the result of the DescribeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DescribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobResponse> describeJob(DescribeJobRequest describeJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                            .withOperationName("DescribeJob").withMarshaller(new DescribeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobRequest));
            CompletableFuture<DescribeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Represents one run of a DataBrew job.
     * </p>
     *
     * @param describeJobRunRequest
     * @return A Java Future containing the result of the DescribeJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DescribeJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DescribeJobRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobRunResponse> describeJobRun(DescribeJobRunRequest describeJobRunRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobRunRequest, DescribeJobRunResponse>()
                            .withOperationName("DescribeJobRun")
                            .withMarshaller(new DescribeJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobRunRequest));
            CompletableFuture<DescribeJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the definition of a specific DataBrew project.
     * </p>
     *
     * @param describeProjectRequest
     * @return A Java Future containing the result of the DescribeProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DescribeProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DescribeProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProjectResponse> describeProject(DescribeProjectRequest describeProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProjectRequest, DescribeProjectResponse>()
                            .withOperationName("DescribeProject")
                            .withMarshaller(new DescribeProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeProjectRequest));
            CompletableFuture<DescribeProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the definition of a specific DataBrew recipe corresponding to a particular version.
     * </p>
     *
     * @param describeRecipeRequest
     * @return A Java Future containing the result of the DescribeRecipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DescribeRecipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DescribeRecipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRecipeResponse> describeRecipe(DescribeRecipeRequest describeRecipeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRecipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRecipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRecipeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRecipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRecipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRecipeRequest, DescribeRecipeResponse>()
                            .withOperationName("DescribeRecipe")
                            .withMarshaller(new DescribeRecipeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRecipeRequest));
            CompletableFuture<DescribeRecipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about the ruleset.
     * </p>
     *
     * @param describeRulesetRequest
     * @return A Java Future containing the result of the DescribeRuleset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DescribeRuleset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DescribeRuleset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRulesetResponse> describeRuleset(DescribeRulesetRequest describeRulesetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRulesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRuleset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRulesetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRulesetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRulesetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRulesetRequest, DescribeRulesetResponse>()
                            .withOperationName("DescribeRuleset")
                            .withMarshaller(new DescribeRulesetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRulesetRequest));
            CompletableFuture<DescribeRulesetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the definition of a specific DataBrew schedule.
     * </p>
     *
     * @param describeScheduleRequest
     * @return A Java Future containing the result of the DescribeSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.DescribeSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DescribeSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScheduleResponse> describeSchedule(DescribeScheduleRequest describeScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScheduleRequest, DescribeScheduleResponse>()
                            .withOperationName("DescribeSchedule")
                            .withMarshaller(new DescribeScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeScheduleRequest));
            CompletableFuture<DescribeScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the DataBrew datasets.
     * </p>
     *
     * @param listDatasetsRequest
     * @return A Java Future containing the result of the ListDatasets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetsResponse> listDatasets(ListDatasetsRequest listDatasetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatasetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                            .withOperationName("ListDatasets").withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDatasetsRequest));
            CompletableFuture<ListDatasetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the previous runs of a particular DataBrew job.
     * </p>
     *
     * @param listJobRunsRequest
     * @return A Java Future containing the result of the ListJobRuns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListJobRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListJobRuns" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobRunsResponse> listJobRuns(ListJobRunsRequest listJobRunsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobRuns");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobRunsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobRunsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobRunsRequest, ListJobRunsResponse>()
                            .withOperationName("ListJobRuns").withMarshaller(new ListJobRunsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJobRunsRequest));
            CompletableFuture<ListJobRunsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the DataBrew jobs that are defined.
     * </p>
     *
     * @param listJobsRequest
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listJobsRequest));
            CompletableFuture<ListJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the DataBrew projects that are defined.
     * </p>
     *
     * @param listProjectsRequest
     * @return A Java Future containing the result of the ListProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProjectsResponse> listProjects(ListProjectsRequest listProjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                            .withOperationName("ListProjects").withMarshaller(new ListProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProjectsRequest));
            CompletableFuture<ListProjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the versions of a particular DataBrew recipe, except for <code>LATEST_WORKING</code>.
     * </p>
     *
     * @param listRecipeVersionsRequest
     * @return A Java Future containing the result of the ListRecipeVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListRecipeVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListRecipeVersions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecipeVersionsResponse> listRecipeVersions(ListRecipeVersionsRequest listRecipeVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecipeVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecipeVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecipeVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRecipeVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecipeVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecipeVersionsRequest, ListRecipeVersionsResponse>()
                            .withOperationName("ListRecipeVersions")
                            .withMarshaller(new ListRecipeVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRecipeVersionsRequest));
            CompletableFuture<ListRecipeVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the DataBrew recipes that are defined.
     * </p>
     *
     * @param listRecipesRequest
     * @return A Java Future containing the result of the ListRecipes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListRecipes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListRecipes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecipesResponse> listRecipes(ListRecipesRequest listRecipesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecipesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecipes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecipesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRecipesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecipesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecipesRequest, ListRecipesResponse>()
                            .withOperationName("ListRecipes").withMarshaller(new ListRecipesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRecipesRequest));
            CompletableFuture<ListRecipesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all rulesets available in the current account or rulesets associated with a specific resource (dataset).
     * </p>
     *
     * @param listRulesetsRequest
     * @return A Java Future containing the result of the ListRulesets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListRulesets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListRulesets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRulesetsResponse> listRulesets(ListRulesetsRequest listRulesetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRulesetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRulesets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRulesetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRulesetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRulesetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRulesetsRequest, ListRulesetsResponse>()
                            .withOperationName("ListRulesets").withMarshaller(new ListRulesetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRulesetsRequest));
            CompletableFuture<ListRulesetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the DataBrew schedules that are defined.
     * </p>
     *
     * @param listSchedulesRequest
     * @return A Java Future containing the result of the ListSchedules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListSchedules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListSchedules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchedulesResponse> listSchedules(ListSchedulesRequest listSchedulesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchedulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchedules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchedulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSchedulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchedulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchedulesRequest, ListSchedulesResponse>()
                            .withOperationName("ListSchedules")
                            .withMarshaller(new ListSchedulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSchedulesRequest));
            CompletableFuture<ListSchedulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the tags for a DataBrew resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service failure occurred.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Publishes a new version of a DataBrew recipe.
     * </p>
     *
     * @param publishRecipeRequest
     * @return A Java Future containing the result of the PublishRecipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.PublishRecipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/PublishRecipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PublishRecipeResponse> publishRecipe(PublishRecipeRequest publishRecipeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, publishRecipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PublishRecipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PublishRecipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PublishRecipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PublishRecipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PublishRecipeRequest, PublishRecipeResponse>()
                            .withOperationName("PublishRecipe")
                            .withMarshaller(new PublishRecipeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(publishRecipeRequest));
            CompletableFuture<PublishRecipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Performs a recipe step within an interactive DataBrew session that's currently open.
     * </p>
     *
     * @param sendProjectSessionActionRequest
     * @return A Java Future containing the result of the SendProjectSessionAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.SendProjectSessionAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/SendProjectSessionAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendProjectSessionActionResponse> sendProjectSessionAction(
            SendProjectSessionActionRequest sendProjectSessionActionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendProjectSessionActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendProjectSessionAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendProjectSessionActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendProjectSessionActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendProjectSessionActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendProjectSessionActionRequest, SendProjectSessionActionResponse>()
                            .withOperationName("SendProjectSessionAction")
                            .withMarshaller(new SendProjectSessionActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendProjectSessionActionRequest));
            CompletableFuture<SendProjectSessionActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Runs a DataBrew job.
     * </p>
     *
     * @param startJobRunRequest
     * @return A Java Future containing the result of the StartJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.StartJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/StartJobRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartJobRunResponse> startJobRun(StartJobRunRequest startJobRunRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartJobRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartJobRunRequest, StartJobRunResponse>()
                            .withOperationName("StartJobRun").withMarshaller(new StartJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startJobRunRequest));
            CompletableFuture<StartJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an interactive session, enabling you to manipulate data in a DataBrew project.
     * </p>
     *
     * @param startProjectSessionRequest
     * @return A Java Future containing the result of the StartProjectSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.StartProjectSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/StartProjectSession" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartProjectSessionResponse> startProjectSession(
            StartProjectSessionRequest startProjectSessionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startProjectSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartProjectSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartProjectSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartProjectSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartProjectSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartProjectSessionRequest, StartProjectSessionResponse>()
                            .withOperationName("StartProjectSession")
                            .withMarshaller(new StartProjectSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startProjectSessionRequest));
            CompletableFuture<StartProjectSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a particular run of a job.
     * </p>
     *
     * @param stopJobRunRequest
     * @return A Java Future containing the result of the StopJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.StopJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/StopJobRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopJobRunResponse> stopJobRun(StopJobRunRequest stopJobRunRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopJobRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopJobRunRequest, StopJobRunResponse>().withOperationName("StopJobRun")
                            .withMarshaller(new StopJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopJobRunRequest));
            CompletableFuture<StopJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds metadata tags to a DataBrew resource, such as a dataset, project, recipe, job, or schedule.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service failure occurred.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes metadata tags from a DataBrew resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service failure occurred.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the definition of an existing DataBrew dataset.
     * </p>
     *
     * @param updateDatasetRequest
     * @return A Java Future containing the result of the UpdateDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access to the specified resource was denied.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.UpdateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/UpdateDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatasetResponse> updateDataset(UpdateDatasetRequest updateDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatasetRequest, UpdateDatasetResponse>()
                            .withOperationName("UpdateDataset")
                            .withMarshaller(new UpdateDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDatasetRequest));
            CompletableFuture<UpdateDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the definition of an existing profile job.
     * </p>
     *
     * @param updateProfileJobRequest
     * @return A Java Future containing the result of the UpdateProfileJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access to the specified resource was denied.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.UpdateProfileJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/UpdateProfileJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProfileJobResponse> updateProfileJob(UpdateProfileJobRequest updateProfileJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProfileJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProfileJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProfileJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateProfileJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProfileJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProfileJobRequest, UpdateProfileJobResponse>()
                            .withOperationName("UpdateProfileJob")
                            .withMarshaller(new UpdateProfileJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProfileJobRequest));
            CompletableFuture<UpdateProfileJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the definition of an existing DataBrew project.
     * </p>
     *
     * @param updateProjectRequest
     * @return A Java Future containing the result of the UpdateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectResponse> updateProject(UpdateProjectRequest updateProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                            .withOperationName("UpdateProject")
                            .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProjectRequest));
            CompletableFuture<UpdateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the definition of the <code>LATEST_WORKING</code> version of a DataBrew recipe.
     * </p>
     *
     * @param updateRecipeRequest
     * @return A Java Future containing the result of the UpdateRecipe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.UpdateRecipe
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/UpdateRecipe" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRecipeResponse> updateRecipe(UpdateRecipeRequest updateRecipeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRecipeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRecipe");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRecipeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateRecipeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRecipeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRecipeRequest, UpdateRecipeResponse>()
                            .withOperationName("UpdateRecipe").withMarshaller(new UpdateRecipeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRecipeRequest));
            CompletableFuture<UpdateRecipeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the definition of an existing DataBrew recipe job.
     * </p>
     *
     * @param updateRecipeJobRequest
     * @return A Java Future containing the result of the UpdateRecipeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException Access to the specified resource was denied.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.UpdateRecipeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/UpdateRecipeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRecipeJobResponse> updateRecipeJob(UpdateRecipeJobRequest updateRecipeJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRecipeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRecipeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRecipeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRecipeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRecipeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRecipeJobRequest, UpdateRecipeJobResponse>()
                            .withOperationName("UpdateRecipeJob")
                            .withMarshaller(new UpdateRecipeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRecipeJobRequest));
            CompletableFuture<UpdateRecipeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates specified ruleset.
     * </p>
     *
     * @param updateRulesetRequest
     * @return A Java Future containing the result of the UpdateRuleset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.UpdateRuleset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/UpdateRuleset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRulesetResponse> updateRuleset(UpdateRulesetRequest updateRulesetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRulesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRuleset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRulesetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateRulesetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRulesetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRulesetRequest, UpdateRulesetResponse>()
                            .withOperationName("UpdateRuleset")
                            .withMarshaller(new UpdateRulesetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRulesetRequest));
            CompletableFuture<UpdateRulesetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the definition of an existing DataBrew schedule.
     * </p>
     *
     * @param updateScheduleRequest
     * @return A Java Future containing the result of the UpdateSchedule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ServiceQuotaExceededException A service quota is exceeded.</li>
     *         <li>ValidationException The input parameters for this request failed validation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataBrewException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataBrewAsyncClient.UpdateSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/UpdateSchedule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateScheduleResponse> updateSchedule(UpdateScheduleRequest updateScheduleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScheduleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataBrew");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSchedule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateScheduleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateScheduleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateScheduleRequest, UpdateScheduleResponse>()
                            .withOperationName("UpdateSchedule")
                            .withMarshaller(new UpdateScheduleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateScheduleRequest));
            CompletableFuture<UpdateScheduleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final DataBrewServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DataBrewException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
