/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connectcases;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.connectcases.model.AccessDeniedException;
import software.amazon.awssdk.services.connectcases.model.BatchGetFieldRequest;
import software.amazon.awssdk.services.connectcases.model.BatchGetFieldResponse;
import software.amazon.awssdk.services.connectcases.model.BatchPutFieldOptionsRequest;
import software.amazon.awssdk.services.connectcases.model.BatchPutFieldOptionsResponse;
import software.amazon.awssdk.services.connectcases.model.ConflictException;
import software.amazon.awssdk.services.connectcases.model.ConnectCasesException;
import software.amazon.awssdk.services.connectcases.model.CreateCaseRequest;
import software.amazon.awssdk.services.connectcases.model.CreateCaseResponse;
import software.amazon.awssdk.services.connectcases.model.CreateDomainRequest;
import software.amazon.awssdk.services.connectcases.model.CreateDomainResponse;
import software.amazon.awssdk.services.connectcases.model.CreateFieldRequest;
import software.amazon.awssdk.services.connectcases.model.CreateFieldResponse;
import software.amazon.awssdk.services.connectcases.model.CreateLayoutRequest;
import software.amazon.awssdk.services.connectcases.model.CreateLayoutResponse;
import software.amazon.awssdk.services.connectcases.model.CreateRelatedItemRequest;
import software.amazon.awssdk.services.connectcases.model.CreateRelatedItemResponse;
import software.amazon.awssdk.services.connectcases.model.CreateTemplateRequest;
import software.amazon.awssdk.services.connectcases.model.CreateTemplateResponse;
import software.amazon.awssdk.services.connectcases.model.DeleteDomainRequest;
import software.amazon.awssdk.services.connectcases.model.DeleteDomainResponse;
import software.amazon.awssdk.services.connectcases.model.GetCaseEventConfigurationRequest;
import software.amazon.awssdk.services.connectcases.model.GetCaseEventConfigurationResponse;
import software.amazon.awssdk.services.connectcases.model.GetCaseRequest;
import software.amazon.awssdk.services.connectcases.model.GetCaseResponse;
import software.amazon.awssdk.services.connectcases.model.GetDomainRequest;
import software.amazon.awssdk.services.connectcases.model.GetDomainResponse;
import software.amazon.awssdk.services.connectcases.model.GetLayoutRequest;
import software.amazon.awssdk.services.connectcases.model.GetLayoutResponse;
import software.amazon.awssdk.services.connectcases.model.GetTemplateRequest;
import software.amazon.awssdk.services.connectcases.model.GetTemplateResponse;
import software.amazon.awssdk.services.connectcases.model.InternalServerException;
import software.amazon.awssdk.services.connectcases.model.ListCasesForContactRequest;
import software.amazon.awssdk.services.connectcases.model.ListCasesForContactResponse;
import software.amazon.awssdk.services.connectcases.model.ListDomainsRequest;
import software.amazon.awssdk.services.connectcases.model.ListDomainsResponse;
import software.amazon.awssdk.services.connectcases.model.ListFieldOptionsRequest;
import software.amazon.awssdk.services.connectcases.model.ListFieldOptionsResponse;
import software.amazon.awssdk.services.connectcases.model.ListFieldsRequest;
import software.amazon.awssdk.services.connectcases.model.ListFieldsResponse;
import software.amazon.awssdk.services.connectcases.model.ListLayoutsRequest;
import software.amazon.awssdk.services.connectcases.model.ListLayoutsResponse;
import software.amazon.awssdk.services.connectcases.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.connectcases.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.connectcases.model.ListTemplatesRequest;
import software.amazon.awssdk.services.connectcases.model.ListTemplatesResponse;
import software.amazon.awssdk.services.connectcases.model.PutCaseEventConfigurationRequest;
import software.amazon.awssdk.services.connectcases.model.PutCaseEventConfigurationResponse;
import software.amazon.awssdk.services.connectcases.model.ResourceNotFoundException;
import software.amazon.awssdk.services.connectcases.model.SearchCasesRequest;
import software.amazon.awssdk.services.connectcases.model.SearchCasesResponse;
import software.amazon.awssdk.services.connectcases.model.SearchRelatedItemsRequest;
import software.amazon.awssdk.services.connectcases.model.SearchRelatedItemsResponse;
import software.amazon.awssdk.services.connectcases.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.connectcases.model.TagResourceRequest;
import software.amazon.awssdk.services.connectcases.model.TagResourceResponse;
import software.amazon.awssdk.services.connectcases.model.ThrottlingException;
import software.amazon.awssdk.services.connectcases.model.UntagResourceRequest;
import software.amazon.awssdk.services.connectcases.model.UntagResourceResponse;
import software.amazon.awssdk.services.connectcases.model.UpdateCaseRequest;
import software.amazon.awssdk.services.connectcases.model.UpdateCaseResponse;
import software.amazon.awssdk.services.connectcases.model.UpdateFieldRequest;
import software.amazon.awssdk.services.connectcases.model.UpdateFieldResponse;
import software.amazon.awssdk.services.connectcases.model.UpdateLayoutRequest;
import software.amazon.awssdk.services.connectcases.model.UpdateLayoutResponse;
import software.amazon.awssdk.services.connectcases.model.UpdateTemplateRequest;
import software.amazon.awssdk.services.connectcases.model.UpdateTemplateResponse;
import software.amazon.awssdk.services.connectcases.model.ValidationException;
import software.amazon.awssdk.services.connectcases.transform.BatchGetFieldRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.BatchPutFieldOptionsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateCaseRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateFieldRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateLayoutRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateRelatedItemRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.CreateTemplateRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetCaseEventConfigurationRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetCaseRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetDomainRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetLayoutRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.GetTemplateRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListCasesForContactRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListFieldOptionsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListFieldsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListLayoutsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.ListTemplatesRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.PutCaseEventConfigurationRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.SearchCasesRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.SearchRelatedItemsRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UpdateCaseRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UpdateFieldRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UpdateLayoutRequestMarshaller;
import software.amazon.awssdk.services.connectcases.transform.UpdateTemplateRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ConnectCasesAsyncClient}.
 *
 * @see ConnectCasesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectCasesAsyncClient implements ConnectCasesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultConnectCasesAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ConnectCasesServiceClientConfiguration serviceClientConfiguration;

    protected DefaultConnectCasesAsyncClient(ConnectCasesServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns the description for the list of fields in the request parameters.
     * </p>
     *
     * @param batchGetFieldRequest
     * @return A Java Future containing the result of the BatchGetField operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.BatchGetField
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/BatchGetField" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetFieldResponse> batchGetField(BatchGetFieldRequest batchGetFieldRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetFieldRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetField");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetFieldResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BatchGetFieldResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetFieldResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetFieldRequest, BatchGetFieldResponse>()
                            .withOperationName("BatchGetField")
                            .withMarshaller(new BatchGetFieldRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGetFieldRequest));
            CompletableFuture<BatchGetFieldResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates and updates a set of field options for a single select field in a Cases domain.
     * </p>
     *
     * @param batchPutFieldOptionsRequest
     * @return A Java Future containing the result of the BatchPutFieldOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded. For a list of service quotas, see
     *         <a href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.BatchPutFieldOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/BatchPutFieldOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchPutFieldOptionsResponse> batchPutFieldOptions(
            BatchPutFieldOptionsRequest batchPutFieldOptionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutFieldOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutFieldOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchPutFieldOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchPutFieldOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchPutFieldOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchPutFieldOptionsRequest, BatchPutFieldOptionsResponse>()
                            .withOperationName("BatchPutFieldOptions")
                            .withMarshaller(new BatchPutFieldOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchPutFieldOptionsRequest));
            CompletableFuture<BatchPutFieldOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a case in the specified Cases domain. Case system and custom fields are taken as an array id/value pairs
     * with a declared data types.
     * </p>
     * <note>
     * <p>
     * The following fields are required when creating a case:
     * </p>
     * 
     * <pre>
     * <code> &lt;ul&gt; &lt;li&gt; &lt;p&gt; &lt;code&gt;customer_id&lt;/code&gt; - You must provide the full customer profile ARN in this format: &lt;code&gt;arn:aws:profile:your AWS Region:your AWS account ID:domains/profiles domain name/profiles/profile ID&lt;/code&gt; &lt;/p&gt; &lt;/li&gt; &lt;li&gt; &lt;p&gt; &lt;code&gt;title&lt;/code&gt; &lt;/p&gt; &lt;/li&gt; &lt;/ul&gt; &lt;/note&gt; </code>
     * </pre>
     *
     * @param createCaseRequest
     * @return A Java Future containing the result of the CreateCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.CreateCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCaseResponse> createCase(CreateCaseRequest createCaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCaseRequest, CreateCaseResponse>().withOperationName("CreateCase")
                            .withMarshaller(new CreateCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCaseRequest));
            CompletableFuture<CreateCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a domain, which is a container for all case data, such as cases, fields, templates and layouts. Each
     * Amazon Connect instance can be associated with only one Cases domain.
     * </p>
     * <important>
     * <p>
     * This will not associate your connect instance to Cases domain. Instead, use the Amazon Connect <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateIntegrationAssociation.html"
     * >CreateIntegrationAssociation</a> API. You need specific IAM permissions to successfully associate the Cases
     * domain. For more information, see <a href=
     * "https://docs.aws.amazon.com/connect/latest/adminguide/required-permissions-iam-cases.html#onboard-cases-iam"
     * >Onboard to Cases</a>.
     * </p>
     * </important>
     *
     * @param createDomainRequest
     * @return A Java Future containing the result of the CreateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded. For a list of service quotas, see
     *         <a href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.CreateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDomainResponse> createDomain(CreateDomainRequest createDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                            .withOperationName("CreateDomain").withMarshaller(new CreateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDomainRequest));
            CompletableFuture<CreateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a field in the Cases domain. This field is used to define the case object model (that is, defines what
     * data can be captured on cases) in a Cases domain.
     * </p>
     *
     * @param createFieldRequest
     * @return A Java Future containing the result of the CreateField operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded. For a list of service quotas, see
     *         <a href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.CreateField
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateField" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFieldResponse> createField(CreateFieldRequest createFieldRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFieldRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateField");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFieldResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFieldResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFieldResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFieldRequest, CreateFieldResponse>()
                            .withOperationName("CreateField").withMarshaller(new CreateFieldRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFieldRequest));
            CompletableFuture<CreateFieldResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a layout in the Cases domain. Layouts define the following configuration in the top section and More Info
     * tab of the Cases user interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Fields to display to the users
     * </p>
     * </li>
     * <li>
     * <p>
     * Field ordering
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Title and Status fields cannot be part of layouts since they are not configurable.
     * </p>
     * </note>
     *
     * @param createLayoutRequest
     * @return A Java Future containing the result of the CreateLayout operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded. For a list of service quotas, see
     *         <a href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.CreateLayout
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateLayout" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLayoutResponse> createLayout(CreateLayoutRequest createLayoutRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLayoutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLayout");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLayoutResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLayoutResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLayoutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLayoutRequest, CreateLayoutResponse>()
                            .withOperationName("CreateLayout").withMarshaller(new CreateLayoutRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLayoutRequest));
            CompletableFuture<CreateLayoutResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a related item (comments, tasks, and contacts) and associates it with a case.
     * </p>
     * <note>
     * <p>
     * A Related Item is a resource that is associated with a case. It may or may not have an external identifier
     * linking it to an external resource (for example, a <code>contactArn</code>). All Related Items have their own
     * internal identifier, the <code>relatedItemArn</code>. Examples of related items include <code>comments</code> and
     * <code>contacts</code>.
     * </p>
     * </note>
     *
     * @param createRelatedItemRequest
     * @return A Java Future containing the result of the CreateRelatedItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded. For a list of service quotas, see
     *         <a href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.CreateRelatedItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateRelatedItem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRelatedItemResponse> createRelatedItem(CreateRelatedItemRequest createRelatedItemRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRelatedItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRelatedItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRelatedItemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRelatedItemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRelatedItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRelatedItemRequest, CreateRelatedItemResponse>()
                            .withOperationName("CreateRelatedItem")
                            .withMarshaller(new CreateRelatedItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRelatedItemRequest));
            CompletableFuture<CreateRelatedItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a template in the Cases domain. This template is used to define the case object model (that is, to define
     * what data can be captured on cases) in a Cases domain. A template must have a unique name within a domain, and it
     * must reference existing field IDs and layout IDs. Additionally, multiple fields with same IDs are not allowed
     * within the same Template. A template can be either Active or Inactive, as indicated by its status. Inactive
     * templates cannot be used to create cases.
     * </p>
     *
     * @param createTemplateRequest
     * @return A Java Future containing the result of the CreateTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded. For a list of service quotas, see
     *         <a href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.CreateTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/CreateTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTemplateResponse> createTemplate(CreateTemplateRequest createTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTemplateRequest, CreateTemplateResponse>()
                            .withOperationName("CreateTemplate")
                            .withMarshaller(new CreateTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTemplateRequest));
            CompletableFuture<CreateTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a domain.
     * </p>
     *
     * @param deleteDomainRequest
     * @return A Java Future containing the result of the DeleteDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.DeleteDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/DeleteDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDomainResponse> deleteDomain(DeleteDomainRequest deleteDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                            .withOperationName("DeleteDomain").withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDomainRequest));
            CompletableFuture<DeleteDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specific case if it exists.
     * </p>
     *
     * @param getCaseRequest
     * @return A Java Future containing the result of the GetCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.GetCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCaseResponse> getCase(GetCaseRequest getCaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCaseRequest, GetCaseResponse>().withOperationName("GetCase")
                            .withMarshaller(new GetCaseRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCaseRequest));
            CompletableFuture<GetCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the case event publishing configuration.
     * </p>
     *
     * @param getCaseEventConfigurationRequest
     * @return A Java Future containing the result of the GetCaseEventConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.GetCaseEventConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetCaseEventConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCaseEventConfigurationResponse> getCaseEventConfiguration(
            GetCaseEventConfigurationRequest getCaseEventConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCaseEventConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCaseEventConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCaseEventConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCaseEventConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCaseEventConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCaseEventConfigurationRequest, GetCaseEventConfigurationResponse>()
                            .withOperationName("GetCaseEventConfiguration")
                            .withMarshaller(new GetCaseEventConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCaseEventConfigurationRequest));
            CompletableFuture<GetCaseEventConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specific domain if it exists.
     * </p>
     *
     * @param getDomainRequest
     * @return A Java Future containing the result of the GetDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.GetDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainResponse> getDomain(GetDomainRequest getDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainRequest, GetDomainResponse>().withOperationName("GetDomain")
                            .withMarshaller(new GetDomainRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDomainRequest));
            CompletableFuture<GetDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details for the requested layout.
     * </p>
     *
     * @param getLayoutRequest
     * @return A Java Future containing the result of the GetLayout operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.GetLayout
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetLayout" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLayoutResponse> getLayout(GetLayoutRequest getLayoutRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLayoutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLayout");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLayoutResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLayoutResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLayoutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLayoutRequest, GetLayoutResponse>().withOperationName("GetLayout")
                            .withMarshaller(new GetLayoutRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLayoutRequest));
            CompletableFuture<GetLayoutResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details for the requested template.
     * </p>
     *
     * @param getTemplateRequest
     * @return A Java Future containing the result of the GetTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.GetTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/GetTemplate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTemplateResponse> getTemplate(GetTemplateRequest getTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTemplateRequest, GetTemplateResponse>()
                            .withOperationName("GetTemplate").withMarshaller(new GetTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTemplateRequest));
            CompletableFuture<GetTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists cases for a given contact.
     * </p>
     *
     * @param listCasesForContactRequest
     * @return A Java Future containing the result of the ListCasesForContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.ListCasesForContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListCasesForContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCasesForContactResponse> listCasesForContact(
            ListCasesForContactRequest listCasesForContactRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCasesForContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCasesForContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCasesForContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCasesForContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCasesForContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCasesForContactRequest, ListCasesForContactResponse>()
                            .withOperationName("ListCasesForContact")
                            .withMarshaller(new ListCasesForContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCasesForContactRequest));
            CompletableFuture<ListCasesForContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all cases domains in the Amazon Web Services account. Each list item is a condensed summary object of the
     * domain.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDomainsRequest));
            CompletableFuture<ListDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the field options for a field identifier in the domain.
     * </p>
     *
     * @param listFieldOptionsRequest
     * @return A Java Future containing the result of the ListFieldOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.ListFieldOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListFieldOptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFieldOptionsResponse> listFieldOptions(ListFieldOptionsRequest listFieldOptionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFieldOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFieldOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFieldOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFieldOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFieldOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFieldOptionsRequest, ListFieldOptionsResponse>()
                            .withOperationName("ListFieldOptions")
                            .withMarshaller(new ListFieldOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFieldOptionsRequest));
            CompletableFuture<ListFieldOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all fields in a Cases domain.
     * </p>
     *
     * @param listFieldsRequest
     * @return A Java Future containing the result of the ListFields operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.ListFields
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListFields" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFieldsResponse> listFields(ListFieldsRequest listFieldsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFieldsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFields");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFieldsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFieldsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFieldsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFieldsRequest, ListFieldsResponse>().withOperationName("ListFields")
                            .withMarshaller(new ListFieldsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFieldsRequest));
            CompletableFuture<ListFieldsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all layouts in the given cases domain. Each list item is a condensed summary object of the layout.
     * </p>
     *
     * @param listLayoutsRequest
     * @return A Java Future containing the result of the ListLayouts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.ListLayouts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListLayouts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLayoutsResponse> listLayouts(ListLayoutsRequest listLayoutsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLayoutsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLayouts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLayoutsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLayoutsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLayoutsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLayoutsRequest, ListLayoutsResponse>()
                            .withOperationName("ListLayouts").withMarshaller(new ListLayoutsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLayoutsRequest));
            CompletableFuture<ListLayoutsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the templates in a Cases domain. Each list item is a condensed summary object of the template.
     * </p>
     *
     * @param listTemplatesRequest
     * @return A Java Future containing the result of the ListTemplates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.ListTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/ListTemplates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTemplatesResponse> listTemplates(ListTemplatesRequest listTemplatesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTemplatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTemplatesRequest, ListTemplatesResponse>()
                            .withOperationName("ListTemplates")
                            .withMarshaller(new ListTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTemplatesRequest));
            CompletableFuture<ListTemplatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * API for adding case event publishing configuration
     * </p>
     *
     * @param putCaseEventConfigurationRequest
     * @return A Java Future containing the result of the PutCaseEventConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.PutCaseEventConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/PutCaseEventConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutCaseEventConfigurationResponse> putCaseEventConfiguration(
            PutCaseEventConfigurationRequest putCaseEventConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putCaseEventConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutCaseEventConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutCaseEventConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutCaseEventConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutCaseEventConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutCaseEventConfigurationRequest, PutCaseEventConfigurationResponse>()
                            .withOperationName("PutCaseEventConfiguration")
                            .withMarshaller(new PutCaseEventConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putCaseEventConfigurationRequest));
            CompletableFuture<PutCaseEventConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for cases within their associated Cases domain. Search results are returned as a paginated list of
     * abridged case documents.
     * </p>
     * <note>
     * <p>
     * For <code>customer_id</code> you must provide the full customer profile ARN in this format:
     * <code> arn:aws:profile:your AWS Region:your AWS account ID:domains/profiles domain name/profiles/profile ID</code>
     * .
     * </p>
     * </note>
     *
     * @param searchCasesRequest
     * @return A Java Future containing the result of the SearchCases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.SearchCases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/SearchCases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchCasesResponse> searchCases(SearchCasesRequest searchCasesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchCasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchCases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchCasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchCasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchCasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchCasesRequest, SearchCasesResponse>()
                            .withOperationName("SearchCases").withMarshaller(new SearchCasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchCasesRequest));
            CompletableFuture<SearchCasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for related items that are associated with a case.
     * </p>
     * <note>
     * <p>
     * If no filters are provided, this returns all related items associated with a case.
     * </p>
     * </note>
     *
     * @param searchRelatedItemsRequest
     * @return A Java Future containing the result of the SearchRelatedItems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.SearchRelatedItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/SearchRelatedItems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchRelatedItemsResponse> searchRelatedItems(SearchRelatedItemsRequest searchRelatedItemsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchRelatedItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchRelatedItems");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchRelatedItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchRelatedItemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchRelatedItemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchRelatedItemsRequest, SearchRelatedItemsResponse>()
                            .withOperationName("SearchRelatedItems")
                            .withMarshaller(new SearchRelatedItemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchRelatedItemsRequest));
            CompletableFuture<SearchRelatedItemsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Untags a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the values of fields on a case. Fields to be updated are received as an array of id/value pairs identical
     * to the <code>CreateCase</code> input .
     * </p>
     * <p>
     * If the action is successful, the service sends back an HTTP 200 response with an empty HTTP body.
     * </p>
     *
     * @param updateCaseRequest
     * @return A Java Future containing the result of the UpdateCase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.UpdateCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UpdateCase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCaseResponse> updateCase(UpdateCaseRequest updateCaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateCaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCaseRequest, UpdateCaseResponse>().withOperationName("UpdateCase")
                            .withMarshaller(new UpdateCaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateCaseRequest));
            CompletableFuture<UpdateCaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the properties of an existing field.
     * </p>
     *
     * @param updateFieldRequest
     * @return A Java Future containing the result of the UpdateField operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.UpdateField
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UpdateField" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFieldResponse> updateField(UpdateFieldRequest updateFieldRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFieldRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateField");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFieldResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateFieldResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFieldResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFieldRequest, UpdateFieldResponse>()
                            .withOperationName("UpdateField").withMarshaller(new UpdateFieldRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFieldRequest));
            CompletableFuture<UpdateFieldResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the attributes of an existing layout.
     * </p>
     * <p>
     * If the action is successful, the service sends back an HTTP 200 response with an empty HTTP body.
     * </p>
     * <p>
     * A <code>ValidationException</code> is returned when you add non-existent <code>fieldIds</code> to a layout.
     * </p>
     * <note>
     * <p>
     * Title and Status fields cannot be part of layouts because they are not configurable.
     * </p>
     * </note>
     *
     * @param updateLayoutRequest
     * @return A Java Future containing the result of the UpdateLayout operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded. For a list of service quotas, see
     *         <a href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html">Amazon
     *         Connect Service Quotas</a> in the <i>Amazon Connect Administrator Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.UpdateLayout
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UpdateLayout" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLayoutResponse> updateLayout(UpdateLayoutRequest updateLayoutRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLayoutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLayout");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLayoutResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateLayoutResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLayoutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLayoutRequest, UpdateLayoutResponse>()
                            .withOperationName("UpdateLayout").withMarshaller(new UpdateLayoutRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLayoutRequest));
            CompletableFuture<UpdateLayoutResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the attributes of an existing template. The template attributes that can be modified include
     * <code>name</code>, <code>description</code>, <code>layoutConfiguration</code>, <code>requiredFields</code>, and
     * <code>status</code>. At least one of these attributes must not be null. If a null value is provided for a given
     * attribute, that attribute is ignored and its current value is preserved.
     * </p>
     *
     * @param updateTemplateRequest
     * @return A Java Future containing the result of the UpdateTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same Amazon Web Services Region as your request, and try your request again.</li>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException The rate has been exceeded for this API. Please try again after a few minutes.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request. See the
     *         accompanying error message for details.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectCasesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ConnectCasesAsyncClient.UpdateTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcases-2022-10-03/UpdateTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTemplateResponse> updateTemplate(UpdateTemplateRequest updateTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCases");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTemplateRequest, UpdateTemplateResponse>()
                            .withOperationName("UpdateTemplate")
                            .withMarshaller(new UpdateTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTemplateRequest));
            CompletableFuture<UpdateTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ConnectCasesServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectCasesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
