/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Auto Scaling group recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingGroupRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoScalingGroupRecommendation.Builder, AutoScalingGroupRecommendation> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutoScalingGroupRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> AUTO_SCALING_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutoScalingGroupRecommendation::autoScalingGroupArn)).setter(setter(Builder::autoScalingGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoScalingGroupArn").build())
            .build();

    private static final SdkField<String> AUTO_SCALING_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutoScalingGroupRecommendation::autoScalingGroupName)).setter(setter(Builder::autoScalingGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoScalingGroupName").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AutoScalingGroupRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<UtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<UtilizationMetric>> builder(MarshallingType.LIST)
            .getter(getter(AutoScalingGroupRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOK_BACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(AutoScalingGroupRecommendation::lookBackPeriodInDays)).setter(setter(Builder::lookBackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookBackPeriodInDays").build())
            .build();

    private static final SdkField<AutoScalingGroupConfiguration> CURRENT_CONFIGURATION_FIELD = SdkField
            .<AutoScalingGroupConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(AutoScalingGroupRecommendation::currentConfiguration)).setter(setter(Builder::currentConfiguration))
            .constructor(AutoScalingGroupConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentConfiguration").build())
            .build();

    private static final SdkField<List<AutoScalingGroupRecommendationOption>> RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<AutoScalingGroupRecommendationOption>> builder(MarshallingType.LIST)
            .getter(getter(AutoScalingGroupRecommendation::recommendationOptions))
            .setter(setter(Builder::recommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AutoScalingGroupRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(AutoScalingGroupRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(AutoScalingGroupRecommendation::lastRefreshTimestamp)).setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            AUTO_SCALING_GROUP_ARN_FIELD, AUTO_SCALING_GROUP_NAME_FIELD, FINDING_FIELD, UTILIZATION_METRICS_FIELD,
            LOOK_BACK_PERIOD_IN_DAYS_FIELD, CURRENT_CONFIGURATION_FIELD, RECOMMENDATION_OPTIONS_FIELD,
            LAST_REFRESH_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String autoScalingGroupArn;

    private final String autoScalingGroupName;

    private final String finding;

    private final List<UtilizationMetric> utilizationMetrics;

    private final Double lookBackPeriodInDays;

    private final AutoScalingGroupConfiguration currentConfiguration;

    private final List<AutoScalingGroupRecommendationOption> recommendationOptions;

    private final Instant lastRefreshTimestamp;

    private AutoScalingGroupRecommendation(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.autoScalingGroupArn = builder.autoScalingGroupArn;
        this.autoScalingGroupName = builder.autoScalingGroupName;
        this.finding = builder.finding;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookBackPeriodInDays = builder.lookBackPeriodInDays;
        this.currentConfiguration = builder.currentConfiguration;
        this.recommendationOptions = builder.recommendationOptions;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
    }

    /**
     * <p>
     * The AWS account ID of the Auto Scaling group.
     * </p>
     * 
     * @return The AWS account ID of the Auto Scaling group.
     */
    public String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Auto Scaling group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Auto Scaling group.
     */
    public String autoScalingGroupArn() {
        return autoScalingGroupArn;
    }

    /**
     * <p>
     * The name of the Auto Scaling group.
     * </p>
     * 
     * @return The name of the Auto Scaling group.
     */
    public String autoScalingGroupName() {
        return autoScalingGroupName;
    }

    /**
     * <p>
     * The finding classification for the Auto Scaling group.
     * </p>
     * <p>
     * Findings for Auto Scaling groups include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when AWS Compute Optimizer
     * identifies a recommendation that can provide better performance for your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer determines
     * that the group is correctly provisioned to run your workload based on the chosen instance type. For optimized
     * resources, Compute Optimizer might recommend a new generation instance type.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The values that are returned might be <code>NOT_OPTIMIZED</code> or <code>OPTIMIZED</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link Finding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for the Auto Scaling group.</p>
     *         <p>
     *         Findings for Auto Scaling groups include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when AWS Compute
     *         Optimizer identifies a recommendation that can provide better performance for your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
     *         determines that the group is correctly provisioned to run your workload based on the chosen instance
     *         type. For optimized resources, Compute Optimizer might recommend a new generation instance type.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The values that are returned might be <code>NOT_OPTIMIZED</code> or <code>OPTIMIZED</code>.
     *         </p>
     * @see Finding
     */
    public Finding finding() {
        return Finding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification for the Auto Scaling group.
     * </p>
     * <p>
     * Findings for Auto Scaling groups include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when AWS Compute Optimizer
     * identifies a recommendation that can provide better performance for your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer determines
     * that the group is correctly provisioned to run your workload based on the chosen instance type. For optimized
     * resources, Compute Optimizer might recommend a new generation instance type.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The values that are returned might be <code>NOT_OPTIMIZED</code> or <code>OPTIMIZED</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link Finding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for the Auto Scaling group.</p>
     *         <p>
     *         Findings for Auto Scaling groups include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when AWS Compute
     *         Optimizer identifies a recommendation that can provide better performance for your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
     *         determines that the group is correctly provisioned to run your workload based on the chosen instance
     *         type. For optimized resources, Compute Optimizer might recommend a new generation instance type.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The values that are returned might be <code>NOT_OPTIMIZED</code> or <code>OPTIMIZED</code>.
     *         </p>
     * @see Finding
     */
    public String findingAsString() {
        return finding;
    }

    /**
     * Returns true if the UtilizationMetrics property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the Auto Scaling group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUtilizationMetrics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the Auto Scaling group.
     */
    public List<UtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for the Auto Scaling group.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for the Auto Scaling group.
     */
    public Double lookBackPeriodInDays() {
        return lookBackPeriodInDays;
    }

    /**
     * <p>
     * An array of objects that describe the current configuration of the Auto Scaling group.
     * </p>
     * 
     * @return An array of objects that describe the current configuration of the Auto Scaling group.
     */
    public AutoScalingGroupConfiguration currentConfiguration() {
        return currentConfiguration;
    }

    /**
     * Returns true if the RecommendationOptions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasRecommendationOptions() {
        return recommendationOptions != null && !(recommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for the Auto Scaling group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRecommendationOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for the Auto Scaling group.
     */
    public List<AutoScalingGroupRecommendationOption> recommendationOptions() {
        return recommendationOptions;
    }

    /**
     * <p>
     * The time stamp of when the Auto Scaling group recommendation was last refreshed.
     * </p>
     * 
     * @return The time stamp of when the Auto Scaling group recommendation was last refreshed.
     */
    public Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(utilizationMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(lookBackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(currentConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(recommendationOptions());
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingGroupRecommendation)) {
            return false;
        }
        AutoScalingGroupRecommendation other = (AutoScalingGroupRecommendation) obj;
        return Objects.equals(accountId(), other.accountId())
                && Objects.equals(autoScalingGroupArn(), other.autoScalingGroupArn())
                && Objects.equals(autoScalingGroupName(), other.autoScalingGroupName())
                && Objects.equals(findingAsString(), other.findingAsString())
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookBackPeriodInDays(), other.lookBackPeriodInDays())
                && Objects.equals(currentConfiguration(), other.currentConfiguration())
                && Objects.equals(recommendationOptions(), other.recommendationOptions())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AutoScalingGroupRecommendation").add("AccountId", accountId())
                .add("AutoScalingGroupArn", autoScalingGroupArn()).add("AutoScalingGroupName", autoScalingGroupName())
                .add("Finding", findingAsString()).add("UtilizationMetrics", utilizationMetrics())
                .add("LookBackPeriodInDays", lookBackPeriodInDays()).add("CurrentConfiguration", currentConfiguration())
                .add("RecommendationOptions", recommendationOptions()).add("LastRefreshTimestamp", lastRefreshTimestamp())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "autoScalingGroupArn":
            return Optional.ofNullable(clazz.cast(autoScalingGroupArn()));
        case "autoScalingGroupName":
            return Optional.ofNullable(clazz.cast(autoScalingGroupName()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookBackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookBackPeriodInDays()));
        case "currentConfiguration":
            return Optional.ofNullable(clazz.cast(currentConfiguration()));
        case "recommendationOptions":
            return Optional.ofNullable(clazz.cast(recommendationOptions()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoScalingGroupRecommendation, T> g) {
        return obj -> g.apply((AutoScalingGroupRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoScalingGroupRecommendation> {
        /**
         * <p>
         * The AWS account ID of the Auto Scaling group.
         * </p>
         * 
         * @param accountId
         *        The AWS account ID of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Auto Scaling group.
         * </p>
         * 
         * @param autoScalingGroupArn
         *        The Amazon Resource Name (ARN) of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupArn(String autoScalingGroupArn);

        /**
         * <p>
         * The name of the Auto Scaling group.
         * </p>
         * 
         * @param autoScalingGroupName
         *        The name of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupName(String autoScalingGroupName);

        /**
         * <p>
         * The finding classification for the Auto Scaling group.
         * </p>
         * <p>
         * Findings for Auto Scaling groups include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when AWS Compute
         * Optimizer identifies a recommendation that can provide better performance for your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
         * determines that the group is correctly provisioned to run your workload based on the chosen instance type.
         * For optimized resources, Compute Optimizer might recommend a new generation instance type.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The values that are returned might be <code>NOT_OPTIMIZED</code> or <code>OPTIMIZED</code>.
         * </p>
         * </note>
         * 
         * @param finding
         *        The finding classification for the Auto Scaling group.</p>
         *        <p>
         *        Findings for Auto Scaling groups include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when AWS Compute
         *        Optimizer identifies a recommendation that can provide better performance for your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
         *        determines that the group is correctly provisioned to run your workload based on the chosen instance
         *        type. For optimized resources, Compute Optimizer might recommend a new generation instance type.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The values that are returned might be <code>NOT_OPTIMIZED</code> or <code>OPTIMIZED</code>.
         *        </p>
         * @see Finding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Finding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification for the Auto Scaling group.
         * </p>
         * <p>
         * Findings for Auto Scaling groups include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when AWS Compute
         * Optimizer identifies a recommendation that can provide better performance for your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
         * determines that the group is correctly provisioned to run your workload based on the chosen instance type.
         * For optimized resources, Compute Optimizer might recommend a new generation instance type.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The values that are returned might be <code>NOT_OPTIMIZED</code> or <code>OPTIMIZED</code>.
         * </p>
         * </note>
         * 
         * @param finding
         *        The finding classification for the Auto Scaling group.</p>
         *        <p>
         *        Findings for Auto Scaling groups include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when AWS Compute
         *        Optimizer identifies a recommendation that can provide better performance for your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
         *        determines that the group is correctly provisioned to run your workload based on the chosen instance
         *        type. For optimized resources, Compute Optimizer might recommend a new generation instance type.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The values that are returned might be <code>NOT_OPTIMIZED</code> or <code>OPTIMIZED</code>.
         *        </p>
         * @see Finding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Finding
         */
        Builder finding(Finding finding);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Auto Scaling group.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<UtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Auto Scaling group.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(UtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Auto Scaling group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<UtilizationMetric>.Builder} avoiding the
         * need to create one manually via {@link List<UtilizationMetric>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UtilizationMetric>.Builder#build()} is called immediately
         * and its result is passed to {@link #utilizationMetrics(List<UtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on {@link List<UtilizationMetric>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(List<UtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<UtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for the Auto Scaling group.
         * </p>
         * 
         * @param lookBackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookBackPeriodInDays(Double lookBackPeriodInDays);

        /**
         * <p>
         * An array of objects that describe the current configuration of the Auto Scaling group.
         * </p>
         * 
         * @param currentConfiguration
         *        An array of objects that describe the current configuration of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentConfiguration(AutoScalingGroupConfiguration currentConfiguration);

        /**
         * <p>
         * An array of objects that describe the current configuration of the Auto Scaling group.
         * </p>
         * This is a convenience that creates an instance of the {@link AutoScalingGroupConfiguration.Builder} avoiding
         * the need to create one manually via {@link AutoScalingGroupConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoScalingGroupConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #currentConfiguration(AutoScalingGroupConfiguration)}.
         * 
         * @param currentConfiguration
         *        a consumer that will call methods on {@link AutoScalingGroupConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentConfiguration(AutoScalingGroupConfiguration)
         */
        default Builder currentConfiguration(Consumer<AutoScalingGroupConfiguration.Builder> currentConfiguration) {
            return currentConfiguration(AutoScalingGroupConfiguration.builder().applyMutation(currentConfiguration).build());
        }

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Auto Scaling group.
         * </p>
         * 
         * @param recommendationOptions
         *        An array of objects that describe the recommendation options for the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationOptions(Collection<AutoScalingGroupRecommendationOption> recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Auto Scaling group.
         * </p>
         * 
         * @param recommendationOptions
         *        An array of objects that describe the recommendation options for the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationOptions(AutoScalingGroupRecommendationOption... recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Auto Scaling group.
         * </p>
         * This is a convenience that creates an instance of the {@link List
         * <AutoScalingGroupRecommendationOption>.Builder} avoiding the need to create one manually via {@link List
         * <AutoScalingGroupRecommendationOption>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AutoScalingGroupRecommendationOption>.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #recommendationOptions(List<AutoScalingGroupRecommendationOption>)}.
         * 
         * @param recommendationOptions
         *        a consumer that will call methods on {@link List<AutoScalingGroupRecommendationOption>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendationOptions(List<AutoScalingGroupRecommendationOption>)
         */
        Builder recommendationOptions(Consumer<AutoScalingGroupRecommendationOption.Builder>... recommendationOptions);

        /**
         * <p>
         * The time stamp of when the Auto Scaling group recommendation was last refreshed.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The time stamp of when the Auto Scaling group recommendation was last refreshed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String autoScalingGroupArn;

        private String autoScalingGroupName;

        private String finding;

        private List<UtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookBackPeriodInDays;

        private AutoScalingGroupConfiguration currentConfiguration;

        private List<AutoScalingGroupRecommendationOption> recommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private Instant lastRefreshTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingGroupRecommendation model) {
            accountId(model.accountId);
            autoScalingGroupArn(model.autoScalingGroupArn);
            autoScalingGroupName(model.autoScalingGroupName);
            finding(model.finding);
            utilizationMetrics(model.utilizationMetrics);
            lookBackPeriodInDays(model.lookBackPeriodInDays);
            currentConfiguration(model.currentConfiguration);
            recommendationOptions(model.recommendationOptions);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final String getAutoScalingGroupArn() {
            return autoScalingGroupArn;
        }

        @Override
        public final Builder autoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
            return this;
        }

        public final void setAutoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
        }

        public final String getAutoScalingGroupName() {
            return autoScalingGroupName;
        }

        @Override
        public final Builder autoScalingGroupName(String autoScalingGroupName) {
            this.autoScalingGroupName = autoScalingGroupName;
            return this;
        }

        public final void setAutoScalingGroupName(String autoScalingGroupName) {
            this.autoScalingGroupName = autoScalingGroupName;
        }

        public final String getFinding() {
            return finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(Finding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        public final Collection<UtilizationMetric.Builder> getUtilizationMetrics() {
            return utilizationMetrics != null ? utilizationMetrics.stream().map(UtilizationMetric::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder utilizationMetrics(Collection<UtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = UtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(UtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<UtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics).map(c -> UtilizationMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUtilizationMetrics(Collection<UtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = UtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        public final Double getLookBackPeriodInDays() {
            return lookBackPeriodInDays;
        }

        @Override
        public final Builder lookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
            return this;
        }

        public final void setLookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
        }

        public final AutoScalingGroupConfiguration.Builder getCurrentConfiguration() {
            return currentConfiguration != null ? currentConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder currentConfiguration(AutoScalingGroupConfiguration currentConfiguration) {
            this.currentConfiguration = currentConfiguration;
            return this;
        }

        public final void setCurrentConfiguration(AutoScalingGroupConfiguration.BuilderImpl currentConfiguration) {
            this.currentConfiguration = currentConfiguration != null ? currentConfiguration.build() : null;
        }

        public final Collection<AutoScalingGroupRecommendationOption.Builder> getRecommendationOptions() {
            return recommendationOptions != null ? recommendationOptions.stream()
                    .map(AutoScalingGroupRecommendationOption::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder recommendationOptions(Collection<AutoScalingGroupRecommendationOption> recommendationOptions) {
            this.recommendationOptions = AutoScalingGroupRecommendationOptionsCopier.copy(recommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationOptions(AutoScalingGroupRecommendationOption... recommendationOptions) {
            recommendationOptions(Arrays.asList(recommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationOptions(
                Consumer<AutoScalingGroupRecommendationOption.Builder>... recommendationOptions) {
            recommendationOptions(Stream.of(recommendationOptions)
                    .map(c -> AutoScalingGroupRecommendationOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRecommendationOptions(
                Collection<AutoScalingGroupRecommendationOption.BuilderImpl> recommendationOptions) {
            this.recommendationOptions = AutoScalingGroupRecommendationOptionsCopier.copyFromBuilder(recommendationOptions);
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public AutoScalingGroupRecommendation build() {
            return new AutoScalingGroupRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
