/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.computeoptimizer.model.AccessDeniedException;
import software.amazon.awssdk.services.computeoptimizer.model.ComputeOptimizerException;
import software.amazon.awssdk.services.computeoptimizer.model.GetAutoScalingGroupRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetAutoScalingGroupRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2InstanceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2InstanceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2RecommendationProjectedMetricsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2RecommendationProjectedMetricsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationSummariesRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationSummariesResponse;
import software.amazon.awssdk.services.computeoptimizer.model.InternalServerException;
import software.amazon.awssdk.services.computeoptimizer.model.InvalidParameterValueException;
import software.amazon.awssdk.services.computeoptimizer.model.MissingAuthenticationTokenException;
import software.amazon.awssdk.services.computeoptimizer.model.OptInRequiredException;
import software.amazon.awssdk.services.computeoptimizer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.computeoptimizer.model.ServiceUnavailableException;
import software.amazon.awssdk.services.computeoptimizer.model.ThrottlingException;
import software.amazon.awssdk.services.computeoptimizer.model.UpdateEnrollmentStatusRequest;
import software.amazon.awssdk.services.computeoptimizer.model.UpdateEnrollmentStatusResponse;
import software.amazon.awssdk.services.computeoptimizer.transform.GetAutoScalingGroupRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEc2InstanceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEc2RecommendationProjectedMetricsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetRecommendationSummariesRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.UpdateEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ComputeOptimizerAsyncClient}.
 *
 * @see ComputeOptimizerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultComputeOptimizerAsyncClient implements ComputeOptimizerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultComputeOptimizerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultComputeOptimizerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Returns Auto Scaling group recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer currently generates recommendations for Auto Scaling groups that are configured to run
     * instances of the M, C, R, T, and X instance families. The service does not generate recommendations for Auto
     * Scaling groups that have a scaling policy attached to them, or that do not have the same values for desired,
     * minimum, and maximum capacity. In order for Compute Optimizer to analyze your Auto Scaling groups, they must be
     * of a fixed size. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/what-is.html">AWS Compute Optimizer User Guide</a>.
     * </p>
     *
     * @param getAutoScalingGroupRecommendationsRequest
     * @return A Java Future containing the result of the GetAutoScalingGroupRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException You must opt in to the service to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetAutoScalingGroupRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetAutoScalingGroupRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAutoScalingGroupRecommendationsResponse> getAutoScalingGroupRecommendations(
            GetAutoScalingGroupRecommendationsRequest getAutoScalingGroupRecommendationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAutoScalingGroupRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetAutoScalingGroupRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAutoScalingGroupRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAutoScalingGroupRecommendationsRequest, GetAutoScalingGroupRecommendationsResponse>()
                            .withOperationName("GetAutoScalingGroupRecommendations")
                            .withMarshaller(new GetAutoScalingGroupRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getAutoScalingGroupRecommendationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Amazon EC2 instance recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer currently generates recommendations for Amazon Elastic Compute Cloud (Amazon EC2) and
     * Amazon EC2 Auto Scaling. It generates recommendations for M, C, R, T, and X instance families. For more
     * information, see the <a href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/what-is.html">AWS Compute
     * Optimizer User Guide</a>.
     * </p>
     *
     * @param getEc2InstanceRecommendationsRequest
     * @return A Java Future containing the result of the GetEC2InstanceRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException You must opt in to the service to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEC2InstanceRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEC2InstanceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEc2InstanceRecommendationsResponse> getEC2InstanceRecommendations(
            GetEc2InstanceRecommendationsRequest getEc2InstanceRecommendationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEc2InstanceRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEc2InstanceRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEc2InstanceRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEc2InstanceRecommendationsRequest, GetEc2InstanceRecommendationsResponse>()
                            .withOperationName("GetEC2InstanceRecommendations")
                            .withMarshaller(new GetEc2InstanceRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getEc2InstanceRecommendationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the projected utilization metrics of Amazon EC2 instance recommendations.
     * </p>
     *
     * @param getEc2RecommendationProjectedMetricsRequest
     * @return A Java Future containing the result of the GetEC2RecommendationProjectedMetrics operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException You must opt in to the service to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEC2RecommendationProjectedMetrics
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEC2RecommendationProjectedMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEc2RecommendationProjectedMetricsResponse> getEC2RecommendationProjectedMetrics(
            GetEc2RecommendationProjectedMetricsRequest getEc2RecommendationProjectedMetricsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEc2RecommendationProjectedMetricsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetEc2RecommendationProjectedMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEc2RecommendationProjectedMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEc2RecommendationProjectedMetricsRequest, GetEc2RecommendationProjectedMetricsResponse>()
                            .withOperationName("GetEC2RecommendationProjectedMetrics")
                            .withMarshaller(new GetEc2RecommendationProjectedMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getEc2RecommendationProjectedMetricsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the enrollment (opt in) status of an account to the AWS Compute Optimizer service.
     * </p>
     * <p>
     * If the account is a master account of an organization, this operation also confirms the enrollment status of
     * member accounts within the organization.
     * </p>
     *
     * @param getEnrollmentStatusRequest
     * @return A Java Future containing the result of the GetEnrollmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEnrollmentStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnrollmentStatusResponse> getEnrollmentStatus(
            GetEnrollmentStatusRequest getEnrollmentStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnrollmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnrollmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnrollmentStatusRequest, GetEnrollmentStatusResponse>()
                            .withOperationName("GetEnrollmentStatus")
                            .withMarshaller(new GetEnrollmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getEnrollmentStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the optimization findings for an account.
     * </p>
     * <p>
     * For example, it returns the number of Amazon EC2 instances in an account that are under-provisioned,
     * over-provisioned, or optimized. It also returns the number of Auto Scaling groups in an account that are not
     * optimized, or optimized.
     * </p>
     *
     * @param getRecommendationSummariesRequest
     * @return A Java Future containing the result of the GetRecommendationSummaries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException You must opt in to the service to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetRecommendationSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetRecommendationSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecommendationSummariesResponse> getRecommendationSummaries(
            GetRecommendationSummariesRequest getRecommendationSummariesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecommendationSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRecommendationSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecommendationSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationSummariesRequest, GetRecommendationSummariesResponse>()
                            .withOperationName("GetRecommendationSummaries")
                            .withMarshaller(new GetRecommendationSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getRecommendationSummariesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the enrollment (opt in) status of an account to the AWS Compute Optimizer service.
     * </p>
     * <p>
     * If the account is a master account of an organization, this operation can also enroll member accounts within the
     * organization.
     * </p>
     *
     * @param updateEnrollmentStatusRequest
     * @return A Java Future containing the result of the UpdateEnrollmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.UpdateEnrollmentStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/UpdateEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnrollmentStatusResponse> updateEnrollmentStatus(
            UpdateEnrollmentStatusRequest updateEnrollmentStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnrollmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnrollmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnrollmentStatusRequest, UpdateEnrollmentStatusResponse>()
                            .withOperationName("UpdateEnrollmentStatus")
                            .withMarshaller(new UpdateEnrollmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateEnrollmentStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ComputeOptimizerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptInRequiredException")
                                .exceptionBuilderSupplier(OptInRequiredException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingAuthenticationToken")
                                .exceptionBuilderSupplier(MissingAuthenticationTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
