/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides configuration parameters for PII entity redaction.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RedactionConfig implements SdkPojo, Serializable, ToCopyableBuilder<RedactionConfig.Builder, RedactionConfig> {
    private static final SdkField<List<String>> PII_ENTITY_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PiiEntityTypes")
            .getter(getter(RedactionConfig::piiEntityTypesAsStrings))
            .setter(setter(Builder::piiEntityTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PiiEntityTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> MASK_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaskMode").getter(getter(RedactionConfig::maskModeAsString)).setter(setter(Builder::maskMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaskMode").build()).build();

    private static final SdkField<String> MASK_CHARACTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaskCharacter").getter(getter(RedactionConfig::maskCharacter)).setter(setter(Builder::maskCharacter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaskCharacter").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PII_ENTITY_TYPES_FIELD,
            MASK_MODE_FIELD, MASK_CHARACTER_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> piiEntityTypes;

    private final String maskMode;

    private final String maskCharacter;

    private RedactionConfig(BuilderImpl builder) {
        this.piiEntityTypes = builder.piiEntityTypes;
        this.maskMode = builder.maskMode;
        this.maskCharacter = builder.maskCharacter;
    }

    /**
     * <p>
     * An array of the types of PII entities that Amazon Comprehend detects in the input text for your request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPiiEntityTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of the types of PII entities that Amazon Comprehend detects in the input text for your request.
     */
    public final List<PiiEntityType> piiEntityTypes() {
        return ListOfPiiEntityTypesCopier.copyStringToEnum(piiEntityTypes);
    }

    /**
     * Returns true if the PiiEntityTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasPiiEntityTypes() {
        return piiEntityTypes != null && !(piiEntityTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of the types of PII entities that Amazon Comprehend detects in the input text for your request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPiiEntityTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of the types of PII entities that Amazon Comprehend detects in the input text for your request.
     */
    public final List<String> piiEntityTypesAsStrings() {
        return piiEntityTypes;
    }

    /**
     * <p>
     * Specifies whether the PII entity is redacted with the mask character or the entity type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #maskMode} will
     * return {@link PiiEntitiesDetectionMaskMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #maskModeAsString}.
     * </p>
     * 
     * @return Specifies whether the PII entity is redacted with the mask character or the entity type.
     * @see PiiEntitiesDetectionMaskMode
     */
    public final PiiEntitiesDetectionMaskMode maskMode() {
        return PiiEntitiesDetectionMaskMode.fromValue(maskMode);
    }

    /**
     * <p>
     * Specifies whether the PII entity is redacted with the mask character or the entity type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #maskMode} will
     * return {@link PiiEntitiesDetectionMaskMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #maskModeAsString}.
     * </p>
     * 
     * @return Specifies whether the PII entity is redacted with the mask character or the entity type.
     * @see PiiEntitiesDetectionMaskMode
     */
    public final String maskModeAsString() {
        return maskMode;
    }

    /**
     * <p>
     * A character that replaces each character in the redacted PII entity.
     * </p>
     * 
     * @return A character that replaces each character in the redacted PII entity.
     */
    public final String maskCharacter() {
        return maskCharacter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasPiiEntityTypes() ? piiEntityTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maskModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maskCharacter());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RedactionConfig)) {
            return false;
        }
        RedactionConfig other = (RedactionConfig) obj;
        return hasPiiEntityTypes() == other.hasPiiEntityTypes()
                && Objects.equals(piiEntityTypesAsStrings(), other.piiEntityTypesAsStrings())
                && Objects.equals(maskModeAsString(), other.maskModeAsString())
                && Objects.equals(maskCharacter(), other.maskCharacter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RedactionConfig").add("PiiEntityTypes", hasPiiEntityTypes() ? piiEntityTypesAsStrings() : null)
                .add("MaskMode", maskModeAsString()).add("MaskCharacter", maskCharacter()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PiiEntityTypes":
            return Optional.ofNullable(clazz.cast(piiEntityTypesAsStrings()));
        case "MaskMode":
            return Optional.ofNullable(clazz.cast(maskModeAsString()));
        case "MaskCharacter":
            return Optional.ofNullable(clazz.cast(maskCharacter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RedactionConfig, T> g) {
        return obj -> g.apply((RedactionConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RedactionConfig> {
        /**
         * <p>
         * An array of the types of PII entities that Amazon Comprehend detects in the input text for your request.
         * </p>
         * 
         * @param piiEntityTypes
         *        An array of the types of PII entities that Amazon Comprehend detects in the input text for your
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder piiEntityTypesWithStrings(Collection<String> piiEntityTypes);

        /**
         * <p>
         * An array of the types of PII entities that Amazon Comprehend detects in the input text for your request.
         * </p>
         * 
         * @param piiEntityTypes
         *        An array of the types of PII entities that Amazon Comprehend detects in the input text for your
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder piiEntityTypesWithStrings(String... piiEntityTypes);

        /**
         * <p>
         * An array of the types of PII entities that Amazon Comprehend detects in the input text for your request.
         * </p>
         * 
         * @param piiEntityTypes
         *        An array of the types of PII entities that Amazon Comprehend detects in the input text for your
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder piiEntityTypes(Collection<PiiEntityType> piiEntityTypes);

        /**
         * <p>
         * An array of the types of PII entities that Amazon Comprehend detects in the input text for your request.
         * </p>
         * 
         * @param piiEntityTypes
         *        An array of the types of PII entities that Amazon Comprehend detects in the input text for your
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder piiEntityTypes(PiiEntityType... piiEntityTypes);

        /**
         * <p>
         * Specifies whether the PII entity is redacted with the mask character or the entity type.
         * </p>
         * 
         * @param maskMode
         *        Specifies whether the PII entity is redacted with the mask character or the entity type.
         * @see PiiEntitiesDetectionMaskMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PiiEntitiesDetectionMaskMode
         */
        Builder maskMode(String maskMode);

        /**
         * <p>
         * Specifies whether the PII entity is redacted with the mask character or the entity type.
         * </p>
         * 
         * @param maskMode
         *        Specifies whether the PII entity is redacted with the mask character or the entity type.
         * @see PiiEntitiesDetectionMaskMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PiiEntitiesDetectionMaskMode
         */
        Builder maskMode(PiiEntitiesDetectionMaskMode maskMode);

        /**
         * <p>
         * A character that replaces each character in the redacted PII entity.
         * </p>
         * 
         * @param maskCharacter
         *        A character that replaces each character in the redacted PII entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maskCharacter(String maskCharacter);
    }

    static final class BuilderImpl implements Builder {
        private List<String> piiEntityTypes = DefaultSdkAutoConstructList.getInstance();

        private String maskMode;

        private String maskCharacter;

        private BuilderImpl() {
        }

        private BuilderImpl(RedactionConfig model) {
            piiEntityTypesWithStrings(model.piiEntityTypes);
            maskMode(model.maskMode);
            maskCharacter(model.maskCharacter);
        }

        public final Collection<String> getPiiEntityTypes() {
            if (piiEntityTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return piiEntityTypes;
        }

        @Override
        public final Builder piiEntityTypesWithStrings(Collection<String> piiEntityTypes) {
            this.piiEntityTypes = ListOfPiiEntityTypesCopier.copy(piiEntityTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder piiEntityTypesWithStrings(String... piiEntityTypes) {
            piiEntityTypesWithStrings(Arrays.asList(piiEntityTypes));
            return this;
        }

        @Override
        public final Builder piiEntityTypes(Collection<PiiEntityType> piiEntityTypes) {
            this.piiEntityTypes = ListOfPiiEntityTypesCopier.copyEnumToString(piiEntityTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder piiEntityTypes(PiiEntityType... piiEntityTypes) {
            piiEntityTypes(Arrays.asList(piiEntityTypes));
            return this;
        }

        public final void setPiiEntityTypes(Collection<String> piiEntityTypes) {
            this.piiEntityTypes = ListOfPiiEntityTypesCopier.copy(piiEntityTypes);
        }

        public final String getMaskMode() {
            return maskMode;
        }

        @Override
        public final Builder maskMode(String maskMode) {
            this.maskMode = maskMode;
            return this;
        }

        @Override
        public final Builder maskMode(PiiEntitiesDetectionMaskMode maskMode) {
            this.maskMode(maskMode == null ? null : maskMode.toString());
            return this;
        }

        public final void setMaskMode(String maskMode) {
            this.maskMode = maskMode;
        }

        public final String getMaskCharacter() {
            return maskCharacter;
        }

        @Override
        public final Builder maskCharacter(String maskCharacter) {
            this.maskCharacter = maskCharacter;
            return this;
        }

        public final void setMaskCharacter(String maskCharacter) {
            this.maskCharacter = maskCharacter;
        }

        @Override
        public RedactionConfig build() {
            return new RedactionConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
