/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurusecurity.endpoints.internal;

import java.net.URI;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.codegurusecurity.endpoints.CodeGuruSecurityEndpointParams;
import software.amazon.awssdk.services.codegurusecurity.endpoints.CodeGuruSecurityEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultCodeGuruSecurityEndpointProvider implements CodeGuruSecurityEndpointProvider {
    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(CodeGuruSecurityEndpointParams params) {
        Validate.notNull(params.useDualStack(), "Parameter 'UseDualStack' must not be null");
        Validate.notNull(params.useFips(), "Parameter 'UseFIPS' must not be null");
        try {
            Region region = params.region();
            String regionId = region == null ? null : region.id();
            RuleResult result = endpointRule0(params, regionId);
            if (result.canContinue()) {
                throw SdkClientException.create("Rule engine did not reach an error or endpoint result");
            }
            if (result.isError()) {
                String errorMsg = result.error();
                if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                    errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
                }
                throw SdkClientException.create(errorMsg);
            }
            return CompletableFuture.completedFuture(result.endpoint());
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static RuleResult endpointRule0(CodeGuruSecurityEndpointParams params, String region) {
        RuleResult result = endpointRule1(params);
        if (result.isResolved()) {
            return result;
        }
        return endpointRule6(params, region);
    }

    private static RuleResult endpointRule1(CodeGuruSecurityEndpointParams params) {
        if (params.endpoint() != null) {
            if (params.useFips()) {
                return RuleResult.error("Invalid Configuration: FIPS and custom endpoint are not supported");
            }
            return endpointRule3(params);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule3(CodeGuruSecurityEndpointParams params) {
        if (params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: Dualstack and custom endpoint are not supported");
        }
        return RuleResult.endpoint(Endpoint.builder().url(URI.create(params.endpoint())).build());
    }

    private static RuleResult endpointRule6(CodeGuruSecurityEndpointParams params, String region) {
        RuleResult result = endpointRule7(params, region);
        if (result.isResolved()) {
            return result;
        }
        return RuleResult.error("Invalid Configuration: Missing Region");
    }

    private static RuleResult endpointRule7(CodeGuruSecurityEndpointParams params, String region) {
        if (region != null) {
            return endpointRule8(params, region);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule8(CodeGuruSecurityEndpointParams params, String region) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            RuleResult result = endpointRule9(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule14(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule19(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule24(params, region, partitionResult);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule9(CodeGuruSecurityEndpointParams params, RulePartition partitionResult, String region) {
        if (params.useFips() && params.useDualStack()) {
            RuleResult result = endpointRule10(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("FIPS and DualStack are enabled, but this partition does not support one or both");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule10(CodeGuruSecurityEndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsFIPS() && partitionResult.supportsDualStack()) {
            return endpointRule11(params, region, partitionResult);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule11(CodeGuruSecurityEndpointParams params, String region, RulePartition partitionResult) {
        return RuleResult
                .endpoint(Endpoint.builder()
                        .url(URI.create("https://codeguru-security-fips." + region + "." + partitionResult.dualStackDnsSuffix()))
                        .build());
    }

    private static RuleResult endpointRule14(CodeGuruSecurityEndpointParams params, RulePartition partitionResult, String region) {
        if (params.useFips()) {
            RuleResult result = endpointRule15(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("FIPS is enabled but this partition does not support FIPS");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule15(CodeGuruSecurityEndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsFIPS()) {
            return endpointRule16(params, region, partitionResult);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule16(CodeGuruSecurityEndpointParams params, String region, RulePartition partitionResult) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://codeguru-security-fips." + region + "." + partitionResult.dnsSuffix())).build());
    }

    private static RuleResult endpointRule19(CodeGuruSecurityEndpointParams params, RulePartition partitionResult, String region) {
        if (params.useDualStack()) {
            RuleResult result = endpointRule20(params, partitionResult, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("DualStack is enabled but this partition does not support DualStack");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule20(CodeGuruSecurityEndpointParams params, RulePartition partitionResult, String region) {
        if (partitionResult.supportsDualStack()) {
            return endpointRule21(params, region, partitionResult);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule21(CodeGuruSecurityEndpointParams params, String region, RulePartition partitionResult) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://codeguru-security." + region + "." + partitionResult.dualStackDnsSuffix())).build());
    }

    private static RuleResult endpointRule24(CodeGuruSecurityEndpointParams params, String region, RulePartition partitionResult) {
        return RuleResult.endpoint(Endpoint.builder()
                .url(URI.create("https://codeguru-security." + region + "." + partitionResult.dnsSuffix())).build());
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }
}
