/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a set of Amazon ECS tasks in an CodeDeploy deployment. An Amazon ECS task set includes details such
 * as the desired number of tasks, how many tasks are running, and whether the task set serves production traffic. An
 * CodeDeploy application that uses the Amazon ECS compute platform deploys a containerized application in an Amazon ECS
 * service as a task set.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ECSTaskSet implements SdkPojo, Serializable, ToCopyableBuilder<ECSTaskSet.Builder, ECSTaskSet> {
    private static final SdkField<String> IDENTIFER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("identifer").getter(getter(ECSTaskSet::identifer)).setter(setter(Builder::identifer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identifer").build()).build();

    private static final SdkField<Long> DESIRED_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("desiredCount").getter(getter(ECSTaskSet::desiredCount)).setter(setter(Builder::desiredCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("desiredCount").build()).build();

    private static final SdkField<Long> PENDING_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("pendingCount").getter(getter(ECSTaskSet::pendingCount)).setter(setter(Builder::pendingCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pendingCount").build()).build();

    private static final SdkField<Long> RUNNING_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("runningCount").getter(getter(ECSTaskSet::runningCount)).setter(setter(Builder::runningCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("runningCount").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ECSTaskSet::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Double> TRAFFIC_WEIGHT_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("trafficWeight").getter(getter(ECSTaskSet::trafficWeight)).setter(setter(Builder::trafficWeight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("trafficWeight").build()).build();

    private static final SdkField<TargetGroupInfo> TARGET_GROUP_FIELD = SdkField
            .<TargetGroupInfo> builder(MarshallingType.SDK_POJO).memberName("targetGroup")
            .getter(getter(ECSTaskSet::targetGroup)).setter(setter(Builder::targetGroup)).constructor(TargetGroupInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetGroup").build()).build();

    private static final SdkField<String> TASK_SET_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskSetLabel").getter(getter(ECSTaskSet::taskSetLabelAsString)).setter(setter(Builder::taskSetLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskSetLabel").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTIFER_FIELD,
            DESIRED_COUNT_FIELD, PENDING_COUNT_FIELD, RUNNING_COUNT_FIELD, STATUS_FIELD, TRAFFIC_WEIGHT_FIELD,
            TARGET_GROUP_FIELD, TASK_SET_LABEL_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String identifer;

    private final Long desiredCount;

    private final Long pendingCount;

    private final Long runningCount;

    private final String status;

    private final Double trafficWeight;

    private final TargetGroupInfo targetGroup;

    private final String taskSetLabel;

    private ECSTaskSet(BuilderImpl builder) {
        this.identifer = builder.identifer;
        this.desiredCount = builder.desiredCount;
        this.pendingCount = builder.pendingCount;
        this.runningCount = builder.runningCount;
        this.status = builder.status;
        this.trafficWeight = builder.trafficWeight;
        this.targetGroup = builder.targetGroup;
        this.taskSetLabel = builder.taskSetLabel;
    }

    /**
     * <p>
     * A unique ID of an <code>ECSTaskSet</code>.
     * </p>
     * 
     * @return A unique ID of an <code>ECSTaskSet</code>.
     */
    public final String identifer() {
        return identifer;
    }

    /**
     * <p>
     * The number of tasks in a task set. During a deployment that uses the Amazon ECS compute type, CodeDeploy
     * instructs Amazon ECS to create a new task set and uses this value to determine how many tasks to create. After
     * the updated task set is created, CodeDeploy shifts traffic to the new task set.
     * </p>
     * 
     * @return The number of tasks in a task set. During a deployment that uses the Amazon ECS compute type, CodeDeploy
     *         instructs Amazon ECS to create a new task set and uses this value to determine how many tasks to create.
     *         After the updated task set is created, CodeDeploy shifts traffic to the new task set.
     */
    public final Long desiredCount() {
        return desiredCount;
    }

    /**
     * <p>
     * The number of tasks in the task set that are in the <code>PENDING</code> status during an Amazon ECS deployment.
     * A task in the <code>PENDING</code> state is preparing to enter the <code>RUNNING</code> state. A task set enters
     * the <code>PENDING</code> status when it launches for the first time, or when it is restarted after being in the
     * <code>STOPPED</code> state.
     * </p>
     * 
     * @return The number of tasks in the task set that are in the <code>PENDING</code> status during an Amazon ECS
     *         deployment. A task in the <code>PENDING</code> state is preparing to enter the <code>RUNNING</code>
     *         state. A task set enters the <code>PENDING</code> status when it launches for the first time, or when it
     *         is restarted after being in the <code>STOPPED</code> state.
     */
    public final Long pendingCount() {
        return pendingCount;
    }

    /**
     * <p>
     * The number of tasks in the task set that are in the <code>RUNNING</code> status during an Amazon ECS deployment.
     * A task in the <code>RUNNING</code> state is running and ready for use.
     * </p>
     * 
     * @return The number of tasks in the task set that are in the <code>RUNNING</code> status during an Amazon ECS
     *         deployment. A task in the <code>RUNNING</code> state is running and ready for use.
     */
    public final Long runningCount() {
        return runningCount;
    }

    /**
     * <p>
     * The status of the task set. There are three valid task set statuses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PRIMARY</code>: Indicates the task set is serving production traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code>: Indicates the task set is not serving production traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DRAINING</code>: Indicates the tasks in the task set are being stopped and their corresponding targets are
     * being deregistered from their target group.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The status of the task set. There are three valid task set statuses: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PRIMARY</code>: Indicates the task set is serving production traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code>: Indicates the task set is not serving production traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DRAINING</code>: Indicates the tasks in the task set are being stopped and their corresponding
     *         targets are being deregistered from their target group.
     *         </p>
     *         </li>
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The percentage of traffic served by this task set.
     * </p>
     * 
     * @return The percentage of traffic served by this task set.
     */
    public final Double trafficWeight() {
        return trafficWeight;
    }

    /**
     * <p>
     * The target group associated with the task set. The target group is used by CodeDeploy to manage traffic to a task
     * set.
     * </p>
     * 
     * @return The target group associated with the task set. The target group is used by CodeDeploy to manage traffic
     *         to a task set.
     */
    public final TargetGroupInfo targetGroup() {
        return targetGroup;
    }

    /**
     * <p>
     * A label that identifies whether the ECS task set is an original target (<code>BLUE</code>) or a replacement
     * target (<code>GREEN</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskSetLabel} will
     * return {@link TargetLabel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #taskSetLabelAsString}.
     * </p>
     * 
     * @return A label that identifies whether the ECS task set is an original target (<code>BLUE</code>) or a
     *         replacement target (<code>GREEN</code>).
     * @see TargetLabel
     */
    public final TargetLabel taskSetLabel() {
        return TargetLabel.fromValue(taskSetLabel);
    }

    /**
     * <p>
     * A label that identifies whether the ECS task set is an original target (<code>BLUE</code>) or a replacement
     * target (<code>GREEN</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskSetLabel} will
     * return {@link TargetLabel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #taskSetLabelAsString}.
     * </p>
     * 
     * @return A label that identifies whether the ECS task set is an original target (<code>BLUE</code>) or a
     *         replacement target (<code>GREEN</code>).
     * @see TargetLabel
     */
    public final String taskSetLabelAsString() {
        return taskSetLabel;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(identifer());
        hashCode = 31 * hashCode + Objects.hashCode(desiredCount());
        hashCode = 31 * hashCode + Objects.hashCode(pendingCount());
        hashCode = 31 * hashCode + Objects.hashCode(runningCount());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(trafficWeight());
        hashCode = 31 * hashCode + Objects.hashCode(targetGroup());
        hashCode = 31 * hashCode + Objects.hashCode(taskSetLabelAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ECSTaskSet)) {
            return false;
        }
        ECSTaskSet other = (ECSTaskSet) obj;
        return Objects.equals(identifer(), other.identifer()) && Objects.equals(desiredCount(), other.desiredCount())
                && Objects.equals(pendingCount(), other.pendingCount()) && Objects.equals(runningCount(), other.runningCount())
                && Objects.equals(status(), other.status()) && Objects.equals(trafficWeight(), other.trafficWeight())
                && Objects.equals(targetGroup(), other.targetGroup())
                && Objects.equals(taskSetLabelAsString(), other.taskSetLabelAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ECSTaskSet").add("Identifer", identifer()).add("DesiredCount", desiredCount())
                .add("PendingCount", pendingCount()).add("RunningCount", runningCount()).add("Status", status())
                .add("TrafficWeight", trafficWeight()).add("TargetGroup", targetGroup())
                .add("TaskSetLabel", taskSetLabelAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "identifer":
            return Optional.ofNullable(clazz.cast(identifer()));
        case "desiredCount":
            return Optional.ofNullable(clazz.cast(desiredCount()));
        case "pendingCount":
            return Optional.ofNullable(clazz.cast(pendingCount()));
        case "runningCount":
            return Optional.ofNullable(clazz.cast(runningCount()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "trafficWeight":
            return Optional.ofNullable(clazz.cast(trafficWeight()));
        case "targetGroup":
            return Optional.ofNullable(clazz.cast(targetGroup()));
        case "taskSetLabel":
            return Optional.ofNullable(clazz.cast(taskSetLabelAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("identifer", IDENTIFER_FIELD);
        map.put("desiredCount", DESIRED_COUNT_FIELD);
        map.put("pendingCount", PENDING_COUNT_FIELD);
        map.put("runningCount", RUNNING_COUNT_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("trafficWeight", TRAFFIC_WEIGHT_FIELD);
        map.put("targetGroup", TARGET_GROUP_FIELD);
        map.put("taskSetLabel", TASK_SET_LABEL_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ECSTaskSet, T> g) {
        return obj -> g.apply((ECSTaskSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ECSTaskSet> {
        /**
         * <p>
         * A unique ID of an <code>ECSTaskSet</code>.
         * </p>
         * 
         * @param identifer
         *        A unique ID of an <code>ECSTaskSet</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifer(String identifer);

        /**
         * <p>
         * The number of tasks in a task set. During a deployment that uses the Amazon ECS compute type, CodeDeploy
         * instructs Amazon ECS to create a new task set and uses this value to determine how many tasks to create.
         * After the updated task set is created, CodeDeploy shifts traffic to the new task set.
         * </p>
         * 
         * @param desiredCount
         *        The number of tasks in a task set. During a deployment that uses the Amazon ECS compute type,
         *        CodeDeploy instructs Amazon ECS to create a new task set and uses this value to determine how many
         *        tasks to create. After the updated task set is created, CodeDeploy shifts traffic to the new task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredCount(Long desiredCount);

        /**
         * <p>
         * The number of tasks in the task set that are in the <code>PENDING</code> status during an Amazon ECS
         * deployment. A task in the <code>PENDING</code> state is preparing to enter the <code>RUNNING</code> state. A
         * task set enters the <code>PENDING</code> status when it launches for the first time, or when it is restarted
         * after being in the <code>STOPPED</code> state.
         * </p>
         * 
         * @param pendingCount
         *        The number of tasks in the task set that are in the <code>PENDING</code> status during an Amazon ECS
         *        deployment. A task in the <code>PENDING</code> state is preparing to enter the <code>RUNNING</code>
         *        state. A task set enters the <code>PENDING</code> status when it launches for the first time, or when
         *        it is restarted after being in the <code>STOPPED</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingCount(Long pendingCount);

        /**
         * <p>
         * The number of tasks in the task set that are in the <code>RUNNING</code> status during an Amazon ECS
         * deployment. A task in the <code>RUNNING</code> state is running and ready for use.
         * </p>
         * 
         * @param runningCount
         *        The number of tasks in the task set that are in the <code>RUNNING</code> status during an Amazon ECS
         *        deployment. A task in the <code>RUNNING</code> state is running and ready for use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runningCount(Long runningCount);

        /**
         * <p>
         * The status of the task set. There are three valid task set statuses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PRIMARY</code>: Indicates the task set is serving production traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code>: Indicates the task set is not serving production traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DRAINING</code>: Indicates the tasks in the task set are being stopped and their corresponding targets
         * are being deregistered from their target group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the task set. There are three valid task set statuses: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PRIMARY</code>: Indicates the task set is serving production traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code>: Indicates the task set is not serving production traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DRAINING</code>: Indicates the tasks in the task set are being stopped and their corresponding
         *        targets are being deregistered from their target group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The percentage of traffic served by this task set.
         * </p>
         * 
         * @param trafficWeight
         *        The percentage of traffic served by this task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trafficWeight(Double trafficWeight);

        /**
         * <p>
         * The target group associated with the task set. The target group is used by CodeDeploy to manage traffic to a
         * task set.
         * </p>
         * 
         * @param targetGroup
         *        The target group associated with the task set. The target group is used by CodeDeploy to manage
         *        traffic to a task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetGroup(TargetGroupInfo targetGroup);

        /**
         * <p>
         * The target group associated with the task set. The target group is used by CodeDeploy to manage traffic to a
         * task set.
         * </p>
         * This is a convenience method that creates an instance of the {@link TargetGroupInfo.Builder} avoiding the
         * need to create one manually via {@link TargetGroupInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TargetGroupInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #targetGroup(TargetGroupInfo)}.
         * 
         * @param targetGroup
         *        a consumer that will call methods on {@link TargetGroupInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetGroup(TargetGroupInfo)
         */
        default Builder targetGroup(Consumer<TargetGroupInfo.Builder> targetGroup) {
            return targetGroup(TargetGroupInfo.builder().applyMutation(targetGroup).build());
        }

        /**
         * <p>
         * A label that identifies whether the ECS task set is an original target (<code>BLUE</code>) or a replacement
         * target (<code>GREEN</code>).
         * </p>
         * 
         * @param taskSetLabel
         *        A label that identifies whether the ECS task set is an original target (<code>BLUE</code>) or a
         *        replacement target (<code>GREEN</code>).
         * @see TargetLabel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetLabel
         */
        Builder taskSetLabel(String taskSetLabel);

        /**
         * <p>
         * A label that identifies whether the ECS task set is an original target (<code>BLUE</code>) or a replacement
         * target (<code>GREEN</code>).
         * </p>
         * 
         * @param taskSetLabel
         *        A label that identifies whether the ECS task set is an original target (<code>BLUE</code>) or a
         *        replacement target (<code>GREEN</code>).
         * @see TargetLabel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetLabel
         */
        Builder taskSetLabel(TargetLabel taskSetLabel);
    }

    static final class BuilderImpl implements Builder {
        private String identifer;

        private Long desiredCount;

        private Long pendingCount;

        private Long runningCount;

        private String status;

        private Double trafficWeight;

        private TargetGroupInfo targetGroup;

        private String taskSetLabel;

        private BuilderImpl() {
        }

        private BuilderImpl(ECSTaskSet model) {
            identifer(model.identifer);
            desiredCount(model.desiredCount);
            pendingCount(model.pendingCount);
            runningCount(model.runningCount);
            status(model.status);
            trafficWeight(model.trafficWeight);
            targetGroup(model.targetGroup);
            taskSetLabel(model.taskSetLabel);
        }

        public final String getIdentifer() {
            return identifer;
        }

        public final void setIdentifer(String identifer) {
            this.identifer = identifer;
        }

        @Override
        public final Builder identifer(String identifer) {
            this.identifer = identifer;
            return this;
        }

        public final Long getDesiredCount() {
            return desiredCount;
        }

        public final void setDesiredCount(Long desiredCount) {
            this.desiredCount = desiredCount;
        }

        @Override
        public final Builder desiredCount(Long desiredCount) {
            this.desiredCount = desiredCount;
            return this;
        }

        public final Long getPendingCount() {
            return pendingCount;
        }

        public final void setPendingCount(Long pendingCount) {
            this.pendingCount = pendingCount;
        }

        @Override
        public final Builder pendingCount(Long pendingCount) {
            this.pendingCount = pendingCount;
            return this;
        }

        public final Long getRunningCount() {
            return runningCount;
        }

        public final void setRunningCount(Long runningCount) {
            this.runningCount = runningCount;
        }

        @Override
        public final Builder runningCount(Long runningCount) {
            this.runningCount = runningCount;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Double getTrafficWeight() {
            return trafficWeight;
        }

        public final void setTrafficWeight(Double trafficWeight) {
            this.trafficWeight = trafficWeight;
        }

        @Override
        public final Builder trafficWeight(Double trafficWeight) {
            this.trafficWeight = trafficWeight;
            return this;
        }

        public final TargetGroupInfo.Builder getTargetGroup() {
            return targetGroup != null ? targetGroup.toBuilder() : null;
        }

        public final void setTargetGroup(TargetGroupInfo.BuilderImpl targetGroup) {
            this.targetGroup = targetGroup != null ? targetGroup.build() : null;
        }

        @Override
        public final Builder targetGroup(TargetGroupInfo targetGroup) {
            this.targetGroup = targetGroup;
            return this;
        }

        public final String getTaskSetLabel() {
            return taskSetLabel;
        }

        public final void setTaskSetLabel(String taskSetLabel) {
            this.taskSetLabel = taskSetLabel;
        }

        @Override
        public final Builder taskSetLabel(String taskSetLabel) {
            this.taskSetLabel = taskSetLabel;
            return this;
        }

        @Override
        public final Builder taskSetLabel(TargetLabel taskSetLabel) {
            this.taskSetLabel(taskSetLabel == null ? null : taskSetLabel.toString());
            return this;
        }

        @Override
        public ECSTaskSet build() {
            return new ECSTaskSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
