/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about credentials that provide access to a private Docker registry. When this is set:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>imagePullCredentialsType</code> must be set to <code>SERVICE_ROLE</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * images cannot be curated or an Amazon ECR image.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/codebuild/latest/userguide/sample-private-registry.html">Private Registry with
 * Secrets Manager Sample for CodeBuild</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RegistryCredential implements SdkPojo, Serializable,
        ToCopyableBuilder<RegistryCredential.Builder, RegistryCredential> {
    private static final SdkField<String> CREDENTIAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("credential").getter(getter(RegistryCredential::credential)).setter(setter(Builder::credential))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("credential").build()).build();

    private static final SdkField<String> CREDENTIAL_PROVIDER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("credentialProvider").getter(getter(RegistryCredential::credentialProviderAsString))
            .setter(setter(Builder::credentialProvider))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("credentialProvider").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREDENTIAL_FIELD,
            CREDENTIAL_PROVIDER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String credential;

    private final String credentialProvider;

    private RegistryCredential(BuilderImpl builder) {
        this.credential = builder.credential;
        this.credentialProvider = builder.credentialProvider;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) or name of credentials created using Secrets Manager.
     * </p>
     * <note>
     * <p>
     * The <code>credential</code> can use the name of the credentials only if they exist in your current Amazon Web
     * Services Region.
     * </p>
     * </note>
     * 
     * @return The Amazon Resource Name (ARN) or name of credentials created using Secrets Manager. </p> <note>
     *         <p>
     *         The <code>credential</code> can use the name of the credentials only if they exist in your current Amazon
     *         Web Services Region.
     *         </p>
     */
    public final String credential() {
        return credential;
    }

    /**
     * <p>
     * The service that created the credentials to access a private Docker registry. The valid value, SECRETS_MANAGER,
     * is for Secrets Manager.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #credentialProvider} will return {@link CredentialProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #credentialProviderAsString}.
     * </p>
     * 
     * @return The service that created the credentials to access a private Docker registry. The valid value,
     *         SECRETS_MANAGER, is for Secrets Manager.
     * @see CredentialProviderType
     */
    public final CredentialProviderType credentialProvider() {
        return CredentialProviderType.fromValue(credentialProvider);
    }

    /**
     * <p>
     * The service that created the credentials to access a private Docker registry. The valid value, SECRETS_MANAGER,
     * is for Secrets Manager.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #credentialProvider} will return {@link CredentialProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #credentialProviderAsString}.
     * </p>
     * 
     * @return The service that created the credentials to access a private Docker registry. The valid value,
     *         SECRETS_MANAGER, is for Secrets Manager.
     * @see CredentialProviderType
     */
    public final String credentialProviderAsString() {
        return credentialProvider;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(credential());
        hashCode = 31 * hashCode + Objects.hashCode(credentialProviderAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RegistryCredential)) {
            return false;
        }
        RegistryCredential other = (RegistryCredential) obj;
        return Objects.equals(credential(), other.credential())
                && Objects.equals(credentialProviderAsString(), other.credentialProviderAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RegistryCredential").add("Credential", credential())
                .add("CredentialProvider", credentialProviderAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "credential":
            return Optional.ofNullable(clazz.cast(credential()));
        case "credentialProvider":
            return Optional.ofNullable(clazz.cast(credentialProviderAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RegistryCredential, T> g) {
        return obj -> g.apply((RegistryCredential) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RegistryCredential> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) or name of credentials created using Secrets Manager.
         * </p>
         * <note>
         * <p>
         * The <code>credential</code> can use the name of the credentials only if they exist in your current Amazon Web
         * Services Region.
         * </p>
         * </note>
         * 
         * @param credential
         *        The Amazon Resource Name (ARN) or name of credentials created using Secrets Manager. </p> <note>
         *        <p>
         *        The <code>credential</code> can use the name of the credentials only if they exist in your current
         *        Amazon Web Services Region.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder credential(String credential);

        /**
         * <p>
         * The service that created the credentials to access a private Docker registry. The valid value,
         * SECRETS_MANAGER, is for Secrets Manager.
         * </p>
         * 
         * @param credentialProvider
         *        The service that created the credentials to access a private Docker registry. The valid value,
         *        SECRETS_MANAGER, is for Secrets Manager.
         * @see CredentialProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CredentialProviderType
         */
        Builder credentialProvider(String credentialProvider);

        /**
         * <p>
         * The service that created the credentials to access a private Docker registry. The valid value,
         * SECRETS_MANAGER, is for Secrets Manager.
         * </p>
         * 
         * @param credentialProvider
         *        The service that created the credentials to access a private Docker registry. The valid value,
         *        SECRETS_MANAGER, is for Secrets Manager.
         * @see CredentialProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CredentialProviderType
         */
        Builder credentialProvider(CredentialProviderType credentialProvider);
    }

    static final class BuilderImpl implements Builder {
        private String credential;

        private String credentialProvider;

        private BuilderImpl() {
        }

        private BuilderImpl(RegistryCredential model) {
            credential(model.credential);
            credentialProvider(model.credentialProvider);
        }

        public final String getCredential() {
            return credential;
        }

        public final void setCredential(String credential) {
            this.credential = credential;
        }

        @Override
        public final Builder credential(String credential) {
            this.credential = credential;
            return this;
        }

        public final String getCredentialProvider() {
            return credentialProvider;
        }

        public final void setCredentialProvider(String credentialProvider) {
            this.credentialProvider = credentialProvider;
        }

        @Override
        public final Builder credentialProvider(String credentialProvider) {
            this.credentialProvider = credentialProvider;
            return this;
        }

        @Override
        public final Builder credentialProvider(CredentialProviderType credentialProvider) {
            this.credentialProvider(credentialProvider == null ? null : credentialProvider.toString());
            return this;
        }

        @Override
        public RegistryCredential build() {
            return new RegistryCredential(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
