/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.waiters;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.cloudwatch.CloudWatchClient;
import software.amazon.awssdk.services.cloudwatch.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.cloudwatch.model.CloudWatchRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsResponse;
import software.amazon.awssdk.services.cloudwatch.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultCloudWatchWaiter implements CloudWatchWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final CloudWatchClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeAlarmsResponse> alarmExistsWaiter;

    private final Waiter<DescribeAlarmsResponse> compositeAlarmExistsWaiter;

    private DefaultCloudWatchWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = CloudWatchClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.alarmExistsWaiter = Waiter.builder(DescribeAlarmsResponse.class).acceptors(alarmExistsWaiterAcceptors())
                .overrideConfiguration(alarmExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.compositeAlarmExistsWaiter = Waiter.builder(DescribeAlarmsResponse.class)
                .acceptors(compositeAlarmExistsWaiterAcceptors())
                .overrideConfiguration(compositeAlarmExistsWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeAlarmsResponse> waitUntilAlarmExists(DescribeAlarmsRequest describeAlarmsRequest) {
        return alarmExistsWaiter.run(() -> client.describeAlarms(applyWaitersUserAgent(describeAlarmsRequest)));
    }

    @Override
    public WaiterResponse<DescribeAlarmsResponse> waitUntilAlarmExists(DescribeAlarmsRequest describeAlarmsRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return alarmExistsWaiter.run(() -> client.describeAlarms(applyWaitersUserAgent(describeAlarmsRequest)),
                alarmExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeAlarmsResponse> waitUntilCompositeAlarmExists(DescribeAlarmsRequest describeAlarmsRequest) {
        return compositeAlarmExistsWaiter.run(() -> client.describeAlarms(applyWaitersUserAgent(describeAlarmsRequest)));
    }

    @Override
    public WaiterResponse<DescribeAlarmsResponse> waitUntilCompositeAlarmExists(DescribeAlarmsRequest describeAlarmsRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return compositeAlarmExistsWaiter.run(() -> client.describeAlarms(applyWaitersUserAgent(describeAlarmsRequest)),
                compositeAlarmExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeAlarmsResponse>> alarmExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAlarmsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("MetricAlarms").flatten().length()
                    .compare(">", input.constant(new BigDecimal("0"))).value(), true);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAlarmsResponse>> compositeAlarmExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAlarmsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(
                    input.field("CompositeAlarms").flatten().length().compare(">", input.constant(new BigDecimal("0"))).value(),
                    true);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration alarmExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration compositeAlarmExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static CloudWatchWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends CloudWatchRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements CloudWatchWaiter.Builder {
        private CloudWatchClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public CloudWatchWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public CloudWatchWaiter.Builder client(CloudWatchClient client) {
            this.client = client;
            return this;
        }

        public CloudWatchWaiter build() {
            return new DefaultCloudWatchWaiter(this);
        }
    }
}
