/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The customizations that you specified for the distribution tenant for geographic restrictions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GeoRestrictionCustomization implements SdkPojo, Serializable,
        ToCopyableBuilder<GeoRestrictionCustomization.Builder, GeoRestrictionCustomization> {
    private static final SdkField<String> RESTRICTION_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RestrictionType")
            .getter(getter(GeoRestrictionCustomization::restrictionTypeAsString))
            .setter(setter(Builder::restrictionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RestrictionType")
                    .unmarshallLocationName("RestrictionType").build()).build();

    private static final SdkField<List<String>> LOCATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Locations")
            .getter(getter(GeoRestrictionCustomization::locations))
            .setter(setter(Builder::locations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Locations")
                    .unmarshallLocationName("Locations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Location")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Location").unmarshallLocationName("Location").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESTRICTION_TYPE_FIELD,
            LOCATIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String restrictionType;

    private final List<String> locations;

    private GeoRestrictionCustomization(BuilderImpl builder) {
        this.restrictionType = builder.restrictionType;
        this.locations = builder.locations;
    }

    /**
     * <p>
     * The method that you want to use to restrict distribution of your content by country:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>none</code>: No geographic restriction is enabled, meaning access to content is not restricted by client
     * geo location.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
     * CloudFront to distribute your content.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want CloudFront to
     * distribute your content.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #restrictionType}
     * will return {@link GeoRestrictionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #restrictionTypeAsString}.
     * </p>
     * 
     * @return The method that you want to use to restrict distribution of your content by country:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>none</code>: No geographic restriction is enabled, meaning access to content is not restricted by
     *         client geo location.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
     *         CloudFront to distribute your content.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want
     *         CloudFront to distribute your content.
     *         </p>
     *         </li>
     * @see GeoRestrictionType
     */
    public final GeoRestrictionType restrictionType() {
        return GeoRestrictionType.fromValue(restrictionType);
    }

    /**
     * <p>
     * The method that you want to use to restrict distribution of your content by country:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>none</code>: No geographic restriction is enabled, meaning access to content is not restricted by client
     * geo location.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
     * CloudFront to distribute your content.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want CloudFront to
     * distribute your content.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #restrictionType}
     * will return {@link GeoRestrictionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #restrictionTypeAsString}.
     * </p>
     * 
     * @return The method that you want to use to restrict distribution of your content by country:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>none</code>: No geographic restriction is enabled, meaning access to content is not restricted by
     *         client geo location.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
     *         CloudFront to distribute your content.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want
     *         CloudFront to distribute your content.
     *         </p>
     *         </li>
     * @see GeoRestrictionType
     */
    public final String restrictionTypeAsString() {
        return restrictionType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Locations property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasLocations() {
        return locations != null && !(locations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The locations for geographic restrictions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLocations} method.
     * </p>
     * 
     * @return The locations for geographic restrictions.
     */
    public final List<String> locations() {
        return locations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(restrictionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLocations() ? locations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GeoRestrictionCustomization)) {
            return false;
        }
        GeoRestrictionCustomization other = (GeoRestrictionCustomization) obj;
        return Objects.equals(restrictionTypeAsString(), other.restrictionTypeAsString())
                && hasLocations() == other.hasLocations() && Objects.equals(locations(), other.locations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GeoRestrictionCustomization").add("RestrictionType", restrictionTypeAsString())
                .add("Locations", hasLocations() ? locations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RestrictionType":
            return Optional.ofNullable(clazz.cast(restrictionTypeAsString()));
        case "Locations":
            return Optional.ofNullable(clazz.cast(locations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RestrictionType", RESTRICTION_TYPE_FIELD);
        map.put("Locations", LOCATIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GeoRestrictionCustomization, T> g) {
        return obj -> g.apply((GeoRestrictionCustomization) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GeoRestrictionCustomization> {
        /**
         * <p>
         * The method that you want to use to restrict distribution of your content by country:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>none</code>: No geographic restriction is enabled, meaning access to content is not restricted by
         * client geo location.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
         * CloudFront to distribute your content.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want CloudFront
         * to distribute your content.
         * </p>
         * </li>
         * </ul>
         * 
         * @param restrictionType
         *        The method that you want to use to restrict distribution of your content by country:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>none</code>: No geographic restriction is enabled, meaning access to content is not restricted
         *        by client geo location.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't
         *        want CloudFront to distribute your content.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want
         *        CloudFront to distribute your content.
         *        </p>
         *        </li>
         * @see GeoRestrictionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeoRestrictionType
         */
        Builder restrictionType(String restrictionType);

        /**
         * <p>
         * The method that you want to use to restrict distribution of your content by country:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>none</code>: No geographic restriction is enabled, meaning access to content is not restricted by
         * client geo location.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't want
         * CloudFront to distribute your content.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want CloudFront
         * to distribute your content.
         * </p>
         * </li>
         * </ul>
         * 
         * @param restrictionType
         *        The method that you want to use to restrict distribution of your content by country:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>none</code>: No geographic restriction is enabled, meaning access to content is not restricted
         *        by client geo location.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>blacklist</code>: The <code>Location</code> elements specify the countries in which you don't
         *        want CloudFront to distribute your content.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>whitelist</code>: The <code>Location</code> elements specify the countries in which you want
         *        CloudFront to distribute your content.
         *        </p>
         *        </li>
         * @see GeoRestrictionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeoRestrictionType
         */
        Builder restrictionType(GeoRestrictionType restrictionType);

        /**
         * <p>
         * The locations for geographic restrictions.
         * </p>
         * 
         * @param locations
         *        The locations for geographic restrictions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locations(Collection<String> locations);

        /**
         * <p>
         * The locations for geographic restrictions.
         * </p>
         * 
         * @param locations
         *        The locations for geographic restrictions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locations(String... locations);
    }

    static final class BuilderImpl implements Builder {
        private String restrictionType;

        private List<String> locations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GeoRestrictionCustomization model) {
            restrictionType(model.restrictionType);
            locations(model.locations);
        }

        public final String getRestrictionType() {
            return restrictionType;
        }

        public final void setRestrictionType(String restrictionType) {
            this.restrictionType = restrictionType;
        }

        @Override
        public final Builder restrictionType(String restrictionType) {
            this.restrictionType = restrictionType;
            return this;
        }

        @Override
        public final Builder restrictionType(GeoRestrictionType restrictionType) {
            this.restrictionType(restrictionType == null ? null : restrictionType.toString());
            return this;
        }

        public final Collection<String> getLocations() {
            if (locations instanceof SdkAutoConstructList) {
                return null;
            }
            return locations;
        }

        public final void setLocations(Collection<String> locations) {
            this.locations = LocationListCopier.copy(locations);
        }

        @Override
        public final Builder locations(Collection<String> locations) {
            this.locations = LocationListCopier.copy(locations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder locations(String... locations) {
            locations(Arrays.asList(locations));
            return this;
        }

        @Override
        public GeoRestrictionCustomization build() {
            return new GeoRestrictionCustomization(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
