/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Determines whether CloudFront includes the <code>X-Frame-Options</code> HTTP response header and the header's value.
 * </p>
 * <p>
 * For more information about the <code>X-Frame-Options</code> HTTP response header, see <a
 * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in the MDN Web
 * Docs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResponseHeadersPolicyFrameOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<ResponseHeadersPolicyFrameOptions.Builder, ResponseHeadersPolicyFrameOptions> {
    private static final SdkField<Boolean> OVERRIDE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Override")
            .getter(getter(ResponseHeadersPolicyFrameOptions::override))
            .setter(setter(Builder::override))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Override")
                    .unmarshallLocationName("Override").build()).build();

    private static final SdkField<String> FRAME_OPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FrameOption")
            .getter(getter(ResponseHeadersPolicyFrameOptions::frameOptionAsString))
            .setter(setter(Builder::frameOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FrameOption")
                    .unmarshallLocationName("FrameOption").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OVERRIDE_FIELD,
            FRAME_OPTION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean override;

    private final String frameOption;

    private ResponseHeadersPolicyFrameOptions(BuilderImpl builder) {
        this.override = builder.override;
        this.frameOption = builder.frameOption;
    }

    /**
     * <p>
     * A Boolean that determines whether CloudFront overrides the <code>X-Frame-Options</code> HTTP response header
     * received from the origin with the one specified in this response headers policy.
     * </p>
     * 
     * @return A Boolean that determines whether CloudFront overrides the <code>X-Frame-Options</code> HTTP response
     *         header received from the origin with the one specified in this response headers policy.
     */
    public final Boolean override() {
        return override;
    }

    /**
     * <p>
     * The value of the <code>X-Frame-Options</code> HTTP response header. Valid values are <code>DENY</code> and
     * <code>SAMEORIGIN</code>.
     * </p>
     * <p>
     * For more information about these values, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in the MDN
     * Web Docs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #frameOption} will
     * return {@link FrameOptionsList#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #frameOptionAsString}.
     * </p>
     * 
     * @return The value of the <code>X-Frame-Options</code> HTTP response header. Valid values are <code>DENY</code>
     *         and <code>SAMEORIGIN</code>.</p>
     *         <p>
     *         For more information about these values, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in
     *         the MDN Web Docs.
     * @see FrameOptionsList
     */
    public final FrameOptionsList frameOption() {
        return FrameOptionsList.fromValue(frameOption);
    }

    /**
     * <p>
     * The value of the <code>X-Frame-Options</code> HTTP response header. Valid values are <code>DENY</code> and
     * <code>SAMEORIGIN</code>.
     * </p>
     * <p>
     * For more information about these values, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in the MDN
     * Web Docs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #frameOption} will
     * return {@link FrameOptionsList#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #frameOptionAsString}.
     * </p>
     * 
     * @return The value of the <code>X-Frame-Options</code> HTTP response header. Valid values are <code>DENY</code>
     *         and <code>SAMEORIGIN</code>.</p>
     *         <p>
     *         For more information about these values, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in
     *         the MDN Web Docs.
     * @see FrameOptionsList
     */
    public final String frameOptionAsString() {
        return frameOption;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(override());
        hashCode = 31 * hashCode + Objects.hashCode(frameOptionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResponseHeadersPolicyFrameOptions)) {
            return false;
        }
        ResponseHeadersPolicyFrameOptions other = (ResponseHeadersPolicyFrameOptions) obj;
        return Objects.equals(override(), other.override()) && Objects.equals(frameOptionAsString(), other.frameOptionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResponseHeadersPolicyFrameOptions").add("Override", override())
                .add("FrameOption", frameOptionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Override":
            return Optional.ofNullable(clazz.cast(override()));
        case "FrameOption":
            return Optional.ofNullable(clazz.cast(frameOptionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Override", OVERRIDE_FIELD);
        map.put("FrameOption", FRAME_OPTION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResponseHeadersPolicyFrameOptions, T> g) {
        return obj -> g.apply((ResponseHeadersPolicyFrameOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResponseHeadersPolicyFrameOptions> {
        /**
         * <p>
         * A Boolean that determines whether CloudFront overrides the <code>X-Frame-Options</code> HTTP response header
         * received from the origin with the one specified in this response headers policy.
         * </p>
         * 
         * @param override
         *        A Boolean that determines whether CloudFront overrides the <code>X-Frame-Options</code> HTTP response
         *        header received from the origin with the one specified in this response headers policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder override(Boolean override);

        /**
         * <p>
         * The value of the <code>X-Frame-Options</code> HTTP response header. Valid values are <code>DENY</code> and
         * <code>SAMEORIGIN</code>.
         * </p>
         * <p>
         * For more information about these values, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in the
         * MDN Web Docs.
         * </p>
         * 
         * @param frameOption
         *        The value of the <code>X-Frame-Options</code> HTTP response header. Valid values are <code>DENY</code>
         *        and <code>SAMEORIGIN</code>.</p>
         *        <p>
         *        For more information about these values, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a>
         *        in the MDN Web Docs.
         * @see FrameOptionsList
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FrameOptionsList
         */
        Builder frameOption(String frameOption);

        /**
         * <p>
         * The value of the <code>X-Frame-Options</code> HTTP response header. Valid values are <code>DENY</code> and
         * <code>SAMEORIGIN</code>.
         * </p>
         * <p>
         * For more information about these values, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in the
         * MDN Web Docs.
         * </p>
         * 
         * @param frameOption
         *        The value of the <code>X-Frame-Options</code> HTTP response header. Valid values are <code>DENY</code>
         *        and <code>SAMEORIGIN</code>.</p>
         *        <p>
         *        For more information about these values, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a>
         *        in the MDN Web Docs.
         * @see FrameOptionsList
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FrameOptionsList
         */
        Builder frameOption(FrameOptionsList frameOption);
    }

    static final class BuilderImpl implements Builder {
        private Boolean override;

        private String frameOption;

        private BuilderImpl() {
        }

        private BuilderImpl(ResponseHeadersPolicyFrameOptions model) {
            override(model.override);
            frameOption(model.frameOption);
        }

        public final Boolean getOverride() {
            return override;
        }

        public final void setOverride(Boolean override) {
            this.override = override;
        }

        @Override
        public final Builder override(Boolean override) {
            this.override = override;
            return this;
        }

        public final String getFrameOption() {
            return frameOption;
        }

        public final void setFrameOption(String frameOption) {
            this.frameOption = frameOption;
        }

        @Override
        public final Builder frameOption(String frameOption) {
            this.frameOption = frameOption;
            return this;
        }

        @Override
        public final Builder frameOption(FrameOptionsList frameOption) {
            this.frameOption(frameOption == null ? null : frameOption.toString());
            return this;
        }

        @Override
        public ResponseHeadersPolicyFrameOptions build() {
            return new ResponseHeadersPolicyFrameOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
