/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An origin request policy.
 * </p>
 * <p>
 * When it's attached to a cache behavior, the origin request policy determines the values that CloudFront includes in
 * requests that it sends to the origin. Each request that CloudFront sends to the origin includes the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The request body and the URL path (without the domain name) from the viewer request.
 * </p>
 * </li>
 * <li>
 * <p>
 * The headers that CloudFront automatically includes in every origin request, including <code>Host</code>,
 * <code>User-Agent</code>, and <code>X-Amz-Cf-Id</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * All HTTP headers, cookies, and URL query strings that are specified in the cache policy or the origin request policy.
 * These can include items from the viewer request and, in the case of headers, additional ones that are added by
 * CloudFront.
 * </p>
 * </li>
 * </ul>
 * <p>
 * CloudFront sends a request when it can't find an object in its cache that matches the request. If you want to send
 * values to the origin and also include them in the cache key, use <code>CachePolicy</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OriginRequestPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<OriginRequestPolicy.Builder, OriginRequestPolicy> {
    private static final SdkField<String> ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Id")
            .getter(getter(OriginRequestPolicy::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").unmarshallLocationName("Id")
                    .build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime")
            .getter(getter(OriginRequestPolicy::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime")
                    .unmarshallLocationName("LastModifiedTime").build()).build();

    private static final SdkField<OriginRequestPolicyConfig> ORIGIN_REQUEST_POLICY_CONFIG_FIELD = SdkField
            .<OriginRequestPolicyConfig> builder(MarshallingType.SDK_POJO)
            .memberName("OriginRequestPolicyConfig")
            .getter(getter(OriginRequestPolicy::originRequestPolicyConfig))
            .setter(setter(Builder::originRequestPolicyConfig))
            .constructor(OriginRequestPolicyConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginRequestPolicyConfig")
                    .unmarshallLocationName("OriginRequestPolicyConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD,
            LAST_MODIFIED_TIME_FIELD, ORIGIN_REQUEST_POLICY_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final Instant lastModifiedTime;

    private final OriginRequestPolicyConfig originRequestPolicyConfig;

    private OriginRequestPolicy(BuilderImpl builder) {
        this.id = builder.id;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.originRequestPolicyConfig = builder.originRequestPolicyConfig;
    }

    /**
     * <p>
     * The unique identifier for the origin request policy.
     * </p>
     * 
     * @return The unique identifier for the origin request policy.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The date and time when the origin request policy was last modified.
     * </p>
     * 
     * @return The date and time when the origin request policy was last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The origin request policy configuration.
     * </p>
     * 
     * @return The origin request policy configuration.
     */
    public final OriginRequestPolicyConfig originRequestPolicyConfig() {
        return originRequestPolicyConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(originRequestPolicyConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OriginRequestPolicy)) {
            return false;
        }
        OriginRequestPolicy other = (OriginRequestPolicy) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(originRequestPolicyConfig(), other.originRequestPolicyConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OriginRequestPolicy").add("Id", id()).add("LastModifiedTime", lastModifiedTime())
                .add("OriginRequestPolicyConfig", originRequestPolicyConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "OriginRequestPolicyConfig":
            return Optional.ofNullable(clazz.cast(originRequestPolicyConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("LastModifiedTime", LAST_MODIFIED_TIME_FIELD);
        map.put("OriginRequestPolicyConfig", ORIGIN_REQUEST_POLICY_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OriginRequestPolicy, T> g) {
        return obj -> g.apply((OriginRequestPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OriginRequestPolicy> {
        /**
         * <p>
         * The unique identifier for the origin request policy.
         * </p>
         * 
         * @param id
         *        The unique identifier for the origin request policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The date and time when the origin request policy was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The date and time when the origin request policy was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The origin request policy configuration.
         * </p>
         * 
         * @param originRequestPolicyConfig
         *        The origin request policy configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originRequestPolicyConfig(OriginRequestPolicyConfig originRequestPolicyConfig);

        /**
         * <p>
         * The origin request policy configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link OriginRequestPolicyConfig.Builder}
         * avoiding the need to create one manually via {@link OriginRequestPolicyConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OriginRequestPolicyConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #originRequestPolicyConfig(OriginRequestPolicyConfig)}.
         * 
         * @param originRequestPolicyConfig
         *        a consumer that will call methods on {@link OriginRequestPolicyConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #originRequestPolicyConfig(OriginRequestPolicyConfig)
         */
        default Builder originRequestPolicyConfig(Consumer<OriginRequestPolicyConfig.Builder> originRequestPolicyConfig) {
            return originRequestPolicyConfig(OriginRequestPolicyConfig.builder().applyMutation(originRequestPolicyConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private Instant lastModifiedTime;

        private OriginRequestPolicyConfig originRequestPolicyConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(OriginRequestPolicy model) {
            id(model.id);
            lastModifiedTime(model.lastModifiedTime);
            originRequestPolicyConfig(model.originRequestPolicyConfig);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final OriginRequestPolicyConfig.Builder getOriginRequestPolicyConfig() {
            return originRequestPolicyConfig != null ? originRequestPolicyConfig.toBuilder() : null;
        }

        public final void setOriginRequestPolicyConfig(OriginRequestPolicyConfig.BuilderImpl originRequestPolicyConfig) {
            this.originRequestPolicyConfig = originRequestPolicyConfig != null ? originRequestPolicyConfig.build() : null;
        }

        @Override
        public final Builder originRequestPolicyConfig(OriginRequestPolicyConfig originRequestPolicyConfig) {
            this.originRequestPolicyConfig = originRequestPolicyConfig;
            return this;
        }

        @Override
        public OriginRequestPolicy build() {
            return new OriginRequestPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
